#include "misc.h"
#include "file.h"
#include "xy.h"
#include "nsc.h"
#include "genitem.h"
#include "edb.h"
#include "edbproc.h"	/* public  structures */
#include "edbpriv.h"	/* private structures */
#include <assert.h>
#include <stdio.h>
#include <ctype.h>
#include <malloc.h>

#define DIAG(x) x

extern struct castr sct_cf[], lnd_cf[], shp_cf[], pln_cf[];


ObjectDescriptor objTable[] = {
	{ SCT_CLASS, "sctcls", sizeof(struct dchrstr), sct_cf, sect_cf_store },
	{ SCT_OBJECT, "sect", sizeof(struct sctstr), sect_ca, sect_ca_store },
	{ LND_CLASS, "lndcls", sizeof(struct lchrstr), lnd_cf, land_cf_store },
	{ LND_OBJECT,"land",  sizeof(struct lndstr), land_ca, land_ca_store },
	{ SHP_CLASS, "shpcls", sizeof(struct mchrstr), shp_cf, ship_cf_store },
	{ SHP_OBJECT, "ship", sizeof(struct shpstr), ship_ca, ship_ca_store },
	{ PLN_CLASS,  "plncls", sizeof(struct plchrstr), pln_cf, plane_cf_store },
	{ PLN_OBJECT, "plane", sizeof(struct plnstr), plane_ca, plane_ca_store },
	{ EF_BAD, 0, 0, 0 } };


/*
	findObject -- find object type in our object table
*/
ObjectDescriptor *findObject(int code)
{
	ObjectDescriptor *p;
	for (p=objTable; p->objtype!=EF_BAD; p++)
		if (p->objtype==code)
			return p;
	return 0;
}


/*
	 newField -- allocate a new Field structure
 */
Field *newField(s_char *name, int namelen, int kind)
{
	Field *f;
	s_char *fname;
	fname = (s_char *) malloc(namelen+1);
	assert(fname!=0);
	memcpy(	fname, name, namelen);
	fname[namelen] = 0;
	f = (Field *) malloc(sizeof(Field));
	assert(f!=0);
	f->fldname = fname;
	f->fldtype = kind;
	f->next = 0;
	return f;
}

/*
	copyField -- copy constructor for Field structure
*/
Field *copyField(Field *aField)
{
	return newField(aField->fldname, strlen(aField->fldname),
			aField->fldtype);
}

/*
	newDict -- allocate a new Dictionary structure
*/
Dictionary *newDict(int num)
{
	Dictionary *d;
	d = (Dictionary *) malloc(sizeof(Dictionary));
	assert(d!=0);
	d->next = 0;
	d->objnum = num;
	d->flds = 0;
	return d;
}

/*
	findDict -- find a Dictionary entry in the database
*/
Dictionary *findDict(Empire_Database *edb, int code)
{
	Dictionary *p;
	for (p = (Dictionary *)edb->dictionary ; p ; p=p->next) {
		if (p->objnum==code) return p;
	}
	return 0;
}
	
/*
	insertDict -- insert a Dictionary entry into the database
	(simple push down with no check for duplicates)
*/
void insertDict(Empire_Database *edb, Dictionary *p)
{
	assert(edb!=0);
	if (!p) return;
	p->next = (Dictionary *) edb->dictionary;
	edb->dictionary = p;
}

/*
	insertField -- add a field to an object definition
*/
void insertField(Dictionary *d, Field *f)
{
	Field *prev, *p;

	assert(d!=0);

	if (!f) return;
	
	for (prev=0,p=d->flds; p; prev=p,p=p->next)
		/**/ ;
	if (prev)
		prev->next = f;
	else
		d->flds = f;
	f->next = 0;
}

/*
	getnum -- parse a number in the C_SYNC message
*/
int getnum(s_char **pp)
{
	s_char *p;
	int n;
	p = *pp;
	while (*p==' ') p++;
	n = 0;
	while (isdigit(*p)) {
		n = n*10 + (*p++) - '0';
	}
	*pp = p;
	return n;
}

/*
	parseField -- parse a field entry in an object definition
	allocates a Field structure and return it if successful.
*/

Field *parseField(s_char **msg)
{
	/* we're parsing entries that look like this */
	/*   fieldname,num,num:fieldname,num,num */
	s_char *p, *nameptr;
	int namelen, fieldkind;
	nameptr = p = *msg;
	/* find end of fieldname */
	while ( *p!=0 && *p!=':' && *p!=',') p++;
	namelen = p - *msg;
	if (*p!=',') return 0;	/* probably bogus */
	p++;	/* skip comma */
	fieldkind = getnum(&p);
	while ( *p!=0 && *p!=':') p++;	/* skip any other junk */
	*msg = p;
	return newField( nameptr, namelen, fieldkind);
}

/*
	findObjectField -- match fields received with fields known
	This is done only when definitions are received, not
	when objects are received.
*/
struct castr *findObjectField(struct castr *objca, s_char *fldname)
{
	for (;objca->ca_code>0;objca++)
		if (objca->ca_name==0)
			return 0;
		else if (strcmp(objca->ca_name, fldname)==0)
			return objca;
	return 0;
}

void bindFieldDefinitions(Dictionary *dict)
{
	ObjectDescriptor *obj;
	Field *f;
	dict->objdesc = obj = findObject(dict->objnum);
	if (!obj) {
		printf("csync: warning: unknown object type %d\n", dict->objnum);
		return;	/* an object we cannot handle */
	}
	for (f=dict->flds;f;f=f->next) {
		f->boundto = findObjectField(obj->obj_ca, f->fldname);
		if (!f->boundto) {
			printf("csync: warning: field not bound: %s\n", f->fldname);
		}
	}
}

/*
	parse_typedef -- parse a C_SYNC type definition
*/
void parse_typedef(Empire_Database *edb, s_char *msg)
{
	int objnum, baseobjnum;
	Dictionary *baseobj;
	Dictionary *obj;
	Field *fld;
	ObjectDescriptor *od;

	baseobjnum = getnum(&msg);
	objnum = getnum(&msg);
	if (baseobjnum==objnum)
		baseobj = NULL;
	else {
		baseobj = findDict(edb, baseobjnum);
		DIAG(if (!baseobj) {
			printf("csync: typedef, unknown base object %d for object %d\n",
			baseobjnum, objnum);
		} )
	}
	if (findDict(edb, objnum)) return;	/* already done */
	DIAG(
		od = findObject(objnum);

		printf("csync: defining object type: %d (%s)\n",
			objnum, od?od->objdescr:"unknown");
	)
	
	obj = newDict(objnum);
	
#if (0)
	/* base object fields are included with the definition for now */
	if (baseobj) {
		for (fld=baseobj->flds;fld;fld=fld->next) {
			insertField(obj, copyField(fld));
		}
	}
#endif
	/* now parse fields and field types */
	while (*msg==' ') msg++;
	for (;;) {
		insertField(obj, parseField(&msg));
		if (*msg != ':') break;
		msg++;
	}
	/* bind fields in the csync message to fields in the database */
	bindFieldDefinitions(obj);

	insertDict(edb, obj);

}

void storeField(struct genobject *item, Field *fld, s_char *buf)
{
	/* msg is a string delimited with a ":" or a newline or a null */
	
	int fldoffs;
	long fldkind;
	typedef union {
		s_char **ccp;
		s_char *cp;
		u_char *up;
		int *ip;
		short *sp;
		unsigned short *usp;
		long *lp;
		float *fp;
	} Thing;
	
	Thing thing;
	struct castr *captr;
	
	/* get object descriptor */	
	captr = fld->boundto;
	if (!captr) {
		return;
	}

	fldoffs = (captr->ca_code)&0xffff;
	fldkind = (captr->ca_code)&NSC_TMASK;
	
	thing.cp = ((char *)item) + fldoffs;
	switch (fldkind) {
		case NSC_CHAR:
			*thing.cp = atoi(buf);
			break;
		case NSC_UCHAR:
			*thing.up = atoi(buf);
			break;
		case NSC_SHORT:
			*thing.sp = atoi(buf);
			break;
		case NSC_INT:
			*thing.ip = atoi(buf);
			break;
		case NSC_USHORT:
			*thing.usp = atoi(buf);
			break;
		case NSC_LONG:
			*thing.lp = atol(buf);
			break;
		case NSC_XCOORD:
		case NSC_YCOORD:
			*thing.sp = atoi(buf);
			break;
		case NSC_FLOAT:
			*thing.fp = atof(buf);
			break;
		case NSC_CHARP:
			/* warning... this will cause a memory leak */
			*thing.ccp = strdup(buf);
			break;
		default:
			DIAG(
				printf("unknown field kind: %s %08lx\n", 
					fld->fldname, fldkind);
			)
			break;
	}
/*	printf("%4d: %10s = %-10s\n", fldoffs, fld->fldname, buf); */
}

void storeObject(Empire_Database *edb, int code, char *msg)
{
	Dictionary *dict;
	Field *fld;
	ObjectDescriptor *desc;
	struct genobject *item;
	s_char buf[128];
	int i;
	s_char *bp;
	s_char *cp;
	s_char c;
	
	dict = findDict(edb, code);
	if (!dict) {
		/* we never received a definition */
		printf("csync: unknown csync type: %d\n", code);
		return;
	}
	desc = dict->objdesc;

	/* if we cannot store these kind of objects, just ignore */
	if (!desc) {
		printf("csync: no physical representation for object: %d\n", code);
		return;
	}

	/* allocate memory for a new object */
	item = (struct genobject *) malloc(desc->objsize);
	if (!item) return;	/* possibly should exit */

	memset(item, 0, desc->objsize);	/* initialize it */
	item->ef_type = code;
	
	/* store fields one by one */
	for (fld=dict->flds;fld;fld=fld->next) {
		
		// extract field by copying to buf
		bp = buf;
		for (i=0;i<127;i++) {
			c = *msg++;
			if (c==0||c==':'||c=='\n') break;
			*bp++ = c;
		}
		*bp++ = '\0';
		storeField(item, fld, buf);
		if (*msg==':') msg++;
	}
	
	/* now store the item... */
	desc->store(edb, item);
}

/* edb_parse_c_sync -- process a Csync message and update the database */
void edb_parse_c_sync(Empire_Database *edb, s_char *msg)
{
	int code;
	assert(edb!=0);
	assert(msg!=0);
	
	msg += 2;	/* skip over protocol character and blank */
	
	code = getnum(&msg);
	if (code==EDB_TYPE_DEF) 
		parse_typedef(edb, msg+1);
	else
		storeObject(edb, code, msg+1);
}

void dumpDict(Empire_Database *edb)
{
	Dictionary *dict;
	Field *fld;
	for (dict=(Dictionary *)edb->dictionary; dict; dict=dict->next) {
		printf("csync: object dictionary for type %d\n", dict->objnum);
		for (fld=dict->flds;fld;fld=fld->next) {
			printf("	%10s: %d\n", fld->fldname, fld->fldtype);
		}
	}
}
