;;; emputil.el --- Common utilities for GENIE

;; Copyright (C) 1994 Markus Armbruster

;; Author: Markus Armbruster <armbru@pond.sub.org>
;; Version: $Id: emputil.el,v 1.4 1994/07/25 09:16:00 armbru Exp $
;; Keywords: games

;; This file is part of GENIE, the GNU Emacs's Nifty Interface to Empire

;; GENIE is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GENIE is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GENIE; see the file COPYING.  If not, write to the Free
;; Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; For general information on GENIE, see file empire.el.

;; This file defines some general utility functions.


;;; Code:

(provide 'emputil)

(defun empire-current-line ()
  "Return current line number."
  (+ (count-lines 1 (point))
     (if (zerop (current-column)) 1 0)))

(defun empire-current-column ()
  "Return distance of point from beginning of line.
Unlike `current-column', this function ignores the actual width of the
characters' displayed representation."
  (save-excursion
    (let ((opoint (point)))
      (beginning-of-line)
      (- opoint (point)))))

(defun empire-current-indentation ()
  "Return distance of the first non-blank character from beginning of line.
Unlike `current-indentation', this function ignores the actual width of the
characters' displayed representation."
  (save-excursion
    (back-to-indentation)
    (let ((opoint (point)))
      (beginning-of-line)
      (- opoint (point)))))

(defun empire-move-to-column (col)
  "Move point to COL-th character in current line.
Unlike `move-to-column', this function ignores the actual width of the
characters' displayed representation."
  (save-restriction
    (let ((bol (progn (beginning-of-line) (point)))
	  (eol (progn (end-of-line) (point))))
      (narrow-to-region bol eol)
      (goto-char (point-min))
      (forward-char col))))

(defun empire-string-replace-match (newtext string)
  "Replace text matched by last call to `string-match' with NEWTEXT in STRING.
Treat `\' in NEWTEXT as special:
  `\&' in NEWTEXT means substitute original matched text.
  `\N' means substitute what matched the Nth `\(...\)'.
       If Nth parens didn't match, substitute nothing.
  `\\' means insert one `\'."
  (let ((prefix (substring string 0 (match-beginning 0)))
	(suffix (substring string (match-end 0)))
	(replacement nil)
	(start 0))
    (save-match-data
      (while (string-match "\\\\\\(&\\|[0-9]+\\|\\\\\\)" newtext start)
	(setq replacement
	      (cons (cond ((= ?& (aref newtext (1+ (match-beginning 0))))
			   0)
			  ((= ?\ (aref newtext (1+ (match-beginning 0))))
			   "\\")
			  (t
			   (string-to-number (substring newtext
							(1+ (match-beginning 0))
							(match-end 0)))))
		    (cons (substring newtext start (match-beginning 0))
			  replacement)))
	(setq start (match-end 0)))
      (setq replacement (cons (substring newtext start) replacement)))
    (concat prefix
	    (mapconcat (function
			(lambda (elt)
			  (cond ((stringp elt)
				 elt)
				((numberp elt)
				 (substring string
					    (match-beginning elt)
					    (match-end elt))))))
		       (nreverse replacement)
		       "")
	    suffix)))

;; Emacs Lisp has no real mapcar, sigh!
(defun empire-mapcar (fun &rest args)
  "Apply FUNCTION to successive cars of all ARGS.  Return the list of results."
  (let (result)
    ;; While no list is exhausted,
    (while (not (memq 'nil args))
      ;; apply function to CARs.
      (setq result (cons (apply fun (mapcar 'car args))
			 result))
      (setq args (mapcar 'cdr args)))
    (nreverse result)))

;; Alas, nsubstitute is missing, too.
(defun empire-nsubstitute (new old list)
  "Substitute NEW for OLD everywhere in LIST (destructively).
Comparison is done with equal."
  (let ((tail list))
    (while tail
      (if (equal (car tail) old)
	  (setcar tail new))
      (setq tail (cdr tail))))
  list)

;;; emputil.el ends here
