;;; empsnoop.el --- GENIE empire dialog snooper

;; Copyright (C) 1994 Markus Armbruster

;; Author: Markus Armbruster <armbru@pond.sub.org>
;; Version: $Id: empsnoop.el,v 1.7 1994/10/19 11:12:31 armbru Exp $
;; Keywords: games

;; This file is part of GENIE, the GNU Emacs's Nifty Interface to Empire

;; GENIE is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GENIE is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GENIE; see the file COPYING.  If not, write to the Free
;; Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; For general information on GENIE, see file empire.el.

;; This file defines the GENIE empire dialog snooper.  It passively
;; snoops the dialog and snarfs any useful information it recognizes.

;; The natural way to snoop output line by line would be with
;; continuations, eg

;; (defun snoop-something ()
;;   (call/cc (lambda (escape)
;;		snoop first line
;;		(call/cc (lambda (continue)
;;			   (escape continue)))
;;		snoop second line
;;		(call/cc (lambda (continue)
;;			   (escape continue)))
;;		...
;;		snoop last line
;;		(escape some-value))))

;; Calling snoop-something snoops the first line and returns a
;; continuation.  Calling the continuation snoops the next line and
;; returns another continuation, etc.  Calling last continuation
;; snoops the last line and returns some-value.

;; Emacs Lisp has no continuations.  Nevertheless, the following code
;; is written in a (weird) continuation-like style.

;; A snoop function snoops one line of output and returns a list.  The
;; list's car is the snoop function to use for the next line of
;; output, the list's cdr are arguments to pass to this function.
;; This simulates a closure.
;; If a snoop function fails, it returns a descriptive string instead.


;;; Restrictions

;; Can't snoop script execution


;;; To do

;; Snoop more commands, not just the bare minimum, perhaps plagiarize
;; Ken Steven's parse.pl
;; Consider using cl's lexical-let for closures


;;; Code:

(require 'emputil)
(require 'empdata)
(provide 'empsnoop)

;;; Variables

(defvar empire-snooper-alist
  '(("version" empire-snoop-version)
    ("map" empire-snoop-map)
    ("bmap" empire-snoop-map)
    ("sect" empire-snoop-sect)
    ("dump" empire-snoop-dump)
    ("explore" empire-snoop-explore)
    ("country" empire-snoop-country)
    ("relations" empire-snoop-relations))
  "Alist associating empire commands with snoop closures.")

(defvar empire-snoop-closure nil
  "List that specifies how to snoop the next line of output.
The list's car is a function that takes (1+ (length list)) arguments.")


;;; Functions

(add-hook 'empire-login-hook 'empire-init-snoop)

(defun empire-init-snoop ()
  "Initialize the empire dialog snooper."
  (setq empire-output-filter 'empire-snoop
	empire-snoop-closure nil)
  (empire-wait-until-non-busy)
  ;; get world size, etc from version
  (empire-send-command "version >>/dev/null")
  (empire-wait-until-non-busy)
  (or (empire-world-x)
      (error "Can't determine world size"))
  ;; get your country number
  (let ((cnum 1))
    (while (and (not empire-your-cnum)
		(< cnum empire-number-of-countries))
      (empire-send-command (format "relations %d >>/dev/null" cnum))
      (empire-wait-until-non-busy)
      (setq empire-your-cnum (empire-ref-cnum empire-country))
      (setq cnum (1+ cnum))))
  (or empire-your-cnum
      (error "Can't determine country number"))
  ;; double check country number
  (setq empire-command-failed nil)
  (empire-send-command (format "dump * ?own#%d >>/dev/null" empire-your-cnum))
  (empire-wait-until-non-busy)
  (if empire-command-failed
      (setq empire-command-failed nil)
    (setq empire-your-cnum nil)
    (error "Incorrect country number")))

(defun empire-snoop (tag string)
  "Snoop a line TAG, STRING.
This function should be in `empire-client-output-filter'."
  (cond ((eq tag 'prompt)
	 ;; abort current snoop
	 (setq empire-snoop-closure nil))
	((memq tag '(data flush))
	 ;; if there is no current snoop, create one
	 (cond ((not empire-snoop-closure)
		(setq empire-snoop-closure
		      (or (empire-get-snooper empire-current-input)
			  '(ignore)))
		(empire-set-current-coord nil nil)))
	 ;; continue current snoop
	 (let ((cont nil))
	   (unwind-protect
	       (setq cont (apply (car empire-snoop-closure)
				 tag string
				 (cdr empire-snoop-closure)))
	     (cond ((listp cont)
		    (setq empire-snoop-closure
			  (or cont '(ignore))))
		   (t
		    (message "Snooping failed: %s" cont)
		    (setq empire-snoop-closure '(ignore))))))))
  (if (and (eq tag 'data)
	   (string-match "\\(^\".*\" is ambiguous --\\|^\".*\" is not a legal command\\|^You don't have the BTU's\\|^Command not implemented\\|^\\(Update in progress...\\)?command failed\\|^Usage: \\)\\|\\(^You have .* new announcements? waiting\\)\\|\\(^You have .* new telegrams? waiting\\)"
			 string))
      (cond ((match-end 1) (setq empire-command-failed t))
	    ((match-end 2) (setq empire-anno-waiting t))
	    ((match-end 3) (setq empire-tele-waiting t))))
  string)

(defun empire-get-snooper (string)
  "Return the snooper closure to snoop command STRING."
  (and (string-match "[^ \t\n]+" string)
       (let ((cmd-regexp (concat "^"
				 (regexp-quote (substring string
							  (match-beginning 0)
							  (match-end 0)))))
	     (alist empire-snooper-alist))
	 (while (and alist
		     (not (string-match cmd-regexp (car (car alist)))))
	   (setq alist (cdr alist)))
	 (cdr (car alist)))))


;; Version

(defun empire-snoop-version (tag string)
  (cond ((string-match "^World size is \\([0-9]+\\) by \\([0-9]+\\).$"
		       string)
	 (or (empire-set-world-size (empire-string-match-to-number 1 string)
				    (empire-string-match-to-number 2 string))
	     (error "Bad world size")))
	((string-match "^There can be up to \\([0-9]+\\) countries.$" string)
	 (setq empire-number-of-countries
	       (empire-string-match-to-number 1 string))))
  '(empire-snoop-version))


;; Map

;; Snoop a line up to the first line of the map header
(defun empire-snoop-map (tag string)
  (if (eq tag 'flush)
      '(empire-snoop-map)		; ignore flushs
    ;; snoop first line of header
    (if (string-match "^ +\\([-0-9][-0-9]+\\)" string)
	(let ((left (match-beginning 1)))
	  (list 'empire-snoop-map-1
		left
		(match-end 1)
		(substring string left (1+ left))
		(substring string (1+ left) (+ left 2))))
      "Can't find map header")))

;; Snoop a trailing line of a map header
;; LEFT is the ruler's beginnning
;; RIGHT is the ruler's end
;; COL0 is a digit string describing the leftmost x-coordinate
;; COL1 is a digit string describing the next x-coordinate
(defun empire-snoop-map-1 (tag string left right col0 col1)
  (if (and (string-match "^ +\\([0-9][0-9]+\\)" string)
	   (eq (match-beginning 1) left))
      ;; this is a header line
      (list 'empire-snoop-map-1
	    left
	    right
	    (concat col0 (substring string left (1+ left)))
	    (concat col1 (substring string (1+ left) (+ left 2))))
    ;; must be the map body
    (empire-snoop-map-2 tag string
			left right
			(let ((x0 (string-to-number col0)))
			  (if (and (> x0 0)
				   (> x0 (string-to-number col1)))
			      (- x0)
			    x0)))))

;; Snoop a map body line
;; LEFT is the beginning of the map interior
;; RIGHT is the end of the map interior
;; X0 is the leftmost x-coordinate
(defun empire-snoop-map-2 (tag string left right x0)
  (if (string-match "^ *\\(-?[0-9]+\\) .* \\1" string)
      ;; this is a body line
      (let ((y (string-to-number (substring string
					    (match-beginning 1)
					    (match-end 1)))))
	(empire-snoop-map-row string left right x0 y)
	(list 'empire-snoop-map-2 left right x0))
    ;; must be the map footer
    (empire-snoop-map-3 tag string left right)))

;; Snoop a map footer line
;; LEFT is the ruler's beginnning
;; RIGHT is the ruler's end
(defun empire-snoop-map-3 (tag string left right)
  (if (and (string-match "^ +\\([-0-9]+\\)" string)
	   (eq (match-beginning 1) left))
      ;; this is a footer line
      (list 'empire-snoop-map-3 left right)
    ;; we are done
    nil))

(defun empire-snoop-sect (tag string)
  (empire-remove-map-highlight 0)
  (empire-snoop-map tag string))

(defun empire-snoop-map-row (string left right x y)
  "Snoop map row in STRING between LEFT andf RIGHT, leftmost sector is at X, Y."
  (or (empire-x-y-valid-p x y)
      (setq left (1+ left)
	    x (1+ x)))
  (let ((row (make-string (/ (- (1+ right) left) 2) 0))
	(index 0))
    (while (< left right)
      (aset row index (aref string left))
      (setq index (1+ index)
	    left (+ left 2)))
    (empire-update-map-row x y row)))


;; Dump

;; Snoop a line up to `DUMP SECTOR'
(defun empire-snoop-dump (tag string)
  (if (or (eq tag 'flush)
	  (not (string= "DUMP SECTOR\n" string)))
      '(empire-snoop-dump)
    '(empire-snoop-dump-1)))

;; Snoop the dump header line
(defun empire-snoop-dump-1 (tag string)
  (if (string-match "\\<x y\\>" string)
      (list 'empire-snoop-dump-2
	    (mapcar 'empire-dump-index (empire-split string "[ \n]+")))
    "Can't find dump"))

;; Snoop a dump data line or the final number of sectors line
;; HEADER is a list of dump index symbols describing the dump's fields.
(defun empire-snoop-dump-2 (tag string header)
  (if (string-match "^[0-9]+ sector" string)
      nil				; we are done
    (let* ((alist (empire-mapcar 'cons header (empire-split string "[ \n]+")))
	   (x (string-to-number (cdr (assq 'empire-dump-index-x alist))))
	   (y (string-to-number (cdr (assq 'empire-dump-index-y alist))))
	   (des (cdr (assq 'empire-dump-index-des alist)))
	   (dump (cons (cons 'empire-dump-index-owner empire-your-cnum)
		       (empire-dump-filter alist))))
      (if des (empire-update-map-sector x y (string-to-char des)))
      (empire-update-dump-sector x y dump))
    (list 'empire-snoop-dump-2 header)))


;; Explore

;; Snoop a line from prompt or flush up to an explore map's first line
(defun empire-snoop-explore (tag string)
  (cond ((eq tag 'flush)
	 '(empire-snoop-explore))	; ignore flushs
	((string-match " \\([^ \n] [^ \n]\\) +\\(\\<\\).*min" string)
	 ;; this is a explore map's first line
	 (list 'empire-snoop-explore-1
	       (match-beginning 0)
	       (list (substring string (match-beginning 1) (match-end 1)))
	       (match-beginning 2)
	       (empire-split-header string (match-beginning 2))))
	((string-match "\\([0-9]+\\) mob left in \\(-?[0-9]+\\),\\(-?[0-9]+\\)"
		       string)
	 ;; trailing mobility cost line
	 (empire-update-dump-sector (empire-string-match-to-number 2 string)
				    (empire-string-match-to-number 3 string)
				    (list (cons 'empire-dump-index-mob
						(empire-string-match-to-number 1 string))))
	 '(empire-snoop-explore))
	((string-match "Sector \\(-?[0-9]+\\),\\(-?[0-9]+\\) is now yours"
		       string)
	 ;; trailing ownership line
	 (empire-update-dump-sector (empire-string-match-to-number 1 string)
				    (empire-string-match-to-number 2 string)
				    (list (cons 'empire-dump-index-owner
						empire-your-cnum)))
	 '(empire-snoop-explore))
	(t
	 ;; other blabla
	 '(empire-snoop-explore))))

;; Snoop an explore map's second line
;; MAP-INDENT is the map's indentation
;; MAP is a list containing the first map line's map string.
;; RESO-INDENT is the resource table's indentation
;; RESO describes the resource header (produced by empire-split-header)
(defun empire-snoop-explore-1 (tag string map-indent map reso-indent reso)
  (if (and (string-match " \\([^ \n] [^ \n] [^ \n]\\) " string)
	   (= map-indent (match-beginning 1)))
      (list 'empire-snoop-explore-2
	    map-indent
	    (cons (substring string (match-beginning 1) (match-end 1))
		  map)
	    (empire-dump-filter
	     (empire-mapcar 'cons
			    (car reso)
			    (cdr (empire-split-at string
						  (cons reso-indent
							(cdr reso)))))))
    "Can't find explore map"))

;; Snoop an explore map's third line
;; MAP-INDENT is the map's indentation
;; MAP is a list containing the first two map lines' map strings (reversed).
;; RESO is an alist describing the resources
(defun empire-snoop-explore-2 (tag string map-indent map reso)
  (cond ((and (string-match " \\([^ \n] [^ \n]\\)" string)
	      (= map-indent (match-beginning 0)))
	 (list 'empire-snoop-explore-3
	       (cons (substring string (match-beginning 1) (match-end 1))
		     map)
	       reso))
	(t
	 "Can't find explore map")))

;; Snoop explore sub-prompt line
;; MAP-INDENT is the map's indentation
;; MAP is a list containing the map lines' map strings (reversed).
;; RESO is an alist describing the resources
(defun empire-snoop-explore-3 (tag string map reso)
  (if (and (eq tag 'flush)
	   (string-match "^<.*: . \\(-?[0-9]+\\),\\(-?[0-9]+\\)>"
			 string))
      (let ((x (empire-string-match-to-number 1 string))
	    (y (empire-string-match-to-number 2 string)))
	(empire-update-dump-sector x y reso)
	(empire-snoop-map-row (nth 2 map) 0 3 (1- x) (1- y))
	(empire-snoop-map-row (nth 1 map) 0 5 (- x 2) y)
	(empire-snoop-map-row (nth 0 map) 0 3 (1- x) (1+ y))
	(empire-set-current-coord x y)
	;; redo from start
	'(empire-snoop-explore))
    "Can't find explore prompt"))


;; Country

;; Snoop a country line up to header
(defun empire-snoop-country (tag string)
  (if (string-match "^ *# *last access" string)
      '(empire-snoop-country-1)
    '(empire-snoop-country)))

;; Snoop a country body line
(defun empire-snoop-country-1 (tag string)
  (if (not (string-match " *\\([0-9]+\\).+\\([^ ]+\\)$" string))
      nil
    (empire-set-country (empire-string-match-to-number 1 string)
			(substring string (match-beginning 1) (match-end 1)))
    '(empire-snoop-country-1)))


;; Relations

;; Snoop a relations line up to header
(defun empire-snoop-relations (tag string)
  (if (string-match "^ *Formal Relations" string)
      '(empire-snoop-relations-1)
    '(empire-snoop-relations)))

;; Snoop a relations body line
(defun empire-snoop-relations-1 (tag string)
  (if (not (string-match " *\\([0-9]+\\)) +\\([^ ]+\\)" string))
      nil
    ;; TODO: remember relations
    (empire-set-country (empire-string-match-to-number 1 string)
			(substring string (match-beginning 1) (match-end 1)))
    '(empire-snoop-relations-1)))


;; Utilities

(defun empire-string-match-to-number (arg string)
  "Convert ARG-th match in STRING to a number with `string-to-number'."
  (string-to-number (substring string (match-beginning arg) (match-end arg))))

(defun empire-split (string split-regexp)
  "Split STRING at SPLIT-REGEXP, return list of pieces."
  (let ((pieces nil)
	(start (if (string-match (concat "^" split-regexp) string)
		   (match-end 0)	; skip leading split-regexp
		 0)))
    (while (string-match split-regexp string start)
      (setq pieces (cons (substring string start (match-beginning 0))
			 pieces)
	    start (match-end 0)))
    (nreverse (if (= start (length string))
		  pieces
		(cons (substring string start) pieces)))))

(defun empire-split-header (string start)
  "Split STRING beginning with START at space or newline.
Return the list of pieces and the list of pieces' end positions in
STRING consed together."  
  (let ((pieces nil)
	(positions nil))
    (while (string-match "[ \n]+" string start)
      (setq pieces (cons (substring string start (match-beginning 0))
			 pieces)
	    positions (cons (match-beginning 0)
			    positions)
	    start (match-end 0)))
    (if (/= start (length string))
	(setq pieces (cons (substring string start) pieces)
	      positions (cons (length string) positions)))
    (cons (nreverse pieces)
	  (nreverse positions))))

(defun empire-split-at (string positions)
  "Split STRING at POSITIONS, return list of pieces."
  (let ((pieces nil)
	(start 0))
    (while positions
      (setq pieces (cons (substring string start (car positions))
			 pieces)
	    start (car positions)
	    positions (cdr positions)))
    (nreverse pieces)))
