;;; empdata.el --- Access GENIE game data

;; Copyright (C) 1994 Markus Armbruster

;; Author: Markus Armbruster <armbru@pond.sub.org>
;; Version: $Id: empdata.el,v 1.6 1994/07/25 09:14:31 armbru Exp $
;; Keywords: games

;; This file is part of GENIE, the GNU Emacs's Nifty Interface to Empire

;; GENIE is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GENIE is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GENIE; see the file COPYING.  If not, write to the Free
;; Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; For general information on GENIE, see file empire.el.

;; This file defines means to store and retrieve information about an
;; empire.


;;; Terminology

;; Coordinates are pairs (X . Y) of integers.
;; An x-coordinate X is canonical if XMIN<=X<=XMAX, where XMIN, XMAX
;; are determined by the world size.  Y-coordinates likewise.
;; Two x-coordinates X, X' are equivalent if their difference is a
;; multiple of the world's x size.  Y-coordinates likewise.
;; Coordinates are valid if they are both even or both odd.

;; Sector designations are characters as defined by empire.

;; Commodities are characters as defined by empire.


;;; To do

;; Cutoffs & deliveries
;; Units, ships, planes
;; Production, starve, budget, nation
;; Relations, etc.
;; Mail & news
;; Persistency


;;; Code:

(require (if window-system 'empwin 'empterm))
(provide 'empdata)


;;; User options

(defvar empire-login-hook nil
  "*Hook run after successful login.")

(defvar empire-high-min 95
  "*Iron ore contents above or equal this value are considered high.")

(defvar empire-high-gold 60
  "*Gold ore contents above or equal this value are considered high.")

(defvar empire-high-fert 100
  "*Soil fertility above or equal this value is considered high.")

(defvar empire-high-ocontent 80
  "*Oil contents above or equal this value is considered high.")

(defvar empire-high-uran 95
  "*Uranium contents above or equal this value is considered high.")


;;; Variables

;; You should not access these variables directly, except when
;; explicitely stated.  Use the access functions.

;; Empire state

(defvar empire-command-failed nil
  "Non-nil means some command failed.")

(defvar empire-anno-waiting nil
  "Non-nil means there are unread announcements.")

(defvar empire-tele-waiting nil
  "Non-nil means there are unread telegrams.")


;; Empire data

(defvar empire-world-size nil
  "Size of the empire world.
Looks like (X-SIZE . Y-SIZE).")

(defvar empire-current-coord nil
  "Current coordinates.
Just for convenient inter-tool communication.")

(defvar empire-map nil
  "The empire map.
This is a vector of rows which are strings")

(defvar empire-map-version 0
  "Every map change increments this variable.")

(defvar empire-dump nil
  "Empire sector dump.
This is a vector of rows which are vectors of sectors.  A sector is
nil or a vector.")

(defvar empire-dump-version 0
  "Every dump change increments this variable.")

;; dump index symbols
;; values are lists (index-in-sector-vector function-to-convert-from-string)
(defconst empire-dump-index-eff '(0 string-to-number))
(defconst empire-dump-index-civ '(1 string-to-number))
(defconst empire-dump-index-mil '(2 string-to-number))
(defconst empire-dump-index-uw '(3 string-to-number))
(defconst empire-dump-index-min '(4 string-to-number))
(defconst empire-dump-index-gold '(5 string-to-number))
(defconst empire-dump-index-fert '(6 string-to-number))
(defconst empire-dump-index-ocontent '(7 string-to-number))
(defconst empire-dump-index-uran '(8 string-to-number))
(defconst empire-dump-index-owner '(9 nil))
(defconst empire-dump-index-sdes '(10 string-to-char))
(defconst empire-dump-index-mob '(11 string-to-number))
(defconst empire-dump-index-* '(12 string-to-char))
(defconst empire-dump-index-off '(13 string-to-number))
(defconst empire-dump-index-work '(14 string-to-number))
(defconst empire-dump-index-avail '(15 string-to-number))
(defconst empire-dump-index-terr '(16 string-to-number))
(defconst empire-dump-index-food '(17 string-to-number))
(defconst empire-dump-index-shell '(18 string-to-number))
(defconst empire-dump-index-gun '(19 string-to-number))
(defconst empire-dump-index-pet '(20 string-to-number))
(defconst empire-dump-index-iron '(21 string-to-number))
(defconst empire-dump-index-dust '(22 string-to-number))
(defconst empire-dump-index-bar '(23 string-to-number))
(defconst empire-dump-index-oil '(24 string-to-number))
(defconst empire-dump-index-lcm '(25 string-to-number))
(defconst empire-dump-index-hcm '(26 string-to-number))
(defconst empire-dump-index-rad '(27 string-to-number))
;; TODO: cutoffs and deliveries
(defconst empire-dump-index-dist_x '(28 string-to-number))
(defconst empire-dump-index-dist_y '(29 string-to-number))
(defconst empire-dump-index-c_dist '(30 string-to-number))
(defconst empire-dump-index-m_dist '(31 string-to-number))
(defconst empire-dump-index-u_dist '(32 string-to-number))
(defconst empire-dump-index-f_dist '(33 string-to-number))
(defconst empire-dump-index-s_dist '(34 string-to-number))
(defconst empire-dump-index-g_dist '(35 string-to-number))
(defconst empire-dump-index-p_dist '(36 string-to-number))
(defconst empire-dump-index-i_dist '(37 string-to-number))
(defconst empire-dump-index-d_dist '(38 string-to-number))
(defconst empire-dump-index-b_dist '(39 string-to-number))
(defconst empire-dump-index-o_dist '(40 string-to-number))
(defconst empire-dump-index-l_dist '(41 string-to-number))
(defconst empire-dump-index-h_dist '(42 string-to-number))
(defconst empire-dump-index-r_dist '(43 string-to-number))
(defconst empire-dump-length 44)

;; TODO: cutoffs and deliveries
(defconst empire-commodities
  '((?c empire-dump-index-civ   empire-dump-index-c_dist)
    (?m empire-dump-index-mil   empire-dump-index-m_dist)
    (?u empire-dump-index-uw    empire-dump-index-u_dist)
    (?f empire-dump-index-food  empire-dump-index-f_dist)
    (?s empire-dump-index-shell empire-dump-index-s_dist)
    (?g empire-dump-index-gun   empire-dump-index-g_dist)
    (?p empire-dump-index-pet  	empire-dump-index-p_dist)
    (?i empire-dump-index-iron 	empire-dump-index-i_dist)
    (?d empire-dump-index-dust 	empire-dump-index-d_dist)
    (?b empire-dump-index-bar  	empire-dump-index-b_dist)
    (?o empire-dump-index-oil  	empire-dump-index-o_dist)
    (?l empire-dump-index-lcm  	empire-dump-index-l_dist)
    (?h empire-dump-index-hcm  	empire-dump-index-h_dist)
    (?r empire-dump-index-rad  	empire-dump-index-r_dist))
  "Alist associating commodity characters with dump index symbols.
Each association is a list (COMMODITY AMOUNT-SYMBOL THRESHOLD-SYMBOL).")

;; access directly
(defvar empire-your-cnum nil
  "Your country number.")

;; access directly
(defvar empire-number-of-countries nil
  "The maximal number of countries in the game.")

(defvar empire-cnums (make-vector 109 0)
  "Obarray to map country names to numbers.")

(defvar empire-cnames nil
  "Country names, a vector of strings, indexed by country numbers.")


;; Highlights

;; Define highlight symbols and their bit index.  A bit index is a
;; power of two, of course.
(defconst empire-highlight-0 0)
(defconst empire-highlight-1 1)

(defvar empire-highlight-symbols
  (vector 'empire-highlight-0 'empire-highlight-1)
  "Array of symbols used as text properties for highlighting.")

(defvar empire-default-highlight empire-highlight-0
  "The highlight symbol to use if no specific one is requested.")


;;; World size, coordinates.

(defun empire-world-x ()
  "Return world's x-size."
  (car empire-world-size))

(defun empire-world-y ()
  "Return world's y-size."
  (cdr empire-world-size))

(defun empire-set-world-size (x y)
  "Set worlds size to X, Y, return the new world size as pair (X . Y).
If world size was previously unknown, initialize all other empire data."
  (cond (empire-world-size		; world size already known
	 (and (= x (car empire-world-size))
	      (= y (cdr empire-world-size))
	      empire-world-size))
	((and (zerop (% x 2))
	      (zerop (% y 2)))		; new size valid
	 (empire-new-map x y)
	 (empire-new-dump x y)
	 (setq empire-world-size (cons x y)))
	(t nil)))			; new size invalid

(defun empire-canon-x (x)
  "Return the canonical x-coordinate equivalent to X."
  (let ((size (empire-world-x)))
    (- (mod (+ x (/ size 2)) size)
       (/ size 2))))

(defun empire-canon-y (y)
  "Return the canonical y-coordinate equivalent to Y."
  (let ((size (empire-world-y)))
    (- (mod (+ y (/ size 2)) size)
       (/ size 2))))

(defsubst empire-x-y-valid-p (x y)
  "Return non-nil if X, Y are valid coordinates."
  (eq (zerop (% x 2))
      (zerop (% y 2))))

;; More efficient than (empire-canon-x (1+ X)).  X must be canonical.
(defsubst empire-next-x (x)
  "Return the next x-coordinate on the right."
  (let ((max (/ (empire-world-x) 2)))
    (setq x (1+ x))
    (if (>= x max)
	(- x max max)
      x)))

;; More efficient than (empire-canon-y (1+ Y)).  Y must be canonical.
(defsubst empire-next-y (y)
  "Return the next y-coordinate in downwards direction."
  (let ((max (/ (empire-world-y) 2)))
    (setq y (1+ y))
    (if (>= y max)
	(- y max max)
      y)))

(defun empire-current-x ()
  "Return current x-coordinate."
  (car empire-current-coord))

(defun empire-current-y ()
  "Return current y-coordinate."
  (cdr empire-current-coord))

(defun empire-set-current-coord (x y)
  "Set current coordinates to X, Y."
  (setq empire-current-coord (cons x y)))


;;; Access map

(defsubst empire-map-x (x)
  "Return X's column index in map.
X must be canonical."
  (let ((size (empire-world-x)))
    (% (+ x (/ size 2)) size)))

(defsubst empire-map-y (y)
  "Return Y's row index in map.
Y must be canonical."
  (let ((size (empire-world-y)))
    (% (+ y (/ size 2)) size)))

(defun empire-new-map (x y)
  "Initialize the map for world size X, Y."
  (setq empire-map (apply 'vector	; y rows of strings with length x
			  (mapcar (function (lambda (ignored)
					      (make-string x ?\ )))
				  (make-list y nil)))
	empire-map-version (1+ empire-map-version)))

(defun empire-map-changed-p (old-version)
  "Return current map version if it differs from OLD-VERSION, else nil."
  (and (not (eq old-version empire-map-version))
       empire-map-version))

(defun empire-ref-map-sector (x y)
  "Return designation of sector X, Y.
X, Y must be canonical.  Invalid coordinates return ?\\ ."
  (aref (aref empire-map (empire-map-y y))
	(empire-map-x x)))

(defun empire-update-map-row (x y des)
  "Update map row, leftmost sector is X, Y, with string DES.
(aref DES I) is the designation for sector X+I, Y.
X, Y must be canonical and valid."
  (let ((i (length des)))
    (while (> i 0)
      (setq i (1- i))
      (empire-update-map-sector (+ x i i) y (aref des i)))))

(defun empire-update-map-sector (x y des)
  "Update map sector X, Y with DES.
If sector's designation changed, return old one, else nil.
X, Y must be canonical and valid."
  (or (empire-x-y-valid-p x y)
      (error "Bad coordinates"))
  (let* ((row (aref empire-map (empire-map-y y)))
	 (index (empire-map-x x))
	 (old-des (aref row index)))
    ;; server highlight
    (cond ((>= des 128)
	   (setq des (- des 128))
	   (empire-set-map-highlight empire-default-highlight row index)))
    ;; update owner
    (if (and (= des ??)
	     (memq (empire-dump-value (empire-ref-dump-sector x y)
				      'empire-dump-index-owner)
		   '(nil me)))
	(empire-update-dump-sector x y '((empire-dump-index-owner . foreign))))
    ;; update designation
    (cond ((= des ?\ )			; terra incognita
	   nil)
	  ((= des ?0)			; center of a radmap
	   nil)
	  ((and (<= ?A des) (<= des ?Z)) ; ship, plane or unit
	   nil)
	  ((or (/= des ??)
	       (memq old-des '(?\  ?. ?-)))
	   (aset row index des)
	   (setq empire-map-version (1+ empire-map-version))
	   old-des))))

(defun empire-format-map-row (x y len)
  "Format a map row for display, leftmost sector is X, Y, width LEN characters."
  (let ((map-row (empire-highlight-map-row (aref empire-map (empire-map-y y))))
	disp-row)
    (setq disp-row (substring map-row (empire-map-x x)))
    (setq len (- len (length disp-row)))
    (while (>= len 0)
      (setq disp-row (concat disp-row map-row))
      (setq len (- len (length map-row))))
    (substring disp-row 0 len)))

(defun empire-highlight-map-row (row)
  "Return a copy of ROW highlighted for display."
  (let ((start 0)
	end)
    (setq row (concat row ""))		; copy row
    (remove-text-properties 0 (length row) '(face nil) row)
    (while (setq end (next-property-change start row))
      (empire-highlight-string row start end (text-properties-at start row))
      (setq start end))
    row))

(defun empire-remove-map-highlight (n)
  "Remove highlight N from the map."
  (let ((prop (aref empire-highlight-symbols n))
	(row-index (length empire-map)))
    (while (> row-index 0)
      (setq row-index (1- row-index))
      (let ((row (aref empire-map row-index)))
	(remove-text-properties 0 (length row) (list prop nil) row)))))

(defun empire-set-map-highlight (n row index)
  "Set highlight N at ROW, INDEX."
  (put-text-property index (1+ index)
		     (aref empire-highlight-symbols n) t
		     row))


;;; Access dump

(defsubst empire-dump-index (string)
  "Map STRING to its dump index symbol."
  (intern (concat "empire-dump-index-" string)))

(defsubst empire-dump-x (x)
  "Return X's column index in dump."
  (if (< x 0)
      (- -1 x)
    x))

(defsubst empire-dump-y (y)
  "Return Y's row index in dump."
  (if (< y 0)
      (- -1 y y)
    (+ y y)))

(defun empire-new-dump (x y)
  "Initialize the dump for world size X, Y."
  (setq empire-dump (apply 'vector	; y rows of x/2 columns
			   (mapcar (function (lambda (ignored)
					       (make-vector (/ x 2) nil)))
				   (make-list y nil)))
	empire-dump-version (1+ empire-dump-version)))

(defun empire-dump-changed-p (old-version)
  "Return current dump version if it differs from OLD-VERSION, else nil."
  (and (not (eq old-version empire-dump-version))
       empire-dump-version))

(defun empire-ref-dump-sector (x y)
  "Return dump for sector X, Y.
The value can be examined with `empire-dump-value'."
  (or (empire-x-y-valid-p x y)
      (error "Bad coordinates"))
  (aref (aref empire-dump (empire-dump-y y)) (empire-dump-x x)))

(defun empire-dump-value (dump key)
  "Return DUMP's value for KEY.
DUMP is a value obtained with `empire-ref-dump-sector'.
KEY is a dump index symbol."
  (let ((index (car (symbol-value key))))
    (and (< index (length dump))
	 (aref dump index))))

(defun empire-update-dump-sector (x y alist)
  "Update dump at sector X, Y according to ALIST.
ALIST is a list of pairs (KEY . VALUE), where KEY is a dump index symbol."
  (or (empire-x-y-valid-p x y)
      (error "Bad coordinates"))
  (let ((sect (aref (aref empire-dump (empire-dump-y y)) (empire-dump-x x)))
	;; convert (KEY . VALUE) to (INDEX . CONVERTED-VALUE)
	(data (mapcar (function
		       (lambda (asso)
			 (let ((def (symbol-value (car asso))))
			   (cons (nth 0 def)
				 (if (stringp (cdr asso))
				     (funcall (nth 1 def) (cdr asso))
				   (cdr asso))))))
		       alist)))
    ;; enlarge sect if necessary
    (let ((delta-len (- (min empire-dump-length
			     (1+ (apply 'max (mapcar 'car data))))
			(length sect))))
      (or (<= delta-len 0)
	  (setq sect (vconcat sect (make-list delta-len nil)))))
    ;; set new values
    (mapcar (function
	     (lambda (asso)
	       (aset sect (car asso) (cdr asso))))
	    data)
    (aset (aref empire-dump (empire-dump-y y)) (empire-dump-x x) sect)
    (setq empire-dump-version (1+ empire-dump-version))))

(defun empire-dump-filter (alist)
  "Return ALIST with all unrecognized associations removed.
ALIST's keys are dump index symbols or strings.  The latter are mapped
to the former."
  (delq nil
	(mapcar (function
		 (lambda (asso)
		   (if (stringp (car asso))
		       (setcar asso (empire-dump-index (car asso))))
		   (and (boundp (car asso))
			asso)))
		alist)))


;;; Access countries

(defun empire-ref-cname (num)
  "Return country NUM's name."
  (and (< num (length empire-cnames))
       (aref empire-cnames num)))

(defun empire-ref-cnum (name)
  "Return country NAME's number."
  (let ((sym (intern name empire-cnums)))
    (and (boundp sym)
	 (symbol-value sym))))

(defun empire-set-country (num name)
  "Associate country NUM with NAME."
  (or (< num (length empire-cnames))
      (setq empire-cnames
	    (vconcat empire-cnames
		     (make-list (- num (length empire-cnames) -1) nil))))
  (aset empire-cnames num name)
  (set (intern name empire-cnums) num))


;;; Display

(defun empire-number-of-highlights ()
  "Return the number of highlights we can display."
  (length empire-highlight-symbols))

(defun empire-highlight-index (prop-list)
  "Return the highlight index to use for PROP-LIST."
  (apply 'logior (mapcar (function (lambda (prop)
				     (if (memq prop prop-list)
					 (lsh 1 (symbol-value prop))
				       0)))
			 empire-highlight-symbols)))

;;; empdata.el ends here
