/* Prefs.c */

#include "Sys.h"

#include <ctype.h>
#include <signal.h>
#include <setjmp.h>

#include "Util.h"
#include "Cmds.h"
#include "Prefs.h"
#include "RCmd.h"
#include "Recent.h"
#include "Main.h"


jmp_buf gPrefsWinJmp;
extern int gMayUTime, gStartupMsgs, gBlankLines;
extern int gMaxLogSize, gMaxRecents;
extern int gTotalRuns, gRememberLCWD;
extern int gNetworkTimeout, gTrace, gPreferredDataPortMode;
extern longstring gLocalCWD, gPager, gStartupDir;
extern int gMarkTrailingSpace;



PrefOpt gPrefOpts[] = {
	{ "blank-lines", "Blank lines between cmds:",
		kToggleMsg,
		PREFBOOL(gBlankLines) },
	{ "logsize", "User log size:",
		"Enter the maximum number of bytes allowed for log file, or 0 for no log.",
		PREFINT(gMaxLogSize) },
	{ "maxhosts", "Max hosts to save:",
		"Enter the maximum number of hosts allowed for host file, or 0 for no limit.",
		PREFINT(gMaxRecents) },
	{ "pager", "Pager:",
		"Type the pathname of the program you use to view text a page at a time.",
		PREFSTR(gPager, kOkayIfEmpty, kGetAndEcho) },
	{ "startup-lcwd", "Startup in Local Dir:",
		"Type directory to always lcd to, or hit <RETURN> to not lcd at startup.",
		PREFSTR(gStartupDir, kOkayIfEmpty, kGetAndEcho) },
	{ "startup-msgs", "Startup messages:",
		kToggleMsg,
		PREFTOGGLE(gStartupMsgs, kNoStartupMsgs, (kStartupMsg | kTips)) },
	{ "timeout", "Network timeout:",
		"Enter the maximum amount of time to wait on a connection before giving up.",
		PREFINT(gNetworkTimeout) },
	{ "trace", "Trace logging:",
		kToggleMsg,
		PREFBOOL(gTrace) },
};

/* These are options that are for information only, or options I don't feel
 * like wasting screen space on in the prefs editor.
 */
PrefOpt gNonEditPrefOpts[] = {
	{ "show-trailing-space", "Show trailing space:",
		kToggleMsg,
		PREFBOOL(gMarkTrailingSpace) },
	{ "total-runs", NULL, NULL,
		PREFINT(gTotalRuns) },
};

int gNumEditablePrefOpts = ((int)(sizeof(gPrefOpts) / sizeof(PrefOpt)));
#define kNumNonEditablePrefOpts ((int)(sizeof(gNonEditPrefOpts) / sizeof(PrefOpt)))





void TogglePref(int *val, int min, int max)
{
	int newVal;

	newVal = *val + 1;
	if (newVal > max)
		newVal = min;
	*val = newVal;
}	/* TogglePref */





void GetPrefSetting(char *dst, size_t siz, int item)
{
	char *cp;
	string str;
	size_t len;

	*dst = '\0';
	switch (item) {
		case kBlankLinesWinItem:
			cp = gBlankLines ? "yes" : "no";
			(void) Strncpy(dst, cp, siz);
			break;

		case kLogSizePrefsWinItem:
			if (gMaxLogSize == 0)
				(void) Strncpy(str, "no logging", siz);
			else
				sprintf(str, "%d", gMaxLogSize);
			(void) Strncpy(dst, str, siz);
			break;

		case kMaxHostsWinItem:
			if (gMaxRecents == kNoRecentLimit)
				(void) Strncpy(str, "unlimited", siz);
			else
				sprintf(str, "%d", gMaxRecents);
			(void) Strncpy(dst, str, siz);
			break;

		case kPagerPrefsWinItem:
			if ((len = strlen(gPager)) > 47) {
				/* Abbreviate a long program path. */
				STRNCPY(str, "...");
				STRNCAT(str, gPager + len - 44);
			} else {
				if (gPager[0] == '\0')
					STRNCPY(str, "(none)");
				else
					STRNCPY(str, gPager);
			}
			(void) Strncpy(dst, str, siz);
			break;

		case kStartupLCWDWinItem:
			cp = (gStartupDir[0] != '\0') ? gStartupDir :
				"(none)";
			(void) Strncpy(dst, cp, siz);
			break;

		case kStartupMsgsPrefsWinItem:
			if ((gStartupMsgs & (kStartupMsg | kTips)) == (kStartupMsg | kTips))
				cp = "headers and tips";
			else if ((gStartupMsgs & kStartupMsg) != 0)
				cp = "headers only";
			else if ((gStartupMsgs & kTips) != 0)
				cp = "tips only";
			else
				cp = "no startup messages";
			(void) Strncpy(dst, cp, siz);
			break;

		case kTimeoutPrefsWinItem:
			sprintf(dst, "%d", gNetworkTimeout);
			break;

		case kTracePrefsWinItem:
			if (gTrace) {
				cp = "yes";
				OpenTraceLog();
			} else {
				cp = "no";
				CloseTraceLog();
			}
			(void) Strncpy(dst, cp, siz);
			break;

	}
}	/* GetPrefSetting */

void ShowAll(void)
{
	int i;
	string value;

	for (i = kFirstPrefsWinItem; i <= kLastPrefsWinItem; i++) {
		GetPrefSetting(value, sizeof(value), i);
		MultiLinePrintF("%-28s%s\n", gPrefOpts[i].label, value);
	}
}	/* ShowAll */




static
void ShowSetHelp(void)
{
	int i;

	for (i = kFirstPrefsWinItem; i <= kLastPrefsWinItem; i++) {
		MultiLinePrintF("%-15s %s\n", gPrefOpts[i].name, gPrefOpts[i].label);
	}
}	/* ShowSetHelp */





int SetCmd(int argc, char **argv)
{
	int i, j, match;
	size_t len;
	string value;

	if ((argc == 1) || ISTREQ(argv[1], "all")) {
		ShowAll();
	} else if (ISTREQ(argv[1], "help")) {
		ShowSetHelp();
	} else {
		len = strlen(argv[1]);
		for (i = kFirstPrefsWinItem, match = -1; i <= kLastPrefsWinItem; i++) {
			if (ISTRNEQ(gPrefOpts[i].name, argv[1], len)) {
				if (match >= 0) {
					Error(kDontPerror, "Ambiguous option name \"%s.\"\n",
						argv[1]);
					return (kCmdErr);
				}
				match = i;
			}
		}
		if (match < 0) {
			Error(kDontPerror, "Unknown option name \"%s.\"\n",
				argv[1]);
			return (kCmdErr);
		}
		i = match;
		switch (gPrefOpts[i].type) {
			case kPrefInt:
				if (argc > 2)
					AtoIMaybe((int *) gPrefOpts[i].storage, argv[2]);
				break;
			case kPrefToggle:
				if (argc > 2) {
					/* User can set it directly instead of cycling through
					 * the choices.
					 */
					AtoIMaybe((int *) &j, argv[2]);
					if (j < gPrefOpts[i].min)
						j = gPrefOpts[i].min;
					else if (j > gPrefOpts[i].max)
						j = gPrefOpts[i].max;
					* (int *) gPrefOpts[i].storage = j;
				} else {
					/* User toggled to the next choice. */
					TogglePref((int *) gPrefOpts[i].storage,
						gPrefOpts[i].min, gPrefOpts[i].max);
				}
				break;
			case kPrefStr:
				if (argc > 2)
					(void) Strncpy((char *) gPrefOpts[i].storage,
						argv[2], gPrefOpts[i].siz);
				break;
		}

		/* Print the (possibly new) value. */
		GetPrefSetting(value, sizeof(value), i);
		PrintF("%-28s%s\n", gPrefOpts[i].label, value);
	}
	return (kNoErr);
}	/* SetCmd */


void WritePrefs(void)
{
	longstring path;
	FILE *fp;
	int i;

	if (gOurDirectoryPath[0] == '\0')
		return;		/* Don't create in root directory. */

	OurDirectoryPath(path, sizeof(path), kPrefsName);
	fp = fopen(path, "w");
	if (fp == NULL) {
		Error(kDoPerror, "Can't open %s for writing.\n", path);
		return;
	}
	for (i = kFirstPrefsWinItem; i <= kLastPrefsWinItem; i++) {
		switch (gPrefOpts[i].type) {
			case kPrefInt:
			case kPrefToggle:
				fprintf(fp, "%s %d\n", gPrefOpts[i].name,
					* (int *) gPrefOpts[i].storage);
				break;
			case kPrefStr:
				fprintf(fp, "%s %s\n", gPrefOpts[i].name,
					(char *) gPrefOpts[i].storage);
				break;
		}
	}
	for (i = 0; i < kNumNonEditablePrefOpts; i++) {
		switch (gNonEditPrefOpts[i].type) {
			case kPrefInt:
			case kPrefToggle:
				fprintf(fp, "%s %d\n", gNonEditPrefOpts[i].name,
					* (int *) gNonEditPrefOpts[i].storage);
				break;
			case kPrefStr:
				fprintf(fp, "%s %s\n", gNonEditPrefOpts[i].name,
					(char *) gNonEditPrefOpts[i].storage);
				break;
		}
	}

	fclose(fp);
}	/* WritePrefs */





static
int PrefSearchProc(char *key, const PrefOpt *b)
{
	return (ISTRCMP(key, (*b).name));	
}	/* PrefSearchProc */





void ReadPrefs(void)
{
	longstring path;
	FILE *fp;
	string option;
	longstring val;
	longstring line;
	int o;
	PrefOpt *pop;

	OurDirectoryPath(path, sizeof(path), kPrefsName);
	fp = fopen(path, "r");
	if (fp == NULL) {
		/* It's okay if we don't have one. */
		return;
	}

	while (FGets(line, (int) sizeof(line) - 1, fp) != NULL) {
		if (sscanf(line, "%s", option) < 1)
			continue;
		o = strlen(option) + 1;
		STRNCPY(val, line + o);
		pop = (PrefOpt *) BSEARCH(option, gPrefOpts, SZ(gNumEditablePrefOpts),
			sizeof(PrefOpt), PrefSearchProc);
		if (pop == NULL) {
			pop = (PrefOpt *) BSEARCH(option, gNonEditPrefOpts,
				SZ(kNumNonEditablePrefOpts),
				sizeof(PrefOpt), PrefSearchProc);
			if (pop == NULL) {
				Error(kDontPerror, "Unrecognized preference option \"%s\".\n",
					option);
				continue;
			}
		}
		switch (pop->type) {
			case kPrefInt:
			case kPrefToggle:
				* (int *) pop->storage = atoi(val);
				break;
			case kPrefStr:
				(void) Strncpy((char *) pop->storage, val, pop->siz);
				break;
		}
	}

	fclose(fp);
}	/* ReadPrefs */
