/*
 * program.inc
 */

#if defined(DUMP)
static void DumpPhase1Program(), DumpPhase2Program();
static void DumpActionStatement(), DumpOutputList(), DumpSector(),
	    DumpLocation(), DumpExprList();
static char *CommodityName();
#endif

/*
 * Local data.
 */
static Instruction LLprogram;		/* Program being parsed         */
static int next_label;			/* Next free label value        */

#define LABEL_HASH_COUNT	23
static Instruction label_hash_list[LABEL_HASH_COUNT];

void DestroyProgram(program)
Program *program;
{
	Instruction instr, link;

	if (*program == (Program) 0)
		return;

	for (instr = *program; instr != (Instruction) 0; instr = link) {
		if (instr->type == I_EVAL)
			DestroyExpression(&instr->expression);
		else if (instr->type == I_ACTION)
			DestroyAction(&instr->action);
		link = instr->link;
		dofree ((char *) instr);
	}

	*program = (Instruction) 0;
}

Program ParseProgram(strings)
Strings strings;
{
	int token, count;

	parse_error = False;
	LLprogram   = (Instruction) 0;
	next_label  = 3;

	for (count = 0; count < LABEL_HASH_COUNT; count++)
		label_hash_list[count] = (Instruction) 0;

	LinkInstruction(I_EXECUTE);

	LexInitStrings(strings);
	LLParseProgram();

	if (parse_error == True) {
		DestroyProgram(&LLprogram);
		DisplayParseErrors();
		return (Program) 0;
	}

	AddInstructionLabel(1);
	LinkInstruction(I_STOP_PROGRAM);

	AddInstructionLabel(2);
	LinkInstruction(I_ABORT);

#if defined(DUMP) && defined(DEBUG)
	DumpPhase1Program(LLprogram);
#endif

	RunPhase2();

	return LLprogram;
}

static void AddInstructionLabel(number)
int number;
{
	Instruction instr = LinkInstruction(I_LABEL);
	int index = (number % LABEL_HASH_COUNT);

	instr->number          = number;
	instr->label_list      = label_hash_list[index];
	label_hash_list[index] = instr;
}

static void AddInstructionGoto(number)
int number;
{
	Instruction instr = LinkInstruction(I_GOTO);

	instr->next_value = number;
}

static void AddInstructionEval(expression, label_true, label_false)
Stack expression;
int label_true, label_false;
{
	Instruction instr = LinkInstruction(I_EVAL);

	instr->expression  = expression;
	instr->true_value  = label_true;
	instr->false_value = label_false;
}

static void AddInstructionSetCount(count)
int count;
{
	Instruction instr = LinkInstruction(I_SET_COUNT);

	instr->number = count;
}

static void AddInstructionTestCount(false_label)
{
	Instruction instr = LinkInstruction(I_TEST_COUNT);

	instr->false_value = false_label;
}

static void AddInstructionAction(action, label_next)
Action action;
int label_next;
{
	Instruction instr = LinkInstruction(I_ACTION);

	instr->action      = action;
	instr->next_value  = label_next;
	instr->fail_value  = label_next;
	instr->false_value = label_next;
}

static Instruction LastInstruction()
{
	if (LLprogram == (Program) 0)
		return (Instruction) 0;

	return LLprogram->prev;
}

static void SetFailLabel(instr, label)
Instruction instr;
int label;
{	
	instr->fail_value = label;
}

static void SetErrorLabel(instr, label)
Instruction instr;
int label;
{
	instr->false_value = label;
}

static Instruction LinkInstruction(type)
int type;
{
	Instruction instr = (Instruction) doalloc(sizeof(struct _Instruction));

	instr->type        = type;
	instr->referenced  = False;
	instr->next_value  = 0;
	instr->false_value = 0;
	instr->fail_value  = 0;
	instr->link        = (Instruction) 0;

	if (LLprogram == (Instruction) 0) {
		instr->prev = instr;
		LLprogram   = instr;
	}
	else {
		instr->prev           = LLprogram->prev;
		LLprogram->prev->link = instr;
		LLprogram->prev       = instr;
	}

	return instr;
}

static void RunPhase2()
{
	Instruction instr, link_instr;
#if defined(DUMP)
	int seq_nr = 1;
	int i = 0;
	extern char *ptypes[];

	LLprogram->nr = seq_nr++;		/* I_FOREVER/I_EXECUTE */
#endif

/* Substitute labels values for label addresses */
	for (instr = LLprogram; instr->type != I_ABORT; instr = instr->link) {
		instr->next_address  = FindLabelAddress(instr,
						        instr->next_value);
		instr->false_address = FindLabelAddress(instr,
							instr->false_value);
		instr->fail_address  = FindLabelAddress(instr,
						        instr->fail_value);
	}

	for (instr = LLprogram; instr->type != I_ABORT; instr = instr->link) {
		instr->next_address  = FindShortCut(instr,instr->next_address);
		instr->false_address = FindShortCut(instr,instr->false_address);
		instr->fail_address  = FindShortCut(instr,instr->fail_address);
	}

/* Delete unnessecary instructions (goto's and labels) */
	for (instr = LLprogram->link; instr->type != I_ABORT;
				      instr = link_instr) {
		link_instr = instr->link;

		if (instr->referenced == False) {
#if defined(STAND_ALONE) || defined(DEBUG)
			if (instr->type != I_LABEL && instr->type != I_GOTO)
				printf("Warning %d [%d], not reached.\n",
					instr->nr, instr->type);
#endif
			instr->prev->link = instr->link;
			instr->link->prev = instr->prev;
			dofree((char *) instr);
		}
#if defined(DUMP)
		else
			instr->nr = seq_nr++;
#endif
	}

#if defined(DUMP)
	instr->nr = seq_nr;		/* I_ABORT */

	DumpPhase2Program(LLprogram);
#endif
}

static Instruction FindLabelAddress(current_instr, label_value)
Instruction current_instr;
int label_value;
{
	Instruction instr;

	if (label_value == 0)
		return current_instr->link;

	for (instr = label_hash_list[label_value % LABEL_HASH_COUNT];
			     instr != (Instruction) 0;
			     instr = instr->label_list)
		if (instr->number == label_value)
			return instr;

	printf("Internal Compiler Error: Label %d not found.", label_value);

	return current_instr->link;	/* Bogus return value */
}

static Instruction FindShortCut(instr, next_instr)
Instruction instr, next_instr;
{
	while (next_instr->type == I_GOTO || next_instr->type == I_LABEL)
		next_instr = next_instr->next_address;

	next_instr->referenced = True;
	return next_instr;
}

#if defined(DUMP)
static char *ptypes[10] = {
	"I_FOREVER", "I_EXECUTE", "I_EVAL", "I_ACTION", "I_LABEL", "I_GOTO",
	"I_SET_COUNT", "I_TEST_COUNT", "I_STOP_PROGRAM", "I_ABORT"
};

void DumpProgram (program)
Program program;
{
	if (program == (Program) 0) {
		printf("NO PROGRAM.\n");
		return;
	}

	DumpPhase2Program (program);
}

static void DumpPhase1Program(program)
Program program;
{
	Instruction instr;
	int i = 0;

	if (program == (Program) 0) {
		printf("NO PROGRAM.\n");
		return;
	}

	for (instr = program; instr != (Instruction) 0; instr = instr->link) {
		printf("Instruction %2d: (%s)%s", ++i, ptypes[instr->type],
		       instr->referenced == True ? "*" : "-");

		if (instr->type == I_LABEL)
			printf(" [%d]", instr->number);
		else if (instr->type == I_GOTO) {
			if (instr->next_value != 0)
				printf(" [%d]", instr->next_value);
			else
				printf(" [link]");
		}
		else if (instr->type == I_SET_COUNT) {
			if (instr->number < 0)
				printf(" [FOREVER]");
			else
				printf(" [%d]", instr->number);
		}
		putchar('\n');

		if (instr->type == I_ACTION) {
			DumpActionStatement(instr->action);
			printf("          NEXT: ");
			if (instr->next_value == 0)
				printf("[link]\n");
			else
				printf("%d\n", instr->next_value);
			if (instr->fail_value != 0)
				printf("        ONFAIL: %d\n",
				       instr->fail_value);
			if (instr->false_value != 0)
				printf("       ONERROR: %d\n",
				       instr->false_value);
		}
		else if (instr->type == I_EVAL) {
			printf("    Expression: ");
			DumpExpression(instr->expression);
			printf("\n");
			printf("           JNZ: ");
			if (instr->next_value == 0)
				printf("[link]\n");
			else
				printf("%d\n", instr->next_value);
			printf("            JZ: ");
			if (instr->false_value == 0)
				printf("[link]\n");
			else
				printf("[%d]\n", instr->false_value);
		}
	}

	printf(" -- Label Hash List --\n");
	for (i = 0; i < LABEL_HASH_COUNT; i++) {
		if (label_hash_list[i] == (Instruction) 0)
			continue;
		printf("%2d: ", i % LABEL_HASH_COUNT);
		for (instr = label_hash_list[i]; instr != (Instruction) 0;
				 		 instr = instr->label_list)
			printf("[%d] ", instr->number);
		printf("\n");
	}
}

static void DumpPhase2Program(program)
Program program;
{
	Instruction instr;

	for (instr = program; instr->type != I_ABORT; instr = instr->link) {
		printf("Instruction %2d: (%s)",
				instr->nr, ptypes[instr->type]);
		if (instr->type == I_SET_COUNT) {
			if (instr->number < 0)
				printf(" [FOREVER]");
			else
				printf(" [%d]", instr->number);
		}

		putchar('\n');

		if (instr->type == I_EVAL) {
			printf("     Expression: ");
			DumpExpression(instr->expression);
			printf("\n");
			if (instr->next_address->nr != instr->nr + 1)
				printf("     Next instr: %d\n",
				       instr->next_address->nr);
			printf("    False instr: %d\n",
			       instr->false_address->nr);
		}
		else if (instr->type == I_ACTION) {
			DumpActionStatement(instr->action);
			if (instr->next_address->nr != instr->nr + 1)
				printf("           NEXT: %d\n",
					instr->next_address->nr);
			if (instr->false_address->nr != instr->nr + 1)
				printf("       ON ERROR: %d\n",
					instr->false_address->nr);
			if (instr->fail_address->nr != instr->nr + 1)
				printf("         ON FAIL %d\n",
					instr->fail_address->nr);
		}
		else {
			if (instr->next_address->nr != instr->nr + 1)
				printf("           next: %d\n",
					instr->next_address->nr);
			if (instr->false_address->nr != instr->nr + 1)
				printf("    False instr: %d\n",
					instr->false_address->nr);
			if (instr->fail_address->nr != instr->nr + 1)
				printf("     Fail instr: %d\n",
					instr->fail_address->nr);
		}
	}

	printf("Instruction %2d: (%s)\n", instr->nr, ptypes[instr->type]);
}

static void DumpActionStatement(action)
Action action;
{
	printf("         Action: ");

	if (action == (Action) 0) {
		printf("Empty list?\n");
		return;
	}

	switch (action->type) {
	case MOVE:
		printf("MOVE ");
		DumpExpression(action->u_move->quantity);
		printf(" %s FROM ", CommodityName(action->u_move->commodity));
		DumpLocation(&action->u_move->from_loc);
		printf(" ");
		if (action->u_move->via_list != NULL)
			DumpExprList(action->u_move->via_list->next, "VIA");
		printf("TO ");
		DumpLocation(&action->u_move->to_loc);
		break;
	case SET_DISTRIBUTE:
		printf("SET DISTRIBUTE PATH IN SECTOR ");
		DumpSector(action->u_distribute->sector_expr);
		printf(" TO ");
		DumpLocation(action->u_distribute->to_loc);
		break;
	case SET_DELIVER:
		printf("SET DELIVER PATH IN SECTOR ");
		DumpSector(action->u_deliver->sector_expr);
		printf(" %s ", CommodityName(action->u_deliver->commodity));
		printf("TO ");
		DumpSector(action->u_deliver->dest_sector);
		break;
	case SET_DESIGNATION:
		printf("DESIGNATE SECTOR ");
		DumpSector(action->u_designate->sector_expr);
		printf(" TO %c", action->u_designate->flag);
		break;
	case ENLIST:
		printf("ENLIST ");
		DumpExpression(action->u_enlist->quantity);
		printf(" IN SECTOR ");
		DumpSector(action->u_enlist->sector_expr);
		break;
	case DEMOBILIZE:
		printf("DEMOBILIZE ");
		DumpExpression(action->u_demobilize->quantity);
		printf(" [%s] IN SECTOR ",
			action->u_demobilize->flag == 'y' ? "RESERVE" : "");
		DumpSector(action->u_demobilize->sector_expr);
		break;
	case CONVERT:
		printf("CONVERT ");
		DumpExpression(action->u_convert->quantity);
		printf(" IN SECTOR ");
		DumpSector(action->u_convert->sector_expr);
		break;
	case SHOOT:
		printf("SHOOT ");
		DumpExpression(action->u_convert->quantity);
		printf(" %s IN SECTOR ",
			action->u_convert->flag == 'u' ? "uw" : "civilians");
		DumpSector(action->u_convert->sector_expr);
		break;
	case SET_THRESHOLD:
		printf("SET %s THRESHOLD IN SECTOR ",
			CommodityName(action->u_threshold->flag));
		DumpSector(action->u_threshold->sector_expr);
		printf(" TO ");
		DumpExpression(action->u_threshold->quantity);
		break;
	case SET_CUTOFF:
		printf("SET CUTOFF PATH of %s IN SECTOR ",
					CommodityName(action->u_cutoff->flag));
		DumpSector(action->u_cutoff->sector_expr);
		printf(" TO");
		DumpExpression(action->u_cutoff->quantity);
		break;
	case TERRITORY:
		printf("SET TERRITORY IN SECTOR ");
		DumpSector(action->u_territory->sector_expr);
		printf("TO ");
		DumpExpression(action->u_territory->quantity);
		break;
	case PRINT:
		printf ("PRINT ");
		if (action->u_print != NULL)
			DumpOutputList(action->u_print->next);
		break;
	case SEND:
		printf ("SEND ");
		if (action->u_send != NULL)
			DumpOutputList(action->u_send->next);
		break;
	default:
		printf("Invalid Action statement (%d)", action->type);
	}

	putchar ('\n');
}

static void DumpExprList(list, prompt)
ExprList list;
char *prompt;
{
	if (list == (ExprList) 0)
		return;

	printf("\n				%s ", prompt);
	DumpExpression(list->e_expr);
	DumpExprList(list->next, prompt);
}

static void DumpLocation(loc)
Location loc;
{
	switch(loc->type) {
	case LOC_NONE:
		printf("current_sector");
		break;
	case LOC_SECTOR:
		DumpExpression(loc->l_sector_expr);
		break;
	case LOC_WHERE:
		if (loc->l_where_list != NULL)
			DumpExprList(loc->l_where_list->next, "WHERE");
		break;
	default:
		printf ("<<<<<OUT OF NOWHERE>>>>>");
	}

	putchar(' ');
}

static void DumpOutputList(list)
Output list;
{
	if (list == (Output) 0)
		return;

	DumpExpression(list->o_expr);
	if (list->format)
		printf ("FORMAT(%d)", list->format);
	printf("   ");
	DumpOutputList(list->next);
}

static char *CommodityName(commodity)
int commodity;
{
	switch (commodity) {
		case 'c':	return "civilians";
		case 'm':	return "military";
		case 'u':	return "uw";
		case 'f':	return "food";
		case 's':	return "shells";
		case 'g':	return "guns";
		case 'i':	return "iron dust";
		case 'd':	return "gold dust";
		case 'b':	return "bars";
		case 'o':	return "oil";
		case 'p':	return "petroleum";
		case 'l':	return "lcms";
		case 'h':	return "hcms";
		case 'r':	return "radioactive";
		default:	return "unknown *COMMODITY*";
	}
}

static void DumpSector(sector_expr)
Stack sector_expr;
{
	if (sector_expr == (Stack) 0)
		printf ("current_sector");
	else
		DumpExpression(sector_expr);
}
#endif
