#!/usr/local/bin/perl
# $Header: parse.pl,v 1.8.1.1 95/05/11 03:56:02 stevens Exp $
#                      Perl Empire Interface Parser
#
#                Written by Drake Diedrich and Sam Tetherow
#
#
# DESCRIPTION:
# This module parses Empire data and stores it in variables.
#
# INSTALLATION:
# You do not need to do anything to install parse.pl.  It will be automatically
# included into pei when you connect to a game.  Just make sure that it
# is in the same directory as pei when pei is run.  Or you can put parse.pl
# in the directory ~/perl and add the following line to your ~/.login file:
#   setenv PERLLIB ~/perl
#
# AUTHORS:
# Written by Drake Diedrich.
# Modified to use packages by Ken Stevens.
# Completely rewritten by Sam Tetherow.
#
# BUG REPORTS:
# mail your bug-reports and comments to:
# children@empire.net (Ken Stevens)
# tetherow@cse.unl.edu (Sam Tetherow)
 
#       --- Global variables ---
#
# Game related variables:
# $etu                Number of Empire Time Units per update
# $obrate             Civilian babies born per ETU per civilian
# $eatrate            Food eaten per person per ETU
# $babyeat            Food eaten per baby at the update
# $uwbrate            uw babies born per ETU per uw
# $milcost            Cost per active military per ETU
# $reservecost        Cost per reserve per ETU
# $easytech           Tech up to this amount is not reduced
# $logtech            Log base factored into tech above easytech
# $secmob             Monility gained in a sector at the update
# $maxshipmob         Ship mobility gained at the update
# $planemob           Plane mobility gained at the update
# $unitmob            Unit mobility gained at the update
# $seceff             Maximum sector efficiency gained at the update
# $maxshipeff         Maximum ship efficiency gained at the update
# $planeeff           Maximum plane efficiency gained at the update
# $uniteff            Maximum unit efficiency gained at the update
# $width              Width of the world
# $height             Height of the world
# $maxnumcountries    Maximum number of countries
# %option             Array of options
# 
# Country related variables:
# $budget             Estimated treasury after the next update
# $estimated_delta    Change in the treasury at the next update
# %relation           Array of relations
# $coun               Number of player country
# %country            Array of country names
# %number             Array of country numbers
# $cap                Location of player capitol
# %realm              List of realms
# $reserves           Number of reserves
# %money              Current treasury
# %edu                Array of education levels
# %hap                Array of happiness levels
# %tech               Array of technology levels
# %res                Array of research levels
# %status             Array of country statuses
# $maxciv             Maximum number of civs that can reproduce/sector
# $maxuw              Maximum number of uws that can reproduce/sector
# $maxpop             Maximum number of people of each class/sector
# 
# Sector related variables:
# %newdes             Sector designation after the next update
# %neweff             Sector efficiency after the next update
# %mine               Array of mines in a sector.
# %defense_strength   Defensive strength of a sector
# %bdes               Sector designations
# %dump               Values by sector and header field
# %own                Array of sector ownership
# %oldown             Original owners of sectors\n";
# %radar_center       Array of radar stations
# %radar_range        Array of radar station ranges
# %wkfc               Workforce per sector
# %will               Amount of product produced by sectors
# %make               Product produced by sectors
# %prodmax            Maximum production by sectors
# %use1               Amount of material used by sectors
# %comm1              Type of material used by sectors
# %max1               Amount of material that could be used by sectors
# %use2               Amount of material used by sectors
# %comm2              Type of material used by sectors
# %max2               Amount of material that could be used by sectors
# %use3               Amount of material used by sectors
# %comm3              Type of material used by sectors
# %max3               Amount of material that could be used by sectors
# %sectype            sector types
# %movcost            movment cost through sector
# %secoff             sector offensive multiplier
# %secdef             sector defensive multiplier
# $bridgetech         Minimum technology level to build a bridge
# $bridgehcm          hcm necessary to build a brige
# $bridgeworkers      number of civs+uws to build a bridge
# $bridgeavail        available work necessary to build a bridge
# $bridgecost         cost to build a bridge
# 
# Plane, Unit, and Ship related variables:
# %buildmil           mil required to build item
# %buildlcm           lcm required to build item
# %buildhcm           hcm required to build item
# %buildgun           guns required to build unit
# %buildshell         shells required to build unit
# %buildavail         avail required to build unit
# %buildcost          cost to build unit
# %buildtech          tech required to build unit
# %fuelcar            fuel capacity of unit
# %fueluse            fuel use (per 10 mob) of unit
# %capability         capabilities of item
# %cargo              cargo capacities by item and commodity
#
# Planes:
# %planename          types of planes
# %planesect          locations of planes
# %planewing          wing assignments of planes
# %planeeff           plane efficiencies
# %planemob           mobility of planes
# %planetech          tech level planes were constructed at
# %planehard          amounts planes have been hardened
# %planesln           ship, land unit, or nuke carrying or carried by plane
# %planeacc           bombing accuracy
# %planeload          plane cargo limits
# %planeatt           plane attack values
# %planedef           plane defense values
# %planeran           plane ranges
# %planefuel          fuel used by planes
# %planestlth         likelyhood of avoiding detection
#
# Ships:
# %shipflt            fleets ships are in
# %shipeff            efficiencies of ships
# %shipcargo          commodities carried by ships
# %shipmob            mobility of ships
# %shipfuel           fuel aboard ships
# %shiptech           technology ships were constructed at
# %shipdef            armor rating of ship type
# %shipspd            speed of ship type
# %visibility         visibility og ship type
# %shipspy            spy range of ship type
# %shipfrg            double the basic firing range of ship type
# %shiplnd            units carried by ships
# %shippln            planes carried by ships
# %shiphel            helicopters carried by ships
# %shipxpl            sams carried by ships
# %shipname           Ship types
# %shipsect           Location of ships
# %shipown            Owners of ships
#
# Units:
# %unitname           type of each unit
# %unitown            owners of units
# %unitsect           locations of units
# %unitnear		approximate locations of units (if unknown)
# %unitarmy           armies units are in
# %uniteff            efficiencies of units
# %unitfort           fortifications of units
# %unitmob            mobility of units
# %unitcargo          cargo carried by units
# %unitfuel           fuel aboard units
# %unittech           technology level unit was constructed at
# %unitretr           retreat below these efficiencies
# %unitrad            unit reaction radii
# %unitxpl            sams carried by units
# %unitatt            unit attack multipliers
# %unitdef            unit defensive multipliers
# %unitvul            unit vulnerability to shelling
# %unitspd            unit speed
# %visibility         unit visibility
# %unitspy            unit spy range
# %unitrad            unit reaction radius
# %unitfrg            twice basic unit firing range
# %unitacc            unit shelling accuracy
# %unitdam            basic damage unit causes from shelling
# %unitamm            shells unit uses when firing
# %unitaaf            flak guns fired by unit
# %unitxpl            sam capacity of unit 
 
if (!$main'parse_loaded) {
  package main;
 
  %month=('Jan', 1, 'Feb', 2, 'Mar', 3, 'Apr', 4, 'May', 5, 'Jun', 6, 'Jul', 7,
          'Aug', 8, 'Sep', 9, 'Oct', 10, 'Nov', 11, 'Dec', 12);
  %monthname=(0, 'Jan', 1, 'Feb', 2, 'Mar', 3, 'Apr', 4, 'May', 5, 'Jun', 6,
          'Jul', 7, 'Aug', 8, 'Sep', 9, 'Oct', 10, 'Nov', 11, 'Dec');
  %sectype=('gold mine','g', 'highway','+', 'airfield','*', 'warehouse','w',
            'mine','m', 'bridge span','=', 'capitol','c', 'mountain','^',
	    'radar installation',')', 'light manufacturing','j', 'harbor','h',
	    'enlistment center','e', 'bank','b', 'refinery','%', 'park','p',
	    'headquarters','!', 'heavy manufacturing','k', 'agribusiness','a',
	    'technical center','t', 'research lab','r', 'defense plant','d',
	    'library/school','l', 'shell industry','i', 'wilderness','-',
	    'oil field','o', 'fortress','f', 'nuclear plant','n',
	    'sanctuary','s', 'wasteland','\\', 'uranium mine','u', 
            'bridge head','#');

  sub add_parsemap {
    local ($command, $len, $subroutine) = @_;
    local ($i);
 
    for ($i = $len; $i <= length($command); ++$i) {
      $parsemap{substr($command, 0, $i)} = $subroutine;
    }
  }
}

if (!$main'parse_loaded) {
  package main;
 
  &add_parsemap('change', 2, 'parse_change');
  &add_parsemap('version', 1, 'parse_version');
  &add_parsemap('nation',3,'parse_nation');
  &add_parsemap('budget',3,'parse_budget');
  &add_parsemap('relations',3,'parse_relations');
  &add_parsemap('realm',4,'parse_realm');
  &add_parsemap('report',3,'parse_report');
  &add_parsemap('power',3,'parse_power');
  &add_parsemap('country',3,'parse_country');
  &add_parsemap('read',4,'parse_read');
 
  $parsemap{'show sect stats'}='parse_sector_stats';
  $parsemap{'show sect build'}='parse_sector_build';
  &add_parsemap('neweff',4,'parse_neweff');
  &add_parsemap('product',2,'parse_product');
  &add_parsemap('radar',3,'parse_radar');
  &add_parsemap('navigate',3,'parse_look');
  &add_parsemap('march',3,'parse_look');
  &add_parsemap('lookout',3,'parse_look');
  &add_parsemap('llookout',4,'parse_look');
  &add_parsemap('sonar',3,'parse_look');
  &add_parsemap('spy',2,'parse_spy_report');
  &add_parsemap('recon',3,'parse_spy_plane');
  &add_parsemap('bomb',3,'parse_spy_plane');
  &add_parsemap('fly',3,'parse_spy_plane');
  &add_parsemap('paradrop',4,'parse_spy_plane');
  &add_parsemap('drop',3,'parse_spy_plane');
  &add_parsemap('satellite',3,'parse_satellite');
  &add_parsemap('dump',2,'parse_dump');
  &add_parsemap('level',3,'parse_level');
  &add_parsemap('map',2,'parse_map');
  &add_parsemap('bmap',2,'parse_map');
  &add_parsemap('attack',2,'parse_attack');
  &add_parsemap('census',2,'parse_census');
  &add_parsemap('resource',3,'parse_resource');
  &add_parsemap('move',3,'parse_move');
  &add_parsemap('explore',3,'parse_explore');
 
  &add_parsemap('info',1,'parse_show');
  &add_parsemap('show',4,'parse_show');
  $parsemap{'show plane build'}='parse_plane_build';
  $parsemap{'show plane stats'}='parse_plane_stats';
  $parsemap{'show capabilities'}='parse_capabilities';
  &add_parsemap('plane',2,'parse_plane');
  $parsemap{"mission"} = "parse_mission";
  &add_parsemap('mission',3,'parse_mission');
  &add_parsemap('build',3,'parse_build');
 
  $parsemap{'show ship build'}='parse_ship_build';
  $parsemap{'show ship stats'}='parse_ship_stats';
  &add_parsemap('coastwatch',3,'parse_look');
  &add_parsemap('ship',3,'parse_ship');
  &add_parsemap('cargo',3,'parse_cargo');
 
  $parsemap{'show land build'}='parse_land_build';
  $parsemap{'show land stats'}='parse_land_stats';
  &add_parsemap('land',3,'parse_land');
  &add_parsemap('unit',3,'parse_land');
  &add_parsemap('lcargo',3,'parse_cargo');
 
}
 
sub parse_change {
  if (/^Country \#(\d+) is already called/) {
    $coun = $1;
  }
}

sub parse_version {
  if (/An update consists of (\d+) empire time units/) { $etu=$1; return; }
  if (/1000 civilians will give birth to ([\d\.]+) babies per etu./) {
    $obrate = $1 / 1000.0;
    return;
  }
  if (/In one time unit, 1000 people eat ([\d\.]+) units of food./) {
    $eatrate = $1 / 1000.0;
    return;
  }
  if (/1000 babies eat ([\d\.]+) units of food becoming adults./) {
    $babyeat = $1 / 1000.0;
    return;
  }
  if (/1000 uncompensated workers will give birth to ([\d\.]+) babies./) {
    $uwbrate = $1 / 1000.0;
    return;
  }
  if (/1000 active military cost \$([\d.]+)\,/) {
    $milcost=$1 / 1000.0;
    return;
  }
  if (/reserves cost \$([\d.]+)\.$/) {
    $reservecost=$1/1000.0;
    return;
  }
  if (/Tech Buildup is limited to logarithmic growth \(base (.+)\) after (.+)\./) {
    $easytech = $2;
    $logtech = $1;
    return;
  }
  if (/Nation levels \(tech etc.\) decline (\d+)% every (\d+) time units./) {
    $level_decay_rate = $1 * $etu / $2 / 100.0;
  }
  if (/Max mob gain per update\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/) {
    if ($1>0) { $secmob = $1; }
    if ($2>0) { $maxshipmob = $2; }
    if ($3>0) { $planemob = $3;}
    if ($4>0) { $unitmob = $4; }
    return;
  }
  if (/Max eff gain per update\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/) {
    if ($1 > 0) { $seceff = $1; }
    if ($2 > 0) { $maxshipeff = $2; }
    if ($3 > 0) { $planeeff = $3; }
    if ($4 > 0) { $uniteff = $4; }
    return;
  }
  if (/World size is (\d+) by (\d+)\./) {
    $width = $1;
    $height = $2;
    return;
  }
  if (/There can be up to (\d+) countries\./) {
    $maxnumcountries = $1;
    return;
  }
  if (/Options enabled in this game:/) {
    $options=1;
    return;
  }
  if ($options==1) {
    @F = split(/\,/, $_);
    $options=0 if ($#F<1);
    for ($i=0;$i<=$#F;$i++) { $F[$i]=~s/\s+//go; $option{$F[$i]}=""; }
  }
}
 
sub parse_show {
  if (/sector type\s+mcost\s+off\s+def/) { $main'command='show sect stats'; }
  elsif (/^\s*sector type\s+cost to des/) { $main'command='show sect build'; }
  elsif (/^\s+lcm hcm avail tech \$$/) { $main'command='show ship build'; }
  elsif (/ *def  d  s  y  g  r  d  n  l  l/) { $main'command='show ship stats'; }
  elsif (/ *cargos & capabilities/) { $main'command='show capabilities'; }
  elsif (/ *lcm hcm mil guns shells avail tech  \$/) { $main'command='show land build'; }
  elsif (/^ *capabilities$/) { $main'command='show capabilities'; }
  elsif (/ *att def vul  d  s  y  d  g  c  m  m  f  c  u  l/) {
    $main'command='show land stats';
  }
  elsif (/ *lcm hcm crew avail tech  \$$/) { $main'command='show plane build'; }
  elsif (/ *acc load att def ran fuel stlth/) { $main'command='show plane stats'; }
  elsif (/^Bridges require (\d+) tech, (\d+) hcm, (\d+) workers,/) {
    $bridgetech=$1;
    $bridgehcm=$2;
    $bridgeworkers=$3;
  }
  elsif (/^(\d+) available workforce, and cost \$(\d+)/) {
    $bridgeavail=$1;
    $bridgecost=$2;
  }
}
 
sub parse_budget {
  if (/New treasury:\s+(\d+)/) { $budget = $1; }
  elsif (/Estimated Delta:\s+(\d+)/) { $estimated_delta = $1; }
}
 
sub parse_neweff {
  if (/(\S+,\S+)\s+(\S) +(\d+)\%$/) {
    $newdes{$1}=$2;
    $neweff{$1}=$3;
  }
}
 
sub parse_product {
  if (/PRODUCTION SIMULATION/) { 	# undef old values we can't rely on them
    undef %wkfc;
    undef %will;
    undef %make;
    undef %prodmax;
    undef %use1;
    undef %use2;
    undef %use3;
    undef %max1;
    undef %max2;
    undef %max3;
    undef %comm1;
    undef %comm2;
    undef %comm3; 
  }
  elsif (/^\s*(\S+,\S+)\s+(\S)\s+\d+\%\s+(\d+)\s+(\S+)\s+(\S+)\s+\S+\s+\$\d+\s+(\S+)$/) {
    $newdes{$1}=$2;
    $wkfc{$1}=$3;
    $will{$1}=$4;
    $make{$1}=$5;
    $prodmax{$1}=$6;
 
  } elsif (/^\s*(\S+,\S+)\s+(\S)\s+\d+\%\s+(\d+)\s+(\S+)\s+(\S+)\s+\S+\s+\$\d+\s+(\d+)([a-z])\s+(\d+)[a-z]\s+(\S+)$/) {
    $newdes{$1}=$2;
    $wkfc{$1}=$3;
    $will{$1}=$4;
    $make{$1}=$5;
    $use1{$1}=$6;
    $comm1{$1}=$commstr{$7};
    $max1{$1}=$8;
    $prodmax{$1}=$9;
 
  } elsif (/^\s*(\S+,\S+)\s+(\S)\s+\d+\%\s+(\d+)\s+(\S+)\s+(\S+)\s+\S+\s+\$\d+\s+(\d+)([a-z])\s+(\d+)([a-z])\s+(\d+)[a-z]\s+(\d+)[a-z]\s+(\S+)$/) {
    $newdes{$1}=$2;
    $wkfc{$1}=$3;
    $will{$1}=$4;
    $make{$1}=$5;
    $use1{$1}=$6;
    $comm1{$1}=$commstr{$7};
    $use2{$1}=$8;
    $comm2{$1}=$commstr{$9};
    $max1{$1}=$10;
    $max2{$1}=$11;
    $prodmax{$1}=$12;
 
  } elsif (/^\s*(\S+,\S+)\s+(\S)\s+\d+\%\s+(\d+)\s+(\S+)\s+(\S+)\s+\S+\s+\$\d+\s+(\d+)([a-z])\s+(\d+)([a-z])\s+(\d+)([a-z])\s+(\d+)[a-z]\s+(\d+)[a-z]\s+(\d+)[a-z]\s+(\S+)$/) {
    $newdes{$1}=$2;
    $wkfc{$1}=$3;
    $will{$1}=$4;
    $make{$1}=$5;
    $use1{$1}=$6;
    $comm1{$1}=$commstr{$7};
    $use2{$1}=$8;
    $comm2{$1}=$commstr{$9};
    $use3{$1}=$10;
    $comm3{$1}=$commstr{$11};
    $max1{$1}=$12;
    $max2{$1}=$13;
    $max3{$1}=$14;
    $prodmax{$1}=$15;
  }
  if ($make{$1} eq 'petro') { $make{$1}='pet'; }
  if ($make{$1} eq 'bars') { $make{$1}='bar'; }
  if ($make{$1} eq 'guns') { $make{$1}='gun'; }
}
 
sub parse_nation {
  if (/\(\#(\d+)\)\s+(.+) Nation Report/) {
    $coun = $1;
    $country{$coun} = $2;
    $number{$2} = $coun;
    $new_server = 1;
  } elsif (/\s*(.+) Nation Report/) {
    if (defined $number{$1}) { $coun = $number{$1};	}
    if (defined $coun) {
      $country{$coun} = $1;
      $number{$1} = $coun;
    }
  }
  elsif (/^#\s*(\d+)\s+(\S+)\s+#\s*(\d+)\s+(\S+)\s+#\s*(\d+)\s+(\S+)\s*/) {
    $realm{$1}=$2; $realm{$3}=$4; $realm{$5}=$6;
  }
  elsif (/.* eff(| mountain) capit[oa]l at (\S+)/) { $cap = $2; }
 
  elsif (/No capit[oa]l\./) { $cap = ''; }
 
  elsif (/The treasury has \$(\d+)\.00\s+Military reserves:\s+(\d+)/) {
    if (defined $coun) { $money{$coun} = $1;}
    $reserves = $2;
  }
  elsif (/Education\.*\s+(.+) * Happiness\.* (.+)/) {
    if (defined $coun) {
      $edu{$coun} = $1;
      $hap{$coun} = $2;
    }
  }
  elsif (/Technology\.* *(.+) * Research\.*(.+)/) {
    if (defined $coun) {
      $tech{$coun} = $1;
      $res{$coun} = $2;
    }
  }
  elsif (/Technology factor : (.+)%\s+Plague factor/) {
    $techfactor = $1;
  }
  elsif (/^Max population : (\d+)/) { $maxpop=$1; }
 
  elsif (/^Max safe population for civs\/uws: (\d+)\/(\d+)$/) {
    $maxciv=$1;
    $maxuw=$2;
  }
}

sub parse_radar {
# Maybe add in map updating here (replace . with ? or ? with 'sectype')
  if (/(\S+,\S+) efficiency ([0-9]+)%, max range ([0-9]+)/) {
    $radar_center{$1} = $1;
    $radar_range{$1} = $3;
  }
}
 
sub parse_realm {
  if (/^Realm \#(\d{0,2}) is (\S+)$/) {
    $realm{$1} = $2;
  }
}

sub parse_payoff {
}
 
sub parse_orders {
}
 
sub parse_relations {
  if (/(.*) Diplomatic Relations Report/) {
    $last_relation_name=$1;
  } elsif (/^\s*([0-9]+)\) (.*\S)\s+(Neutral|At War|Hostile|Mobilizing|Sitzkrieg|Allied)\s+(Neutral|At War|Hostile|Mobilizing|Sitzkrieg|Allied)/) {
    $number{$2} = $1;
    $country{$1} = $2;
    $relation{$2,$last_relation_name} = $3;
    $relation{$last_relation_name,$2} = $4;
  }
}
 
sub parse_report {
# Not sure if we want to try and do some estimating here or not
  if (/^\s*(\d+)\s+(.+)\s+(backward|similar|primitive|advanced|superior)\s+(primitive|backward|similar|advanced|superior)\s+(primitive|backward|similar|advanced|superior)\s+(.+)\s*$/) {
    $country{$1}=$2;
    $number{$2}=$1;
    $tech{$1}=$3;
    $res{$1}=$4;
    $edu{$1}=$5;
    $status{$1}=$6;
  }
}
 
sub parse_power {
  $number{'worldwide'}=0;
  if (/^\s*(.+)\s+(\d+)\s+(\d+)%\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s*$/) {
    local($name, $sect, $eff, $civ, $mil, $shell, $gun, $pet, $iron, $dust, $oil, $plane, $ship, $unit, $money)=($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15);
    $name=~s/^\s*//; $name=~s/\s*$//;
    $num=$number{$name};
    $sects{$num}=&todec($sect);
    $eff{$num}=&todec($eff);
    $civ{$num}=&todec($civ);
    $mil{$num}=&todec($mil);
    $shell{$num}=&todec($shell);
    $gun{$num}=&todec($gun);
    $petrol{$num}=&todec($pet);
    $iron{$num}=&todec($iron);
    $dust{$num}=&todec($dust);
    $oil{$num}=&todec($oil);
    $plane{$num}=&todec($plane);
    $ship{$num}=&todec($ship);
    $unit{$num}=&todec($unit);
    $money{$num}=&todec($money);
  }
}

sub parse_country {
  if (/\s*(\d+)\s+.+\s+DEITY\s+(\S.*)\s*$/) {
    $number{$2}=$1; $country{$1}=$2; $deity{$1}=$2;
  }
}

sub parse_look {
  ## mine detection from nuke sub 2s needs to be added (and the auto-des of the
  ## bmap accordingly.  Also need to add this into parse_nav.
  if (/^Your (.+) (\d+)% efficient (.*)\s*@ (\S+,\S+)\s*/) {
    local($contents, $sect)=($3, $4);
    $own{$sect}=$coun;
    $dump{$sect, "eff"}=$2;
    $dump{$sect, "des"}=$main'sectype{$1};
    if ($contents=~/with (\d+) civ/) { $dump{$sect, "civ"}=$1; }
    if ($contents=~/with (\d+) mil/) { $dump{$sect, "mil"}=$1; }
  }
  elsif (/^(.+)\s+\(#(\d+)\) (.+) (\d+)% efficient (.*)\s*@ (\S+,\S+)\s*/) {
    local($sect, $contents)=($6, $5);
    $country{$2}=$1; $number{$1}=$2; $own{$6}=$2; 
    $dump{$sect, "eff"}=$4;
    $dump{$sect, "des"}=$main'sectype{$3};
    if ($contents=~/with approx (\d+) civ/) { $dump{$sect, "civ"}=$1+1; }
    if ($contents=~/with approx (\d+) mil/) { $dump{$sect, "mil"}=$1+1; }
    local(@tmp)=localtime;
    $dump{$sect, 'date'}="$tmp[4]/$tmp[3] $tmp[2]:$tmp[1]";
  }
  elsif (/^\s*(.+)\s+\(#\s*(\d+)\)\s+(\S+\D*\d*)\s+#(\d+)\s+@\s*(\S+,\S+)\s*/) {
    $country{$2}=$1;
    $number{$1}=$2;
    if ($shipname{$3}) {
      $shipname{$4}=$3;
      $shipsect{$4}=$5;
      $shipown{$4}=$2;
      local($tmp, $min, $hour, $mday, $mon, $tmp, $tmp, $tmp, $tmp)=localtime;
      $shipdate{$4}="$mon/$mday $hour:$min";
    } elsif ($unitname{$3}) {
      $unitname{$4}=$3;
      $unitsect{$4}=$5;
      $unitown{$4}=$2;
      local($tmp, $min, $hour, $mday, $mon, $tmp, $tmp, $tmp, $tmp)=localtime;
      $unitdate{$4}="$mon/$mday $hour:$min";
    }
  }
}
 
sub parse_spy_plane {
  if (/^Your (.+) (\d+)% efficient (.*)\s*@ (\S+,\S+)\s*/) {
    local($contents, $sect)=($3, $4);
    $own{$sect}=$coun;
    $dump{$sect, "eff"}=$2;
    $dump{$sect, "des"}=$main'sectype{$1};
    if ($contents=~/with (\d+) civ/) { $dump{$sect, "civ"}=$1; }
    if ($contents=~/with (\d+) mil/) { $dump{$sect, "mil"}=$1; }
  } elsif (/^(.+)\s+\(#(\d+)\) (.+) (\d+)% efficient (.*)\s*@ (\S+,\S+)\s*/) {
    local($sect, $contents)=($6, $5);
    $country{$2}=$1; $number{$1}=$2; $own{$6}=$2; 
    $dump{$sect, "eff"}=$4;
    $dump{$sect, "des"}=$main'sectype{$3};
    if ($contents=~/with approx (\d+) civ/) { $dump{$sect, "civ"}=$1+1; }
    if ($contents=~/with approx (\d+) mil/) { $dump{$sect, "mil"}=$1+1; }
    local(@tmp)=localtime;
    $dump{$sect, 'date'}="$tmp[4]/$tmp[3] $tmp[2]:$tmp[1]";
  } elsif (/\s*(\S+,\S+)\s+(\S)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s*/) {
    $own{$1}=$3;
    $dump{$1,"des"}=$2;
    $dump{$1,"eff"}=$4;
    $dump{$1,"civ"}=$5;
    $dump{$1,"mil"}=$6;
    $dump{$1,"shell"}=$7;
    $dump{$1,"gun"}=$8;
    $dump{$1,"iron"}=$9;
    $dump{$1,"pet"}=$10;
    $dump{$1,"food"}=$11;
    local(@tmp)=localtime;
    $dump{$1, 'date'}="$tmp[4]/$tmp[3] $tmp[2]:$tmp[1]";
  } elsif (/flying over\s+(.+)\s+at\s+(\S+,\S+)/) {
    $bmap{$2}=$main'sectype{$1}; $last_overfly_sect=$2;
  } elsif (/^\s*(\d+)\s+(\d+)\s+(\S+\D*\d*)\s+(\S+,\S+)\s+(\d+)%\s*$/) {
    local($own, $unit, $type, $sect, $eff)=($1, $2, $3, $4, $5);
    $type=~s/\s*$//;
    if ($unitname{$type}) {
      $unitname{$unit}=$type; 
      $unitown{$unit}=$own; 
      $unitsect{$unit}=$sect; 
      $uniteff{$unit}=$eff;
      local($tmp, $min, $hour, $mday, $mon, $tmp, $tmp, $tmp, $tmp)=localtime;
      $unitdate{$unit}="$mon/$mday $hour:$min";
    } elsif ($shipname{$type}) {
      $shipname{$unit}=$type; 
      $shipown{$unit}=$own;
      $shipsect{$unit}=$sect;
      $shipeff{$unit}=$eff;
      local($tmp, $min, $hour, $mday, $mon, $tmp, $tmp, $tmp, $tmp)=localtime;
      $shipdate{$unit}="$mon/$mday $hour:$min";
    }
  } elsif (/\s*\(#\s*(\d+)\)\s+(.+)\s+(minimally|partially|moderately|completely)\s+(\S+\D*\d*)\s*$/) {
    local($unit, $countryname, $eff, $type)=($1, $2, $3, $4);
    $countryname=~s/\s*$//; $type=~s/\s*$//;
    local($tmp, $min, $hour, $mday, $mon, $tmp, $tmp, $tmp, $tmp)=localtime;
    if ($unitname{$type}) {
      $uniteff{$unit}=25 if $eff eq 'minimally';  
      $uniteff{$unit}=50 if $eff eq 'partially';  
      $uniteff{$unit}=75 if $eff eq 'moderately'; 
      $uniteff{$unit}=100 if $eff eq 'completely'; 
      $unitown{$unit}=$number{$countryname};
      $unitsect{$unit}=$last_overfly_sect;
      $unitname{$unit}=$type;
      $unitdate{$unit}="$mon/$mday $hour:$min";
    } elsif ($planename{substr($type,0,19)}) {
      $planeeff{$unit}=25 if $eff eq 'minimally';  
      $planeeff{$unit}=50 if $eff eq 'partially';  
      $planeeff{$unit}=75 if $eff eq 'moderately'; 
      $planeeff{$unit}=100 if $eff eq 'completely'; 
      $planeown{$unit}=$number{$countryname};
      $planesect{$unit}=$last_overfly_sect;
      $planename{$unit}=substr($type,0,19);
      $planedate{$unit}="$mon/$mday $hour:$min";
    } elsif ($shipname{$type}) {
      $shipeff{$unit}=25 if $eff eq 'minimally';  
      $shipeff{$unit}=50 if $eff eq 'partially';  
      $shipeff{$unit}=75 if $eff eq 'moderately'; 
      $shipeff{$unit}=100 if $eff eq 'completely'; 
      $shipown{$unit}=$number{$countryname};
      $shipsect{$unit}=$last_overfly_sect;
      $shipname{$unit}=$type;
      $shipdate{$unit}="$mon/$mday $hour:$min";
    }
  }
}
 
sub parse_spy_report {
  if (/\s*(\S+,\S+)\s+(\S)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s*/) {
    $own{$1}=$3;
    $dump{$1,"des"}=$2;
    $dump{$1,"eff"}=$4;
    $dump{$1,"civ"}=$5;
    $dump{$1,"mil"}=$6;
    $dump{$1,"shell"}=$7;
    $dump{$1,"gun"}=$8;
    $dump{$1,"iron"}=$9;
    $dump{$1,"pet"}=$10;
    $dump{$1,"food"}=$11;
    local(@tmp)=localtime;
    $dump{$1, 'date'}="$tmp[4]/$tmp[3] $tmp[2]:$tmp[1]";
  } elsif (/$Spies report .+ unit in (\S+,\S+): (.+) #(\d+) (.*)/) {
    $unitown{$3}=$own{$1};
    $unitsect{$3}=$1;
    $unitname{$3}=$2;
    local($unit, $rest)=($3, $4);
    if ($rest=~/efficiency (\d+)[,\)]/) { $uniteff{$unit}=$1; }      
    if ($rest=~/tech (\d+)\)/) { $unittech{$unit}=$1; }      
    local($tmp, $min, $hour, $mday, $mon, $tmp, $tmp, $tmp, $tmp)=localtime;
    $unitdate{$unit}="$mon/$mday $hour:$min";
  }
}
 
sub parse_satellite {
  if (/\s*(\S+,\S+)\s+(\S)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s*/) {
    $own{$1}=$3;
    $dump{$1,"des"}=$2;
    $dump{$1,"eff"}=$4;
    $dump{$1,"civ"}=$5;
    $dump{$1,"mil"}=$6;
    $dump{$1,"shell"}=$7;
    $dump{$1,"gun"}=$8;
    $dump{$1,"iron"}=$9;
    $dump{$1,"pet"}=$10;
    $dump{$1,"food"}=$11;
    local(@tmp)=localtime;
    $dump{$1, 'date'}="$tmp[4]/$tmp[3] $tmp[2]:$tmp[1]";
  }
  elsif (/^\s*(\d+)\s+(\d+)\s+(\S+\D*\d*)\s+(\S+,\S+)\s+(\d+)%\s*$/) {
    local($own, $unit, $type, $sect, $eff)=($1, $2, $3, $4, $5);
    $type=~s/\s*$//;
    if ($unitname{$type}) {
      $unitname{$unit}=$type; $unitown{$unit}=$own; 
      $unitsect{$unit}=$sect; $uniteff{$unit}=$eff;
      local($tmp, $min, $hour, $mday, $mon, $tmp, $tmp, $tmp, $tmp)=localtime;
      $unitdate{$unit}="$mon/$mday $hour:$min";
    } elsif ($shipname{$type}) {
      $shipname{$unit}=$type; $shipown{$unit}=$own; 
      $shipsect{$unit}=$sect; $shipeff{$unit}=$eff;
      local($tmp, $min, $hour, $mday, $mon, $tmp, $tmp, $tmp, $tmp)=localtime;
      $shipdate{$unit}="$mon/$mday $hour:$min";
    }
  }
}

sub parse_dump {
  local($i,@F,$x,$y);
 
  if (/x y des sdes/) {
    @header = split(/[ \n]+/, $_);
    for ($i=0;$i<=$#header;$i++) {
      if ($header[$i] eq 'x') {
	$xpos=$i;
	$ypos=$i + 1;
	last;
      }
    }
  } else {
    @F = split(/[ \n]+/, $_);
    if ($#header != $#F) {
      if ($#header>0) { undef @header; undef $xpos; undef $ypos; }
      return;
    }
    $x = $F[$xpos];
    $y = $F[$ypos];
    for ($i=0;$i<=$#F;$i++) {
      $dump{$x,$y,$header[$i]} = $F[$i];
    }
## Isn't this a bit backwards?  Shouldn't we be check for defined own for oldown
## before setting it to 0

    if ($dump{$x,$y,'*'} eq '*') {
      $oldown{$x,$y} = 0;
    }  elsif (defined $dump{$x,$y,'own'}) {
      $oldown{$x,$y} = $dump{$x,$y, 'own'};
    } else {
      $oldown{$x,$y} = $coun;
    }
    $own{$x,$y} = $coun;
  }
}
 
sub parse_level {
  if (/^\s*(\S+,\S+)\s+(\S)\s+(\S+),(\S+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s*/) {
    $dump{$1, 'des'}=$2;
    $dump{$1, 'dist_x'}=$3;
    $dump{$1, 'dist_y'}=$4;
    $dump{$1, 'c_dist'}=$5;
    $dump{$1, 'm_dist'}=$6;
    $dump{$1, 'u_dist'}=$7;
    $dump{$1, 'f_dist'}=$8;
    $dump{$1, 's_dist'}=$9;
    $dump{$1, 'g_dist'}=$10;
    $dump{$1, 'p_dist'}=$11;
    $dump{$1, 'i_dist'}=$12;
    $dump{$1, 'd_dist'}=$13;
    $dump{$1, 'b_dist'}=$14;
    $dump{$1, 'o_dist'}=$15;
    $dump{$1, 'l_dist'}=$16;
    $dump{$1, 'h_dist'}=$17;
    $dump{$1, 'r_dist'}=$18;
    $dump{$1, 'own'}=$main'coun;
    $own{$1}=$main'coun;
  }
}

sub parse_ship_build {
  if (/^(\S+\D*\d*) +(\d+) +(\d+) +(\d+) +(\d+) \$(\d+)$/) {
    local($lcm, $hcm, $avail, $tech, $cost)=($2, $3, $4, $5, $6);
    local($type)=$1; $type=~s/\s*$//;
    $shipname{$type}=$type;
    $buildlcm{$type}=$lcm;
    $buildhcm{$type}=$hcm;
    $buildavail{$type}=$avail;
    $buildtech{$type}=$tech;
    $buildcost{$type}=$cost;
  }
}
 
sub parse_capabilities {
  local (@F,$name,$tmp);
  if (/^(\S+( {1,2}\S+)*)  +/) {
    $name=$1;
    @F=split(' ',$');
    $capability{$name}="";
    while ($tmp=pop(@F)) {
      if ($tmp =~ /(\d+)([a-z])/) { $cargo{$name,$commstr{$2}}=$1; }
      else { $capability{$name} .= " $tmp"; }
    }
  }
}
 
sub parse_ship_stats {
  if (/^(\S+\D*\d*) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+)\/(\d+)/) {
    local($def, $spd, $vis, $spy, $frg, $fir, $lnd, $pln, $hel, $xpl, $car, $use)=($2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13);
    local($type)=$1; $type=~s/\s*$//;
    $shipdef{$type} = $def;
    $shipspd{$type} = $spd;
    $visibility{$type} = $vis;
    $shipspy{$type} = $spy;
    $shipfrg{$type} = $frg;
    $shipfir{$type} = $fir;
    $shiplnd{$type} = $lnd;
    $shippln{$type} = $pln;
    $shiphel{$type} = $hel;
    $shipxpl{$type} = $xpl;
    $fuelcar{$type} = $car;
    $fueluse{$type} = $use;
  }
}
 
sub parse_land_stats {
  if (/^(\S+\D*\d*)  +([\d\.]+) +([\d\.]+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+)/) {
    local($att, $def, $vul, $spd, $vis, $spy, $rad, $frg, $acc, $dam, $amm, $aaf, $car, $use, $xpl)=($2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16);
    local($type)=$1; $type=~s/\s*$//;
    $unitatt{$type} = $att;
    $unitdef{$type} = $def;
    $unitvul{$type} = $vul;
    $unitspd{$type} = $spd;
    $visibility{$type} = $vis;
    $unitspy{$type} = $spy;
    $unitrad{$type} = $rad;
    $unitfrg{$type} = $frg;
    $unitacc{$type} = $acc;
    $unitdam{$type} = $dam;
    $unitamm{$type} = $amm;
    $unitaaf{$type} = $aaf;
    $fuelcar{$type} = $car;
    $fueluse{$type} = $use;
    $unitxpl{$type} = $xpl;
  }
}
 
sub parse_land_build {
  if (/^(\S+\D*\d*)  +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +\$(\d+)/) {
    local($lcm, $hcm, $mil, $gun, $shell, $avail, $tech, $cost)=($2, $3, $4, $5, $6, $7, $8, $9);
    local($type)=$1; $type=~s/\s*$//;
    $unitname{$type}=$type;
    $buildlcm{$type}=$lcm;
    $buildhcm{$type}=$hcm;
    $buildmil{$type}=$mil;
    $buildgun{$type}=$gun;
    $buildshell{$type}=$shell;
    $buildavail{$type}=$avail;
    $buildtech{$type}=$tech;
    $buildcost{$type}=$cost;
  }
}
 
 
sub parse_plane_build {
  if (/^\s*(\S+\D*\d*) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +\$(\d+)/){
    local($lcm, $hcm, $mil, $avail, $tech, $cost)=($2, $3, $4, $5, $6, $7);
    local($pname)=$1; $pname=~s/\s*$//;
    local($type)=substr($pname,0,19); $type=~s/\s*$//;
    $planename{$type}=$pname;
    $buildlcm{$pname}=$lcm;
    $buildhcm{$pname}=$hcm;
    $buildmil{$pname}=$mil;
    $buildavail{$pname}=$avail;
    $buildtech{$pname}=$tech;
    $buildcost{$pname}=$cost;
  }
}
 
 
sub parse_plane_stats {
  if (/^(\S+\D*\d*)  +([\d-]+) +([\d-]+) +([\d-]+) +([\d-]+) +([\d-]+) +([\d-]+) +(\d+)\%/) {
    local($acc, $load, $att, $def, $ran, $fuel, $stlth)=($1, $2, $3, $4, $5, $6, $7, $8);
    local($type)=$1; $type=~s/\s*$//;
    $planeacc{$type}=$acc;
    $planeload{$type}=$load;
    $planeatt{$type}=$att;
    $planedef{$type}=$def;
    $planeran{$type}=$ran;
    $planefuel{$type}=$fuel;
    $planestlth{$type}=$stlth;
  }
}
 
sub parse_land {
  if (/^ *(\d+) +(\S+\D*\d*)  +(\S+,\S+) +(\S?) +(\d+)\% +(\d+) +(\S+) +(\d+) +(\d+) +(\d+) +(\d+)\% +(\d+) +(\d+) *(\d*)/) {
    $unitown{$1}=$coun;
    $unitsect{$1}=$3;
    $unitarmy{$1}=$4;
    $uniteff{$1}=$5;
    $unitfort{$1}=$6;
    $unitmob{$1}=$7;
    $unitcargo{$1,"food"}=$8;
    $unitfuel{$1}=$9;
    $unittech{$1}=$10;
    $unitretr{$1}=$11;
    $unitrad{$1}=$12;
    $unitxpl{$1}=$13;
    if (defined $14) { $unitship{$1}=$14; }
    local($unit)=$1;
    $unitname{$1}=$2;
    $unitname{$unit}=~s/\s*$//;
  }
}
 
 
sub parse_ship {
  if (/^ *(\d+) +(\S+\D*\d*)  +(\S+,\S+) +(\S?) +(\d+)\% +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\S+) +(\d+) +(\d+)/) {
    $shipsect{$1}=$3;
    $shipflt{$1}=$4;
    $shipeff{$1}=$5;
    $shipcargo{$1,"civ"}=$6;
    $shipcargo{$1,"mil"}=$7;
    $shipcargo{$1,"uw"}=$8;
    $shipcargo{$1,"food"}=$9;
    $shippln{$1}=$10;
    $shiphel{$1}=$11;
    $shipxpl{$1}=$12;
    $shiplnd{$1}=$13;
    $shipmob{$1}=$14;
    $shipfuel{$1}=$15;
    $shiptech{$1}=$16;
    local($unit)=$1;
    $shipname{$1}=$2;
    $shipname{$unit}=~s/\s*$//;
    $shipown{$unit}=$main'coun;
  }
}
 
sub parse_cargo {
  if (/lnd#/) { $cargo_type=1; }	# lcargo output
  else { $cargo_type=0; }		# cargo output
  if (/^ *(\d+) +([a-z]+( \w+)*)  +(\S+,\S+) +(\S?) +(\d+)\% +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+)/) {
    if ($cargo_type) {
      $unitname{$1}=$2;
      $unitsect{$1}=$4;
      $unitflt{$1}=$5;
      $uniteff{$1}=$6;
      $unitcargo{$1,"shell"}=$7;
      $unitcargo{$1,"gun"}=$8;
      $unitcargo{$1,"pet"}=$9;
      $unitcargo{$1,"iron"}=$10;
      $unitcargo{$1,"dust"}=$11;
      $unitcargo{$1,"bar"}=$12;
      $unitcargo{$1,"oil"}=$13;
      $unitcargo{$1,"lcm"}=$14;
      $unitcargo{$1,"hcm"}=$15;
      $unitcargo{$1,"rad"}=$16;
    } else {
      $shipname{$1}=$2;
      $shipsect{$1}=$4;
      $shipflt{$1}=$5;
      $shipeff{$1}=$6;
      $shipcargo{$1,"shell"}=$7;
      $shipcargo{$1,"gun"}=$8;
      $shipcargo{$1,"pet"}=$9;
      $shipcargo{$1,"iron"}=$10;
      $shipcargo{$1,"dust"}=$11;
      $shipcargo{$1,"bar"}=$12;
      $shipcargo{$1,"oil"}=$13;
      $shipcargo{$1,"lcm"}=$14;
      $shipcargo{$1,"hcm"}=$15;
    }
  }
}
 
 
sub parse_plane {
  if (/^\s*(\d+)\s+(\S+\D*\d*)\s+(\S+,\S+)\s+(\S?)\s+(\d+)\%\s+(\S+)\s+(\S+)\s+([\d-]+)\s+(\d+)\s+(\d+)\s+(\d+)\s*(.*)/) {
    local($unit, $sect, $wing, $eff, $mob, $att, $def, $tech, $ran, $hard, $sln)=($1, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12);
    local($type)=substr($2,0,19); $type=~s/\s*$//;
    $planename{$unit}=$planename{$type};
    $planesect{$unit}=$sect;
    $planewing{$unit}=$wing;
    $planeeff{$unit}=$eff;
    $planemob{$unit}=$mob;
    $planeatt{$unit}=$att;
    $planedef{$unit}=$def;
    $planetech{$unit}=$tech;
    $planeran{$unit}=$ran;
    $planehard{$unit}=$hard;
    $planesln{$unit}=$sln;
    $planeown{$unit}=$main'coun;
  }
}
 
sub parse_mission {
  local($type, $unit, $sect, $missioninfo, $op, $rad, $mission);
 
  if (/\s*(\S+\D*\d*)\s+#(\d+)\s+(\S+,\S+)\s+(.*)$/) {
    $type=$1; $unit=$2; $sect=$3; $missioninfo=$4; $type=~s/\s*$//;
    if ($missioninfo=~/has no mission/) {
      $mission=''; $op=''; $rad='';
    } elsif ($missioninfo=~/(\S+,\S+)\s+(\d+)\s+is on an* (\S+\s*\S*) mission/) {
      $op=$1; $rad=$2; $mission=substr($3,0,1);
    }
    if ($unitname{$unit} eq $type) {
      $unitsect{$unit}=$sect;
      $unitmission{$unit}=$mission;
      $unitopsect{$unit}=$op;
      $unitmissionradius{$unit}=$rad;
    } elsif ($shipname{$unit} eq $type) {
      $shipsect{$unit}=$sect;
      $shipmission{$unit}=$mission;
      $shipopsect{$unit}=$op;
      $shipmissionradius{$unit}=$rad;
    } elsif ($planename{$unit} eq substr($type,0,19)) {
      $planesect{$unit}=$sect;
      $planemission{$unit}=$mission;
      $planeopsect{$unit}=$op;
      $planemissionradius{$unit}=$rad;
    }
  }
}
 
sub parse_build {
  local($tech, $unit, $type, $sect);
 
  if (/building with tech level (\d+)\./) {
    $tech=$1;
  } elsif (/\s*(\S+\D*\d*) +#(\d+) +built in sector (\S+,\S+)/) {
    $type=$1; $unit=$2; $sect=$3; $type=~s/\s*$//;
    if ($planename{$type}) {
      $planename{$unit}=substr($type,0,19);
      $planename{$unit}=~s/\s*$//;
      $planesect{$unit}=$sect; $planewing{$unit}=''; $planeeff{$unit}=10;
      $planemob{$unit}=0; $planetech{$unit}=$tech; $planehard{$unit}=0;
      $planesln{$unit}='';
      $dump{$sect, "avail"}-=$buildavail{$type};
      $money{$coun}-=$buildcost{$type}*0.1;
    } elsif ($shipname{$type}) {
      $shipname{$unit}=$type; $shipsect{$unit}=$sect; $shipflt{$unit}='';
      $shipeff{$unit}=20; $shipcargo{$unit, "civ"}=0;
      $shipcargo{$unit, "mil"}=0; $shipcargo{$unit, "uw"}=0;
      $shipcargo{$unit, "food"}=0; $shipcargo{$unit, "shell"}=0;
      $shipcargo{$unit, "gun"}=0; $shipcargo{$unit, "pet"}=0;
      $shipcargo{$unit, "iron"}=0; $shipcargo{$unit, "dust"}=0;
      $shipcargo{$unit, "bar"}=0; $shipcargo{$unit, "oil"}=0;
      $shipcargo{$unit, "lcm"}=0; $shipcargo{$unit, "hcm"}=0;
      $shipcargo{$unit, "rad"}=0; $shippln{$unit}=0; $shiphel{$unit}=0;
      $shipxpl{$unit}=0; $shiplnd{$unit}=0; $shipmob{$unit}=0;
      $shipfuel{$unit}=$fuelcar{$type}; $shiptech{$unit}=$tech;
      $dump{$sect, "avail"}-=$buildavail{$type};
    } elsif ($unitname{$type}) {
      $unitname{$unit}=$type;
      $unitown{$unit}=$coun; $unitsect{$unit}=$sect; $unitarmy{$unit}='';
      $uniteff{$unit}=10; $unitfort{$unit}=0; $unitmob{$unit}=0;
      $unitfuel{$unit}=$fuelcar{$type}; $unittech{$unit}=$tech;
      $unitxpl{$unit}=0; $unitship{$unit}='';
      $dump{$sect, "avail"}-=$buildavail{$type};
      $money{$coun}-=$buildcost{$type}*0.1;
    }
  } elsif (/Bridge span built over (\S+,\S+)\s*/) {
    $own{$1}=0;
    $dump{$1, "des"}='='; $dump{$1, "own"}=0; $dump{$1, "mil"}=0;
    $dump{$1, "civ"}=0; $dump{$1, "uw"}=0; $dump{$1, "gun"}=0;
    $dump{$1, "shell"}=0;
    $dump{$bridgesect, "avail"}-=$bridgeavail;
    $dump{$bridgesect, "hcm"}-=$bridgehcm;
  } elsif (/Bridge head at (\S+,\S+);/) { $bridgesect=$1; }
  elsif (/That just cost you \$(\d+\.\d+)\s*/) {
    $money{$coun}-=$1;
  }
}
 
sub parse_map {
# Internal variables: 
#   @H		Header lines of the map.
#   %mapcol		The x coordinates.
#   $mapheader	Set to 1 if the header has already been read.
#   $maprow		The current y coordinate.
#   $mappartial	Contains the map data minus the y coordinate.
  local($x, $y, $digit);
  if ($mapheader==0) {
    if (/\s*\S+\s+\S+/) { 
      $mapheader=1;
      for($x=0;$x<length($H[0]); $x++) {
        $mapcol[$x]=0;
	for($y=0;$y<=$#H;$y++) { 
	  $digit=substr($H[$y], $x, 1);
	  if ($digit=~/\d/) {
	    $mapcol[$x]=$mapcol[$x]*10+substr($H[$y],$x,1); 
          }
        }
      }
      for($x=0;$x<$#mapcol;$x++) {
        if ($mapcol[$x]>$mapcol[$x+1]) { $mapcol[$x]*=-1; }
      }
      if ($mapcol[$#mapcol-1]<0 && $mapcol[$#mapcol]<(-1*$mapcol[$#mapcol-1])) {
        $mapcol[$#mapcol]*=-1;
      }
      undef @H; 
    } else {
      undef @mapcol;
      push(@H, $_);
      $H[$#H]=~s/^\s*//; $H[$#H]=~s/\s*$//;
    }
  }
  if (/\s*(\S+) (.+) \S+\s*$/ && $mapheader==1) { 
    $maprow=$1; $mappartial=$2;
    for($x=0; $x<=$#mapcol; $x++) { 
      if (substr($mappartial, $x, 1) cmp ' ') {
        $bdes{"$mapcol[$x],$maprow"}=substr($mappartial, $x, 1);
      }
    }
  } else { $mapheader=0; }
}
    
sub parse_sector_stats {
  if (/(\S) (\S+( \S+)*)  +(\d+) +(\d+) +(\d+)/) {
    $main'sectype{$2}=$1;
    $movcost{$1}=$4;
    $secoff{$1}=$5;
    $secdef{$1}=$6;
  }
}

sub parse_sector_build {
  if (/(\S)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s*/) {
    $descost{$1}=$2;
    $buildcost{$1}=$3;
    $buildlcm{$1}=$4;
    $buildhcm{$1}=$5;
  }
}
 
 
sub parse_attack {
  if (/(\S+,\S+) is a (\d+)% (.+) with approximately (\d+) military\.$/) {
    local($thiscoun, $eff, $mil)=($3, $2, $4);
    $last_attacked_sector=$1;
    local(@info)=split(/ /, $3);
    if ($main'sectype{$info[$#info]}) { 
      local($type)=pop(@info);
      $dump{$last_attacked_sector, 'des'}=$main'sectype{$type}; 
    } else { 
      local($type)=pop(@info); $type=pop(@info) . " " . $type;
      $dump{$last_attacked_sector, 'des'}=$main'sectype{$type}; 
    }
    $thiscoun=join(' ', @info);
    $own{$last_attacked_sector}=$number{$thiscoun};
    if (!$own{$last_attacked_sector} || $own{$last_attacked_sector}==$coun) {
      $own{$last_attacked_sector} = -1;
    }
    $dump{$last_attacked_sector,"mil"} = $mil+1;
    $dump{$last_attacked_sector,"eff"} = $eff;
    local(@tmp)=localtime();
    $dump{$last_attacked_sector, 'date'}="$tmp[4]/$tmp[3] $tmp[2]:$tmp[1]";
  } elsif (/^\s*defender\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s*/) {
    $def{$last_attacked_sector, 'fort'}=$1;
    $def{$last_attacked_sector, 'ship'}=$2;
    $def{$last_attacked_sector, 'land'}=$3;
    $def{$last_attacked_sector, 'plane'}=$4;
  } elsif (/^Final defender strength (\d+)/) {
    $defense_strength{$last_attacked_sector} = $1;
  } elsif (/^\S+\D*\d* #(\d+) occupies (\S+,\S+)\s*/) {
    $unitsect{$1}=$2; 
  } elsif (/^Scouts sight new enemy unit: (\S+\D*\d*) #(\d+)(.*)/) {
    $unitname{$2}=$1;
    $unitown{$2}=$own{$last_attacked_sector};
    $unitnear{$2}=$last_attacked_sector;
    local($unit, $rest)=($2, $3);
    if ($rest=~/efficiency (\d+)[\,\)]/) { $uniteff{$unit}=$1; }
    if ($rest=~/tech (\d+)\)/) { $unittech{$unit}=$1; }
  } elsif (/^Scouts report enemy unit: (\S+\D*\d*) #(\d+)(.*)/) {
    $unitname{$2}=$1;
    $unitown{$2}=$own{$last_attacked_sector};
    $unitsect{$2}=$last_attacked_sector;
    local($unit, $rest)=($2, $3);
    if ($rest=~/efficiency (\d+)[\,\)]/) { $uniteff{$unit}=$1; }
    if ($rest=~/tech (\d+)\)/) { $unittech{$unit}=$1; }
  } elsif (/^\s*\S+\D*\d* #(\d+) dies attacking/) {
    delete $unitname{$1}; delete $unitown{$1}; delete $unitarmy{$1};
    delete $uniteff{$1}; delete $unitfort{$1}; delete $unitmob{$1};
    delete $unitfuel{$1}; delete $unittech{$1}; delete $unitretr{$1};
    delete $unitrad{$1}; delete $unitxpl{$1}; delete $unitnear{$1};
  } elsif (/\s*(.+) #(\d+) retreats at (\d+)% efficiency to (\S+,\S+)!\s*/) {
    $unitname{$2}=$1;
    $uniteff{$2}=$3;
    $unitsect{$2}=$4;
    $unitown{$2}=$coun;
    local($tmp, $min, $hour, $mday, $mon, $tmp, $tmp, $tmp, $tmp)=localtime;
    $unitdate{$2}="$mon/$mday $hour:$min";
  } elsif (/(\d+) of your troops now occupy (\S+,\S+)\s*/) {
    $own{$2}=$main'coun;
    $dump{$2, 'mil'}=$1;
  }
}
 
 
sub parse_read {
# should the mines be in a BULLETIN??
  if (/^Kawhomp! Mine detected by (.+) \(#(\d+)\) in (\S+,\S+)!/) {
      $mine{$3} = 20;
  }
  if ($teletype==0) {
    if (/^> BULLETIN.+dated \S+\s+(\S+)\s+(\S+)\s+(\d+):(\d+):.+/) {
      $teletype=1;
      $bullet_month=$main'month{$1};
      $bullet_day=$2;
      $bullet_hour=$3;
      $bullet_min=$4;
    }
  } elsif ($teletype==1) {
    &parse_bulletin;
  }
}

sub parse_bulletin {
  if (/^> /) { $teletype=0; &parse_read; return; }
  elsif (/^(.+) \(#([0-9]+)\) lost ([0-9]+) troops taking (\S+,\S+)/) {
    ($mon, $mday, $hour, $min)=($dump{$sect, 'date'}=~/(\d+)\/(\d+) (\d+):(\d+)/);
    if (!($mon) || ($mon<=$bullet_month && $mday<=$bullet_day && $hour<=$bullet_hour && $min<=$bullet_min)) {
      $dump{$2, 'date'}="$bullet_month/$bullet_day $bullet_hour:$bullet_min";
      $country{$2}=$1;
      $number{$1}=$2;
      $own{$4}=$2;
    }
    while($#aunits>-1) {
      $unitown{$aunits[0]}=$2;
      $unitnear{$aunits[0]}=$4;
      shift(@aunits);
    }
  } elsif (/^(.+) \(#(\d+)\) lost (\d+) troops attacking (\S+,\S+)/) {
    while($#aunits>-1) {
      $unitown{$aunits[0]}=$2;
      $unitnear{$aunits[0]}=$4;
      shift(@aunits);
    }
  } elsif (/^Scouts report attacking unit: (.+) #(\d+)(.*)/) {
    local($type, $unit, $rest)=($1, $2, $3);
    ($mon, $mday, $hour, $min)=($unitdate{$unit}=~/(\d+)\/(\d+) (\d+):(\d+)/);
    if (!($mon) || ($mon<=$bullet_month && $mday<=$bullet_day && $hour<=$bullet_hour && $min<=$bullet_min)) {
      $type=~s/\s*$//;
      $unitname{$unit}=$type;
      if ($rest=~/efficiency (\d+)[,\)]/) { $uniteff{$unit}=$1; }
      if ($rest=~/tech (\d+)\)/) { $unittech{$unit}=$1; }
      push(@aunits, $unit);
      $unitdate{$2}="$bullet_month/$bullet_day $bullet_hour:$bullet_min";
    }
  } elsif (/\s*(.+) #(\d+) retreats at (\d+)% efficiency to (\S+,\S+)!\s*/) {
    local($type, $unit, $eff, $sect)=($1, $2, $3, $4);
    ($mon, $mday, $hour, $min)=($unitdate{$unit}=~/(\d+)\/(\d+) (\d+):(\d+)/);
    if (!($mon) || ($mon<=$bullet_month && $mday<=$bullet_day && $hour<=$bullet_hour && $min<=$bullet_min)) {
      $unitname{$2}=$1;
      $uniteff{$2}=$3;
      $unitsect{$2}=$4;
      $unitown{$2}=$coun;
      $unitdate{$2}="$bullet_month/$bullet_day $bullet_hour:$bullet_min";
    }
  } elsif (/^\s*Diplomatic relations with (.+) downgraded to \"(.+)\"/) {
    $relation{$coun, $1}=$2;
  } elsif (/^\s*.+ \(#(\d+)\) .+ -- shot down/) {
    local($unit)=$1;
    ($mon, $mday, $hour, $min)=($planedate{$unit}=~/(\d+)\/(\d+) (\d+):(\d+)/);
    if (!($mon) || ($mon<=$bullet_month && $mday<=$bullet_day && $hour<=$bullet_hour && $min<=$bullet_min)) {
      delete $planename{$unit}; delete $planeeff{$unit}; 
      delete $planesect{$unit}; delete $planewing{$unit}; 
      delete $planemob{$unit}; delete $planetech{$unit};
      delete $planehard{$unit}; delete $planesln{$unit}; 
      delete $planeatt{$unit}; delete $planedef{$unit}; 
      delete $planeran{$unit}; delete $planeown{$unit};
      delete $planenear{$unit}; delete $planedate{$unit};
    }
  } elsif (/firing \d+ flak guns in (\S+,[^.]+)\.+/) {
    $last_flak_sector=$1;
  } elsif (/\s*(.+)\s+\(#\s*(\d+)\)\s+takes \d+/) {
    local(@tmp)=split(/ /, $1); local($unit)=$2;
    local($pilot)=join(' ', shift(@tmp));
    while (!(defined($number{$pilot})) && $#tmp > -1) { $pilot=join(' ', shift(@tmp)); }
    local($type)=join(' ', @tmp); $type=substr($type,0,19);
    ($mon, $mday, $hour, $min)=($planedate{$unit}=~/(\d+)\/(\d+) (\d+):(\d+)/);
    if (!($mon) || ($mon<=$bullet_month && $mday<=$bullet_day && $hour<=$bullet_hour && $min<=$bullet_min)) {
      $planeown{$unit}=$number{$pilot};
      $planename{$unit}=$type;
      $planenear{$unit}=$last_flak_sector;
      $planedate{$unit}="$bullet_month/$bullet_day $bullet_hour:$bullet_min";
    }
  } elsif (/^.+ turned (\S+,\S+) into a radioactive wasteland/) {
    $dump{$1, 'own'}=0; $dump{$1, 'des'}='\\'; $own{$1}=0;
    $dump{$1, 'date'}="$bullet_month/$bullet_day $bullet_hour:$bullet_min";
  } elsif (/^\s*(.+) #(\d+) fires at you/) {
    local(@info)=split(/ /, $1); local($unit)=$2;
    local($countryname)=shift(@info);
    while(1) {
      if ($number{$countryname}) { last; }
      else { $countryname=$countryname . " " . shift(@info); }
    }
    $type=join(' ', @info);
    ($mon, $mday, $hour, $min)=($shipdate{$unit}=~/(\d+)\/(\d+) (\d+):(\d+)/);
    if (!($mon) || ($mon<=$bullet_month && $mday<=$bullet_day && $hour<=$bullet_hour && $min<=$bullet_min)) {
      if ($shipname{$type}) {
	$shipown{$unit}=$number{$countryname};
	$shipname{$unit}=$type;
	$shipdate{$unit}="$bullet_month/$bullet_day $bullet_hour:$bullet_min";
      }
    }
  }
  elsif (/Bridge falls at (\S+,\S+)\s*/) {
    $own{$1}=0; $dump{$1, "des"}='.';
  }
}
 
sub parse_census {
  if (/^\s*(\S+,\S+) +(\S+) +(\d+)\% +(no)? +(\d+) .. .. (\S?) +(\d+) +(\d+) +(\d+) +(\d+) +(\d+)\% +(\d+)( +\d+)?$/) {
    $own{$1}=$coun;	
    $dump{$1,'des'}=substr($2,0,1);
    if (substr($2,1,1)) {
      $dump{$1,'sdes'}=substr($2,1,1);
    } else {
      $dump{$1,'sdes'}='_';
    }
    $dump{$1,'eff'} = $3;
    $dump{$1,'off'} = 1 if ($4);
    $dump{$1,'mob'} = $5;
    if ($6) {
      $oldown{$1} = -1;
    } else {
      $oldown{$1} = $own{$1};
    }
    $dump{$1,'civ'} = $7;
    $dump{$1,'mil'} = $8;
    $dump{$1,'uw'} = $9;
    $dump{$1,'food'} = $10;
    $dump{$1,'work'} = $11;
    $dump{$1,'avail'} = $12;
    $dump{$1,'terr'} = int($13);
  }
}
 
sub parse_resource {
  if (/\s*(\S+,\S+)\s+(\S)\s+(\d+)%\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d*)\s*/) {
    $dump{$1, 'des'}=$2; $dump{$1, 'eff'}=$3; $dump{$1, 'min'}=$4;
    $dump{$1, 'gold'}=$5; $dump{$1, 'fert'}=$6; $dump{$1, 'ocontent'}=$6; 
    $dump{$1, 'uran'}=$7; $dump{$1, 'terr'}=$8;
  }
}
 
sub parse_move {
  if (/^(\d+) +mob left in (\S+,\S+)\s*/) {
    $dump{$2, 'mob'}=$1;
  }
}
 
sub parse_explore {
  if (/\s+\S+\s+\S+\s+\S+\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s*/) {
    $expmin=$1; $expgold=$2; $expfert=$3; $expoil=$4; $expuran=$5;
  } elsif (/<\d+\.\d+:\s*(\S)\s+(\S+,\S+)>/) {
    $dump{$2, 'des'}=$1; $dump{$2, 'min'}=$expmin; $dump{$2, 'gold'}=$expgold;
    $dump{$2, 'fert'}=$expfert; $dump{$2, 'oil'}=$expoil; $dump{$2, 'uran'}=$expuran;
    undef $expmin;
    undef $expgold;
    undef $expfert;
    undef $expoil;
    undef $expuran;
  }
  elsif (/(\d+) mob left in (\S+,\S+)$/) {
    $dump{$2,'mob'} = $1;
  } elsif (/^Sector (\S+,\S+) is now yours.$/) {
    $own{$1} = $coun;
    $oldown{$1} = $coun;
    $dump{$1,'own'} = $coun;
  }
}
 
sub save_DB {
  print STDERR "Saving internal database for $main'game...";
  if (defined(%own)) {
    open(DUMP, ">$main'game.dump");
    foreach $sect (keys %own) {
      if ($own{$sect} ne $coun) {
        ($x, $y)=($sect=~/(\S+)\,(\S+)/);
        print DUMP "$x:$y:$own{$sect}:$dump{$sect, 'des'}:$dump{$sect, 'sdes'}:";
        print DUMP "$dump{$sect, 'eff'}:$dump{$sect, 'mob'}:$dump{$sect, '*'}:";
        print DUMP "$dump{$sect, 'off'}:$dump{$sect, 'min'}:$dump{$sect, 'gold'}:";
        print DUMP "$dump{$sect, 'fert'}:$dump{$sect, 'ocontent'}:";
        print DUMP "$dump{$sect, 'uran'}:$dump{$sect, 'work'}:";
        print DUMP "$dump{$sect, 'avail'}:$dump{$sect, 'terr'}:";
        print DUMP "$dump{$sect, 'civ'}:$dump{$sect, 'mil'}:$dump{$sect, 'uw'}:";
        print DUMP "$dump{$sect, 'food'}:$dump{$sect, 'shell'}:";
        print DUMP "$dump{$sect, 'gun'}:$dump{$sect, 'pet'}:$dump{$sect, 'iron'}:";
        print DUMP "$dump{$sect, 'dust'}:$dump{$sect, 'bar'}:$dump{$sect, 'oil'}:";
        print DUMP "$dump{$sect, 'lcm'}:$dump{$sect, 'hcm'}:$dump{$sect, 'rad'}:";
        print DUMP "$def{$sect, 'fort'}:$def{$sect, 'ship'}:$def{$sect, 'land'}:$def{$sect, 'plane'}:";
        print DUMP "$dump{$sect, 'date'}\n";
      }
    }
    close DUMP;
  }
  if (defined(%planeown)) {
    open(DUMP, ">$main'game.planes");
    foreach $unit (keys %planeown) {
      next if $planeown{$unit}==$main'coun;
      print DUMP "$unit:$planename{$unit}:$planesect{$unit}:$planenear{$unit}:$planewing{$unit}:";
      print DUMP "$planeeff{$unit}:$planemob{$unit}:$planetech{$unit}:";
      print DUMP "$planehard{$unit}:$planesln{$unit}:$planeatt{$unit}:";
      print DUMP "$planedef{$unit}:$planeran{$unit}:$planeown{$unit}:";
      print DUMP "$planedate{$unit}\n";
    }
    close(DUMP);
  }
  if (defined(%shipown)) {
    open(DUMP, ">$main'game.ships");
    foreach $unit (keys %shipown) {
      next if $shipown{$unit}==$main'coun;
      print DUMP "$unit:$shipname{$unit}:$shipsect{$unit}:$shipnear{$unit}:$shipflt{$unit}:";
      print DUMP "$shipeff{$unit}:$shipmob{$unit}:$shiptech{$unit}:";
      print DUMP "$shipfuel{$unit}:$shiplnd{$unit}:$shippln{$unit}:";
      print DUMP "$shiphel{$unit}:$shipxpl{$unit}:$shipown{$unit}:";
      print DUMP "$shipdate{$unit}\n";
    }
    close(DUMP);
  }
  if (defined(%unitown)) {
    open(DUMP, ">$main'game.units");
    foreach $unit (keys %unitown) {
      next if $unitown{$unit}==$main'coun;
      print DUMP "$unit:$unitname{$unit}:$unitsect{$unit}:$unitnear{$unit}:$unitarmy{$unit}:";
      print DUMP "$uniteff{$unit}:$unitmob{$unit}:$unittech{$unit}:";
      print DUMP "$unitfort{$unit}:$unitfuel{$unit}:$unitretr{$unit}:";
      print DUMP "$unitxpl{$unit}:$unitrad{$unit}:$unitown{$unit}:";
      print DUMP "$unitdate{$unit}\n";
    }
    close(DUMP);
  }
  print STDERR "done\n";
}
    
sub load_DB {
  if (-f "$main'game.dump") {
    local(@d);
    open(DUMP, "$main'game.dump");
    while(<DUMP>) {
      chop($_);
      (@d)=split(/:/, $_);
      $x=shift(@d); $y=shift(@d);
      if (!($x =~ /^-*\d+$/) || !($y =~ /^-*\d+$/)) {
	print STDERR "ERROR: Your internal database file $main'game.dump was corrupt--pei will now rename it $main'game.dump.bad and abort load_DB.\n";
	close (DUMP);
	system "mv $main'game.dump $main'game.dump.bad";
	last;
      }
      $sect="$x,$y"; $own{$sect}=shift(@d); $dump{$sect, 'own'}=$own{$sect};
      $dump{$sect, 'des'}=shift(@d); $dump{$sect, 'sdes'}=shift(@d);
      $dump{$sect, 'eff'}=shift(@d); $dump{$sect, 'mob'}=shift(@d); 
      $dump{$sect, '*'}=shift(@d); $dump{$sect, 'off'}=shift(@d); 
      $dump{$sect, 'min'}=shift(@d); $dump{$sect, 'gold'}=shift(@d);
      $dump{$sect, 'fert'}=shift(@d); $dump{$sect, 'ocontent'}=shift(@d);
      $dump{$sect, 'uran'}=shift(@d); $dump{$sect, 'work'}=shift(@d);
      $dump{$sect, 'avail'}=shift(@d); $dump{$sect, 'terr'}=shift(@d);
      $dump{$sect, 'civ'}=shift(@d); $dump{$sect, 'mil'}=shift(@d);
      $dump{$sect, 'uw'}=shift(@d); $dump{$sect, 'food'}=shift(@d);
      $dump{$sect, 'shell'}=shift(@d); $dump{$sect, 'gun'}=shift(@d);
      $dump{$sect, 'pet'}=shift(@d); $dump{$sect, 'iron'}=shift(@d);
      $dump{$sect, 'dust'}=shift(@d); $dump{$sect, 'bar'}=shift(@d);
      $dump{$sect, 'oil'}=shift(@d); $dump{$sect, 'lcm'}=shift(@d);
      $dump{$sect, 'hcm'}=shift(@d); $dump{$sect, 'rad'}=shift(@d);
      $def{$sect, 'fort'}=shift(@d); $def{$sect, 'ship'}=shift(@d);
      $def{$sect, 'land'}=shift(@d); $def{$sect, 'plane'}=shift(@d);
      $dump{$sect, 'date'}=shift(@d); $dump{$sect, 'date'}.=':';
      $dump{$sect, 'date'}.=shift(@d);
    }
    close(DUMP);
  }
  if (-f "$main'game.planes") {
    local(@p);
    open(DUMP, "$main'game.planes");
    while(<DUMP>) {
      chop($_);
      (@p)=split(/:/, $_);
      $unit=shift(@p); $planename{$unit}=shift(@p); $planesect{$unit}=shift(@p);
      $planenear{$unit}=shift(@p); $planewing{$unit}=shift(@p); $planeeff{$unit}=shift(@p); 
      $planemob{$unit}=shift(@p); $planetech{$unit}=shift(@p);
      $planehard{$unit}=shift(@p); $planesln{$unit}=shift(@p);
      $planeatt{$unit}=shift(@p); $planedef{$unit}=shift(@p);
      $planeran{$unit}=shift(@p); $planeown{$unit}=shift(@p);
      $planedate{$unit}=shift(@p); $planedate{$unit}.=':';
      $planedate{$unit}.=shift(@p);
    }
    close(DUMP);
  }
  if (-f "$main'game.ships") {
    local(@s);
    open(DUMP, "$main'game.ships");
    while(<DUMP>) {
      chop($_);
      (@s)=split(/:/, $_);
      $unit=shift(@s); $shipname{$unit}=shift(@s); $shipsect{$unit}=shift(@s);
      $shipnear{$unit}=shift(@s); $shipflt{$unit}=shift(@s); $shipeff{$unit}=shift(@s); 
      $shipmob{$unit}=shift(@s); $shiptech{$unit}=shift(@s);
      $shipfuel{$unit}=shift(@s); $shiplnd{$unit}=shift(@s); 
      $shippln{$unit}=shift(@s); $shiphel{$unit}=shift(@s);
      $shipxpl{$unit}=shift(@s); $shipown{$unit}=shift(@s);
      $shipdate{$unit}=shift(@s); $shipdate{$unit}.=':';
      $shipdate{$unit}.=shift(@s);
    }
    close(DUMP);
  }
  if (-f "$main'game.units") {
    local(@u);
    open(DUMP, "$main'game.units");
    while(<DUMP>) {
      chop($_);
      (@u)=split(/:/, $_);
      $unit=shift(@u); $unitname{$unit}=shift(@u); $unitsect{$unit}=shift(@u);
      $unitnear{$unit}=shift(@u); $unitarmy{$unit}=shift(@u); $uniteff{$unit}=shift(@u);
      $unitmob{$unit}=shift(@u); $unittech{$unit}=shift(@u);
      $unitfort{$unit}=shift(@u); $unitfuel{$unit}=shift(@u);
      $unitretr{$unit}=shift(@u); $unitxpl{$unit}=shift(@u);
      $unitrad{$unit}=shift(@u); $unitown{$unit}=shift(@u);
      $unitdate{$unit}=shift(@u); $unitdate{$unit}.=':';
      $unitdate{$unit}.=shift(@u);
    }
    close(DUMP);
  }
}
    
$main'parse_loaded = 1;
