/*
 *                        C O M M O D I T Y . C
 *
 *  implements commodity command
 *  -  init and show dialog
 *  -  send command
 *  -  interpret answer
 *
 *  Version      : $Revision: 1.6 $
 *
 *  Created      : Fri Jun 24 02:22:02 1994
 *  Author       : Ulrich Drepper <drepper@mydec>
 *
 *  Last modified: Thu Jul 14 02:01:23 1994
 *  Author       : Ulrich Drepper <drepper@mydec>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#if !defined(lint)
static const char *vcid = "$Id: commodity.c,v 1.6 1994/07/14 09:39:32 drepper Exp $";
#endif /* lint */

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Toggle.h>

#include "empire.h"

/*
 * prototypes of local functions
 */
static void commandInitCommodity(void);
static void commodityEditAdd(Bool active, XawTextBlock *textBlock, Bool async);
static void callbackCommodity(Widget widget, XtPointer closure,
			      XtPointer callData);
static void commodityNotify(Widget w, XEvent *event, String *params,
			    Cardinal *numParams);
static Bool processCommodity(Bool first, Bool last, char *str, void *data);

/*
 * local variables
 */
static Widget commodityPopup;
static Widget form;
static Widget sectorText;
static Widget okButton;
static void (*oldEditAdd)(Bool, XawTextBlock *, Bool);

/*
 * exported functions
 */
void
queryCommandCommodity(void)
{
    static Bool firstCall = True;
    Position fx, fy;
    Position x, y;

    if (firstCall) {
	commandInitCommodity();
	firstCall = False;
    }

    XtVaGetValues(
	commandForm,
	XtNx, &fx,
	XtNy, &fy,
	NULL);

    XtVaGetValues(
	horizPane,
	XtNx, &x,
	XtNy, &y,
	NULL);

    XtTranslateCoords(
	topLevel,
	x+fx, y+fy,
	&x, &y);

    XtVaSetValues(
	commodityPopup,
	XtNx, x,
	XtNy, y,
	NULL);

    /* prepare edit widget communication */
    oldEditAdd = currentEditAdd;
    oldEditAdd(False, NULL, False);

    currentEditAdd = commodityEditAdd;
    currentEditAdd(True, NULL, False);
    
    XawTextSetInsertionPoint(sectorText, 0);
    XtVaSetValues(
	sectorText,
        XtNstring, "",
        NULL);

    XtPopup(commodityPopup, XtGrabNonexclusive);
    XtAddGrab(showViewport, False, False);
}

Bool
commodityCommand(int xMin, int yMin, int xMax, int yMax, Bool redisplay)
{
    return sendCmdStr(processCommodity, (void*)redisplay, NULL,
		      "commodity %d:%d,%d:%d\n", xMin, xMax, yMin, yMax);
}

/*
 * local functions
 */
static void
commandInitCommodity(void)
{
    static String textTranslations = 
	"#override\n"
	"<Key>Return: commodityNotify()\n"
	"<Key>Linefeed: commodityNotify()\n"
	"Ctrl<Key>M: commodityNotify()\n"
	"Ctrl<Key>J: commodityNotify()\n";
    static XtActionsRec defaultAction[1] = {
	{ "commodityNotify", commodityNotify },
    };
    XtTranslations textTrans;
    Widget tmp;

    XtAppAddActions(appContext, defaultAction, 1);
    textTrans = XtParseTranslationTable(textTranslations);

    commodityPopup =  XtVaCreatePopupShell(
	"commoditypopupshell",
	topLevelShellWidgetClass,
	topLevel,
	NULL);
    form = XtVaCreateManagedWidget(
	"form",
	formWidgetClass,
	commodityPopup,
	NULL);
    tmp = XtVaCreateManagedWidget(
	"commodity",
	labelWidgetClass,
	form,
	NULL);
    tmp = XtVaCreateManagedWidget(
	"sectorlabel",
	labelWidgetClass,
	form,
	NULL);
    sectorText = XtVaCreateManagedWidget(
	"sectortext",
	asciiTextWidgetClass,
	form,
	XtNeditType, XawtextEdit,
	XtNtranslations, textTrans,
	NULL);

    okButton = XtVaCreateManagedWidget(
	"OK",
	commandWidgetClass,
	form,
	NULL);
    XtAddCallback(
	okButton,
	XtNcallback, callbackCommodity,
	(XtPointer)True);
    tmp = XtVaCreateManagedWidget(
	"Cancel",
	commandWidgetClass,
	form,
	NULL);
    XtAddCallback(
	tmp,
	XtNcallback, callbackCommodity,
	(XtPointer)False);

    XtRealizeWidget(commodityPopup);
}

/* ARGSUSED */
static void
commodityEditAdd(Bool active, XawTextBlock *textBlock, Bool async)
{
    assert(async==False);
    if (textBlock) {
	XtVaSetValues(
	    sectorText,
	    XtNstring, textBlock->ptr,
	    NULL);
	XawTextSetInsertionPoint(sectorText, strlen(textBlock->ptr));
    } else {
	XawTextDisplayCaret(sectorText, active);
	if (active) {
	    XtSetKeyboardFocus(form, sectorText);
	    XtSetKeyboardFocus(vertPane, sectorText);
	}
    }
}

static void
callbackCommodity(Widget widget, XtPointer closure, XtPointer callData)
{
    XtPopdown(commodityPopup);

    currentEditAdd(False, NULL, False);
    currentEditAdd = oldEditAdd;
    oldEditAdd(True, NULL, False);
    
    if ((Bool)closure) {
	char *sector;
	int x1, y1, x2, y2;
	
	XtVaGetValues(
	    sectorText,
	    XtNstring, &sector,
	    NULL);

	if (strToRange(&sector, &x1, &y1, &x2, &y2)) {
	    commodityCommand(x1, y1, x2, y2, True);
	}
    }
}

/* ARGSUSED */
static void
commodityNotify(Widget w, XEvent *event, String *params, Cardinal *numParams)
{
    XtCallActionProc(okButton, "set", NULL, NULL, 0);
    XtCallActionProc(okButton, "notify", NULL, NULL, 0);
    XtCallActionProc(okButton, "unset", NULL, NULL, 0);
}

/* ARGSUSED */
static Bool
processCommodity(Bool first, Bool last, char *str, void *data)
{
#define COMMODITYHEADERLINES 3
    static dataLine = 0;
    static XRectangle rectangle;
    static needRedisplay = False;
    Bool redisplay = (Bool)data;
    int x, y;
    EmpireSector *sector;
    int tmp;
    char typeChar;

    if (first) {
	dataLine = 0;
	needRedisplay = False;
    }

    if (last) {
	if (redisplay && needRedisplay && dataLine > COMMODITYHEADERLINES) {
	    redisplaySectors(mapWidget, rectangle.x, rectangle.y,
			     rectangle.width, rectangle.height);
	}
	return True;
    }

    if (++dataLine<=COMMODITYHEADERLINES || !isspace(str[0])) return True;

    /* First info is coord */
    strToCoord(&str, &x, &y);

    /* combine to minimal rectangle */
    if (dataLine == COMMODITYHEADERLINES+1) {
	rectangle.x = x;
	rectangle.y = y;
	rectangle.width = x;
	rectangle.height = y;
    } else {
	rectangle.x = min(rectangle.x,x);
	rectangle.width = max(rectangle.width,x);
	rectangle.y = min(rectangle.y,y);
	rectangle.height = max(rectangle.height,y);
    }

    /* change to array base */
    x = MAP2ARRX(x);
    y = MAP2ARRY(y);

    /* we can access the sector */
    sector = &empire.map[x+y*empire.rowBytes];

    /* next is the designation */
    while (isspace(*str)) str++;
    typeChar = *str++;
    if (sector->typeChar != typeChar) {
	sector->typeChar = typeChar;
	sector->status = OwnSector;
	sector->typePixmap = getTypePixmap(x, y, OwnSector, False);
	needRedisplay = True;
    }

    /* get delivery paths */
    sector->path[commIdxShell] = *++str;
    sector->path[commIdxGun]   = *++str;
    sector->path[commIdxPetrol]= *++str;
    sector->path[commIdxIron]  = *++str;
    sector->path[commIdxDust]  = *++str;
    sector->path[commIdxBars]  = *++str;
    sector->path[commIdxOil]   = *++str;
    sector->path[commIdxLcm]   = *++str;
    sector->path[commIdxHcm]   = *++str;
    sector->path[commIdxRad]   = *++str;

    /* FIXME: can we use the distrib columns ?? */
    str += 12;
    while (isspace(*str)) str++;

    sscanf(str, "%d %d %d %d %d %d %d %d %d %d",
	   &sector->product[commIdxShell],
	   &sector->product[commIdxGun],
	   &sector->product[commIdxPetrol],
	   &sector->product[commIdxIron],
	   &sector->product[commIdxDust],
	   &sector->product[commIdxBars],
	   &sector->product[commIdxOil],
	   &sector->product[commIdxLcm],
	   &sector->product[commIdxHcm],
	   &sector->product[commIdxRad]);
    
    return True;
}

/*
 * Local Variables:
 *  mode:c
 *  c-indent-level:4
 *  c-continued-statement-offset:4
 *  c-continued-brace-offset:0
 *  c-brace-offset:0
 *  c-imaginary-offset:0
 *  c-argdecl-indent:4
 *  c-label-offset:-2
 * End:
 */
