/*
 *                           C E N S U S . C
 *
 *  implements census command
 *  -  init and show dialog
 *  -  send command
 *  -  interpret answer
 *
 *  Version      : $Revision: 1.10 $
 *
 *  Created      : Thu Jun 16 19:59:56 1994
 *  Author       : Ulrich Drepper <drepper@mydec>
 *
 *  Last modified: Thu Jul 14 21:27:13 1994
 *  Author       : Ulrich Drepper <drepper@mydec>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#if !defined(lint)
static const char *vcid = "$Id: census.c,v 1.10 1994/07/15 00:47:11 drepper Exp $";
#endif /* lint */

#include <assert.h>
#include <ctype.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>

#include "empire.h"

/*
 * prototypes of local functions
 */
static void commandInitCensus(void);
static void censusEditAdd(Bool active, XawTextBlock *textBlock, Bool async);
static void callbackCensus(Widget widget, XtPointer closure,
			   XtPointer callData);
static void censusNotify(Widget w, XEvent *event, String *params,
			 Cardinal *numParams);
static Bool processCensus(Bool first, Bool last, char *str, void *data);

/*
 * local variables
 */
static Widget censusPopup;
static Widget form;
static Widget sectorText;
static Widget okButton;
static void (*oldEditAdd)(Bool, XawTextBlock *, Bool);

/*
 * exported functions
 */
void
queryCommandCensus(void)
{
    static Bool firstCall = True;
    Position fx, fy;
    Position x, y;

    if (firstCall) {
	commandInitCensus();
	firstCall = False;
    }

    XtVaGetValues(
	commandForm,
	XtNx, &fx,
	XtNy, &fy,
	NULL);

    XtVaGetValues(
	horizPane,
	XtNx, &x,
	XtNy, &y,
	NULL);

    XtTranslateCoords(
	topLevel,
	x+fx, y+fy,
	&x, &y);

    XtVaSetValues(
	censusPopup,
	XtNx, x,
	XtNy, y,
	NULL);

    /* prepare edit widget communication */
    oldEditAdd = currentEditAdd;
    oldEditAdd(False, NULL, False);

    currentEditAdd = censusEditAdd;
    currentEditAdd(True, NULL, False);
    
    XawTextSetInsertionPoint(sectorText, 0);
    XtVaSetValues(
	sectorText,
        XtNstring, "",
        NULL);

    XtPopup(censusPopup, XtGrabNonexclusive);
    XtAddGrab(showViewport, False, False);
}

Bool
censusCommand(int xMin, int yMin, int xMax, int yMax, Bool redisplay)
{
    return sendCmdStr(processCensus, (void*)redisplay, NULL,
		      "census %d:%d,%d:%d\n", xMin, xMax, yMin, yMax);
}

/*
 * local functions
 */
static void
commandInitCensus(void)
{
    static String textTranslations = 
	"#override\n"
	"<Key>Return: censusNotify()\n"
	"<Key>Linefeed: censusNotify()\n"
	"Ctrl<Key>M: censusNotify()\n"
	"Ctrl<Key>J: censusNotify()\n";
    static XtActionsRec defaultAction[1] = {
	{ "censusNotify", censusNotify },
    };
    XtTranslations textTrans;
    Widget tmp;

    XtAppAddActions(appContext, defaultAction, 1);
    textTrans = XtParseTranslationTable(textTranslations);

    censusPopup =  XtVaCreatePopupShell(
	"censuspopupshell",
	topLevelShellWidgetClass,
        topLevel,
	NULL);
    form = XtVaCreateManagedWidget(
	"form",
	formWidgetClass,
	censusPopup,
	NULL);
    tmp = XtVaCreateManagedWidget(
	"census",
	labelWidgetClass,
	form,
	NULL);
    tmp = XtVaCreateManagedWidget(
	"sectorlabel",
	labelWidgetClass,
	form,
	NULL);
    sectorText = XtVaCreateManagedWidget(
	"sectortext",
	asciiTextWidgetClass,
	form,
	XtNeditType, XawtextEdit,
	XtNtranslations, textTrans,
	NULL);

    okButton = XtVaCreateManagedWidget(
	"OK",
	commandWidgetClass,
	form,
	NULL);
    XtAddCallback(
	okButton,
	XtNcallback, callbackCensus,
	(XtPointer)True);
    tmp = XtVaCreateManagedWidget(
	"Cancel",
	commandWidgetClass,
	form,
	NULL);
    XtAddCallback(
	tmp,
	XtNcallback, callbackCensus,
	(XtPointer)False);

    XtRealizeWidget(censusPopup);
}

/* ARGSUSED */
static void
censusEditAdd(Bool active, XawTextBlock *textBlock, Bool async)
{
    assert(async==False);
    if (textBlock) {
	XtVaSetValues(
	    sectorText,
	    XtNstring, textBlock->ptr,
	    NULL);
	XawTextSetInsertionPoint(sectorText, strlen(textBlock->ptr));
    } else {
	XawTextDisplayCaret(sectorText, active);
	if (active) {
	    XtSetKeyboardFocus(form, sectorText);
	    XtSetKeyboardFocus(vertPane, sectorText);
	}
    }
}

static void
callbackCensus(Widget widget, XtPointer closure, XtPointer callData)
{
    XtPopdown(censusPopup);

    currentEditAdd(False, NULL, False);
    currentEditAdd = oldEditAdd;
    oldEditAdd(True, NULL, False);
    
    if ((Bool)closure) {
	char *sector;
	int x1, y1, x2, y2;
	
	XtVaGetValues(
	    sectorText,
	    XtNstring, &sector,
	    NULL);

	if (strToRange(&sector, &x1, &y1, &x2, &y2)) {
	    censusCommand(x1, y1, x2, y2, True);
	}
    }
}

/* ARGSUSED */
static void
censusNotify(Widget w, XEvent *event, String *params, Cardinal *numParams)
{
    XtCallActionProc(okButton, "set", NULL, NULL, 0);
    XtCallActionProc(okButton, "notify", NULL, NULL, 0);
    XtCallActionProc(okButton, "unset", NULL, NULL, 0);
}

/* ARGSUSED */
static Bool
processCensus(Bool first, Bool last, char *str, void *data)
{
#define CENSUSHEADERLINES 3
    static dataLine = 0;
    static XRectangle rectangle;
    Bool redisplay = (Bool)data;
    int x, y;
    EmpireSector *sector;
    int tmp;

    if (first) {
	dataLine = 0;
    }

    if (last) {
	if (redisplay && dataLine > CENSUSHEADERLINES) {
	    redisplaySectors(mapWidget, rectangle.x, rectangle.y,
			     rectangle.width, rectangle.height);
	}
	return True;
    }

    if (++dataLine<=CENSUSHEADERLINES || !isspace(str[0])) return True;

    /* First info is coord */
    strToCoord(&str, &x, &y);

    /* combine to minimal rectangle */
    if (dataLine == CENSUSHEADERLINES+1) {
	rectangle.x = x;
	rectangle.y = y;
	rectangle.width = x;
	rectangle.height = y;
    } else {
	rectangle.x = min(rectangle.x,x);
	rectangle.width = max(rectangle.width,x);
	rectangle.y = min(rectangle.y,y);
	rectangle.height = max(rectangle.height,y);
    }

    /* change to array base */
    x = MAP2ARRX(x);
    y = MAP2ARRY(y);

    /* we can access the sector */
    sector = &empire.map[x+y*empire.rowBytes];

    /* next is the designation */
    while (isspace(*str)) str++;
    if (sector->typeChar != *str ||
	sector->status != OwnSector) {
	sector->typeChar = *str;
	sector->status = OwnSector;
	sector->typePixmap = getTypePixmap(x, y, OwnSector, False);
    }
    str++;
    
    /* sector->newType = */ *str++;
     
    /* efficience */
    strToInt(&str, &tmp);
    sector->eff = tmp;

    /* will produce ? */
    sector->stopped = *(str+=2) == 'n';
    str+=2;

    /* mobility */
    strToInt(&str, &sector->mobility);

    /* FIX ME: how to deal with uf del/dst ??? */
    str += 6;

    /* occupied sector ? */
    sector->occupied = *++str != ' ';
    str++;

    /* civiliance */
    strToInt(&str, &sector->product[commIdxCiv]);

    /* military */
    strToInt(&str, &sector->product[commIdxMil]);

    /* uncompensated workers */
    strToInt(&str, &sector->product[commIdxUW]);

    /* food */
    strToInt(&str, &sector->product[commIdxFood]);

    /* percentage working */
    strToInt(&str, &sector->work);
    assert(*str == '%');
    str++;
    
    /* available workforce */
    strToInt(&str, &sector->avail);

    /* no use for territory number for now */

    return True;
}

/*
 * Local Variables:
 *  mode:c
 *  c-indent-level:4
 *  c-continued-statement-offset:4
 *  c-continued-brace-offset:0
 *  c-brace-offset:0
 *  c-imaginary-offset:0
 *  c-argdecl-indent:4
 *  c-label-offset:-2
 * End:
 */
