;;; empterm.el --- Functions used by GENIE on character terminals

;; Copyright (C) 1994 Markus Armbruster

;; Author: Markus Armbruster <armbru@pond.sub.org>
;; Version: $Id: empterm.el,v 1.2 1994/07/25 09:25:14 armbru Exp $
;; Keywords: games

;; This file is part of GENIE, the GNU Emacs's Nifty Interface to Empire

;; GENIE is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GENIE is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GENIE; see the file COPYING.  If not, write to the Free
;; Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; For general information on GENIE, see file empire.el.

;; This file contains definitions specific to character terminals.
;; See empwin.el for window system specific stuff.

;; Entry points:
;; empire-highlight-string


;; To do:

;; Reuse display characters & glyphs
;; Terminal independence, don't assume VT-style escapes
;; Document the non-obvious glyph hackery


;;; Code:

(require 'disp-table)
(provide 'empterm)


(defvar empire-highlight-display-table (make-display-table)
  "Display table to display map highlights.")

(defvar empire-glyph-formats
  (vector nil "\e[1m%c\e[m" "\e[7m%c\e[m" "\e[1;7m\%c\e[m")
  "Array of highlight glyph formats.
The array index encodes the combination highlights: bit I is on iff
highlight I is active.
Array elements are used to make glyph strings from characters with
(format array-element char).")

(defvar empire-highlight-xlat (vector nil (make-string 94 ?\ ))
  "Array of translation tables mapping ordinary characters to display characters.
To display C with highlight index I use (aref (aref xlat I) C).")

(defvar empire-next-highlight-xlat 0
  "The next character to use as display character.")

(defun empire-highlight-string (string start end prop-list)
  "Modify STRING between START and END to highlight it according to PROP-LIST."
  (let ((index (empire-highlight-index prop-list)))
    (if (/= 0 index)
	(let ((xlat (aref empire-highlight-xlat index)))
	  (while (< start end)
	    (let* ((char (aref string start))
		   (xlat-index (empire-highlight-xlat-index char)))
	      (cond ((and (<= 0 xlat-index) (< xlat-index (length xlat)))
		     (if (= ?\  (aref xlat xlat-index))
			 (empire-create-highlight-xlat char index))
		     (aset string start (aref xlat xlat-index))))
	      (setq start (1+ start))))))
    string))

(defun empire-create-highlight-xlat (char index)
  "Create a translation for CHAR with highlight INDEX." 
  (if (> empire-next-highlight-xlat 255)
      (error "Highlights exhausted display table"))
  (aset empire-highlight-display-table
	empire-next-highlight-xlat
	(vector (create-glyph (format (aref empire-glyph-formats index)
				      char))))
  (aset (aref empire-highlight-xlat index)
	(empire-highlight-xlat-index char)
	empire-next-highlight-xlat)
  (setq empire-next-highlight-xlat (1+ empire-next-highlight-xlat))
  (if (= empire-next-highlight-xlat ?\ )
      (setq empire-next-highlight-xlat 128)))

(defun empire-highlight-xlat-index (char)
  "Return CHAR's index in an element of `empire-highlight-xlat'."
  (- char ?\  1))

;;; empterm.el ends here
