;;; empsubst.el --- GENIE Variable & Function expansion for Empclient

;; Copyright (C) 1994 Markus Armbruster

;; Author: Markus Armbruster <armbru@pond.sub.org>
;; Version: $Id: empsubst.el,v 1.7 1994/10/19 11:15:01 armbru Exp $
;; Keywords: games

;; This file is part of GENIE, the GNU Emacs's Nifty Interface to Empire

;; GENIE is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GENIE is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GENIE; see the file COPYING.  If not, write to the Free
;; Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; For general information on GENIE, see file empire.el.

;; This file extends Empclient by a variable and function expansion
;; mechanism.  Some functions to get help, set variables and define
;; aliases are predefined.


;; Meta-Characters

;; The following characters have special meanings:
;; \\	backslash
;; '	single quote
;; \"	double quote
;; @	jolly alpha
;; $	dollar
;; \t	tab
;; \n	newline
;; ;	semicolon
;; ?	question mark
;; [ ]	brackets

;; Quoting

;; A backslash quotes the following character.
;; Everything between single quotes is quoted.
;; Space, tab, single quote, semicolon, newline and jolly alpha are
;; quoted between double quotes.
;; Quoted meta-characters loose their special meaning.

;; Comments

;; A jolly alpha introduces a comment that extends up to the end of
;; the line excluding the newline.

;; Variable Expansion

;; `$NAME' or `${NAME}' expands into the value bound to variable NAME.
;; NAME consists of letters, digits and underscores.

;; Function Expansion

;; Newlines and semicolons separate commands.  Every command is
;; scanned from left to right.  Everything between brackets is an
;; function call and is expanded recursively.  Then the command is
;; split into words separated by spaces or tabs.  If the first word is
;; bound as a function, the call is expanded into the result of
;; applying this function to the remaining words.  If a top level
;; command function-expands into the empty string, the command is
;; removed completely.

;; Command Splitting

;; Every line is split into commands at semicolons.  A command
;; starting with a question mark is called sub-command; the question
;; mark is removed.  Non-sub-commands are called proper commands.
;; Proper commands after semicolons abort the preceding command if it
;; is still prompting for arguments.
;; Sub-commands are ignored unless they are prompted for.
;; Empty proper commands after semicolons are ignored.  Hence, if you
;; terminate a line with a semicolon, the last command completes or is
;; aborted.  Useful for scripts.

;; Defining Functions

;; The macro `empire-subst-defun' feels very much like `defun' but
;; binds words for function expansion instead.  The value of the such
;; a function must be a string and is used as expansion.  The function
;; may call `insert' to print messages.  The predefined functions are
;; implemented this way and serve as examples.

;; Predefined Functions

;; The predefined functions are self-documenting.  Use function `help'
;; to browse their documentation.

;; Accessing Variables and Functions from Lisp

;; Use `(empire-user-symbol NAME)' to get the symbol used for the
;; string NAME.


;;; Summary of User Options

;; empire-unbound-var-error	signal error on unbound variables 
;; empire-unbound-fun-error	signal error on unbound functions
;;				(only for bracketed calls)


;;; Restrictions

;; Neither variables nor functions can be made buffer local.


;;; To do

;; Function "source"


;;; Code:

(require 'empclient)
(provide 'empsubst)


;;; User options

(defvar empire-unbound-var-error nil
  "*If non-nil, expanding unbound variables givs an error instead of \"\".")

(defvar empire-unbound-fun-error nil
  "*If non-nil, expanding unbound functions below top level gives an error.")


;;; Variables

(defvar empire-user-obarray (make-vector 109 0)
  "Obarray for user symbols.")


;;; Functions

(add-hook 'empire-client-mode-hook 'empire-init-subst)

(defun empire-init-subst ()
  "Initialize the empire substitution tool."
  (add-hook 'empire-input-filter-functions 'empire-substitute)
  (setq empire-input-split-function 'empire-split-command))

(defun empire-substitute (string)
  "Return STRING with substitutions performed."
  (empire-subst-fun (empire-subst-var (empire-unquote string))))

(defun empire-split-command (string)
  "Split string at semicolons into commands.
Strip question mark form sub-commands and prefix them with
`at-flush-or-ignore'.
Remove non-leading empty proper commands.
Prefix non-leading proper commands with `at-prompt-or-abort'."
  (let ((cmds
	 (apply 'append
		(mapcar (function
			 (lambda (cmd)
			   (let ((first (string-match "[^ \t]" cmd)))
			     (cond ((not first)
				    (list 'at-prompt-or-abort))
				   ((eq ?? (aref cmd first))
				    (list 'at-flush-or-ignore
					  (substring cmd (1+ first))))
				   (t
				    (list 'at-prompt-or-abort
					  (substring cmd first)))))))
			(empire-split-unquoted string ";")))))
    (if (eq (car cmds) 'at-prompt-or-abort)
	(setq cmds (cdr cmds)))		; undo prefixing leading proper
    (if (or (stringp (car cmds))
	    (not string))
	cmds
      (cons "" cmds))))			; undo removal of leading empty proper

(defun empire-user-symbol (name)
  "Map string NAME to symbol."
  (intern name empire-user-obarray))

(defun empire-subst-error (fmt &rest args)
  "Like function `error', but inserts message in current buffer."
  (insert (apply 'format fmt args))
  (insert "\n")
  (error "Bad substitution"))

(defun empire-split-unquoted (string split-regexp)
  "Split STRING at unquoted SPLIT-REGEXP, return list of pieces."
  (and string
       (let ((cmds nil)
	     (start 0)
	     (copied 0))
	 (while (string-match split-regexp string start)
	   (if (and (not (get-text-property (match-beginning 0)
					    'empire-quoted string))
		    (= (next-single-property-change (match-beginning 0)
						    'empire-quoted
						    string
						    (match-end 0))
		       (match-end 0)))
	       (setq cmds (cons (substring string
					   copied
					   (match-beginning 0))
				cmds)
		     copied (match-end 0)))
	   (setq start (match-end 0)))
	 (nreverse (cons (substring string copied) cmds)))))


;; Quote & comment removal

(defun empire-unquote (string)
  "Return string with quotes and comments removed.
Quoted characters get a non-nil empire-quoted text property instead.
Everything between single quotes is quoted.  Spaces, tabs, newlines
and semicolons between double quotes are quoted.  The character after
a backslash is quoted.  A comment starts with a jolly alpha and
extends to the end of the line.
Quoted empty strings are replaced by an unquoted quote, which is
recognized and removed by function `empire-unquote-hack'.  This is a
hack."
  (let ((result "")
	(start 0))
    (while (string-match "['\"\\\\@]" string start)
      (setq result
	    (concat result (substring string start (match-beginning 0))))
      (setq start (match-end 0))
      (let ((found (aref string (1- start)))
	    sub-str)
	(cond ((= ?' found)
	       (or (string-match "'" string start)
		   (empire-subst-error "Unclosed quote %s"
				       (substring string (1- start))))
	       (setq sub-str (substring string start (match-beginning 0))
		     start (match-end 0))
	       (put-text-property 0 (length sub-str)
				  'empire-quoted t
				  sub-str))
	      ((= ?\" found)
	       (or (string-match "\"" string start)
		   (empire-subst-error "Unclosed doublequote %s"
				       (substring string (1- start))))
	       (setq sub-str (substring string start (match-beginning 0))
		     start (match-end 0))
	       (let ((sub-start 0))
		 (while (string-match "[ \t';\n@]+" sub-str sub-start)
		   (put-text-property (match-beginning 0) (match-end 0)
				      'empire-quoted t
				      sub-str)
		   (setq sub-start (match-end 0)))))
	      ((= ?\\ found)
	       (or (< start (length string))
		   (empire-subst-error "Trailing \\"))
	       (setq sub-str (substring string start (1+ start))
		     start (1+ start))
	       (put-text-property 0 1 'empire-quoted t sub-str))
	      ((= ?@ found)
	       (setq sub-str "\n")
	       (setq start (if (string-match "\n" string (match-end 0))
			       (match-end 0)
			     (length string)))))
	(setq result (concat result
			     (if (zerop (length sub-str)) "'" sub-str)))))
    (concat result (substring string start))))

(defun empire-unquote-hack (string)
  "Return string with unquoted single quotes removed."
  (let ((start 0))
    (while (string-match "'" string start)
      (if (get-text-property (match-beginning 0) 'empire-quoted string)
	  (setq start (match-end 0))
	(setq string (empire-string-replace-match "" string)
	      start (match-beginning 0))))
    string))


;; Variable expansion

(defun empire-subst-var (string)
  "Return STRING with variables expanded.
A variable looks like `$name' or `${name}', where name matches [a-zA-Z_0-9]+.
It is expanded into the value of (empire-user-symbol name)."
  (let ((result "")
	(start 0))
    (while (string-match "$[{a-zA-Z_0-9]" string start)
      (cond ((or (get-text-property (match-beginning 0) 'empire-quoted string)
		 (get-text-property (1- (match-end 0)) 'empire-quoted string))
	     (setq result (concat result
				  (substring string start (1- (match-end 0)))))
	     (setq start (1- (match-end 0))))
	    (t
	     (setq result
		   (concat result
			   (substring string start (match-beginning 0))))
	     (let (var)
	       (cond ((= ?{ (aref string (1- (match-end 0))))
		      (let* ((vstart (match-end 0))
			     (vend (string-match "}" string vstart)))
			(setq var (substring string vstart vend))
			(or (and vend
				 (string-match "\\`[a-zA-Z_0-9]+\\'" var))
			    (empire-subst-error "Bad substitution `%s'" var))
			(setq start (1+ vend))))
		     (t
		      (let* ((vstart (1- (match-end 0)))
			     (vend (or (string-match "[^a-zA-Z_0-9]"
						     string
						     vstart)
				       (length string))))
			(setq var (substring string vstart vend))
			(setq start vend))))
	       (let ((varsym (empire-user-symbol var)))
		 (cond ((boundp varsym)
			(setq result (concat result (symbol-value varsym))))
		       (empire-unbound-var-error
			(empire-subst-error "Unbound variable `%s'" var))))))))
    (concat result (substring string start))))


;; Function expansion

(defun empire-subst-fun (string)
  "Return STRING with function calls expanded.
A top level function call looks like `fun arg1 arg2 ...', inner
function calls are bracketed with `[' and `]'.  Calls are expanded
into the value of (apply (empire-user-symbol fun) (list arg1 arg2 ...)).
Unquoted single quotes are removed.  See function `empire-unquote'."
  (setq string (concat string "\n"))
  (let ((pieces nil)
	(start 0))
    (while (< start (length string))
      (let ((result (empire-subst-fun-1 string start t)))
	(if (cdr result)
	    (setq pieces (cons (cdr result) pieces)))
	(setq start (car result))))
    (and pieces
	 (mapconcat 'identity (nreverse pieces) ";"))))

;; Expand function call in STRING starting at START.  If TOP, this is
;; at top level.  Return pair (END . EXPANSION), where END is the end
;; of the call and EXPANSION its expansion.
(defun empire-subst-fun-1 (string start top)
  (catch 'return
    (let ((result "")
	  (copied start))
      (while (string-match "[][\n;]" string start)
	(if (get-text-property (match-beginning 0) 'empire-quoted string)
	    (setq start (match-end 0))
	  (let ((found (aref string (match-beginning 0))))
	    (if (/= copied (match-beginning 0))
		(setq result
		      (concat result
			      (substring string copied (match-beginning 0)))))
	    (cond ((= ?\[ found)
		   (let ((sub (empire-subst-fun-1 string (match-end 0) nil)))
		     (setq result (concat result (cdr sub)))
		     (setq start (car sub))
		     (setq copied start)))
		  (t			; ?\[ ?\n ?\;
		   (if (eq top (= ?\] found))
		       (error "Mismatched brackets"))
		   (let ((end (match-end 0))
			 (pieces (mapcar 'empire-unquote-hack
					 (delete "" (empire-split-unquoted result "[ \t]")))))
		     (cond ((and (car pieces)
				 (fboundp (empire-user-symbol (car pieces))))
			    (setq result
				  (apply (empire-user-symbol (car pieces))
					 (cdr pieces)))
			    (and top
				 (zerop (length result))
				 (setq result nil)))
			   ((and (not top) empire-unbound-fun-error)
			    (empire-subst-error "Unbound function `%s'"
					       (car pieces)))
			   (t
			    (empire-unquote-hack result)
			    (if (= found ?\])
				(setq result (concat "[" result "]")))))
		     (throw 'return
			    (cons end result))))))))
      (empire-subst-error "Mismatched brackets"))))


;; Predefined Functions

(defmacro empire-subst-defun (name arglist docstring &rest body)
  "Define function expansion method for NAME.
Syntax like defun: (empire-subst-defun NAME ARGLIST DOCSTRING BODY...),
but NAME is a string and DOCSTRING is mandatory."
  (` (fset (empire-user-symbol (, name))
	   (function (lambda (, arglist)
		       (, docstring)
		       (,@ body))))))

(empire-subst-defun "help" (&rest names)
  "help [name ...]
Print documentation about the specified names, expand to the empty string.
Without arguments print brief help for all functions, except aliases."
  (if names
      (mapcar (function
	       (lambda (name)
		 (insert (if (fboundp (empire-user-symbol name))
			     (empire-subst-documentation (empire-user-symbol name))
			   (format "%s is not bound" name)))
		 (insert "\n")))
	      names)
    (insert "Type `help NAME' to find out more about NAME.\n")
    (mapatoms (function
	       (lambda (symbol)
		 (if (and (fboundp symbol)
			  (not (get symbol 'empire-alias)))
		     (insert (format "%s\n"
				     (empire-subst-documentation symbol t))))))
	      empire-user-obarray))
  "")

(defun empire-subst-documentation (symbol &optional brief)
  "Return SYMBOL's documentation as user function.
If optional arg BRIEF is non-nil, return at most one line."
  (let ((name (symbol-name symbol)))
    (if (get symbol 'empire-alias)
	(format "%s is aliased to %s" name (get symbol 'empire-alias))
      (let ((docstring (documentation symbol)))
	(cond ((not docstring)
	       (format "%s is not documented" name))
	      (brief
	       (substring docstring 0 (or (string-match "\n" docstring)
					  (length docstring))))
	      (t
	       docstring))))))

(empire-subst-defun ":" (&rest args)
   ": ...
No effect, just expand to the empty string."
   "")

(empire-subst-defun "set" (&optional name value)
  "set [name [value]]
Set variable NAME to VALUE, expand to the empty string.
If VALUE is omitted, assume the empty string.
If both are omitted, print variable bindings."
  (if name
      (set (empire-user-symbol name) (or value ""))
    (mapatoms (function
	       (lambda (symbol)
		 (if (boundp symbol)
		     (insert (format "%s=%s\n"
				     (symbol-name symbol)
				     (symbol-value symbol))))))
	      empire-user-obarray))
  "")

(empire-subst-defun "unset" (name &rest more-names)
  "unset name ...
Remove the specified variables, expand to the empty string."
  (mapcar (function (lambda (name)
		      (makunbound (empire-user-symbol name))))
	  (cons name more-names))
  "")

(empire-subst-defun "print" (&rest args)
  "print ...
Print arguments, expand to the empty string."
  (insert (mapconcat 'identity args " "))
  (insert "\n")
  "")

(empire-subst-defun "alias" (&optional name &rest value)
  "alias [name [value ...]]
Define NAME to be an alias for VALUE ..., expand to the empty string.
An alias is a special kind of function.  It expands into VALUE ...
with occurences of !N replaced by the N-th argument, !N-M by the N-th
to the M-th argument separated by spaces and likewise !N- by all
arguments from the N-th on.
If VALUE is omitted, print alias NAME.
Without arguments, print all aliases."
  (if name
      (let ((symbol (empire-user-symbol name)))
	(cond (value
	       (let ((string (mapconcat 'identity value " ")))
		 (remove-text-properties 0 (length string)
					 '(empire-quoted nil)
					 string)
		 (empire-define-alias symbol string)))
	      ((and (fboundp symbol)
		    (get symbol 'empire-alias))
	       (insert (empire-subst-documentation symbol))
	       (insert "\n"))
	      (t
	       (empire-subst-error "alias: `%s' not found" name))))
    (mapatoms (function
	       (lambda (symbol)
		 (cond ((and (fboundp symbol)
			     (get symbol 'empire-alias))
			(insert (empire-subst-documentation symbol))
			(insert "\n")))))
	      empire-user-obarray))
  "")

(empire-subst-defun "unalias" (name &rest more-names)
  "unalias name ...
Remove the specified aliases, expand to the empty string."
  (mapcar (function (lambda (name)
		      (let ((symbol (empire-user-symbol name)))
			(if (not (get symbol 'empire-alias))
			    (empire-subst-error "unalias: `%s' not found" name)
			  (fmakunbound symbol)
			  (put symbol 'empire-alias nil)))))
	  (cons name more-names))
  "")

(defun empire-define-alias (symbol string)
  "Set SYMBOL's function cell to an alias expander for STRING."
  (let ((pieces nil)
	(start 0)
	(copied 0))
    ;; Parse STRING into list PIECES.
    ;; Elements are strings or lisp expressions that access the
    ;; argument list ARGS.
    (while (string-match "!\\([0-9]+\\)\\(-\\([0-9]+\\)?\\)?" string start)
      (let ((n (string-to-number (substring string
					    (match-beginning 1)
					    (match-end 1))))
	    (m (and (match-end 2)
		    (if (match-end 3)
			(string-to-number (substring string
						     (match-beginning 3)
						     (match-end 3)))
		      'rest))))
	(setq start (match-end 0))
	(or (> n 0)
	    (not m)
	    (eq m 'rest)
	    (<= n m)
	    (empire-subst-error "alias: bad substitution `%s'"
		   (substring string (match-beginning 0) (match-end 0))))
	(cond ((not (get-text-property (match-beginning 0) 'empire-quoted
				       string))
	       (setq pieces (cons (substring string copied (match-beginning 0))
				  pieces))
	       (cond ((not m)
		      (setq pieces (cons (list ', (list 'nth (1- n) 'args))
					 pieces)))
		     ((eq m 'rest)
		      (setq pieces (cons (list ', (list 'mapconcat
							'(quote identity)
							(list 'nthcdr (1- n)
							      'args)
							" "))
					 pieces)))
		     (t
		      (while (<= n m)
			(setq pieces (cons " "
					   (cons (list ', (list 'nth (1- n)
								'args))
						 pieces)))
			(setq n (1+ n)))
		      (setq pieces (cdr pieces))))))
	(setq copied start)))
    (if pieces
	(setq pieces (cons (substring string copied (length string))
			   pieces))
      ;; no !, append all arguments
      (setq pieces
	    (cons (list ', (list 'mapconcat '(quote identity) 'args " "))
		  (list (concat string " ")))))
    ;; build function from PIECES
    (fset symbol
	  (` (lambda (&rest args)
	       (let ((save (symbol-function (quote (, symbol)))))
		 (unwind-protect
		     (progn
		       (fmakunbound (quote (, symbol)))
		       (empire-filter-input (apply 'concat
						   ((, (intern "`"))
						    (, (nreverse pieces))))
					    empire-input-filter-functions))
		   (fset (quote (, symbol)) save))))))
    (put symbol 'empire-alias string)))

(empire-subst-defun "eval" (&rest args)
  "eval ...
Re-expand the arguments."
  (let ((string (mapconcat 'identity args " ")))
    (remove-text-properties 0 (length string) '(empire-quoted nil) string)
    (empire-filter-input string empire-input-filter-functions)))

(empire-subst-defun "lisp" (&rest args)
  "lisp ...
Expand into the arguments evaluated as lisp expression.
Precisely: concatenate arguments separated with spaces, evaluate,
convert to string."
  (let* ((string (mapconcat 'identity args " "))
	 (pair (read-from-string string)))
    (or (= (cdr pair) (length string))
	(error "Trailing garbage following expression"))
    (format "%s" (eval (car pair)))))

;;; empsubst.el ends here
