;;; empmap.el --- GENIE map display

;; Copyright (C) 1994 Markus Armbruster

;; Author: Markus Armbruster <armbru@pond.sub.org>
;; Version: $Id: empmap.el,v 1.6 1994/07/25 09:41:01 armbru Exp $
;; Keywords: games

;; This file is part of GENIE, the GNU Emacs's Nifty Interface to Empire

;; GENIE is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GENIE is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GENIE; see the file COPYING.  If not, write to the Free
;; Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; For general information on GENIE, see file empire.el.

;; This file defines the map display (empire map mode) and related
;; functions.


;;; Summary of Commands & Bindings

;; empire-map			Create a map
;; empire-map-mode		Major mode
;; empire-pop-to-client	TAB	Pop to empire client
;; empire-map-recenter	C-l	Recenter map, redisplay frame
;; empire-map-goto	t	Goto coordinates
;; empire-set-map-mark	C-@	Set map mark, bound like set-mark-command
;; empire-exchange-point-and-map-mark	Exchange point and map mark, bound
;;			C-x C-x	  like exchange-point-and-mark
;; empire-scroll-down	C-v	Scroll map down, bound like scroll-down
;; empire-scroll-up	M-v	Scroll map up, bound like scroll-up
;; empire-scroll-right	<	Scroll map right, also bound like scroll-right
;; empire-scroll-left	>	Scroll map left, also bound like scroll-left
;; empire-map-move	y,u,g,j,b,n	Move point in this direction
;; empire-goto-current	h	Goto current coordinates
;; empire-insert-move	Y,U,G,H,J,B,N	dto. and insert movement character
;; empire-insert-sect	s	Insert sector coordinates
;; empire-insert-sects	S	Insert rectangle coordinates
;; empire-send-and-goto-current	Send current command and goto resulting
;;			RET	  coordinates
;; empire-dump-sector	d	Dump sector
;; empire-brief-sector	SPC	Display sector summary
;; empire-brief-census	c	Display sector's census
;; empire-brief-resources r	Display sector's resources
;; empire-brief-commodities m	Display sector's commodities
;; empire-highlight	l	Highlight sectors (not yet implemented)

;; empire-mouse-insert-sect	[mouse-2]	Insert sector coordinates
;; empire-mouse-brief-sector	[C-mouse-2]	Display sector summary

;; Note: `bound like foo' means this function is bound like function
;; `foo' in global-map.


;;; Summary of User Options

;; empire-display-coordinates	Toggle coordinate display in mode line
;; empire-map-mode-hook		Major mode hook


;;; To do

;; Highlight sectors that match a user expression
;; Extend empire-insert-move to all allowed movement characters, eg v
;; Means to undo a empire-insert-move
;; Scroll map with mouse
;; Recenter map on mouse click
;; Rewrite empire-mouse-insert-sects.


;;; Code:

(require 'empclient)
(require 'emputil)
(require 'empdata)
(provide 'empmap)


;;; User options

(defvar empire-display-coordinates t
  "*Non-nil mean display current coordinates in the mode line.
This may be slow.
Changing this variable does not affect existing maps.")

(defvar empire-map-mode-hook nil
  "*Hook for customising Empire Map mode.")


;;; Variables

;; Empire map mode keymap

(defvar empire-map-mode-map nil
  "Keymap used in Empire Map mode.")
(cond ((not empire-map-mode-map)
       (setq empire-map-mode-map (make-keymap))
       (suppress-keymap empire-map-mode-map)
       (define-key empire-map-mode-map [tab] 'empire-pop-to-client)
       (define-key empire-map-mode-map "\C-l" 'empire-map-recenter)
       (define-key empire-map-mode-map "t" 'empire-map-goto)
       (define-key empire-map-mode-map "\C-@" 'empire-set-map-mark)
       (substitute-key-definition 'set-mark-command 'empire-set-map-mark
				  empire-map-mode-map global-map)
       (define-key empire-map-mode-map "\C-x\C-x" 'empire-exchange-point-and-map-mark)
       (substitute-key-definition 'exchange-point-and-mark
				  'empire-exchange-point-and-map-mark
				  empire-map-mode-map global-map)
       (define-key empire-map-mode-map "\C-v" 'empire-scroll-map-down)
       (substitute-key-definition 'scroll-down 'empire-scroll-map-down
				  empire-map-mode-map global-map)
       (define-key empire-map-mode-map "\M-v" 'empire-scroll-map-up)
       (substitute-key-definition 'scroll-up 'empire-scroll-map-up
				  empire-map-mode-map global-map)
       (define-key empire-map-mode-map ">" 'empire-scroll-map-right)
       (substitute-key-definition 'scroll-right 'empire-scroll-map-right
				  empire-map-mode-map global-map)
       (define-key empire-map-mode-map "<" 'empire-scroll-map-left)
       (substitute-key-definition 'scroll-left 'empire-scroll-map-left
				  empire-map-mode-map global-map)
       (define-key empire-map-mode-map "y" 'empire-map-move)
       (define-key empire-map-mode-map "u" 'empire-map-move)
       (define-key empire-map-mode-map "g" 'empire-map-move)
       (define-key empire-map-mode-map "h" 'empire-goto-current)
       (define-key empire-map-mode-map "j" 'empire-map-move)
       (define-key empire-map-mode-map "b" 'empire-map-move)
       (define-key empire-map-mode-map "n" 'empire-map-move)
       (define-key empire-map-mode-map "Y" 'empire-insert-move)
       (define-key empire-map-mode-map "U" 'empire-insert-move)
       (define-key empire-map-mode-map "G" 'empire-insert-move)
       (define-key empire-map-mode-map "H" 'empire-insert-move)
       (define-key empire-map-mode-map "J" 'empire-insert-move)
       (define-key empire-map-mode-map "B" 'empire-insert-move)
       (define-key empire-map-mode-map "N" 'empire-insert-move)
       (define-key empire-map-mode-map "L" 'empire-insert-move)
       (define-key empire-map-mode-map "R" 'empire-insert-move)
       (define-key empire-map-mode-map "M" 'empire-insert-move)
       (define-key empire-map-mode-map "S" 'empire-insert-move)
       (define-key empire-map-mode-map "V" 'empire-insert-move)
       (define-key empire-map-mode-map "s" 'empire-insert-sect)
       (define-key empire-map-mode-map "\M-S" 'empire-insert-sects)
       (define-key empire-map-mode-map [return] 'empire-send-and-goto-current)
       (define-key empire-map-mode-map "d" 'empire-dump-sector)
       (define-key empire-map-mode-map " " 'empire-brief-sector)
       (define-key empire-map-mode-map "c" 'empire-brief-census)
       (define-key empire-map-mode-map "m" 'empire-brief-commodities)
       (define-key empire-map-mode-map "r" 'empire-brief-resources)
       (define-key empire-map-mode-map "l" 'empire-highlight)
       (define-key empire-map-mode-map [mouse-2] 'empire-mouse-insert-sect)
       (define-key empire-map-mode-map [C-mouse-2] 'empire-mouse-brief-sector)))


;; Internal variables

(defvar empire-map-mode-line-coord nil
  "Mode-line control for displaying empire map's current coordinates")

(defvar empire-map-buffers nil
  "List of active map buffers.")

(defvar empire-map-center nil
  "Coordinates of displayed map's center.
Permanently buffer local.")

(defvar empire-map-size nil
  "Displayed map's size, looks like (X-SIZE . Y-SIZE).
Permanently buffer local.")

(defvar empire-displayed-version nil
  "Displayed map's version.
Permanently buffer local.")

(defvar empire-map-mark nil
  "Coordinates of mark.
Permanently buffer local.")

(defconst empire-x-ruler-height 2)

(defconst empire-y-ruler-width 5)

(defconst empire-directions
  '((?y -1 -1)
    (?u  1 -1)
    (?g -2  0)
    (?j  2  0)
    (?b -1  1)
    (?n  1  1))
  "Alist mapping the empire direction characters to lists (DELTA-X DELTA-Y).")



;;; Empire map mode

(defun empire-map-mode ()
  "Major mode to display an empire map.

\\{empire-map-mode-map}

Entry to this mode runs the hook `empire-map-mode-hook'."
  (kill-all-local-variables)
  (setq major-mode 'empire-map-mode)
  (setq mode-name "Empire-Map")
  (cond (empire-display-coordinates
	 (setq mode-line-format
	       '(""
		 mode-line-modified
		 mode-line-buffer-identification
		 "   "
		 global-mode-string
		 "   %[("
		 mode-name
		 mode-line-process
		 minor-mode-alist
		 "%n"
		 ")%]--@"
		 empire-map-mode-line-coord
		 "-%-"))
	 (make-local-variable 'post-command-hook)
	 (make-local-variable 'empire-map-current-coord)
	 (setq post-command-hook 'empire-update-map-coord)))
  (toggle-read-only 1)
  (use-local-map empire-map-mode-map)
  (make-local-variable 'empire-map-center)
  (put 'empire-map-center 'permanent-local t)
  (make-local-variable 'empire-map-size)
  (put 'empire-map-size 'permanent-local t)
  (make-local-variable 'empire-displayed-version)
  (put 'empire-displayed-version 'permanent-local t)
  (make-local-variable 'empire-map-mark)
  (put 'empire-map-mark 'permanent-local t)
  ;; TODO: move to empterm?
  (if (not window-system)
      (setq buffer-display-table empire-highlight-display-table))
  (run-hooks 'empire-map-mode-hook))

(defun empire-map ()
  "Create an empire map in buffer *empire map*."
  (interactive)
  (or (empire-world-x)
      (error "World size unknown"))
  (switch-to-buffer (get-buffer-create "*empire-map*"))
  (if empire-displayed-version
      (empire-redraw-maps (list (current-buffer)))
    (empire-map-mode)
    (empire-draw-map 0 0)
    (empire-map-goto 0 0)
    (setq empire-map-mode-line-coord "0,0"))
  (or (memq (current-buffer) empire-map-buffers)
      (setq empire-map-buffers (cons (current-buffer) empire-map-buffers)))
  (add-hook 'empire-after-output-hook 'empire-redraw-changed-maps)
  (setq empire-tool-buffer (current-buffer))
  (current-buffer))


;;; Commands

(defun empire-pop-to-client ()
  "Select *empire* buffer in some window, preferably a different one."
  (interactive)
  (setq empire-tool-buffer (current-buffer))
  (pop-to-buffer (get-buffer "*empire*"))
  (empire-redraw-changed-maps))

(defun empire-redraw-maps (buffers)
  "Redraw displayed maps for BUFFERS.
When called interactively, redraw either the current buffer or, with
prefix argument, all map buffers."
  (interactive (list (if current-prefix-arg
			 empire-map-buffers
		       (list (current-buffer)))))
  (empire-forget-killed-maps)
  (mapcar (function
	   (lambda (buffer)
	     (let ((window (get-buffer-window buffer t))
		   new-point)
	       (cond (window
		      (save-excursion
			(set-buffer buffer)
			;; TODO: shrink window if larger than map
			(let ((where (empire-map-where (point))))
			  (empire-draw-map (empire-map-x-center)
					   (empire-map-y-center))
			  (empire-map-goto (car where) (cdr where))
			  (setq new-point (point))))
		      ;; set point outside of excursion
		      (set-window-point window new-point))))))
	  buffers))

(defun empire-map-recenter (&optional arg)
  "Center point in window and redisplay frame.
With prefix arg don't redisplay frame."
  (interactive "P")
  (let ((where (empire-map-where (point))))
    (empire-draw-map (car where) (cdr where))
    (empire-map-goto (car where) (cdr where))
    (or arg (recenter))))

(defun empire-map-goto (x y)
  "Move point to sector X, Y.
If called interactively, prompt for coordinates."
  (interactive "nGoto x: \nnGoto x: %s, y: ")
  (let ((xoffs (empire-display-x-offs x))
	(yoffs (empire-display-y-offs y)))
    (cond ((and xoffs yoffs)
	   (goto-char (point-min))
	   (forward-line (+ empire-x-ruler-height yoffs))
	   (empire-move-to-column (+ empire-y-ruler-width xoffs)))
	  (t
	   (empire-draw-map x y)
	   (empire-map-goto x y)))))	; recursion depth <= 1

;; TODO: mark ring?
(defun empire-set-map-mark (pos)
  "Set the map mark to the coordinates for POS."
  (interactive "d")
  (set-mark pos)
  (setq empire-map-mark (empire-map-where pos)))

(defun empire-exchange-point-and-map-mark ()
  "Set the map mark to the coordinates for point and move to the old mark."
  (interactive)
  (let ((omark empire-map-mark))
    (empire-set-map-mark (point))
    (empire-map-goto (car omark) (cdr omark))))

(defun empire-scroll-map-up (&optional arg)
  "Scroll map upward ARG lines; or near full screen if no ARG.
A near full screen is `next-screen-context-lines' less than a full screen.
Negative ARG means scroll downward."
  (interactive "P")
  (let ((count (if arg
		   (prefix-numeric-value arg)
		 (max 1 (- (empire-map-height) next-screen-context-lines)))))
    (empire-scroll-map 0 count)))

(defun empire-scroll-map-down (&optional arg)
  "Scroll map downward ARG lines; or near full screen if no ARG.
A near full screen is `next-screen-context-lines' less than a full screen.
Negative ARG means scroll upward."
  (interactive "P")
  (let ((count (if arg
		   (prefix-numeric-value arg)
		 (max 1 (- (empire-map-height) next-screen-context-lines)))))
    (empire-scroll-map 0 (- count))))

(defun empire-scroll-map-left (&optional arg)
  "Scroll map left ARG columns; or near full screen if no ARG.
A near full screen is `next-screen-context-lines' less than a full screen.
Negative ARG means scroll right."
  (interactive "P")
  (let ((count (if arg
		   (prefix-numeric-value arg)
		 (max 1 (- (empire-map-width) next-screen-context-lines)))))
    (empire-scroll-map count 0)))

(defun empire-scroll-map-right (&optional arg)
  "Scroll map right ARG lines; or near full screen if no ARG.
A near full screen is `next-screen-context-lines' less than a full screen.
Negative ARG means scroll left."
  (interactive "P")
  (let ((count (if arg
		   (prefix-numeric-value arg)
		 (max 1 (- (empire-map-width) next-screen-context-lines)))))
    (empire-scroll-map (- count) 0)))

(defun empire-map-move (arg)
  "Move point ARG sectors.
Whichever character you type to run this command determines the direction."
  (interactive "p")
  (let ((direction (assq (downcase last-command-char) empire-directions))
	(where (empire-map-where (point))))
    (empire-map-goto (+ (car where) (* arg (nth 1 direction)))
		     (+ (cdr where) (* arg (nth 2 direction))))))

(defun empire-goto-current ()
  "Move point to current coordinates, if any."
  (interactive)
  (if (empire-current-x)
      (empire-map-goto (empire-current-x) (empire-current-y))))

(defun empire-insert-move (arg)
  "Move point ARG sectors and insert the route into *empire* buffer.
Whichever character you type to run this command determines the direction."
  (interactive "p")
  (let ((cmd (downcase last-command-char)))
    (cond ((memq cmd '(?h ?l ?r ?m ?s ?v))
	   (empire-insert-input (char-to-string cmd)))
	  ((> arg 0)
	   (empire-insert-input (make-string arg cmd))
	   (empire-map-move arg)))))

(defun empire-send-and-goto-current ()
  "Send input and move point to the resulting current coordinates, if any."
  (interactive)
  (empire-insert-input "\n")
  (empire-wait-until-non-busy t)
  (empire-goto-current))

(defun empire-insert-sect (pos)
  "Insert coordinates of sector near POS into *empire* buffer."
  (interactive "d")
  (let ((where (empire-map-where pos t)))
    (empire-insert-input (format "%d,%d " (car where) (cdr where)))))

(defun empire-mouse-insert-sect (click)
  "Insert coordinates of sector clicked on into *empire* buffer."
  (interactive "e")
  (let ((where (save-excursion
		 (set-buffer (window-buffer (posn-window (event-end click))))
		 (empire-map-where (posn-point (event-end click)) t))))
    (empire-insert-input (format "%d,%d " (car where) (cdr where)))))

(defun empire-insert-sects ()
  "Insert coordinates of current rectangle into *empire* buffer.
The current rectangle has corners at the map mark and point."
  (interactive)
  (or empire-map-mark
      (error "No mark set in this map."))
  (let* ((where (empire-map-where (point)))
	 (x-range
	  (empire-pick-range (car where)
			     (car empire-map-mark)
			     (empire-display-x-offs (car where))
			     (empire-display-x-offs (car empire-map-mark))))
	 (y-range
	  (empire-pick-range (cdr where)
			     (cdr empire-map-mark)
			     (empire-display-y-offs (cdr where))
			     (empire-display-y-offs (cdr empire-map-mark)))))
    (empire-insert-input (format "%d:%d,%d:%d "
				 (car x-range) (cdr x-range)
				 (car y-range) (cdr y-range)))))

;; Defunct, TODO: rewrite from scratch
;;(defun empire-mouse-insert-sects (click)
;;  "Insert coordinates of dragged rectangle into *empire* buffer."
;;  (interactive "e")
;;  (if (eq (posn-window (event-start click))
;;	  (posn-window (event-end click)))
;;      ;; can't use save-excursion, need to move point in *empire*
;;      ;; which may be current
;;      (let ((saved-buffer (current-buffer)))
;;	(unwind-protect
;;	    (let ((start (posn-point (event-start click)))
;;		  (end   (posn-point (event-end click))))
;;	      (set-buffer (window-buffer (posn-window (event-end click))))
;;	      (empire-insert-sects (min start end) (max start end)))
;;	  (set-buffer saved-buffer)))))

(defun empire-brief-sector (x y)
  "Display data for sector X, Y in the echo area.
If called interactively in a map, display for sector near point."
  (interactive (empire-x-y-arg t))
  (let* ((dump (empire-ref-dump-sector x y))
	 (des (empire-ref-map-sector x y))
	 (data (list (list "%d,%d" x y)
		     (list " %c" des)
		     (empire-brief-owner dump)
		     (empire-brief-value "%c" dump
					 'empire-dump-index-sdes ?_)
		     (empire-brief-value " %d%%" dump
					 'empire-dump-index-eff)
		     (empire-brief-value " mob:%d" dump
					 'empire-dump-index-mob)
		     (empire-brief-value " %c" dump
					 'empire-dump-index-* ?.)
		     (empire-brief-value " work:%d%%" dump
					 'empire-dump-index-work 100)
		     (empire-brief-value " av:%d" dump
					 'empire-dump-index-avail 0)
		     (empire-brief-commo dump "cmufsgpidbolhr")
		     (empire-brief-reso dump des))))
    (apply 'message (empire-concat-format-lists data))))

(defun empire-mouse-brief-sector (click)
  "Display data for sector clicked on in the echo area."
  (interactive "e")
  (save-excursion
    (set-buffer (window-buffer (posn-window (event-end click))))
    (let ((where (empire-map-where (posn-point (event-end click)) t)))
      (empire-brief-sector (car where) (cdr where)))))

(defun empire-brief-census (x y)
  "Display census for sector X, Y in the echo area.
If called interactively in a map, display for sector near point."
  (interactive (empire-x-y-arg t))
  (let* ((dump (empire-ref-dump-sector x y))
	 (data (list (list "%d,%d" x y)
		     (list " %c" (empire-ref-map-sector x y))
		     (empire-brief-owner dump)
		     (empire-brief-value "%c" dump
					 'empire-dump-index-sdes ?_)
		     (empire-brief-value " %d%%" dump
					 'empire-dump-index-eff)
		     (empire-brief-value " mob:%d" dump
					 'empire-dump-index-mob)
		     (empire-brief-commo dump "cmuf")
		     (empire-brief-value " work:%d%%" dump
					 'empire-dump-index-work 100)
		     (empire-brief-value " av:%d" dump
					 'empire-dump-index-avail 0))))
    (apply 'message (empire-concat-format-lists data))))

(defun empire-brief-resources (x y)
  "Display resources for sector X, Y in the echo area.
If called interactively in a map, display for sector near point."
  (interactive (empire-x-y-arg t))
  (let* ((dump (empire-ref-dump-sector x y))
	 (data (list (list "%d,%d" x y)
		     (list " %c" (empire-ref-map-sector x y))
		     (empire-brief-value "%c" dump
					 'empire-dump-index-sdes ?_)
		     (empire-brief-value " %d%%" dump
					 'empire-dump-index-eff)
		     (empire-brief-reso dump))))
    (apply 'message (empire-concat-format-lists data))))

(defun empire-brief-commodities (x y)
  "Display commodities for sector X, Y in the echo area.
If called interactively in a map, display for sector near point."
  (interactive (empire-x-y-arg t))
  (let* ((dump (empire-ref-dump-sector x y))
	 (data (list (list "%d,%d" x y)
		     (list " %c" (empire-ref-map-sector x y))
		     (empire-brief-commo dump "sgpidbolhr"))))
    (apply 'message (empire-concat-format-lists data))))

(defun empire-dump-sector (x y)
  "Send dump command for sector X, Y.
If called interactively in a map, dump sector near point."
  (interactive (empire-x-y-arg t))
  (if empire-busy
      (message "Can't dump now, server is busy")
    (empire-send-command (format "dump %d,%d >>/dev/null" x y))))

;; TODO: implement
(defun empire-highlight (n expr)
  (interactive "nHighlight # \nsHighlight #%s expression ")
  (empire-remove-map-highlight n)
  (empire-redraw-maps empire-map-buffers))


;;; Functions

(defun empire-update-map-coord ()
  "Set `empire-map-current-coord' and run default value of `post-command-hook'.
This may be put in a *buffer-local* `post-command-hook'."
  (let ((where (empire-map-where (point))))
    (setq empire-map-mode-line-coord (format "%d,%d" (car where) (cdr where)))
    (force-mode-line-update))
  (run-hooks (default-value post-command-hook)))

(defun empire-forget-killed-maps ()
  "Remove killed buffers from `empire-map-buffers'."
  (setq empire-map-buffers
	(delq nil
	      (mapcar (function (lambda (buffer)
				  (and (buffer-name buffer) buffer)))
		      empire-map-buffers))))

(defun empire-redraw-changed-maps ()
  "Redraw displayed maps that are out of date."
  (empire-redraw-maps
   (delq nil
	 (mapcar (function
		  (lambda (buffer)
		    (let ((window (get-buffer-window buffer t)))
		      (and window
			   (save-excursion
			     (set-buffer buffer)
			     (or (empire-map-changed-p empire-displayed-version)
				 (/= (empire-map-width)
				     (empire-map-width-for-window window))
				 (/= (empire-map-height)
				     (empire-map-height-for-window window))))
			   buffer))))
		 empire-map-buffers))))

(defun empire-draw-map (x y)
  "Draw map centered around X, Y.
The current buffer must be displayed in a window."
  (let* ((inhibit-read-only t)
	 (window (or (get-buffer-window (current-buffer) t)
		     (error "Buffer not displayed, can't draw map.")))
	 (width (empire-map-width-for-window window))
	 (height (empire-map-height-for-window window))
	 (x0 (empire-canon-x (- x (/ width 2))))
	 (y0 (empire-canon-y (- y (/ height 2))))
	 (x-ruler (empire-map-x-ruler x0 width))
	 (y-left-ruler-format (format "%%%dd " (1- empire-y-ruler-width)))
	 (i 0))
    (setq empire-map-center nil
	  empire-map-size nil)
    (erase-buffer)
    (insert x-ruler)
    (let ((i 0)
	  (yi y0))
      (while (< i height)
	(insert (format y-left-ruler-format yi))
	(insert (empire-format-map-row x0 yi width))
	(insert (format " %d" yi))
	(insert "\n")
	(setq i (1+ i)
	      yi (empire-next-y yi))))
    (insert x-ruler)
    (delete-backward-char 1)		; delete final newline
    (setq empire-map-center (cons x y)
	  empire-map-size (cons width height)
	  empire-displayed-version (empire-map-changed-p nil))))

(defun empire-map-width-for-window (window)
  (min (- (window-width window) (* 2 empire-y-ruler-width) 1)
       (empire-world-x)))

(defun empire-map-height-for-window (window)
  (min (- (window-height window) (* 2 empire-x-ruler-height) 1)
       (empire-world-y)))

(defun empire-map-x-ruler (x0 width)
  "Insert a horizontal ruler starting with X0, WIDTH sectors wide."
  (let ((ten (make-string (+ width empire-y-ruler-width) ?\ ))
	(one (make-string (+ width empire-y-ruler-width) ?\ ))
	(index 0)
	(xi x0))
    (while (< index width)
      (let ((x (substring (format "%02d" xi) -2)))
	(aset ten (+ index empire-y-ruler-width) (aref x 0))
	(aset one (+ index empire-y-ruler-width) (aref x 1))
	(setq index (1+ index)
	      xi (empire-next-x xi))))
    (concat ten "\n" one "\n")))

(defun empire-scroll-map (delta-x delta-y)
  "Scroll map by DELTA-X, DELTA-Y."
  (let* ((where (empire-map-where (point)))
	 (x (car where))
	 (y (cdr where)))
    (empire-draw-map (empire-canon-x (+ (empire-map-x-center) delta-x))
		     (empire-canon-y (+ (empire-map-y-center) delta-y)))
    (or (empire-display-x-offs x)
	(setq x (if (< delta-x 0)
		    (+ (empire-map-left-x) (empire-map-width) -1)
		  (empire-map-left-x))))
    (or (empire-display-y-offs y)
	(setq y (if (< delta-y 0)
		    (+ (empire-map-top-y) (empire-map-height) -1)
		  (empire-map-top-y))))
    (empire-map-goto x y)))

(defun empire-x-y-arg (&optional valid)
  "Return coordinates as a list (x y).
If the current buffer is a map return coordinates of point, else ask user.
If optional argument VALID is non-nil, return coordinates of a valid sector."
  (if (empire-map-x-center)
      (let ((coord (empire-map-where (point) t)))
	(list (car coord) (cdr coord)))
    (call-interactively (function
			 (lambda (x y)
			   (interactive "nSector x: \nnSector x: %s, y: ")
			   (or (not valid)
			       (empire-x-y-valid-p x y)
			       (error "Bad coordinates"))
			   (list x y))))))

(defun empire-display-x-offs (x)
  "Return column X's offset from `(empire-map-left-x)' or nil if not displayed."
  (let ((xoffs (- x (empire-map-left-x))))
    (and (<= 0 xoffs) (< xoffs (empire-map-width))
	 xoffs)))

(defun empire-display-y-offs (y)
  "Return row Y's offset from `(empire-map-top-y)' or nil if not displayed."
  (let ((yoffs (- y (empire-map-top-y))))
    (and (<= 0 yoffs) (< yoffs (empire-map-height))
	 yoffs)))

(defun empire-pick-range (r1 r2 offs1 offs2)
  (cond ((not (and offs1 offs2))
	 ;; range not visible, pick arbitrary one
	 (cons (min r1 r2) (max r1 r2)))
	((< offs1 offs2)
	 ;; range visible, r1 left
	 (cons r1 r2))
	(t
	 ;; range visible, r1 right
	 (cons r2 r1))))

(defun empire-map-where (pos &optional valid)
  "Convert buffer position POS to map coordinates.
If optional argument VALID is non-nil, return coordinates of a valid
sector near POS."
  (save-excursion
    (goto-char pos)
    (let ((x (+ (empire-current-column)
		(- empire-y-ruler-width)
		(empire-map-left-x)))
	  (y (+ (empire-current-line)
		(- empire-x-ruler-height)
		-1
		(empire-map-top-y))))
      (or (not valid)
	  (empire-x-y-valid-p x y)
	  (setq x (1- x)))
      (cons (empire-canon-x x)
	    (empire-canon-y y)))))

(defun empire-map-x-center ()
  "Return map center's x-coordinate."
  (car empire-map-center))

(defun empire-map-y-center ()
  "Return map center's y-coordinate."
  (cdr empire-map-center))

(defun empire-map-left-x ()
  "Return map's left x-coordinate."
  (- (car empire-map-center)
     (/ (car empire-map-size) 2)))

(defun empire-map-top-y ()
  "Return map's top y-coordinate."
  (- (cdr empire-map-center)
     (/ (cdr empire-map-size) 2)))

(defun empire-map-width ()
  "Return map's width."
  (car empire-map-size))

(defun empire-map-height ()
  "Return map's height."
  (cdr empire-map-size))

(defun empire-brief-value (fmt dump key &optional uninteresting)
  "Return a format list to format with FMT DUMP's value for KEY.
If the value is unknown or equal to optional arg UNINTERESTING, return
(\"\"), else return (FMT value)."
  (let ((value (empire-dump-value dump key)))
    (if (and value
	     (not (equal value uninteresting)))
	(list fmt value)
      '(""))))

(defun empire-brief-owner (dump)
  "Return a format list to format DUMP's owner."
  (let ((owner (empire-dump-value dump 'empire-dump-index-owner)))
    (cond ((eq owner empire-your-cnum)
	   '(""))
	  ((numberp owner)
	   (list " own:%d" owner))
	  (t
	   (list " own:%c" ??)))))

(defun empire-brief-commo (dump commodities)
  "Return a format list to format DUMP's COMMODITIES.
COMMODITIES is a string of commodity characters \"cmufsgpidbolhr\".
A format list is a list to which `format' can be applied."
  (let ((fmt "")
	(rev-data nil)
	(com-list (append commodities nil)))
    (while com-list
      (let* ((com-def (assq (car com-list) empire-commodities))
	     (value   (empire-dump-value dump (nth 1 com-def)))
	     (thres   (empire-dump-value dump (nth 2 com-def))))
	(cond ((and thres
		    (not (eq 0 thres)))
	       (setq fmt (concat fmt " %d/%d%c")
		     rev-data (append (list (car com-list) thres value)
				      rev-data)))
	      ((and value
		    (not (eq 0 value)))
	       (setq fmt (concat fmt " %d%c")
		     rev-data (append (list (car com-list) value)
				      rev-data))))
	(setq com-list (cdr com-list))))
    (cons fmt (nreverse rev-data))))

(defun empire-brief-reso (dump &optional des)
  "Return a format list to format DUMP's resources.
With optional argument DES only show resources interesting for this
designation."
  (let ((des-interest (cdr (assq des
				 '((?m empire-dump-index-min)
				   (?g empire-dump-index-gold)
				   (?a empire-dump-index-fert)
				   (?o empire-dump-index-ocontent)
				   (?u empire-dump-index-uran)
				   (?- . t)
				   (?+ . t)
				   (nil . t)
				   (?. empire-dump-index-fert
				       empire-dump-index-ocontent)))))
	(resources (list (list 'empire-dump-index-min
			       " min:%d" empire-high-min)
			 (list 'empire-dump-index-gold
			       " gold:%d" empire-high-gold)
			 (list 'empire-dump-index-fert
			       " fert:%d" empire-high-fert)
			 (list 'empire-dump-index-ocontent
			       " oil:%d" empire-high-ocontent)
			 (list 'empire-dump-index-uran
			       " uran:%d" empire-high-uran))))
    (empire-concat-format-lists
     (mapcar (function (lambda (sym)
			 (let ((reso (assq sym resources))
			       (value (empire-dump-value dump sym)))
			   (and value
				(or (eq des-interest t)
				    (memq sym des-interest)
				    (>= value (nth 2 reso)))
				(list (nth 1 reso) value)))))
	     '(empire-dump-index-min
	       empire-dump-index-gold
	       empire-dump-index-fert
	       empire-dump-index-ocontent
	       empire-dump-index-uran)))))

(defun empire-concat-format-lists (lists)
  "Concatenate format lists in LISTS into a single format list.
A format list is a list to which `format' can be applied."
  (cons (apply 'concat (mapcar 'car lists))
	(apply 'append (mapcar 'cdr lists))))

;;; empmap.el ends here
