/* Socket.c */

#include "Sys.h"

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <arpa/telnet.h>
#include <netdb.h>
#include <errno.h>
#ifdef HAVE_NET_ERRNO_H
#	include <net/errno.h>
#endif
#include <setjmp.h>
#include <ctype.h>

#include "proto.h"
#include "Util.h"
#include "Socket.h"
#include "RCmd.h"

#ifdef HAVE_LIBTERMNET
#	ifdef HAVE_TERMNET_H
#		include <termnet.h>
#	else
#		ifdef HAVE_TERM_TERMNET_H
#			include <term/termnet.h>
#		endif
#	endif
#endif

#ifdef HAVE_LIBSOCKS5
#	define SOCKS 5
#	include <socks.h>
#else
#	ifdef HAVE_LIBSOCKS
#		define accept		Raccept
#		define connect		Rconnect
#		define getsockname	Rgetsockname
#		define listen		Rlisten
#	endif
#endif


/* Need to special case if trying to read the startup message from the
 * server, so our command handler won't jump to the wrong spot if
 * the server hangs up at that point.
 */
int gReadingStartup = 0;

HangupProc gHangupProc;

/* Network addresses of the sockets we use. */
struct sockaddr_in gServerCtrlAddr;
struct sockaddr_in gOurClientCtrlAddr;

/* The control stream we read responses to commands from. */
FILE *gControlIn = NULL;

/* The control stream we write our request to. */
FILE *gControlOut = NULL;

/* Real name (not alias) registered to the host we're connected to. */
string gActualHostName;

/* Internet Protocol address of host we're connected to, as a string. */
char gIPStr[32];

extern int gConnected;

extern struct hostent *GetHostEntry(char *host, struct in_addr *ip_address);

/* Kind of silly, but I wanted to keep this module as self-contained
 * as possible.
 */
void SetPostHangupOnServerProc(HangupProc proc)
{
	gHangupProc = proc;
}	/* SetPostHangupOnServerProc */


static
int GetSocketAddress(int sockfd, struct sockaddr_in *saddr)
{
	int len = (int) sizeof (struct sockaddr_in);
	int result = 0;

	if (getsockname(sockfd, (struct sockaddr *)saddr, &len) < 0) {
		Error(kDoPerror, "Could not get socket name.\n");
		result = -1;
	}
	return (result);
}	/* GetSocketAddress */




void SetInlineOutOfBandData(int sockfd)
{
#ifdef SO_OOBINLINE
	int on = 1;

	if (setsockopt(sockfd, SOL_SOCKET, SO_OOBINLINE, (char *) &on, (int) sizeof(on)) < 0)
		DebugMsg("Note: May not be able to handle out-of-band data.");
#endif /* SO_OOBINLINE */
}	/* SetInlineOutOfBandData */





int OpenControlConnection(char *host, unsigned int port)
{
	struct in_addr ip_address;
	int err = 0;
	int result;
	int sockfd = -1, sock2fd = -1;
	char **curaddr;
	struct hostent *hp;

	/* Since we're the client, we just have to get a socket() and
	 * connect() it.
	 */
	ZERO(gServerCtrlAddr);

	/* Assume it's a fatal error, unless we say otherwise. */
	result = kConnectErrFatal;

	/* Make sure we use network byte-order. */
	port = (unsigned int) htons((unsigned short) port);

	gServerCtrlAddr.sin_port = port;

	hp = GetHostEntry(host, &ip_address);

	if (hp == NULL) {
		/* Okay, no host entry, but maybe we have a numeric address
		 * in ip_address we can try.
		 */
		if (ip_address.s_addr == INADDR_NONE) {
			Error(kDontPerror, "%s: Unknown host.\n", host);
			return (result);
		}
		gServerCtrlAddr.sin_family = AF_INET;
		gServerCtrlAddr.sin_addr.s_addr = ip_address.s_addr;
	} else {
		gServerCtrlAddr.sin_family = hp->h_addrtype;
		/* We'll fill in the rest of the structure below. */
	}
	
	if ((sockfd = socket(gServerCtrlAddr.sin_family, SOCK_STREAM, 0)) < 0) {
		Error(kDoPerror, "Could not get a socket.\n");
		return (result);
	}

	/* Okay, we have a socket, now try to connect it to a remote
	 * address.  If we didn't get a host entry, we will only have
	 * one thing to try (ip_address);  if we do have one, we can try
	 * every address in the list from the host entry.
	 */

	if (hp == NULL) {
		/* Since we're given a single raw address, and not a host entry,
		 * we can only try this one address and not any other addresses
		 * that could be present for a site with a host entry.
		 */
		err = connect(sockfd, (struct sockaddr *) &gServerCtrlAddr,
			      (int) sizeof (gServerCtrlAddr));
	} else {
		/* We can try each address in the list.  We'll quit when we
		 * run out of addresses to try or get a successful connection.
		 */
		for (curaddr = hp->h_addr_list; *curaddr != NULL; curaddr++) {
			/* This could overwrite the address field in the structure,
			 * but this is okay because the structure has a junk field
			 * just for this purpose.
			 */
			memcpy(&gServerCtrlAddr.sin_addr, *curaddr, (size_t) hp->h_length);
			err = connect(sockfd, (struct sockaddr *) &gServerCtrlAddr,
				      (int) sizeof (gServerCtrlAddr));
			if (err == 0)
				break;
		}
	}
	
	if (err < 0) {
		/* Could not connect.  Close up shop and go home. */

		/* If possible, tell the caller if they should bother
		 * calling back later.
		 */
		switch (errno) {
			case ENETDOWN:
			case ENETUNREACH:
			case ECONNABORTED:
			case ETIMEDOUT:
			case ECONNREFUSED:
			case EHOSTDOWN:
				result = kConnectErrReTryable;
				/*FALLTHROUGH*/
			default:
				Error(kDoPerror, "Could not connect to %s.\n", host);
		}
		goto fatal;
	}

	/* Get our end of the socket address for later use. */
	if (GetSocketAddress(sockfd, &gOurClientCtrlAddr) < 0)
		goto fatal;

	/* We want Out-of-band data to appear in the regular stream,
	 * since we can handle TELNET.
	 */
	SetInlineOutOfBandData(sockfd);
	
#if defined(IP_TOS) && defined(IPTOS_LOWDELAY)
	/* Control connection is somewhat interactive, so quick response
	 * is desired.
	 */
	SetTypeOfService(sockfd, IPTOS_LOWDELAY);
#endif

	if ((sock2fd = dup(sockfd)) < 0) {
		Error(kDoPerror, "Could not duplicate a file descriptor.\n");
		goto fatal;
	}

	/* Now setup the FILE pointers for use with the Std I/O library
	 * routines.
	 */
	if ((gControlIn = fdopen(sockfd, "r")) == NULL) {
		Error(kDoPerror, "Could not fdopen.\n");
		goto fatal;
	}

	if ((gControlOut = fdopen(sock2fd, "w")) == NULL) {
		Error(kDoPerror, "Could not fdopen.\n");
		CloseFile(&gControlIn);
		sockfd = kClosedFileDescriptor;
		goto fatal;
	}

	/* We'll be reading and writing lines, so use line buffering.  This
	 * is necessary since the stdio library will use full buffering
	 * for all streams not associated with the tty.
	 */

#ifdef HAVE_SETLINEBUF
	setlinebuf(gControlIn);
	setlinebuf(gControlOut);
#else
	(void) SETVBUF(gControlIn, NULL, _IOLBF, (size_t) BUFSIZ);
	(void) SETVBUF(gControlOut, NULL, _IOLBF, (size_t) BUFSIZ);
#endif

	(void) STRNCPY(gIPStr, inet_ntoa(gServerCtrlAddr.sin_addr));
	if ((hp == NULL) || (hp->h_name == NULL))
		(void) STRNCPY(gActualHostName, host);
	else
		(void) STRNCPY(gActualHostName, (char *) hp->h_name);

	/* Read the startup message from the server. */	
	gReadingStartup = 1;
	if (!GotFromServer(2, "Empire server ready")) {
		/* They probably hung up on us right away.  That's too bad,
		 * but we can tell the caller that they can call back later
		 * and try again.
		 */
		gReadingStartup = 0;
		result = kConnectErrReTryable;
		CloseFile(&gControlIn);
		CloseFile(&gControlOut);
		sockfd = kClosedFileDescriptor;
		sock2fd = kClosedFileDescriptor;
		goto fatal;
	}
	gReadingStartup = 0;

	gConnected = 1;

	return (kConnectNoErr);
	
fatal:
	if (sockfd > 0)
		close(sockfd);
	if (sock2fd > 0)
		close(sock2fd);		
	return (result);
}	/* OpenControlConnection */


void CloseControlConnection(void)
{
	/* This will close each file, if it was open. */
	CloseFile(&gControlIn);
	CloseFile(&gControlOut);
	gConnected = 0;
}	/* CloseControlConnection */

void HangupOnServer(void)
{
	int wasConn;

	wasConn = gConnected;

	/* Since we want to close both sides of the connection for each
	 * socket, we can just have them closed with close() instead of
	 * using shutdown().
	 */
	if (wasConn != 0) {
			CloseControlConnection();
			if (gHangupProc != (HangupProc)0)
				(*gHangupProc)();
	}
}	/* HangupOnServer */

