/* Class 2 Faxmodem Protocol Functions
 *
 * Taken from EIA Standards Proposal No. 2388: Proposed New Standard 
 *	"Asynchronous Facsimile DCE Control Standard" (if approved,
 *       to be published as EIA/TIA-592)
 *
 * Henry Minsky
 * hqm@ai.mit.edu
 * 
 */

#include <stdio.h>
#include <setjmp.h>
#include <signal.h>
#include <fcntl.h>
#include <sgtty.h>

#include "class2protocol.h"

#define TRUE  1
#define FALSE 0

#define XON 	'\021'
#define XOFF	'\023'

#define debug TRUE

static int sigALRM();
static jmp_buf timeoutbuf;

unsigned char swaptable[256];
int swaptableinit = FALSE;

struct faxmodem_response gModem_Response;


int writestr(str, fd)
     char *str;	
     int fd;
{
  write(fd, str, strlen(str));  
}

/* reads a line of characters, terminated by a newline */
int fdgets(buf, nbytes, fd)
     char *buf;
     int nbytes, fd;
{
  int i;
  char *bp;
  unsigned char ch;
  
  bp = buf;

  for (i = 0; i < nbytes; i++) {
    read(fd, &ch, 1);
    if (ch == '\n') break; 
    *bp++ = ch;
  }
  
  *bp = '\0';
}


/* reverses the low order 8 bits of a byte */
unsigned char bit_reverse(ch)
     unsigned char ch;
{ if (!swaptableinit) 
    init_swaptable();  
  return(swaptable[ch]);
  }

int init_swaptable()
{
  int i,j;
  for (i = 0; i < 256; i++) {
    /* swap the low order 4 bitswith the high order */
    j = ( ((i & 0x01) << 7) |
	 ((i & 0x02) << 5) |
	 ((i & 0x04) << 3) |
	 ((i & 0x08) << 1) |
	 ((i & 0x10) >> 1) |
	 ((i & 0x20) >> 3) |
	 ((i & 0x40) >> 5) |
	 ((i & 0x80) >> 7) );
    swaptable[i] = j;
  }
  swaptableinit = TRUE;
}

/****************************************************************
 * Initialize a faxmodem_response struct
 *
 */

int init_modem_response(response)
     struct faxmodem_response *response;
{
  response->connect = FALSE;
  response->nreplies = 0;
  response->fk = FALSE;
  response->result = -1;
  response->hangup_code = -1;
  response->post_page_response_code = -1;
  response->post_page_message_code = -1;

}

int print_modem_response(response)
     struct faxmodem_response *response;
{
  int i;

  printf("MODEM RESPONSE: \n");
  printf("%d text responses\n", response->nreplies);
  for (i = 0; i < response->nreplies; i++) 
    printf("%s\n", &response->replies[i][0]);

  printf("CONNECT: %d\n", response->connect);
  printf("FK: %d\n", response->fk);
  printf("RESULT: %d\n", response->result);
  printf("FHNG: %d\n", response->hangup_code);
  printf("FPTS: %d\n", response->post_page_response_code);
  printf("FET: %d\n", response->post_page_message_code);
  print_T30(&response->T30parameters);

}

int print_T30(T30parameters)
     struct T30Params *T30parameters;
{
  int vr,br,wd,ln,df,ec,bf,st;

  vr = T30parameters->vr;
  br = T30parameters->br;
  wd = T30parameters->wd;
  ln = T30parameters->ln;
  df = T30parameters->df;
  ec = T30parameters->ec;
  bf = T30parameters->bf;
  st = T30parameters->st;


  printf("+FDCS: %d,%d,%d,%d,%d,%d,%d,%d\n",
	 vr,br,wd,ln,df,ec,bf,st);

}





/* This function parses numeric responses from the faxmodem.
 * It is assumed that the modem is in numeric response mode.
 *
 * INPUTS: serial_fd   a serial stream file descriptor from which to read
 *         response    a faxmodem_response structure

 *
 * It fills in any relevant slots of the faxmodem_response structure
 * which is passed to it. 
 */

typedef enum {START, NUMERIC_RESPONSE, TEXT_RESPONSE} response_state;

int get_modem_result_code(serial_fd)
     int serial_fd;
{     	/* responses are a single digit followed by <CR> */
    struct faxmodem_response *response = &gModem_Response;
    char ch, last_ch, *bufptr;
    int numeric_result = -1;
    response_state state = START;

    response->nreplies = 0;

    while (TRUE) {
	switch (state) {
	  case START: 
	    bufptr = &response->replies[response->nreplies][0];

	    read(serial_fd, &ch, 1);

	    if (isdigit(ch)) state = NUMERIC_RESPONSE;
	    else if (ch == '\r' || ch == '\n' || ch == XON || ch == XOFF) state = START;
	    else state = TEXT_RESPONSE;
	    break;

	  case NUMERIC_RESPONSE: 
	    last_ch = ch;
	    read(serial_fd, &ch, 1);
	    if (ch == '\r') { 
		if (debug) printf("get_modem_result = %d\n", last_ch - '0');
		numeric_result = (last_ch - '0');
		goto done;
	    }
	    else state = TEXT_RESPONSE;
	    break;

	  case TEXT_RESPONSE:
	    *bufptr++ = ch;	/* append char into current reply string */
	    read(serial_fd, &ch, 1);
	    if (ch == '\r' || ch == '\n' || ch == XON) {
	      state = START;
	      *bufptr++ = '\0';
	      response->nreplies++; }
	    break;
	  default: 
	    numeric_result = 0;
	    if (debug) printf("Error  in get_modem_result; illegal state\n");
	    goto done;
	}
	if (debug) {
	    printf("%c", ch); 
	    fflush(stdout);
	}
    }
  done:
    parse_text_responses(response);
    response->result = numeric_result;

    print_modem_response(response);

    return(numeric_result);
}

parse_text_responses(response)
     struct faxmodem_response *response;
{
  /* Look for +FCON, +FDCS, +FDIS, +FHNG, +FPTS +FK*/
  int n;

  for (n = 0; n < response->nreplies; n++) {
    char *str;

    str = &response->replies[n][0];

    if (!strncmp("+FCON", str, 4)) 
      response->connect = TRUE;

    if (!strncmp("+FDCS", str, 4) || !strncmp("+FDIS", str, 4))
      parse_T30_params(str, &response->T30parameters);

    if (!strncmp("+FHNG", str, 4)) 
      sscanf(str+6, "%d", &response->hangup_code);

    if (!strncmp("+FK", str, 3)) 
      response->fk = TRUE;

    if (!strncmp("+FPTS", str, 4)) 
      sscanf(str+6, "%d", &response->post_page_response_code);

    if (!strncmp("+FET", str, 4)) 
      sscanf(str+6, "%d", &response->post_page_message_code);

  }
}

int parse_T30_params(str, T30parameters)
     char *str;
     struct T30Params *T30parameters;
{
  int vr,br,wd,ln,df,ec,bf,st;

  sscanf(str+6, "%d,%d,%d,%d,%d,%d,%d,%d",
	 &vr,&br,&wd,&ln,&df,&ec,&bf,&st);

  T30parameters->vr = vr;
  T30parameters->br = br;
  T30parameters->wd = wd;
  T30parameters->ln = ln;
  T30parameters->df = df;
  T30parameters->ec = ec;
  T30parameters->bf = bf;
  T30parameters->st = st;
}


/****************************************************************
 * This routine tries to force the modem to a known state
 */
int
faxmodem_sync(fd)
int fd;
{
    register int tries;
    static int zero = 0;		/* for TIOCFLUSH */
    char code = 0, cr = 0, result[128];
    void (*f)();

    f = signal(SIGALRM, sigALRM);

    for (tries = 0; tries < 3; tries++) {
	/*
	 * Command +FCLASS=2 puts us in class 2 faxmodem mode
	 * and supposedly resets other parameters
	 *
	 * After reseting the modem, initialize all
	 * parameters to required vaules:
	 *
	 *	V0	- result codes are digits
	 *	Q0	- result codes ARE sent
	 *	E0	- do not echo
	 *      M0      - speaker off
	 *	S0=0	- dont automatically answer phone
	 *      S7=90   - wait 90 seconds for carrier event
	 *	S2=255	- disable escape character
	 *	S12=255	- longest possible escape guard time
	 *
	 *      +FCLASS=2 - enable faxmodem commands
	 *	+FCR=1    - enable fax reception
	 */
	writestr("AT\r", fd);
	sleep(1);		

	writestr("ATV0Q0E0M0S0=0S2=255S12=255 +FCLASS=2\r", fd);
	sleep(1);

	/* +FCR=1 means enable fax reception */
	writestr("ATS7=90 +FCR=1\r", fd);	/* timeout 90 seconds */
	sleep(1);

	/* flush any echoes or return codes */
	ioctl(fd, TIOCFLUSH, &zero);


	/* now see if the modem is talking to us properly */
	writestr("AT\r", fd);
	if (setjmp(timeoutbuf) == 0) {
	    alarm(2); 
	    read(fd, result, 1);
	    read(fd, result+1, 1);
	    if (!strncmp("0\r", result, 2)) {
		alarm(0);
		signal(SIGALRM, f);
		return (FAXINSYNC);
	    }
	}
    }

    fprintf(stderr, "can't synchronize with faxmodem\n");
    alarm(0);
    signal(SIGALRM, f);
    return(FAXNOSYNC);
}


static int
sigALRM()
{
	longjmp(timeoutbuf, 1);
}


/****************************************************************
 * Faxmodem commands are divided into the following categories:
 *
 * [8] Facsimile Service Class 2 Commands and Responses
 *  [8.2] Service Class Identification and Selection
 *  [8.3] Service Class 2 Action Commands
 *  [8.4] Service Class 2 Responses
 *  [8.5] Service Class 2 Parameters
 *   [8.5.1] T.30 Session Parameters
 *   [8.5.2] T.30 Procedure Control Parameters
 *   [8.5.3] Phase C Data Format Parmeters
 *   [8.5.4] Miscellaneous Parameters
 *  [8.6] Session HDLC Frame Report Responses
 *

 ****************************************************************
 * Action Commands
 *
*/

/* 8.3.1 Dial Modem
 * ATDT 
 */
int faxmodem_dial(phonenum, fd)
     char *phonenum;
     int fd;
{ 
  int i;
  char cmd[256];
  
  sprintf(cmd, "ATDT%s\r", phonenum);
  if (debug) printf("Dialing...%s\n", cmd);
  
  writestr(cmd, fd);

  init_modem_response(&gModem_Response);

  return(get_modem_result_code(fd));
}
    
/* 8.3.2 Answer a Call
 * ATA
 */
int faxmodem_answer_phone(fd)
{  
   char cmd[16];

   sprintf(cmd, "ATA\r");
   if (debug) printf("%s\n", cmd);

   writestr(cmd, fd);

   init_modem_response(&gModem_Response);

   return(get_modem_result_code(fd));
 }




/* +FDT Begin or continue sending
 *
 */

/* Set desired transmission params with +FDT=DF,VR,WD,LN
* DF = Data Format : 	0  [1-d huffman]
* VR = Vertical Res : 	1 [196 dpi (fine)]
* WD = width : 			0 [ 1728 pixels]
* LN = page length :	2 [ Unlimited ]
*/
int faxmodem_start_data_transmission(df, vr, wd, ln, serial_fd)
     int df, vr, wd, ln, serial_fd;
{  char cmd[64];

   sprintf(cmd, "AT+FDT=%d,%d,%d,%d\r", df, vr, wd, ln);
   if (debug) printf("%s\n", cmd);

   write(serial_fd, cmd, strlen(cmd));


 }

/* Receive a page
 * +FDR
 * This is somewhat ugly, because the FDR command can return
 * a couple of possible results;
 * If the connection is valid, it returns something like
 *  +FCFR
 *  +FDCS: <params>
 *  CONNECT
 *
 * Yes, even in numeric-result mode (ATV0) it still says "CONNECT" in ascii. 
 * Swell.
 * 
 * If, on the other hand, the other machine has hung up, it returns
 * +FHNG: <code>
 *
 * and if the connection was never made at all, it returns ERROR (actually numeric
 * code 4)
 *
 * faxmodem_receive_page returns values:
 * 0 page reception OK, data coming
 * 1 sender hung up
 * 2 error
 */


int faxmodem_receive_page(serial_fd)
     int serial_fd;
{
  int result;
  char buf[BUFSIZ];
  void (*f)();

  writestr("AT+FDR\r", serial_fd);
  if (debug) printf("AT+FDR\n");

  f = signal(SIGALRM, sigALRM);

  /* I want to open the serial port as a stream in order
   * to do fgets() on it.
   * This seems kind of kludgy 
   */

  /* We wait until either a string "CONNECT" is seen or
   * a "+FHNG"
   * Or until 60 seconds for a response.
   */

  if (setjmp(timeoutbuf) == 0) {
    alarm(60); 
    while (TRUE) {
      fdgets(buf, sizeof(buf), serial_fd);
      printf("%s\n", buf);
      if (strncmp(buf, "+FHNG:", 5) == 0) {
	result = END_OF_DOCUMENT;
	break;
      }
      else if (strncmp(buf, "CONNECT", 7) == 0) {
	result = 0;
	break;
      }
    }
  } else {
    result = PAGE_ERROR;
  }
  
  alarm(0);
  signal(SIGALRM, f);

  return(result);
}


/* +FET End page or document */

int faxmodem_end_page(fd)
     int fd;
{ 	/* This page done  */
  if (debug) 
    printf("sending AT+FET=0\n");
  writestr("AT+FET=0\r", fd);

  return(get_modem_result_code(fd));

}

int faxmodem_end_transmission(fd)
     int fd;
{ 	/* post no more pages or documents */
  if (debug) 
    printf("sending AT+FET=2\n");

  writestr("AT+FET=2\r", fd);

  return(get_modem_result_code(fd));

}


/* Service Class 2 Option Parameters */

/* +FBOR bit reversal options */
int faxmodem_bitreverse(code, fd)
     int fd;
     int code;
{	
  char cmd[64];

  sprintf(cmd, "AT+FBOR=%d\r", code);
  if (debug) printf("%s\n", cmd);
  write(fd, cmd, strlen(cmd));
  
  return(get_modem_result_code(fd));

}


/* +FREL bit reversal options :
 * 0 = data is bit aligned as received
 * 1 = dat is byte aligned at EOLS
 */
int faxmodem_byte_align(code, fd)
     int fd;
     int code;
{	
  char cmd[64];

  sprintf(cmd, "AT+FREL=%d\r", code);
  if (debug) printf("%s\n", cmd);
  write(fd, cmd, strlen(cmd));
  
  return(get_modem_result_code(fd));

}


/* 8.5.1.1 DCE capabilities parameters, +FDCC
   Write Syntax: +FDCC=VR,BR,LN,DF,EC,BF,ST
   Default Values: 1,3,2,2,0,0,0,0

   This just sets the VR and BR, currently.

   */

int faxmodem_set_capabilities(vr,br,fd)
     int vr,br,fd;
{
  char cmd[64];

  sprintf(cmd, "AT+FDCC=%d,%d\r",vr,br);
  if (debug) printf("%s\n", cmd);
  write(fd, cmd, strlen(cmd));
  
  return(get_modem_result_code(fd));

}  

/****************************************************************/
/* Standard AT commands 
 */


/* Read the S1 (ring count) register.
   returns number of rings, or -1 if error */
int faxmodem_count_rings(fd)
     int fd;
{
  char cmd[64];

  return(read_S_register("S1", fd));
}


/* Read an S register with numeric contents */
int read_S_register(regname, fd)
     char *regname;
     int fd;
{
  char cmd[64];
  static int zero = 0;		/* for TIOCFLUSH */

  ioctl(fd, TIOCFLUSH, &zero);
  sprintf(cmd, "AT%s?", regname);
    /* flush any echoes or return codes */

  if (debug) printf("%s\n", cmd);
  write(fd, cmd, strlen(cmd));
  ioctl(fd, TIOCFLUSH, &zero);
  sprintf(cmd, "\r");
  write(fd, cmd, strlen(cmd));
  
  /* read a line containing an ascii integer */

  { char ch;
    char response[256];
    char *ptr = response;
    int rings;

    /* discard Ctrl-J ctrl-M */
    read(fd, &ch, 1);
    read(fd, &ch, 1);

    /* read digits. */
    while (TRUE) {
      read(fd, &ch, 1);
      if (isdigit(ch)) *ptr++ = ch;
      else if (ch == '\r' || ch == '\n') break;
    }

    *ptr   = '\0';

    if (debug) printf("read '%s'", response);
    if ((sscanf(response, "%d", &rings)) != 1)
      return(-1);
    else return(rings);
  }  
}
