/* This file contains routines to dial a Class 2 fax modem and send
 * g3 format (1-d Huffman) fax image files.
 * 
 * It is part of the netfax fax processing system. 
 * 
 * Henry Minsky
 * hqm@ai.mit.edu
 *
 */

#include <fcntl.h>
#include <sgtty.h>
#include <sys/types.h>
#include <sys/time.h>
#include <ctype.h>
#include <signal.h>
#include <stdio.h>
#include <setjmp.h>
#include <errno.h>

#include "../include/conf.h"
#include "../libfax/class2protocol.h"


#define debug TRUE

static int sigALRM();
static jmp_buf timeoutbuf;


char *filename;
FILE *faxfile;			/* g3 fax data file descriptor */

/*  (phonenumber, g3filename1, g3filename2, ...}  */
faxsend(dialstring, nfiles, filenames, serial_fd)
     char *dialstring;
     int nfiles;
     char *filenames[];
     int serial_fd;
{
  char *phonenumber;
  int err = 0, result;
  
  if (faxmodem_sync(serial_fd) != FAXINSYNC)
    return(FAXNOSYNC);

  /* set phase C data bit order */
  /* The interfax firmware has a bug with received data in
     bitreverse=1 mode, so I will leave it in bor=0, and fudge the
     data in software. */
  faxmodem_bitreverse(0, serial_fd); 


  result = faxmodem_dial(dialstring, serial_fd);
  if (debug) printf("dial returned %d\n", result);
  if (result == OK) {
    result = send_fax_files(nfiles, filenames, serial_fd);
    if (result != OK) return(result);
    result = faxmodem_end_transmission(serial_fd); 
  }
  if (debug) printf("final result %d\n", result);
  return(result);
}



static int
sigALRM()
{
	longjmp(timeoutbuf, 1);
}

int send_fax_files(nfiles, filenames, fd)
     int nfiles;
     char *filenames[];
     int fd;
{   
  int pages;
  int result;
  
  for (pages = 0; pages < nfiles; pages++) {

    result = send_fax_file(filenames[pages], fd);

    if (debug) printf("send_fax_file return = %d\n", result);

    if (result != 0) return(result);

    if ( (pages + 1) < nfiles) {
      if ((result = faxmodem_end_page(fd)) != 0) return(result);
    }
  }
  
  return(result);
}


int send_fax_file(fname, serial_fd)
     int serial_fd;
     char *fname;
{
   unsigned char ch;
   int nbytes, result, i;
   void (*f)();

   faxfile = fopen(fname, "r");
   if (faxfile == NULL) {
     fprintf(stderr, "Couldn't open file %s\n", fname);
     return(FAXERROR); }

   faxmodem_start_data_transmission(DF_1DHUFFMAN, VR_FINE, WD_1728, LN_UNLIMITED, serial_fd);
   if (debug) printf("Sending fax file %s\n", fname);

   while (TRUE) {
     read(serial_fd, &ch, 1);
     if (ch == XON) {
       if (debug) printf(" --- XON Recv'd, sending phase C data\n");
       break;
     }
   }

   f = signal(SIGALRM, sigALRM); /* timeout in case we hang here */

   if (setjmp(timeoutbuf) == 0) {
     alarm(600);			/* wait 10 minutes for page to be sent */
     
     while (TRUE) {

       unsigned char chbuf[BUFSIZ];
       int nchars;

       nchars = fread(chbuf, sizeof(unsigned char), sizeof(chbuf), faxfile);
       if (nchars == 0) break;

       for (i = 0; i < nchars; i++) {
	 int flow;
	 /* Flow control of sorts; checks every 100 chars 
	    for the presence of an XOFF from modem */
	 if ((i % 100) == 0) {
	   flow = (int) read_no_hang(serial_fd);
	   if ((flow & 0x7f) == (int) XOFF) {
	     if (debug) printf("Recv'd XOFF, waiting for XON\n");
	     wait_for_char((int)XON, serial_fd);
	     if (debug) printf("   Got XON, continue sending\n");
	   }
	 }
	 ch = bit_reverse(chbuf[i]);
	 write(serial_fd, &ch, 1);
	 if (ch == DLE)		/* stuff DLE with extra DLE */
	   write(serial_fd, &ch, 1);
       }
     }

     ch = DLE; write(serial_fd, &ch, 1);
     ch = ETX; write(serial_fd, &ch, 1);

     result = (get_modem_result_code(serial_fd));
     
   } else  {
     alarm(0);
     signal(SIGALRM, f);		/* restore old signal value */
     return(FAXHANG); 
   }

   alarm(0);
   signal(SIGALRM, f);		/* restore old signal value */
   return(result);

 }

