/* 
 * 
 * This recieves a fax from a Class 2 Protocol faxmodem.
 * It assumes that the phone is ringing; it answers the phone
 * and recives a multipage fax document, putting in in the 
 * INCOMING directory as a set of files with a base name
 * and the extensions xxxxxxx.g3.0 xxxxxxxxx.g3.1 ...
 * 
 * only arg is the received fax basename.
 *
 * Henry Minsky 11/02/90
 * hqm@ai.mit.edu
 *
 * 
 *
 */

#include <sys/file.h>
#include <sgtty.h>
#include <sys/types.h>
#include <sys/time.h>
#include <ctype.h>
#include <signal.h>
#include <stdio.h>
#include <setjmp.h>
#include <errno.h>

#include "../include/conf.h"
#include "../libfax/class2protocol.h"

#define debug TRUE

static int sigALRM();
static jmp_buf timeoutbuf;

/* assumes phone is ringing; answer phone, receive fax files into basefilename (argv[1]) 
 */


faxreceive(basefilename, serial_fd)
     char *basefilename;
     int serial_fd;
{	

  int err = 0, result;

  /* set +FBOR=0 because of bug in beta release of interfax firmware.
     This creates the need to reverse the bytes as they are received, 
     in order to have them in the (arbitrary) canonical bit order which
     pbmtog3 defaultly generates/expects
  */
  faxmodem_bitreverse(0, serial_fd);

  result = get_fax_files(basefilename, serial_fd);
  
  if (debug) printf("final result %d\n", result);

  return(result);
}

static int
sigALRM()
{
	longjmp(timeoutbuf, 1);
}


int gEnd_of_document;


int get_fax_files(basename, fd)
     char *basename;
     int fd;
{   
  int result, page;

  if ( (result = faxmodem_answer_phone(fd)) != OK)
    return(result);

  gEnd_of_document = FALSE;

  for (page = 0; gEnd_of_document == FALSE; page++) {
    result = get_fax_file(basename, page, fd);
    if (debug) printf("get_fax_file returns = %d\n", result);
    switch (result) {
    case PAGE_GOOD:		 continue;
    case END_OF_DOCUMENT:	 return(0);
    default:			 return(1);
    }
  }
  return(0);
}

/* This executes the +FDR receive page command, and looks for
 * the proper CONNECT response, or, if the document is finished,
 * looks for the FHNG code.
 * 
 * returns:
 *  PAGE_GOOD        no error conditions occured during reception
 *  END_OF_DOCUMENT  no page this time, end of document
 *  PAGE_ERROR       something's wrong
 */
int
get_fax_file(base, page, fd)
     char *base;
     int page;
{
  char buf[BUFSIZ];
  char tmp[BUFSIZ];
  FILE *fp;
  int  result;

  sprintf(buf, "%s/%s.g3.%d", INCOMING, base, page);

  if ( (result = faxmodem_receive_page(fd)) == 0) {
    if (debug) printf("Receiving fax file %s\n", buf);
    
    /* filename to create for this page of document */
    if ((fp = fopen(buf, "w")) == 0) {
      fprintf(stderr, "Couldn't create output file %s\n", buf);
      return(FAXERROR); }
    
    result = read_g3_stream(fd, fp);
  } else {
    gEnd_of_document = TRUE;
  }
   
  fclose(fp);
  if (debug) printf(" file received %s\n", buf);
  
  return(result);
}


/* Reads a data stream from the faxmodem, unstuffing DLE characters.
 * Returns the +FET value (2 if no more pages).
 */
int read_g3_stream(serial_fd, fp)
     int serial_fd;
     FILE *fp;
{ 
  unsigned char ch;
  int done;
  static int zero = 0;		/* for TIOCFLUSH */

  done = FALSE;

  sleep(1);

  /* flush any echoes or return codes */
  ioctl(serial_fd, TIOCFLUSH, &zero);

  /* Send XON to start phase C data stream */

  ch = XON;
  write(serial_fd, &ch, 1);
  if (debug) printf("sent XON\n");

  while (TRUE) {
    read(serial_fd, &ch, 1);

    /* I am reversing the bit order of recvd bytes in software
       because I don't trust the firmware in our beta-release 
       class 2 modem. Apparently there is a problem with the 
       DLE byte stuffing in bit-reverse mode.
       */
    if (ch == DLE) {
      read(serial_fd, &ch, 1);
      if (ch == DLE) 
	fputc(DLE, fp);
      else 
	if (ch == ETX) break;
    } else
      fputc(ch, fp);    
  }
  return(get_modem_result_code(serial_fd));
}



