/*
 * dviFile.c --
 *
 *      This file implements access routines for DVI files.
 *
 * Copyright  1999 Anselm Lingnau <lingnau@tm.informatik.uni-frankfurt.de>
 * See file COPYING for conditions on use and distribution.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <limits.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#if HAVE_MMAP
#include <sys/types.h>
#include <sys/mman.h>
#endif /* HAVE_MMAP */

#include "dviInt.h"
#include "dviOps.h"

#ifndef lint
static char rcsid[] VAR_UNUSED = "$Id: dviFile.c,v 1.3 1999/06/18 10:11:29 lingnau Exp $";
#endif /* lint */

static Dvi_File *dviFileList = (Dvi_File *)0;
static Dvi_FileInfo *dviFileInfoList = (Dvi_FileInfo *)0;

static Dvi_FileInfo *OpenDviFile _ANSI_ARGS_((Tcl_Interp *, const char *name));
static int CloseDviFile _ANSI_ARGS_((Dvi_FileInfo *dviFileInfo,
				     int forceClose));

static U8 *FindPostamble _ANSI_ARGS_((Dvi_FileInfo *dviFile));
static U8 **MakePageTable _ANSI_ARGS_((Dvi_FileInfo *dviFile));

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_OpenFile --
 *
 *      Given the name of a DVI file, return a pointer to a Dvi_File
 *      structure corresponding to it. The file is opened and mapped into
 *      memory if necessary, or a pointer to an existing Dvi_File structure
 *      is returned if the file has been loaded before.
 *
 * Results:
 *      Returns a standard Tcl result. If this is TCL_OK, then
 *      *dviFilePtr will be set to a pointer to the Dvi_File structure
 *      for the file. Otherwise TCL_ERROR is returned, and an error
 *      message will be left in interp->result.
 *
 * Side effects:
 *      The file named by name will be opened and mapped into memory,
 *      or read into a dynamically allocated array if the underlying
 *      operating system doesn't support memory-mapped files.
 *
 * ------------------------------------------------------------------------
 */

Dvi_File *
Dvi_OpenFile (interp, name, reloadProc, clientData)
    Tcl_Interp *interp;		/* Interpreter to use for error reporting */
    const char *name;		/* String containing name of the DVI file */
    Dvi_ReloadProc *reloadProc;	/* Callback for reloads */
    ClientData clientData;
{
    Dvi_FileInfo *dviFileInfo;	/* Temporary pointer to actual file info */
    Dvi_File *dviFile;		/* Temporary pointer to DVI file record */

    if ((dviFileInfo = OpenDviFile(interp, name)) == (Dvi_FileInfo *)0) {
	return (Dvi_File *)0;
    }

    if ((dviFile = (Dvi_File *)ckalloc(sizeof(Dvi_File))) == (Dvi_File *)0) {
	CloseDviFile(dviFileInfo, 0);
	Tcl_SetResult(interp, "not enough memory", TCL_STATIC);
	return (Dvi_File *)0;
    }

    dviFile->nextPtr = dviFileList;
    dviFile->infoPtr = dviFileInfo;
    dviFile->reloadProc = reloadProc;
    dviFile->clientData = clientData;

    dviFileList = dviFile;

    return dviFile;
}

static Dvi_FileInfo *
OpenDviFile (interp, name)
    Tcl_Interp *interp;
    const char *name;
{
    struct stat statBuf;	/* Used for getting the file size */
    caddr_t addr;		/* Return value of mmap() */
    Dvi_FileInfo *dviFileInfo;	/* Temporary pointer to new structure */
    int fileDesc;		/* O/S file descriptor for DVI file */
    Tcl_Obj *resultPtr = (Tcl_Obj *)0;

    /*
     * If a pointer to a Tcl interpreter has been given we will use this
     * interpreter to return verbose error messages (rather than just a
     * null pointer as the function result) if things go wrong.
     */

    if (interp) {
	resultPtr = Tcl_GetObjResult(interp);
    }

    /*
     * Open the file tentatively and find out its device and inode numbers
     * so we can later check whether it has already been opened earlier.
     */

    fileDesc = open(name, O_RDONLY);
    if (fileDesc < 0) {
	if (resultPtr) {
	    Tcl_AppendStringsToObj(resultPtr, 
				   "couldn't open \"", name, "\": ",
				   Tcl_PosixError(interp), (char *)0);
	}
        return (Dvi_FileInfo *)0;
    }

    if (fstat(fileDesc, &statBuf) < 0) {
        close(fileDesc);
	if (resultPtr) {
	    Tcl_AppendStringsToObj(resultPtr,
				   "couldn't stat \"", name, "\": ",
				   Tcl_PosixError(interp), (char *)0);
	}
        return (Dvi_FileInfo *)0;
    }

    /*
     * Check whether the file is a regular file or symbolic link. If not,
     * then refuse to deal with this file.
     */

    if (!(S_ISREG(statBuf.st_mode) || S_ISLNK(statBuf.st_mode))) {
	close(fileDesc);
	if (resultPtr) {
	    Tcl_AppendStringsToObj(resultPtr,
				   "cannot handle this type of file",
				   (char *)0);
	}
	return (Dvi_FileInfo *)0;
    }

    /*
     * Check whether file is already open. If so, increment reference
     * count and return pointer to the already-opened file.
     */
    
    for (dviFileInfo = dviFileInfoList;
	 dviFileInfo != (Dvi_FileInfo *)0
	     && (statBuf.st_dev != dviFileInfo->devNo
		 || (statBuf.st_dev == dviFileInfo->devNo
		     && statBuf.st_ino != dviFileInfo->inodeNo));
	 dviFileInfo = dviFileInfo->nextPtr)
	;
    if (dviFileInfo != (Dvi_FileInfo *)0) {
	dviFileInfo->refCount++;
	return dviFileInfo;
    }
	
    /*
     * Allocate and fill in a Dvi_File structure for the file.
     */
    
    if ((dviFileInfo = ckalloc(sizeof(Dvi_FileInfo))) == 0) {
	if (interp) {
	    Tcl_SetResult(interp, "not enough memory", TCL_STATIC);
	}
	return (Dvi_FileInfo *)0;
    }

    dviFileInfo->fileDesc = fileDesc;
    dviFileInfo->fileSize = statBuf.st_size;
    dviFileInfo->devNo = statBuf.st_dev;
    dviFileInfo->inodeNo = statBuf.st_ino;
    dviFileInfo->lastModTime = statBuf.st_mtime;
    dviFileInfo->pageCount = -1;

    /*
     * Map the file into memory if the O/S allows it; else read it
     * into a byte array to speed things up. The latter may not work
     * well with big DVI files on machines with little memory. Tough
     * luck, use xdvi.
     */

#if HAVE_MMAP
    if ((addr = mmap(0, (size_t)statBuf.st_size, PROT_READ, MAP_SHARED,
                     dviFileInfo->fileDesc, (off_t)0)) == (caddr_t)-1) {
        close(dviFileInfo->fileDesc);
	if (resultPtr) {
	    Tcl_AppendStringsToObj(resultPtr,
				   "couldn't map file \"", name, "\": ",
				   Tcl_PosixError(interp), (char *)0);
	}
        ckfree(dviFileInfo);
        return (Dvi_FileInfo *)0;
    }
#else /* not HAVE_MMAP */
    if ((addr = ckalloc((size_t)statBuf.st_size)) == 0) {
        close(dviFileInfo->fileDesc);
	if (resultPtr) {
	    Tcl_SetResult(interp, "not enough memory", TCL_STATIC);
	}
        return (Dvi_FileInfo *)0;
    }
    if (read(dviFileInfo->fileDesc, addr, (size_t)statBuf.st_size)
	    != statBuf.st_size) {
        close(dviFileInfo->fileDesc);
        ckfree(addr);
        ckfree(dviFileInfo);
	if (resultPtr) {
	    Tcl_AppendStringsToObj(resultPtr,
				   "couldn't read file \"", name, "\": ",
				   Tcl_PosixError(interp), (char *)0);
	}
        return (Dvi_FileInfo *)0;
    }
    close(dviFileInfo->fileDesc);
#endif /* not HAVE_MMAP */

    /*
     * Check whether the file is actually a DVI file, and obtain basic
     * parameters from the preamble.
     */

    dviFileInfo->code = (unsigned char *)addr;
    if (dviFileInfo->code[0] != D_PRE || dviFileInfo->code[1] != D_ID) {
	if (resultPtr) {
	    Tcl_AppendStringsToObj(resultPtr, "file \"", name,
				   "\" is not a DVI file (bad beginning)",
				   (char *)0);
	}
#if HAVE_MMAP
        munmap((caddr_t)dviFileInfo->code, dviFileInfo->fileSize);
        close(dviFileInfo->fileDesc);
#else /* not HAVE_MMAP */
        ckfree(dviFileInfo->code);
#endif /* not HAVE_MMAP */
        ckfree(dviFileInfo);
	return (Dvi_FileInfo *)0;
    }
    
    dviFileInfo->num = DviGetS32(&(dviFileInfo->code)[2]);
    dviFileInfo->den = DviGetS32(&(dviFileInfo->code)[6]);
    dviFileInfo->mag = DviGetS32(&(dviFileInfo->code)[10]);
    
    /*
     * Try to get the page count, page table and stack size from the
     * postamble. If there is no postamble, assume reasonable defaults.
     * Note that dviFile->pageTable may be 0 even if there is a postamble
     * because of insufficient memory. It is unlikely that the program
     * will run much further in this case.
     *
     * If there is no postamble this can be because the DVI file isn't
     * finished yet. In theory the program should be able to get by
     * in this case. With TeX running times being what they are these
     * days, this may not be a real issue.
     */
    
    if ((dviFileInfo->postamble = FindPostamble(dviFileInfo)) != (U8 *)0) {
        dviFileInfo->stackSize = DviGetU16(dviFileInfo->postamble + 25);
        dviFileInfo->pageCount = DviGetU16(dviFileInfo->postamble + 27);
        dviFileInfo->pageTable = MakePageTable(dviFileInfo);
    } else {
        dviFileInfo->stackSize = DVI_MIN_STACK;
        dviFileInfo->pageCount = 0;	/* no postamble -- no page count */
        dviFileInfo->pageTable = 0;
    }
    
    dviFileInfo->refCount = 1;
    dviFileInfo->name = DviSaveStr(name);
    dviFileInfo->nextPtr = dviFileInfoList;
    dviFileInfoList = dviFileInfo;

    return dviFileInfo;
}

int
Dvi_ReloadFile (interp, dviFile)
    Tcl_Interp *interp;
    Dvi_File *dviFile;
{
    Dvi_FileInfo *oldDviFileInfo = dviFile->infoPtr;
    Dvi_FileInfo *newDviFileInfo;
    Dvi_File *listPtr;

    char *name = DviSaveStr(oldDviFileInfo->name);
    unsigned int oldGeneration = oldDviFileInfo->generation;
    CloseDviFile(oldDviFileInfo, 1);
    newDviFileInfo = OpenDviFile(interp, name);
    ckfree(name);

    if (newDviFileInfo == (Dvi_FileInfo *)0) {
	return 0;
    }

    newDviFileInfo->generation = oldGeneration + 1;
    newDviFileInfo->refCount = 0; /* will be reconstructed shortly */

    for (listPtr = dviFileList; listPtr; listPtr = listPtr->nextPtr) {
	if (listPtr->infoPtr == oldDviFileInfo) {
	    listPtr->infoPtr = newDviFileInfo;
	    newDviFileInfo->refCount++;
	    if (listPtr->reloadProc) {
		(* listPtr->reloadProc)(listPtr->clientData, listPtr);
	    }
	}
    }
    return 1;
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_CloseFile --
 *
 *      Closes a DVI file.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The file described by `*dviFile' will no longer be used by the
 *      caller. If DviOpenFile has been invoked several times over for
 *      the same DVI file, a matching number of invocations to this
 *      function is necessary before the file is actually closed and
 *      its memory freed.
 *
 * ------------------------------------------------------------------------
 */

int
Dvi_CloseFile (dviFile)
    Dvi_File *dviFile;		/* Pointer to Dvi_File structure to free */
{
    Dvi_File *listPtr;

    CloseDviFile(dviFile->infoPtr, 0);

    if (dviFile == dviFileList) {
	dviFileList = dviFile->nextPtr;
    } else {
	for (listPtr = dviFileList;
	     listPtr != (Dvi_File *)0;
	     listPtr = listPtr->nextPtr) {

	    if (listPtr->nextPtr == dviFile) {
		listPtr->nextPtr = listPtr->nextPtr->nextPtr;
	    }
	}
    }

    ckfree((char *)dviFile);
    return 0;
}

static int
CloseDviFile (dviFileInfo, forceClose)
    Dvi_FileInfo *dviFileInfo;
    int forceClose;
{ 
    Dvi_FileInfo *listPtr;
    Dvi_FileInfo *prevPtr;

    if (forceClose || --dviFileInfo->refCount == 0) {
	if (dviFileInfo->code != 0) {
#if HAVE_MMAP
	    munmap((caddr_t)dviFileInfo->code, dviFileInfo->fileSize);
	    close(dviFileInfo->fileDesc);
#else /* not HAVE_MMAP */
	    ckfree(dviFileInfo->code);
#endif /* not HAVE_MMAP */
	}
	ckfree(dviFileInfo->name);
	if (dviFileInfo->pageTable != 0) {
	    ckfree(dviFileInfo->pageTable);
	}

	for (listPtr = dviFileInfoList, prevPtr = (Dvi_FileInfo *)0;
	     listPtr != (Dvi_FileInfo *)0 && listPtr != dviFileInfo;
	     prevPtr = listPtr, listPtr = listPtr->nextPtr)
	    ;
	if (listPtr == (Dvi_FileInfo *)0) {
	    return 0;
	}
	if (prevPtr == (Dvi_FileInfo *)0) {
	    dviFileInfoList = listPtr->nextPtr;
	} else {
	    prevPtr->nextPtr = listPtr->nextPtr;
	}
	ckfree((char *)dviFileInfo);
	return 0;
    }
    return dviFileInfo->refCount;
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_FileChanged --
 *
 *      Determine whether a DVI file was changed.
 *
 * Results:
 *      Returns 1 if the underlying file has been modified since it was
 *      read, 0 if it is still the same, <0 if there was an error.
 *
 * Side effects:
 *      None.
 *
 * ------------------------------------------------------------------------
 */

int
Dvi_FileChanged (dviFile)
    Dvi_File *dviFile;
{
    int result;
    struct stat statBuf;

    if ((result = stat(dviFile->infoPtr->name, &statBuf)) < 0) {
	return result;
    }

    return statBuf.st_mtime > dviFile->infoPtr->lastModTime;
}


/*
 * ------------------------------------------------------------------------
 *
 * FindPostamble --
 *
 *      Locate the postamble of a DVI file.
 *
 * Results:
 *      The return value is a pointer to the D_POST command at the
 *      beginning of the postamble or the null pointer if the file
 *      does not contain a postamble.
 *
 * Side effects:
 *      None.
 *
 * ------------------------------------------------------------------------
 */

static U8 *
FindPostamble (dviFileInfo)
    Dvi_FileInfo *dviFileInfo;
{
    U8 *startPtr = dviFileInfo->code;
    U8 *ptr = &startPtr[dviFileInfo->fileSize - 1];
    S32 offset;

    /*
     * Work forward from the end of the file, skipping 223's.
     */

    while (ptr >= startPtr && *ptr == D_PAD) {
	--ptr;
    }

    /*
     * There must be at least 5 bytes in front of the current position,
     * and the current byte must be a 2 (D_ID). Before the D_ID, there
     * is the backpointer and then a D_POSTPOST byte. If these conditions
     * aren't true, then the file has no postamble.
     */

    if (ptr - startPtr < 5 || *ptr != D_ID || ptr[-5] != D_POSTPOST) {
	return (U8 *)0;
    }

    /*
     * Ensure that the backpointer has a sensible value (i.e., points
     * to a place between the start of the file and the current offset.
     * Follow the backpointer to the start of the postamble. If the
     * postamble doesn't start with D_POST, something's wrong.
     */

    if ((offset = DviGetS32(&ptr[-4])) < 0 || offset >= ptr - startPtr) {
	return (U8 *)0;
    }
    ptr = startPtr + offset;
    if (*ptr != D_POST) {
	return (U8 *)0;
    }
    return ptr;
}

/*
 * ------------------------------------------------------------------------
 *
 * MakePageTable --
 *
 *      Construct a table of pointers to the pages in a DVI file. This
 *      assumes that the file is complete, i.e., has a postamble.
 *
 * Results:
 *      The return value is a pointer to the page offset table or
 *      the null pointer if there isn't enough memory.
 *
 * Side effects:
 *      Dynamically allocates an array of sufficient size to hold the
 *      pointers and fills it in.
 *
 * ------------------------------------------------------------------------
 */

static U8 **
MakePageTable (dviFileInfo)
    Dvi_FileInfo *dviFileInfo;	/* The DVI file in question */
{
    U8 **tablePtr;		/* Page table while it is constructed */
    U8 *pagePtr;		/* Working pointer */
    int i;			/* Page counter */

    /*
     * Sanity checks: bail out if there is no postamble or if there is
     * not enough room for the page table.
     */

    if (dviFileInfo->postamble == 0) {
	return (U8 **)0;
    }

    if ((tablePtr = ckalloc(dviFileInfo->pageCount * sizeof(U8 *))) == 0) {
	return (U8 **)0;
    }

    /*
     * Fill in the page table, working backwards from the postamble.
     */

    pagePtr = dviFileInfo->code + DviGetU32(dviFileInfo->postamble + 1);
    for (i = dviFileInfo->pageCount - 1; i >= 0; i--) {
	tablePtr[i] = pagePtr;
	pagePtr = dviFileInfo->code + DviGetS32(pagePtr + 41);
    }
    return tablePtr;
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_ListPageNumbers --
 *
 *      Construct a Tcl list of (logical) page numbers from the DVI file.
 *
 * Results:
 *      A Tcl list containing the logical page number for each physical
 *      page in the file. The logical page number is constructed from the
 *      arguments to the BOP op-code, which in turn derive from the
 *      values of the TeX counters \count0 ... \count9 at the moment the
 *      page was shipped out. The individual counter values are separated
 *      by dots. Trailing zeroes are suppressed if at least one counter
 *      is non-zero; if all counters are zero a `0' is output.
 *
 * ------------------------------------------------------------------------
 */

Tcl_Obj *
Dvi_ListPageNumbers (dviFile)
    Dvi_File *dviFile;
{
    Tcl_Obj *resultPtr = Tcl_NewObj();
    unsigned int pageNo;

    /*
     * We need a page table for this to work.
     */

    if (dviFile->infoPtr->pageTable == (U8 **)0) {
	return (Tcl_Obj *)0;
    }

    for (pageNo = 0; pageNo < dviFile->infoPtr->pageCount; pageNo++) {
	Tcl_Obj *pageObj = Tcl_NewObj();
	int countNo;		/* general variable for indexing counters */
	int lastUsed;		/* index of last non-zero counter, 0..9 */
	S32 count[10];		/* buffer for the counter values */
	char buf[20];		/* buffer for formatting a single counter */

	U8 *pagePtr = dviFile->infoPtr->pageTable[pageNo];

	/*
	 * Locate the last non-zero counter and put its index into
	 * `lastUsed'.
	 */

	lastUsed = 0;
	for (countNo = 0; countNo < 10; countNo++) {
	    count[countNo] = DviGetS32(pagePtr + 1 + countNo * 4);
	    if (count[countNo] != 0) {
		lastUsed = countNo;
	    }
	}

	/*
	 * Output the first counter followed by all counters up to
	 * the last non-zero counter.
	 */

	sprintf(buf, S32FMT, count[0]);
	Tcl_AppendToObj(pageObj, buf, strlen(buf));
	for (countNo = 1; countNo <= lastUsed; countNo++) {
	    sprintf(buf, "." S32FMT, count[countNo]);
	    Tcl_AppendToObj(pageObj, buf, strlen(buf));
	}
	Tcl_ListObjAppendElement((Tcl_Interp *)0, resultPtr, pageObj);
    }
    return resultPtr;
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_FindPage --
 *
 *     Find start of DVI code for a page in the DVI file.
 *
 * Results:
 *     A pointer to the first opcode (after the BOP and page numbers)
 *     of the specified page in the DVI file, if that page exists. In
 *     that case the absolute 1-based page number of the page in
 *     question is put into *currPageNoPtr. If the page doesn't exist
 *     or can't be found, an error indicator is put into
 *     *currPageNoPtr and the function returns a null pointer. NOTE:
 *     This function presumes that there is a working page table.
 *
 * ------------------------------------------------------------------------
 */

U8 *
Dvi_FindPage (dviFile, pageSpecPtr, currPageNoPtr)
    Dvi_File *dviFile;		/* The DVI file in question */
    Dvi_PageSpec *pageSpecPtr;	/* The page to be searched */
    unsigned int *currPageNoPtr; /* The `current' page */
{
    if (pageSpecPtr->countersUsed == DVI_PS_ABSOLUTE) {
	int pageNo = pageSpecPtr->number[0];

	/*
	 * Absolute page number 0 is shorthand for `next page'.
	 */

	if (pageNo == 0) {
	    /*
	     * Complain if we have no current page.
	     */

	    if (*currPageNoPtr == (unsigned int)-1) {
		*currPageNoPtr = DVI_PS_NOCURRPAGE;
		return (U8 *)0;
	    }

	    /*
	     * Find number of new current page (counting from 1)
	     */

	    (*currPageNoPtr)++;
	    if (*currPageNoPtr > dviFile->infoPtr->pageCount) {
		*currPageNoPtr = 1;
	    }

	    if (dviFile->infoPtr->pageTable != 0) {
		return dviFile->infoPtr->pageTable[ *currPageNoPtr - 1 ];
	    } else {
		*currPageNoPtr = DVI_PS_NOMATCH;
		return (U8 *)0;
	    }
	}

	/*
	 * Otherwise, if a page table is available and the desired page
	 * is in range, return a pointer to the start of that page.
	 */

	if (dviFile->infoPtr->pageTable != 0 && pageNo > 0
	        && (unsigned int)pageNo <= dviFile->infoPtr->pageCount) {
	    *currPageNoPtr = pageNo;
	    return dviFile->infoPtr->pageTable[ pageNo - 1 ];
	}

	/*
	 * Otherwise, complain.
	 */

        *currPageNoPtr = DVI_PS_NOSUCHPAGE;
	return (U8 *)0;

    } else {
	/*
	 * Document-style page numbers.
	 */

	int occurrenceCount;	/* counts occurrences of page number */
	unsigned int pageNo;		/* counter for loop over pages */

	occurrenceCount = pageSpecPtr->occurrences;
	for (pageNo = 0; pageNo < dviFile->infoPtr->pageCount; pageNo++) {
	    int match = 1;	/* page still under consideration? */
	    U8 *pagePtr;	/* page numbers of page pageNo */
	    int i;		/* scratch counter */

	    pagePtr = dviFile->infoPtr->pageTable[pageNo];
	    for (i = 0; match && i < pageSpecPtr->countersUsed; i++) {
		match &= DVI_PS_IGNORE(pageSpecPtr, i)
		    || pageSpecPtr->number[i] == DviGetS32(pagePtr + 1 + 4*i);
	    }
	    if (match && --occurrenceCount == 0) {
		*currPageNoPtr = pageNo + 1;
		return pagePtr;
	    }
	    *currPageNoPtr = DVI_PS_NOMATCH;
	}
    }
    return (U8 *)0;
}

U8 *
Dvi_FindCodeForAbsolutePage (dviFile, pageNo)
    Dvi_File *dviFile;
    unsigned int pageNo;
{
    if (dviFile->infoPtr->pageTable == 0) {
	return (U8 *)0;
    }

    pageNo = pageNo - 1;
    if (pageNo >= dviFile->infoPtr->pageCount) {
	return (U8 *)0;
    }

    return dviFile->infoPtr->pageTable[pageNo];
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_GetPageSpec --
 *
 *      Converts page number specification string to Dvi_PageSpec.
 *
 * Results:
 *      The return value is normally TCL_OK, in this case the interpreter
 *      result will be empty. If the string is improperly formed then
 *      TCL_ERROR is returned and an error message will be left in
 *      interp->result.
 *
 * Side effects:
 *      *pageSpecPtr is filled in according to the string.
 *
 * ------------------------------------------------------------------------
 */

#define PS_ABS          '='     /* leader for absolute page number */
#define PS_SEP          '.'     /* separator in multi-number page numbers */
#define PS_NUMBER       '#'     /* separator between page number and multi */
#define PS_WILD         '*'     /* page number wildcard */

#define PS_DONT_CARE(p, i)      ((p)->careVector &= ~(1 << (i)))
#define PS_DO_CARE(p, i)        ((p)->careVector |= (1 << (i)))

int
Dvi_GetPageSpec(interp, specString, pageSpecPtr)
    Tcl_Interp *interp;		/* Interpreter to use for error reporting */
    char *specString;		/* Page number specification string */
    Dvi_PageSpec *pageSpecPtr;	/* Page specification to be filled in */
{
    char *string = specString;	/* Pointer for parsing the specification */
				/* (We keep the original for error msgs) */
    int count;			/* Number of counters used */

    /*
     * An `absolute' (non-TeX) page number starts out with PS_ABS
     * (normally `='). The convention is to put DVI_PS_ABSOLUTE into
     * `countersUsed' and the absolute page number into `number[0]'.
     */

    if (*string == PS_ABS) {
	if (*(string + 1) == '\0') {
	    goto error;
	}
	if ((pageSpecPtr->number[0] = strtol(string + 1, (char **)0, 10)) < 0
	        || pageSpecPtr->number[0] > 65535)
	    goto error;
	pageSpecPtr->countersUsed = DVI_PS_ABSOLUTE;
	return TCL_OK;
    }

    /*
     * Otherwise, we have to deal with a TeX page number. The page
     * number consists of up to ten parts; each of these parts corresponds
     * to one of the ten counters exported into the DVI file with each BOP
     * command. Each part can either be PS_WILD (`*'), meaning `don't
     * care', or a signed 32-bit value (the page number). The TeX page
     * number may optionally be followed by a PS_NUMBER (`#') character
     * and an unsigned int value. If this value, say i, is given, it
     * stands for the i-th occurrence of the given page number in the DVI
     * file. This allows us to deal properly with things like LaTeX,
     * which can cause the same page number to be used several times over.
     * The number of parts given in the specification is assigned to
     * `countersUsed'; the unused parts up to 10 are implicitly ignored when
     * searching for a specific page.
     */

    pageSpecPtr->countersUsed = 0;
    pageSpecPtr->careVector = 0;
    for (count = 0; count < 10; count++) {
        long n;
        
        /*
         * Check for multiplicity (invalid at start of string) or
         * end of the string.
         */
        
        if (*string == '\0' || *string == PS_NUMBER) {
            if (count == 0) {
                goto error;
            }
            if (*string == PS_NUMBER) {
                if ((pageSpecPtr->occurrences = strtoul(string+1, &string, 10))
                    > 65535) {
                    goto error;
		}
                break;
            } else {
                pageSpecPtr->occurrences = 1;
            }
            break;
        }
        
        /*
         * Check whether a wildcard character (PS_WILD) or valid number
         * is coming up. Anything that is outside the S32 range is an error;
         * on 32-bit machines it may be difficult to detect this because
         * the borderline values are what strtol() returns in case of
         * over/underflow; we check this by looking at errno.
         */
        
        if (!isdigit(*string) && *string != '*' && *string != '-') {
            goto error;
        }
        
        errno = 0;
        if (*string == PS_WILD) {
	    pageSpecPtr->number[count] = 0;
            PS_DONT_CARE(pageSpecPtr, count);
            string++;
        } else if ((n = strtol(string, &string, 10)) <= MAX_S32
                   && n >= MIN_S32
                   && errno != ERANGE) {
            pageSpecPtr->number[count] = (S32)n;
            PS_DO_CARE(pageSpecPtr, count);
        } else {
            goto error;
        }
        
        pageSpecPtr->countersUsed++;
        if (*string != '\0' && *string != PS_SEP && *string != PS_NUMBER) {
            goto error;
        }
        if (*string == PS_SEP) {
            string++;
        }
    }
    if (*string != '\0') {
        goto error;
    }
    return TCL_OK;
    
  error:
    Tcl_AppendResult(interp, "page specification \"", specString,
                     "\" is invalid", (char *)0);
    return TCL_ERROR;
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_LoadFileBinary --
 *
 *     Load a named file into dynamically allocated memory.
 *
 * Returns:
 *     A pointer to an area of memory holding the contents of the file
 *     specified by `name' or a null pointer if loading the file fails.
 *
 * Side effects:
 *     The procedure tries to open the file specified by `name', allocate
 *     sufficient memory to hold its contents and read the file into
 *     the memory area.
 *
 *     This is used for loading font files.
 *
 * ------------------------------------------------------------------------
 */

U8 *
Dvi_LoadFileBinary (const char *name)
{
    int fileDesc;		/* File descriptor for reading the file */
    struct stat statBuf;	/* To find out the file size */
    U8 *buffer;			/* Pointer to buffer for reading the file */

    fileDesc = open(name, O_RDONLY);
    if (fileDesc < 0) {
	return (U8 *)0;
    }
    if (fstat(fileDesc, &statBuf) < 0) {
	close(fileDesc);
	return (U8 *)0;
    }
    if ((buffer = ckalloc((size_t)statBuf.st_size)) == 0) {
	close(fileDesc);
	return (U8 *)0;
    }
    if (read(fileDesc, (char *)buffer, (size_t)statBuf.st_size)
        != statBuf.st_size) {
        ckfree((char *)buffer);
        close(fileDesc);
        return (U8 *)0;
    }
    close(fileDesc);
    return buffer;
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_PurgeFiles --
 *
 *     Purge all DVI files from memory
 *
 *     This procedure is used to `clear the slate' during the automatic
 *     test runs.
 *
 * ------------------------------------------------------------------------
 */

#ifdef DVI_DEBUG
int
Dvi_PurgeFiles ()
{
    while (dviFileList != 0) {
	Dvi_CloseFile(dviFileList);
    }
    return TCL_OK;
}
#endif /* DVI_DEBUG */
