/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  Codec,
  TranslatedString,
  buildCodecForObject,
  codecForBoolean,
} from "@gnu-taler/taler-util";
import {
  buildStorageKey,
  useLocalStorage,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { UiSettings } from "../settings.js";
import { codecOptionalDefault } from "@gnu-taler/taler-util";

const TALER_SCREEN_ID = 102;

interface Preferences {
  showWithdrawalSuccess: boolean;
  hideDemo: boolean;
  showInstallWallet: boolean;
  fastWithdrawalForm: boolean;
}

export const codecForPreferences = (): Codec<Preferences> =>
  buildCodecForObject<Preferences>()
    .allowExtra()
    .property("showWithdrawalSuccess", codecForBoolean())
    .property("hideDemo", codecOptionalDefault(codecForBoolean(), false))
    .property("showInstallWallet", codecForBoolean())
    .property("fastWithdrawalForm", codecForBoolean())
    .build("Preferences");

const defaultPreferences: Preferences = {
  showWithdrawalSuccess: true,
  hideDemo: true,
  showInstallWallet: true,
  fastWithdrawalForm: false,
};

const BANK_PREFERENCES_KEY = buildStorageKey(
  "bank-preferences",
  codecForPreferences(),
);
/**
 * User preferences.
 *
 * @returns tuple of [state, update()]
 */
export function usePreferences(): [
  Readonly<Preferences>,
  <T extends keyof Preferences>(key: T, value: Preferences[T]) => void,
] {
  const { value, update } = useLocalStorage(
    BANK_PREFERENCES_KEY,
    defaultPreferences,
  );

  function updateField<T extends keyof Preferences>(k: T, v: Preferences[T]) {
    const newValue = { ...value, [k]: v };
    update(newValue);
  }
  return [value, updateField];
}

export function getAllBooleanPreferences(
  settings: UiSettings,
): Array<keyof Preferences> {
  if (settings.showDemoDescription) {
    return [
      "hideDemo",
      "showInstallWallet",
      "showWithdrawalSuccess",
      "fastWithdrawalForm",
    ];
  }
  return ["showInstallWallet", "showWithdrawalSuccess", "fastWithdrawalForm"];
}

export function getLabelForPreferences(
  k: keyof Preferences,
  i18n: ReturnType<typeof useTranslationContext>["i18n"],
): TranslatedString {
  switch (k) {
    case "showWithdrawalSuccess":
      return i18n.str`Show withdrawal confirmation`;
    case "fastWithdrawalForm":
      return i18n.str`Withdraw without setting amount`;
    case "hideDemo":
      return i18n.str`Hide demo hint.`;
    case "showInstallWallet":
      return i18n.str`Show install wallet first`;
    // case "showDebugInfo":
    //   return i18n.str`Show debug info`;
  }
}
