#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <netinet6/in6.h>
#include <netinet6/ipv6.h>

size_t inet6_srcrt_space(int type, int segments)
{
	if (type != 0 || segments > 24)
		return 0;

	return (sizeof(struct cmsghdr) + sizeof(struct rt0_hdr) +
		segments * sizeof(struct in6_addr));
}

extern struct cmsghdr *	inet6_srcrt_init(void *bp, int type)
{
	struct cmsghdr *cmsg;

	if (type)
	{
		return NULL;
	}

	memset(bp, 0, sizeof(struct cmsghdr) + sizeof(struct rt0_hdr));
	cmsg = (struct cmsghdr *) bp;

	cmsg->cmsg_len = sizeof(struct cmsghdr) + sizeof(struct rt0_hdr);
	cmsg->cmsg_level = SOL_IPV6;
	cmsg->cmsg_type = SCM_SRCRT;

	return cmsg;
}

int inet6_srcrt_add(struct cmsghdr *cmsg, const struct in6_addr *addr,
		    unsigned int flags)
{
	struct rt0_hdr *hdr;
	
	if (flags & ~IPV6_SRCRT_STRICT)
	{
		return -EINVAL;
	}
	
	hdr = (struct rt0_hdr *) cmsg->cmsg_data;

	if (hdr->rt_hdr.segments_left >= 23)
	{
		return -ENOSPC;
	}

	cmsg->cmsg_len += sizeof(struct in6_addr);
	hdr->rt_hdr.hdrlen += sizeof(struct in6_addr) / 8;

	if (flags)
	{
		hdr->bitmap |= htonl(IPV6_SRCRT_STRICT << 
				     (23 - hdr->rt_hdr.segments_left));
	}

	memcpy(&hdr->addr[hdr->rt_hdr.segments_left++], addr,
	       sizeof(struct in6_addr));
		
	return 0;
}

/*
 *	in and out are allowed to point to the same location
 */

int inet6_srcrt_reverse(const struct cmsghdr *cmi, struct cmsghdr *cmo)
{
	struct cmsghdr *cmsg;
	struct rt0_hdr *rhi,*rho;
	__u8 data[cmi->cmsg_len];
	__u32 ibmap;
	__u32 obmap;
	int copy = 0;
	int addr_num;
	int i;

	if (cmi == cmo)
	{
		cmsg = (struct cmsghdr *) data;
		copy = 1;
	}
	else
	{
		cmsg = cmo;
	}

	memcpy(cmsg, cmi, sizeof(struct cmsghdr));
	
	rhi = (struct rt0_hdr *) cmi->cmsg_data;
	rho = (struct rt0_hdr *) cmsg->cmsg_data;

	memset(rho, 0, sizeof(struct rt0_hdr));
	
	addr_num = rhi->rt_hdr.hdrlen >> 1;

	rho->rt_hdr.hdrlen = rhi->rt_hdr.hdrlen;
	rho->rt_hdr.segments_left = addr_num;

	ibmap = ntohl(rhi->bitmap);
	obmap = 0;

	for (i=0; i < addr_num; i++)
	{
		int compl;

		compl = addr_num - i - 1;
		memcpy(&rho->addr[i], &rho->addr[compl],
		       sizeof(struct in6_addr));
		
		if (ibmap & (IPV6_SRCRT_STRICT << (23 - compl)))
		{
			obmap |= (IPV6_SRCRT_STRICT << (23 - i));
		}
	}
	
	rho->bitmap = htonl(obmap);

	if (copy)
	{
		memcpy(cmo, cmsg, cmsg->cmsg_len);
	}

	return 0;
}

struct in6_addr* inet6_srcrt_getaddr(struct cmsghdr *cmsg, int offset)
{
	struct rt0_hdr *hdr = (struct rt0_hdr *) cmsg->cmsg_data;
	
	if (offset > (hdr->rt_hdr.hdrlen >> 1))
	{
		return NULL;
	}

	return &(hdr->addr[offset]);
}


unsigned int inet6_srcrt_getflags(struct cmsghdr *cmsg, int offset)
{
	struct rt0_hdr *hdr = (struct rt0_hdr *) cmsg->cmsg_data;
	__u32 map;

	if (offset > (hdr->rt_hdr.hdrlen >> 1))
	{
		return (unsigned int) -1;
	}
	
	map = ntohl(hdr->bitmap);
	return (map & (IPV6_SRCRT_STRICT << (23 - offset)));
}
