# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

from __future__ import absolute_import, print_function, unicode_literals

from mozboot.base import BaseBootstrapper
from mozboot.linux_common import LinuxBootstrapper


class OpenSUSEBootstrapper(
        LinuxBootstrapper, BaseBootstrapper):
    '''openSUSE experimental bootstrapper.'''

    SYSTEM_PACKAGES = [
        'nodejs',
        'npm',
        'which',
        'rpmconf',
        'libcurl-devel',
        'libpulse-devel',
    ]

    BROWSER_PACKAGES = [
        'alsa-devel',
        'gcc-c++',
        'python3-dbus-python',
        'gtk3-devel',
        'dbus-1-glib-devel',
        'gconf2-devel',
        'glibc-devel-static',
        'libstdc++-devel',
        'libXt-devel',
        'libproxy-devel',
        'libuuid-devel',
        'clang-devel',
        'patterns-gnome-devel_gnome',
    ]

    BROWSER_GROUP_PACKAGES = [
        'devel_C_C++',
        'devel_gnome',
    ]

    MOBILE_ANDROID_COMMON_PACKAGES = [
        'java-1_8_0-openjdk',
        'wget',
    ]

    def __init__(self, version, dist_id, **kwargs):
        print('Using an experimental bootstrapper for openSUSE.')
        BaseBootstrapper.__init__(self, **kwargs)

    def install_system_packages(self):
        self.zypper_install(*self.SYSTEM_PACKAGES)

    def install_browser_packages(self):
        self.ensure_browser_packages()

    def install_browser_group_packages(self):
        self.ensure_browser_group_packages()

    def install_browser_artifact_mode_packages(self):
        self.ensure_browser_packages(artifact_mode=True)

    def install_mobile_android_packages(self):
        self.ensure_mobile_android_packages()

    def install_mobile_android_artifact_mode_packages(self):
        self.ensure_mobile_android_packages(artifact_mode=True)

    def install_mercurial(self):
        self.run_as_root(['pip', 'install', '--upgrade', 'pip'])
        self.run_as_root(['pip', 'install', '--upgrade', 'Mercurial'])

    def ensure_clang_static_analysis_package(self, state_dir, checkout_root):
        from mozboot import static_analysis
        self.install_toolchain_static_analysis(
            state_dir, checkout_root, static_analysis.LINUX_CLANG_TIDY)

    def ensure_browser_packages(self, artifact_mode=False):
        # TODO: Figure out what not to install for artifact mode
        self.zypper_install(*self.BROWSER_PACKAGES)

    def ensure_browser_group_packages(self, artifact_mode=False):
        # TODO: Figure out what not to install for artifact mode
        self.zypper_patterninstall(*self.BROWSER_GROUP_PACKAGES)

    def ensure_mobile_android_packages(self, artifact_mode=False):
        # Multi-part process:
        # 1. System packages.
        # 2. Android SDK. Android NDK only if we are not in artifact mode. Android packages.

        # 1. This is hard to believe, but the Android SDK binaries are 32-bit
        # and that conflicts with 64-bit Arch installations out of the box.  The
        # solution is to add the multilibs repository; unfortunately, this
        # requires manual intervention.
        try:
            self.zypper_install(*self.MOBILE_ANDROID_COMMON_PACKAGES)
        except Exception as e:
            print('Failed to install all packages.  The Android developer '
                  'toolchain requires 32 bit binaries be enabled')
            raise e

        # 2. Android pieces.
        from mozboot import android
        android.ensure_android('linux', artifact_mode=artifact_mode,
                               no_interactive=self.no_interactive)

    def generate_mobile_android_mozconfig(self, artifact_mode=False):
        from mozboot import android
        return android.generate_mozconfig('linux', artifact_mode=artifact_mode)

    def generate_mobile_android_artifact_mode_mozconfig(self):
        return self.generate_mobile_android_mozconfig(artifact_mode=True)

    def _update_package_manager(self):
        self.zypper_update

    def upgrade_mercurial(self, current):
        self.run_as_root(['pip3', 'install', '--upgrade', 'pip'])
        self.run_as_root(['pip3', 'install', '--upgrade', 'Mercurial'])

    def ensure_nasm_packages(self, state_dir, checkout_root):
        self.zypper_install('nasm')

    def zypper_install(self, *packages):
        command = ['zypper', 'install']
        if self.no_interactive:
            command.append('-n')

        command.extend(packages)

        self.run_as_root(command)

    def zypper_update(self, *packages):
        command = ['zypper', 'update']
        if self.no_interactive:
            command.append('-n')

        command.extend(packages)

        self.run_as_root(command)

    def zypper_patterninstall(self, *packages):
        command = ['zypper', 'install', '-t', 'pattern']
        if self.no_interactive:
            command.append('-y')

        command.extend(packages)

        self.run_as_root(command)
