/* ==== wait.c ============================================================
 * Copyright (c) 1994 by Chris Provenzano, proven@mit.edu
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *  This product includes software developed by Chris Provenzano.
 * 4. The name of Chris Provenzano may not be used to endorse or promote 
 *	  products derived from this software without specific prior written
 *	  permission.
 *
 * THIS SOFTWARE IS PROVIDED BY CHRIS PROVENZANO ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL CHRIS PROVENZANO BE LIABLE FOR ANY 
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * Description : All the appropriate wait routines.
 *
 *  1.38 94/06/13 proven
 *      -Started coding this file.
 *
 */

#ifndef lint
static const char rcsid[] = "wait.c,v 1.50 1994/08/08 03:45:46 proven Exp";
#endif

#include <pthread.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <pthread/posix.h>

/* This is an UGLY hack to get wait to compile, something better is needed. */
#define _POSIX_SOURCE
#include <sys/wait.h>
#undef _POSIX_SOURCE

struct pthread_queue wait_queue = { NULL, NULL, NULL };
semaphore wait_semaphore = SEMAPHORE_CLEAR;
extern void sig_handler_real();

/* ==========================================================================
 * wait_wakeup()
 *
 * This routine is called by the interrupt handler. It cannot call
 * pthread_yield() therefore it returns NOTOK to inform the handler
 * that it will have to be called at a later time.
 */
int wait_wakeup()
{
	struct pthread *pthread;
	semaphore *lock, *plock;
	int ret;

	/* Lock wait queue */
	lock = &(wait_semaphore);
	if (SEMAPHORE_TEST_AND_SET(lock)) {
		return(NOTOK);
	}

	if (pthread = pthread_queue_get(& wait_queue)) {
		/*
	 	* Try to wakeup all threads, if we fail return failure so 
	 	* scheduler can attempt to wakeup threads at a later time
	 	*/
		do {
			plock = &(pthread->lock);
			if (SEMAPHORE_TEST_AND_SET(plock)) {
				SEMAPHORE_RESET(lock);
				return(NOTOK);
			}

			pthread_queue_deq(&wait_queue);
			pthread->state = PS_RUNNING;
			SEMAPHORE_RESET(plock);

		} while (pthread = pthread_queue_get(&wait_queue));
		/* Deinstall signal handler */
		signal(SIGCHLD, SIG_DFL);
		ret = OK;
	} else {
		ret = 1;
	}

	SEMAPHORE_RESET(lock);
	return(ret);
}

/* ==========================================================================
 * For the wait calls, it is important that the wait semaphore is locked
 * before the apropriate wait syscall is preformed. This way we ensure
 * that there is never a case where a thread is waiting for a child but
 * missed the interrupt for that child.
 */

/* ==========================================================================
 * waitpid()
 */
pid_t waitpid(pid_t pid, int *status, int options)
{
	semaphore * lock;
	pid_t ret;

	/* Lock wait queue */
	lock = &(wait_semaphore);
	if (SEMAPHORE_TEST_AND_SET(lock)) {
		pthread_yield();
	}
	while ((ret = machdep_sys_waitpid(pid, status, options | WNOHANG)) == OK) {
		if (pthread_queue_get(& wait_queue) == NULL) {
			/* Install signal handler */
			signal(SIGCHLD, sig_handler_real);
		}

		/* Enqueue thread */
		pthread_queue_enq(&wait_queue, pthread_run);

		SEMAPHORE_RESET(lock);
		reschedule(PS_WAIT_WAIT);

		/* Lock wait queue */
		lock = &(wait_semaphore);
		if (SEMAPHORE_TEST_AND_SET(lock)) {
			return(NOTOK);
		}
	}
	SEMAPHORE_RESET(lock);
	return(ret);
}

/* ==========================================================================
 * wait3()
 */
pid_t wait3(__WAIT_STATUS status, int options, void * rusage)
{
	semaphore * lock;
	pid_t ret;

	/* Lock wait queue */
	lock = &(wait_semaphore);
	if (SEMAPHORE_TEST_AND_SET(lock)) {
		pthread_yield();
	}
	while ((ret = machdep_sys_wait3(status, options | WNOHANG, rusage)) == OK) {
		if (pthread_queue_get(& wait_queue) == NULL) {
			/* Install signal handler */
			signal(SIGCHLD, sig_handler_real);
		}

		/* Enqueue thread */
		pthread_queue_enq(&wait_queue, pthread_run);

		SEMAPHORE_RESET(lock);
		reschedule(PS_WAIT_WAIT);

		/* Lock wait queue */
		lock = &(wait_semaphore);
		if (SEMAPHORE_TEST_AND_SET(lock)) {
			return(NOTOK);
		}
	}
	SEMAPHORE_RESET(lock);
	return(ret);
}

/* ==========================================================================
 * wait()
 */
pid_t wait(__WAIT_STATUS status)
{
	wait3(status, 0, NULL);
}
