/*
 * exception.c
 * Handle exceptions for the interpreter.
 *
 * Copyright (c) 1996 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>
 */

#define	DBG(s)

#include "config.h"
#include "config-std.h"
#include <signal.h>
#include <setjmp.h>
#include "gtypes.h"
#include "access.h"
#include "object.h"
#include "constants.h"
#include "classMethod.h"
#include "exception.h"
#include "baseClasses.h"
#include "lookup.h"
#include "thread.h"
#include "errors.h"
#include "itypes.h"
#include "md.h"

static void dispatchException(throwable*);

static void nullException(int);
static void arithmeticException(int);

extern object* exceptionObject;
extern vmException* cjbuf;

/*
 * Setup the internal exceptions.
 */
void
initExceptions(void)
{
DBG(	printf("initExceptions()\n");				)
#if !defined(DEBUG)
	/* Catch signal we need to convert to exceptions */
#if defined(SIGSEGV)
	signal(SIGSEGV, (SIG_T)nullException);
#endif
#if defined(SIGBUS)
	signal(SIGBUS, (SIG_T)nullException);
#endif
#if defined(SIGFPE)
	signal(SIGFPE, (SIG_T)arithmeticException);
#endif
#if defined(SIGPIPE)
	signal(SIGPIPE, SIG_IGN);
#endif
#endif
}

/*
 * Throw an internal exception.
 */
void
throwException(object* eobj)
{
	((throwable*)eobj)->backtrace = buildStackTrace(0);
	throwExternalException(eobj);
}

/*
 * Throw an exception.
 */
void
throwExternalException(object* eobj)
{
	if (eobj == 0) {
		fprintf(stderr, "Exception thrown on null object ... aborting\n");
		abort();
		exit(1);
	}
	dispatchException((throwable*)eobj);
}

static
void
dispatchException(throwable* eobj)
{
	exceptionInfo info;
	int i;
	object* str;
	vmException* frame;

	/* Search down exception stack for a match */
	for (frame = cjbuf; frame != 0; frame = frame->prev) {
		exceptionObject = (object*)eobj;

		/* Look for handler - jump into method if found */
		if (findExceptionBlockInMethod(frame->pc, exceptionObject->dtable->class, frame->meth, &info) == true) {
			cjbuf = frame;
			cjbuf->pc = info.handler;
			longjmp(cjbuf->jbuf, 1);
		}
		/* If not here, exit monitor if synchronised. */
		if (frame->mobj != 0) {
			soft_monitorexit(frame->mobj);
		}
	}

	fprintf(stderr, "%s\n", eobj->head.dtable->class->name);
	if (eobj->backtrace != 0) {
		for (i = 0; i < eobj->backtrace->size; i++) {
			str = ((object**)(eobj->backtrace+1))[i];
			if (str != 0) {
				fprintf(stderr, "%s\n", (char*)(str+1));
			}
		}
	}
	else {
		fprintf(stderr, "\t[no backtrace available]\n");
	}
	exit(1);
}

/*
 * Build an array of char[] for the current stack backtrace.
 */
object*
buildStackTrace(struct _exceptionFrame* dummy)
{
	char buf[100];
	object* str;
	object* strarray;
	int cnt;
	int i;
	int len;
	vmException* frame;

	/* First count the stack frames */
	cnt = 0;
	for (frame = cjbuf; frame != 0; frame = frame->prev) {
		cnt++;
	}

	/* Build an array of strings */
	strarray = alloc_objectarray(cnt, "[[C");
	assert(strarray != 0);

	cnt = 0;
	for (frame = cjbuf; frame; frame = frame->prev) {
		if (frame->meth != 0) {
			sprintf(buf, "\tat %s.%s(0x%x)",
				frame->meth->class->name,
				frame->meth->pair->s1,
				frame->pc);
			len = strlen(buf);
			str = alloc_array(len, TYPE_Char);
			assert(str != 0);
			strncpy((char*)(str+1), buf, len);
		}
		else {
			str = 0;
		}
		((object**)(strarray+1))[cnt] = str;
		cnt++;
	}
	return (strarray);
}

/*
 * Null exception - catches bad memory accesses.
 */
static
void
nullException(int sig)
{
#if !defined(DEBUG)
	/* Reset signal handler - necessary for SysV, does no harm for BSD */
	signal(sig, (SIG_T)nullException);
#endif
	throwException(NullPointerException);
}

/*
 * Division by zero.
 */
static
void
arithmeticException(int sig)
{
#if !defined(DEBUG)
        /* Reset signal handler - necessary for SysV, does no harm for BSD */
        signal(sig, (SIG_T)arithmeticException);
#endif
	throwException(ArithmeticException);
}
