TITLE High-Level Language to ISR interface
SUBTTL Interface Code/Declarations
NAME   DCBNFS
PAGE   60,132

COMMENT @

	See files COPYING and COPYWRT for copyright information. Copyright (C)
	Agricon International Inc. 1993

	This module provides the interrupt handler that allows communication
	to the network drive routines. It also contains a few modules to
	allow a High-Level Language to provide a test-bed interface for
	development. This avoids trying to debug via DOS once the program
	is resident in memory.

	Assembly for stand-alone TSR work is done with this command:

		masm /ML nfs;

	Assembly for test-bed support is done with this command:

		masm /ML /DTESTMODE nfs;

	This allows the High-level language to manipulate the functions without
	knowing too much about them. (Where they are, etc.) It also allows us
	to keep the functions the same as the ones used in the actual routine.

	@

;***** Equates *****************************************************************
;*******************************************************************************

ETN	equ	11		; Standard DOS name of 8 chars. + 3 char ext.
MAXPTH	equ	67		; Maximum path allowed, with 3 chars. overhead
BADF	equ	0feh	; Bad function code
DATDUMP	equ	69h	; requests data on NFS/packet protocol.

; System functions and calls

DOSCALL	equ	21h
EXIT	equ	4ch		; Exit with a return code
GETPSP3	equ	62h
SETPSP	equ	50h
GETVECT	equ	35h
SETVECT	equ	25h
NWREDIR	equ	11h		; Network redirector functions
MPX2F	equ	2fh
CHRDEV	equ	1223h		; Is file name same as character device?

; Other manifest constants

NEWLINE		equ	0ah	; standard UNIX newline char.
RETCHAR		equ	0dh	; the carriage return for our DOS console
cr		equ	RETCHAR
lf		equ	NEWLINE
FAIL		equ	2
TMRVCTR		equ	8	; Hardware timer interrupt vector.

; Frame management & other important constants

FRAMESIZ	equ	256	; Allow a max frame of 256 bytes
MAXFRAMES	equ	8	; Up to 8 levels of re-entry

; Installable file system/Network redirector constants

IFSSEEK		equ	21h	; Seek from end of file
IFSCLOSE	equ	6	; Close active file
IFSUNLCK	equ	0bh	; Unlock region of file
IFSSIZE		equ	0ch	; Return disk size (free space, etc.)
IFSUNKN		equ	2dh	; What is this function? Blindly ACK it.

;***** Records *****************************************************************
;*******************************************************************************

; Manage the flag bits in the current directory structure

flag	RECORD	NW:1,PH:1,JO:1,SB:1,SPARE:12=0	; Network, physical, join, subst

;***** Structures **************************************************************
;*******************************************************************************

include DOSSTRUC.INC

;**** Segments resident after TSR ********************************************
;*****************************************************************************

;**** Resident code **********************************************************
;	This segment is declared first so the linker will load it first. It is
;	where the routines that service the interrupts go. The TSR is started
;	only in the event of an interrupt. The startup code comes later and
;	is jettisoned.

ifdef	TESTMODE
MYCODE	equ	86	; just say anything...
else
	extrn	MYCODE:abs
endif

Restext	SEGMENT	PARA	PUBLIC	'TEXT'
	extrn	FCTTAB:byte,MAXFCT:abs,closemn:near
Restext	ENDS

ResStack	SEGMENT	WORD	'RSTACK'
	public	ResBase, Ftop

ResBase	equ	$
	db	FRAMESIZ * MAXFRAMES dup (0)
Ftop	equ	$-2
ResStack	ENDS

BUFFER	SEGMENT	PARA	PUBLIC	'BUFR'
BUFFER	ENDS

Resdata	SEGMENT	WORD	PUBLIC	'STATIC'
	extrn	OurSDA:dword,OurCDS:dword
	extrn	rpcfail:word,rpcxidf:word
	extrn	udpbsum:word,udpblen:word,udptot:word
	extrn	ipbdsum:word,ipfrgs:word,ipbdlen:word
	extrn	rxfails:word,txfails:word,sndfrgs:word,sndIP:word
ifdef	BLINDALLEY
	extrn	arpspin:byte
endif
	extrn	OurOS:byte
Resdata	ENDS

	ResGroup	GROUP	Restext,ResStack,BUFFER,Resdata

;**** Offsets used in program data structures

parent	equ	16h	; Where DOS stores the parent's PSP in our PSP
ljump	equ	0ah	; where execution resumes upon our termination

	extrn	dispatch:near

SUBTTL Resident code
PAGE+

Resdata	SEGMENT	WORD	PUBLIC	'STATIC'

	PUBLIC	drive_no,RootOffs,vlabel,OurPSP
	PUBLIC	INITLPATH,INITLPATHLEN,OldVect2F

ifdef	BLINDALLEY
	PUBLIC	OldVect8

OldVect8	dd	0
endif

OldVect2F	dd	cyaproc	; Original vectors we shall restore upon end
OurPSP		dw	0	; The segment of our psp before TSR activation
ResFrame	dw	offset ResGroup:Ftop
drive_no	db	5	;3

ifdef	TESTMODE
INITLPATH	db	'/usr/HOME\',0	; Mount point for HLL test-bed program
else
	PUBLIC	DRV_LTR

INITLPATH	db	'Megalon '	; Mount point for Normal DOS TSR.
DRV_LTR		db	'@:\',0
endif
INITLPATHLEN	equ	$-INITLPATH

vlabel		db	'AN ILLUS.ION',0

		even
RootOffs	dw	INITLPATHLEN-2

ifdef	TESTMODE
	PUBLIC	fnm1ea,fcb1ea

fnm1ea	dw	fname1
fcb1ea	dw	nm1fcb
endif

stats		db	256 dup (0)	; Track function calls
Resdata	ENDS

;**** Resident interrupt handling code **************************************
;****************************************************************************

;	Every effort has been made to keep segments "pure" --in other words,
;	segments that are to be executed are not written in this code. This
;	is to allow compatibility with protected environments. It is unlikely
;	that this TSR will be ported to such an environment, but portions of
;	it may be adaptable to other uses. The overhead of writing with this
;	in mind will pay off in those cases.

Restext	SEGMENT	PARA	PUBLIC	'TEXT'

	public	mpx_isr
	ASSUME	cs:ResGroup,ds:ResGroup,ss:ResStack,es:NOTHING

; Just in case there was no previous INT 2F handler, use this one to
; return harmlessly.

cyaproc	PROC	FAR
	iret
cyaproc	ENDP

; DOS communicates among multiple TSRs via this interrupt. If our ID matches
; see if we are being probed or asked to terminate. If termination is desired,
; do so only if we can release the vectors we chained. If we cannot then
; deactivate and report the problem.
;
mpx_isr	PROC	FAR
	pushf		; We test AL so save the flags from the caller

	cld
	cmp	ah,NWREDIR
	jz	test00		; wasn't meant for us...
	popf
	jmp	cs:[OldVect2F]
test00:	and	al,al		;is this roll call?
	jnz	test01
	mov	ax,MYCODE	; yes, answer w/ special code
	popf
	iret
test01:	cmp	al,0ffh		; terminate request?
	jne	test02

	push	ds		; get local data seg. for constants. Load
	mov	ax,ResGroup	; proper locations, unhook & terminate!
	mov	ds,ax
	mov	dx,es				; ES points to our original psp.
	mov	es,[OurPSP]			; the address was given in ES:BX
	mov	word ptr es:[ljump],bx		; We can trash all registers since
	mov	word ptr es:[ljump+2],dx	; they called us & should expect it
	call	turnoff				; Turn off the network drive
ifndef	TESTMODE
	call	closemn				; unmount file system.
endif
	call	getPSP		; Switch DOS back to our ORIGINAL psp so when we
	call	set_PSP		; ask to quit it re-starts the calling proc. as
	call	clrvect		; the parent.
	jc	no_joy
	xor	al,al
	mov	ah,EXIT
	int	DOSCALL
no_joy:	mov	ax,FAIL
	pop	ds
	popf
	iret

test02:	push	ds		; get resident data, check params.
	push	bx
	mov	bx,ResGroup
	mov	ds,bx
	call	ich		; is the call for us? CY if not.
	jnc	wasus

	pop	bx		; nope, restore context & chain to
	pop	ds		; next Network redirector
	popf
	jmp	cs:[OldVect2F]
wasus:	mov	bx,[ResFrame]		; We need a local stack for the level
	cmp	bx,offset ds:ResBase	; of function asked. Use a frame with
	jg	frmok			; many levels to allow re-entrancy.
	pop	bx			; Ensure we have enough stack left for
	pop	ds			; another frame. Gracefully fail if not.
	popf
	push	bp			; Locate flags saved by microcode during
	mov	bp,sp			; the interrupt.
	or	byte ptr [bp]+6,1	; Set carry to indicate failure.
	pop	bp
	iret

frmok:	mov	[bx],ss		; First, switch to new stack.
	mov	[bx-6],ax
	pop	ax		; clean out bx from stack b/c we want
	mov	[bx-4],ax	; to be able to set it here for some fcts
	mov	[bx-2],sp	; and save our old one at the top
	sub	bx,6		; of the frame. Use AX & BX so that
	mov	ax,ds		; AX on the new stack.
	mov	ss,ax
	mov	sp,bx
	sub	[ResFrame],FRAMESIZ	; point to next avail. frame
	pop	ax			; restore AX & allocate global
	push	ax		; variable for function value AX. We will
	push	bp		; be here a while so allow interrupts.
	mov	bp,sp
	sti
ifdef	BLINDALLEY
	mov	[arpspin],0ffh	; Disable ARP replies while we're active
	call	dispatch	; FCT dispatcher does the work...
	mov	[arpspin],0	; Re-enable ARP replies.
else
	call	dispatch	; FCT dispatcher does the work...
endif
	pop	bp		; frame pointer no longer needed.
	cli			; Prevent ints during stack restore
	mov	bx,[ResFrame]		; point to current frame
	mov	ss,[bx+FRAMESIZ]	; switch atomically
	mov	sp,[bx+FRAMESIZ-2]	; to caller's stack
	push	bp
	mov	bp,sp
	jc	badfct			; adjust original flags
	and	byte ptr [bp]+10,0feh
	jmp short coolxit
badfct:	or	byte ptr [bp]+10,1
coolxit:
	pop	bp
	add	bx,FRAMESIZ
	mov	ax,[bx-4]	; retrieve bx if modified in call
	push	ax		; put on old stack
	mov	ax,[bx]-6	; function return value
	mov	[ResFrame],bx	; update frame pointer
	pop	bx		; restore context and leave
	pop	ds
	popf
	iret

mpx_isr	ENDP

; This function retrieves the PSP of the current process. (Our caller)

getPSP	PROC	NEAR
	mov	ah,GETPSP3
	int	DOSCALL		; Get the PSP of the current process and put
	mov	es:[parent],bx	; that in our ORIGINAL psp as our parent.
	ret
getPSP	ENDP

; Here we want to set the PSP back to our ORIGINAL psp. Then when we terminate
; DOS will check it for the parent PSP, original control-break & etc vectors, and
; the address to resume execution in the parent.

set_PSP	PROC	NEAR
	mov	bx,[OurPSP]
	mov	ah,SETPSP
	int	DOSCALL
	ret
set_PSP	ENDP

; Simply turn off the network & physical bits of a drive in it's CDS. This
; turns off the drive.

turnoff	PROC	NEAR
	push	es
	les	di,[OurCDS]
	and	es:[di].flags,NOT ((MASK NW) OR (MASK PH))
	pop	es
	ret
turnoff	ENDP

; Here we attempt to restore the vectors before exiting. If we see they are not
; the same as the ones we replaced, then somebody has chained into this interrupt
; before us. All we can do is report the problem and exit.

clrvect	PROC	NEAR
	mov	ah,GETVECT	; Get the vector we desire to replace from DOS.
	mov	al,2fh
	int	DOSCALL
	cmp	bx,offset mpx_isr	; Compare the offset the original value
	jnz	tubed			; and if they differ we've trouble.
	mov	bx,es			; check the segment as well.
	mov	ax,cs
	cmp	ax,bx
	jnz	tubed

	push	ds
	lds	dx,[OldVect2F]	; Get the original vector and then ask DOS to
	mov	ah,SETVECT	; set that.
	mov	al,2fh
	int	DOSCALL
	pop	ds
ifdef	BLINDALLEY
	push	ds
	lds	dx,[OldVect8]	; Unhook the timer interrupt
	mov	ah,SETVECT
	mov	al,TMRVCTR
	int	DOSCALL
	pop	ds
endif
	ret
clrvect	ENDP

; If we couldn't clear the vectors, reset the PSP and abandon hopes of
; any uninstall.

tubed	PROC	NEAR
	mov	ax,[OurPSP]	; We couldn't clear some vectors. Load the
	mov	es,ax		; psp of the caller back as the current psp
	mov	bx,es:[parent]	; and return.
	mov	ah,SETPSP
	int	DOSCALL
	stc
	ret
tubed	ENDP

; Perform a few test to see if the call's for us.

ich	PROC	NEAR
	xor	bx,bx		; Make a note of all function calls
	mov	bl,al
	lea	bx,[bx+stats]
	inc	byte ptr [bx]

	cmp	al,DATDUMP	; User query about statistics?
	jne	other
	call	inform		; Yup, copy over the info & take
	stc			; easy way out.
	ret

other:	cmp	al,MAXFCT	; Don't index past table's end
	jle	in_range
	stc
	ret

in_range:
	push	ax			; beat up the stack some more...
	mov	bx,offset FCTTAB	; table lookup tells if function
	xlat	byte ptr cs:[0]		; is supported.
	cmp	al,BADF
	jne	testit
	pop	ax
	stc
	ret

testit:	cmp	al,IFSSEEK	; for Seek and close - unlock, simply
	jz	chkdrv		; verify the requested drive is us.
	cmp	al,IFSCLOSE	; otherwise we must check the path in
	jb	chkcds		; the CDS to be sure.
	cmp	al,IFSUNLCK
	ja	chkcds
	cmp	al,IFSUNKN
	jz	chkdrv
chkdrv:	mov	ax,es:[di].dev	; easy job. If = our drive # we're it!
	and	al,1fh		; es:di GIVEN IN FCT CALL!
	cmp	al,drive_no
	jz	kosher
	stc
kosher:	pop	ax
	ret

chkcds:	push	si
	push	di
	push	es
	les	di,dword ptr [OurSDA]		; Check against the
	cmp	[OurOS],3			; Use different offset if
	ja	DOS4				; not DOS version 3
	les	di,dword ptr es:[di].cdrvcds	; directory currently
	jmp short DOSfix
DOS4:	les	di,dword ptr es:[di].cdrvcds4
DOSfix:	mov	si,offset ds:INITLPATH		; active, up to the
	mov	bx,cx				; Root offset. Save CX here
	mov	cx,RootOffs
	rep cmpsb
ifdef	TESTMODE
	jcxz	big104
else
	jcxz	chkchr
endif
	stc

big104:	pop	es
	pop	di
	pop	si
	pop	ax
	mov	cx,bx		; Restore CX. (It's faster than stack)
	ret

ifndef	TESTMODE
chkchr:	cmp	al,IFSSIZE	; Disk space function
	jz	big104
	push	bx
	push	dx
	push	bp
	mov	ax,ss		; Restore DOS' data segment
	mov	ds,ax
	mov	ax,CHRDEV	; Is file a character device?
	int	MPX2F
	mov	ax,ResGroup
	mov	ds,ax
	pop	bp
	pop	dx
	pop	bx
	cmc			; CY means no, change CY to mean FAIL.
	jmp	big104
endif
ich	ENDP

; Write statistics collected into the address given in ES:DI. Note
; that these count values are only 0-65535 unless noted otherwise.
; Result is that 14 words are stored in the address pointed by ES:DI.

inform	PROC	NEAR
	mov	ax,[rpcfail]	; General Sun RPC failure count
	stosw
	mov	ax,[rpcxidf]	; Bad RPC call exchange ID count
	stosw
	mov	ax,[udpbsum]	; UDP packets with bad checksums
	stosw
	mov	ax,[udpblen]	; UDP packets with wrong lengths
	stosw
	mov	ax,[udptot]	; Total UDP packets received
	stosw
	mov	ax,[ipbdsum]	; IP packets with bad checksums
	stosw
	mov	ax,[ipfrgs]	; Total IP fragments RECEIVED
	stosw
	mov	ax,[ipbdlen]	; IP packets with incorrect lengths
	stosw
	mov	ax,[rxfails]	; Receiver failures (other than packet.)
	stosw
	mov	ax,[txfails]	; Transmitter failures (unavail, etc.)
	stosw
	mov	ax,[sndfrgs]	; Total IP fragments SENT
	stosw
	mov	ax,[sndIP]	; Total whole IP packets sent
	stosw
	lea 	ax,[stats]	; Network redirector function call counts
	stosw			; (Just SEG:OFS of table, counts are 0-256)
	mov	ax,ds
	stosw
	ret
inform	ENDP

Restext	ENDS

ifdef	TESTMODE

; Provide a high-level interface for the low-level file work.

_BSS	SEGMENT	WORD	PUBLIC	'BSS'
_BSS	ENDS

_DATA	SEGMENT	WORD	PUBLIC	'DATA'
_DATA	ENDS

CONST	SEGMENT	WORD	PUBLIC	'CONST'
CONST	ENDS

FAR_BSS	SEGMENT	PARA		'FAR_BSS'

	extrn	_AX:word,_BX:word,_CX:word,_DX:word
	extrn	_SI:word,_DI:word,_ES:word,_dver:byte

FAR_BSS	ENDS

	DGROUP	GROUP	_DATA,_BSS,CONST

data	SEGMENT	WORD	PUBLIC	'DATA'
data	ENDS

trans	SEGMENT	PARA	PUBLIC	'CODE'
	ASSUME	CS:trans

extrn	opts:near

opthook	PROC	FAR
call	opts
	ret
opthook	ENDP
trans	ENDS
	ASSUME	NOTHING

nfs_text	SEGMENT	WORD	PUBLIC	'CODE'

	public	_dosiface, _initstat

; Provide a way to fake the interrupt to our TSR. This is the interface from
; the high-level language to our routine. The registers are saved so that the
; calling program may inspect them from its data segment.

_dosiface	PROC	FAR

	ASSUME	NOTHING
	ASSUME	ds:FAR_BSS

	push	bp
	mov	bp,sp
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	push	es
	push	ds

	mov	ax,FAR_BSS	; Save this so that we can dump
	push	ax		; registers upon return
	push	ds
	mov	ds,ax
	mov	bx,[_BX]	; Allow simulation of register
	mov	cx,[_CX]	; parameters from DOS
	mov	dx,[_DX]
	mov	si,[_SI]
	mov	di,[_DI]
	mov	es,[_ES]
	pop	ds
	mov	ax,[bp+8]	; Get any extra arguments passed
	push	ax		; Simulate DOS' stack
	mov	ax,[bp+6]	; recover function request code
	or	ah,NWREDIR	; Multiplex interrupt function ID
	pushf			; fake an interrupt.
	call	mpx_isr
	add	sp,2		; throw away old arguments.
	pop	ds		; Now dump those registers
	mov	[_BX],bx
	mov	[_AX],ax
	mov	[_CX],cx
	mov	[_DX],dx
	mov	[_SI],si
	mov	[_DI],di
	mov	[_ES],es
	pop	ds

	pop	es
	pop	di
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	bp
	ret
_dosiface	ENDP

; This is the linkage from the high-level language to our routine's "private"
; data segments. Global addresses that would otherwise have been set via its
; startup routines are set here. A far call from the high-level language brings
; us here. The addresses of the simulated SDA and CDS are passed on the stack.

	ASSUME	cs:nfs_text,ds:NOTHING,es:ResGroup

_initstat	PROC	FAR
	push	bp
	mov	bp,sp
	push	si
	push	ds
	push	di
	push	es

	mov	di,data
	mov	ds,di
	call	opthook
	mov	di,ResGroup
	mov	es,di
	push	ds
	mov	di,FAR_BSS
	mov	ds,di
	mov	al,ds:[_dver]
	mov	[OurOS],al
	pop	ds
	lea	di,ResGroup:[OurSDA]
	lea	si,[bp+6]
	movs	word ptr [di],word ptr ss:[si]
	movs	word ptr [di],word ptr ss:[si]
	mov	di,word ptr [OurSDA]
	cmp	[OurOS],3
	ja	newdos
	lea	ax,[di].nm1fcb
	mov	[fcb1ea],ax
	lea	ax,[di].fname1
	mov	[fnm1ea],ax
	jmp short alldos
newdos:	lea	ax,[di].nm1fcb4
	mov	[fcb1ea],ax
	lea	ax,[di].fname14
	mov	[fnm1ea],ax

alldos:	lea	di,ResGroup:[OurCDS]
	movs	word ptr [di],word ptr ss:[si]
	movs	word ptr [di],word ptr ss:[si]

	lds	si,ResGroup:[OurCDS]
	mov	si,[si].rootoff
	mov	ResGroup:[RootOffs],si

	pop	es
	pop	di
	pop	ds
	pop	si
	pop	bp
	ret
_initstat	ENDP
nfs_text	ENDS
endif

SUBTTL	Documentation and Illustrations
PAGE+

COMMENT	@

Discription of frames:

					ResStack
		Level 0		+-----------------------+  T
		+-------------->|	   SS		|  |
		|		+-----------------------+
		|		|	   SP		|  F 
		|		+-----------------------+  R
		|		|	   AX		|  A
		|		+-----------------------+  M
		|		|	   BP		|  E
		|		+-----------------------+  S
		|		|			|  I
	+---------------+	|	   . 		|  Z
	|   ResFrame	|	|	   .		|  E
	+---------------+	|	   .		|
	   :	    !		|			|  |
	   :	    !		+-----------------------+  V
	   :	    +---------->|	   SS		|
	   :	    Level 1	+-----------------------+
	   :			|	   SP		|
	   :			+-----------------------+
	   :			|	   AX		|
	   :			+-----------------------+
	   :			|	   BP		|
	   :			+-----------------------+
	   :			|			|
	   :			|	   . 		|
	   :			|	   .		|
	   :			|	   .		|
	   :			|			|
	   :			+-----------------------+
	   +------------------->|	   SS		|
		Level 2		+-----------------------+
				|	   SP		|
				+-----------------------+
				|	   AX		|
				+-----------------------+
				|	   BP		|
				+-----------------------+

	Next we show what is saved on DOS' old stack and our
	new stack in the event you need to access DOS variables.

	Upon entry, the resident frame has the following values

	   FNFS stack				DOS' stack
	+-----------------------+	+-----------------------+
	|	SS (DOS)	|	| DOS call parameter(s) |
	+-----------------------+ ====> +-----------------------+
	|       SP (DOS)        |	| flags (from INT 2F)   |
	+-----------------------+	+-----------------------+
        |       BX (preserved)  |	| DOS' CS before INT    |
	+-----------------------+	+-----------------------+
	|       AX (see note 1) |	| DOS' IP (next instr.) |
	+-----------------------+	+-----------------------+
	|       BP (DOS' frame) |	| DOS' DS  (we pushed)  |
	+-----------------------+	+-----------------------+
	| IP from CALL CS:[BX]  |
	+-----------------------+
	| Any other registers   |
	|	*		|
	|	*		|
	|	*		|
	+-----------------------+
	| BP (for proc. frame)	|
	+-----------------------+
	| local variables	|
	|	*		|
	|	*		|
	|	*		|
	+-----------------------+
	| and so on...		|
	+-----------------------+

  NOTE 1
	There are some cases where DOS requires that BX and/or AX contain
	return values. YOU MUST GET THE 'IMAGES' OF THE REGISTERS ON THE
	STACK IF YOU WANT THOSE VALUES TO STICK. Setting the register in
	your function won't matter because exiting from FNFS will cause
	what was saved on the stack to go to the registers instead. Presently
	all the other registers must be saved in the procedure code if they
	are modified. NOTE also that BX is sometimes commented out of the
	save/restore process, depending on who last modified the DISPATCH
	procedure in DOSMAP.ASM!

	The following is a basic discussion of stack frames. Skip it if
	you know about such things in the context of high-level languages.

	Note that we use frames for some procedures. This way the BP always
	points to a copy of it's previous value. By using offsets from the
	BP we can access local variables saved when we subtract a value from
	the SP. We can also load the BP with the previous value and reach
	other stack frames from our parents.

	Ultimately we can get DOS' SS/SP or the registers we saved upon
	entry to FNFS. Thus we can retrive parameters DOS passed on *ITS*
	stack. (Happens for Open functions.)

	This trick lets us "push" parameters onto DOS' stack before we make
	it active. In this way we preserve "reentrancy" to a limited extent:
	we are limited by the amount of Master frames we can allocate before
	we run out of reserved stack space. Otherwise there is no limit to
	the number of times we can call ourselves. [CAVEAT: Presently there
	are SOME sections of FNFS that don't allow reentrancy. They must be
	fixed. ALSO we MUST call DOS with function 120C from INT 2FH when
	opening or creating FCB files. We cannot call DOS reentrantly this
	way.]
	
	@

		END
