TITLE	Network interface routines (UDP/IP)
SUBTTL	Packetizing functions
PAGE	60,132
NAME	IPUDPmod

COMMENT	@

	Agricon International Inc	David C. Brown IV	October 16,1992

	See files COPYING and COPYWRT for copyright information. Copyright (C)
	Agricon International Inc. 1993

	These functions manage the actual TCP/IP interface & packetizing.
	We send packets out, fragmenting as necessary. UDP protocol is used
	Which runs underneath TCP/IP & is CONNECTIONLESS. (No ACKs) Recevies
	are handled by interrupt. Upon a READ request, we wait for a reasonable
	time for data. If none occurs we timeout, else we re-assemble the IP
	frags as necessary, apply the TCP checksum, then the UDP sum. We return
	with the pointer to the start of the underlying protocol. (Usually SUN
	RPC) We check for ARPs via timer int. Packet driver interfaces are
	used.

	@

;***** Equates *****************************************************************
;*******************************************************************************

;***** Manifest constants ******************************************************

	PUBLIC	UDPMAX

UDPMAX	equ	4096			; Allow 4k data
MPKTSIZ	equ	UDPMAX + 196 + 68	; Above + 3 frags, RPC & NFS overhead.
cr	equ	0dh
lf	equ	0ah
TIMCH	equ	43h	; command register for timer/counter chip
TIMR0	equ	40h	; timer/counter register 0

;***** Packet Driver Equates ***************************************************

PKTADR	equ	6	; Get our hardware address.
PKTSND	equ	400h	; send a packet function
PKTCLS	equ	300h	; close packet handle
PKTOPN	equ	2	; open a packet handle (Internet interface)
PKTINF	equ	1ffh	; information on this packet driver
PKTTYP	equ	1400h	; request a given packet type
PKTINTR	equ	62h	; packet driver interrupt request

EADRS	equ	6	; size in bytes of ethernet address

;***** TCP/IP Equates **********************************************************

PROTUDP	equ	11h	; UDP protocol spec
TTLUDP	equ	30	; UDP time to live
IPVER	equ	45h	; Actually ip header len / 512 | IP version
IPMTU	equ	1500	; Max packet we can transmit over IP w/o fragmenting.
RUNT	equ	64	; Minimum packet size
IPFRAGM	equ	2000h	; Host ordered longword to flag fragmented packets
FRGTST	equ	3fffh	; Mask to hide "Don't fragment" flag.

IPTIMLIM	equ	546	; 30s limit for packet arrival

PM_PORT		equ	111	; port mapper's listen port

DOSDS	equ	40h	; segment for DOS' data
DOSTMR	equ	6ch	; offset to DOS' clock ticks

;****** Structures *************************************************************
;*******************************************************************************

;***** TCP/IP structures *******************************************************

; All fields must be in NETWORK order. (Reverse Endian for 80x86 family)

UDP	STRUC
srcport	dw	200h
dstport	dw	PM_PORT
udplen	dw	0	; Packet + udp header length
udpsum	dw	0	; zero this while calculating
UDP	ENDS

; Need to convert these fields to NETWORK order before transmission

IP	STRUC
iphlen	db	IPVER	; won't change unless we add ip options
iptos	db	0	; always for our UDP NFS implementation
iplen	dw	0	; size of entire packet w/o ether header
ipid	dw	0	; not necessarily sequential packet id
ipfrag	dw	0	; again, zero for UDP NFS
ipttl	db	TTLUDP	; time to live of 30 for UDP protocol
ipproto	db	PROTUDP	; UDP protocol
ipsum	dw	0	; Fill w/ UDP len for UDP sum, zero for IP sum calcs
ipsrc	dd	80e3a4c2h
ipdst	dd	80e3a4c1h
IP	ENDS

; Already in NETWORK order.

ETHER	STRUC
edest	db	0,0,1bh,01dh,0ch,0f2h
esrc	db	2,60h,8ch,1bh,0a3h,0a1h
etype	db	8,0
ETHER	ENDS

ARPTX	STRUC
brdcst	db	0ffh,0ffh,0ffh,0ffh,0ffh,0ffh	; Ether header dest & source
arpsrc	db	2,60h,8ch,1bh,0a3h,0a1h		; hardware addr. & ethertype
arptyp	dw	608h			; That's 806h in network order
arphwt	dw	100h			; Signal Ether H/W, N/W ordered
arpprot	dw	8			; Signal IP protocol, N/W ordered
arphwsz	db	6			; 6-byte H/W address size
arpprsz	db	4			; 4-byte protocol address size
arpopcd	dw	200h			; Signal ARP reply, N/W ordered
arptxet	db	2,60h,8ch,1bh,0a3h,0a1h	; Sender's H/W address again
arpip	db	80h,0e3h,0a4h,0c2h	; Sender's IP address
unktxet	db	0,0,0,0,0,0		; Unknown station's H/W address
unkip	db	80h,0e3h,0a4h,0c1h	; Unknown station's IP address
	dw	11 dup (0)		; Pad to a runt packet length
ARPTX	ENDS

SUBTTL Segments
PAGE+

;**** Segments ***************************************************************
;*****************************************************************************

Restext	SEGMENT	PARA	PUBLIC	'TEXT'
Restext	ENDS

ResStack	SEGMENT	WORD	'RSTACK'
ResStack	ENDS

BUFFER	SEGMENT	PARA	PUBLIC	'BUFR'

ARPpkttx	ARPTX	<>			; ARP reply buffer space
ARPpktrx	ARPTX	<,,,,,,,100h,,,,,>	; buffer for incoming ARP requests

HWPKTL	equ	size UDP + size IP + size ETHER

RXpkth	ETHER	<>		; Simplify packet filling/addressing by locating
	IP	<>		; fields we know we'll need anyway.
	UDP	<>
RXpktb	db	MPKTSIZ	dup (?)

public	TXpktb
TXpkth	ETHER	<>
	IP	<>
	UDP	<>
TXpktb	db	MPKTSIZ	dup (?)
BUFFER	ENDS

Resdata	SEGMENT	WORD	PUBLIC	'STATIC'
ifdef	BLINDALLEY
	extrn	OldVect8:dword
endif
Resdata	ENDS

	ResGroup	GROUP	Restext,ResStack,BUFFER,Resdata

ifdef	DEBUG
dupseg	SEGMENT	WORD	PUBLIC	'STATIC'
pktcpy	dw	?
dupseg	ENDS
endif
Resdata	SEGMENT	WORD	PUBLIC	'STATIC'

	PUBLIC	cannedip

ARPtxad	dd	ARPpkttx
ARPrxad	dd	ARPpktrx
IPtxad	dd	TXpkth
IPrxad	dd	RXpkth
IPrxpt	dd	RXpkth

ipseq	dw	1962	; sequence number for outbound IP packets

timerem	dw	0	; how many cycles of timer port before we quit?
lastval	dw	0	; timer port 0's last count.

arphndl	dw	0
iphndl	dw	0
arpsiz	dw	0	; ARP packet size received
ipsiz	dw	0	; IP packet size received
bufsiz	dw	HWPKTL + MPKTSIZ

cannedip	db	80h,0e3h,0a4h,0c2h,80h,0e3h,0a4h,0c1h
hosteadr	dw	3 dup (0)

; Protocol statisitcs

PUBLIC	udpbsum, udpblen, udptot, ipbdsum, ipfrgs, ipbdlen, iptot
PUBLIC	rxfails, txfails, sndfrgs, sndIP

	even
udpbsum	dw	0	; # UDP checksum errors
udpblen	dw	0	; UDP length wrong
udptot	dw	0	; Total UDP packets received
ipbdsum	dw	0	; # IP checksum errors
ipfrgs	dw	0	; Total IP fragments received
ipbdlen	dw	0	; IP length wrong
iptot	dw	0	; Total atomic/assembled-fragment IP packets
rxfails	dw	0	; # received packet failures
txfails	dw	0	; # transmit packet failures
sndfrgs	dw	0	; # IP fragments sent
sndIP	dw	0	; Total atomic/assembled-fragment IP packets

ifdef	BLINDALLEY
	PUBLIC	arpspin

arpspin	db	0ffh	; Hold off auto ARP... Number of timer ticks remaining this second.
endif
ifdef	DEBUG
pktdup	dd	pktcpy
endif
Resdata	ENDS

SUBTTL Resident code
PAGE+

;	Every effort has been made to keep segments "pure" --in other words,
;	segments that are to be executed are not written in this code. This
;	is to allow compatibility with protected environments.

Restext	SEGMENT	PARA	PUBLIC	'TEXT'

	PUBLIC	ippkttx, ippktrx, pkthang
	ASSUME	cs:ResGroup,ds:ResGroup,ss:ResStack,es:BUFFER

; Close the packet driver handles & get out. (This IS important because the
; packet driver will attempt to call a receiver routine for each packet it
; gets matching types we have handles for. If an ARP comes in after we've quit
; we get a call to nowhere with dire consequences.)

pkthang	PROC	NEAR
	push	bx		; First close the IP handle. (Broadcasts will
	push	dx		; also be closed with this.)
	mov	ax,PKTCLS
	mov	bx,[iphndl]
	call	PKTCALL		; Ignore errors. (What more could we do?)

	mov	ax,PKTCLS	; Close ARP handle & be gone.
	mov	bx,[arphndl]
	call	PKTCALL
	pop	dx
	pop	bx
	ret
pkthang	ENDP

; This function takes the static packet filled by an upper level routine and
; prepares the UDP & IP headers & zaps off the packet. The caller should call
; this routine and then call ippktrx to poll for the received packet. This is
; because ippktrx itself checks for any ARP requests and sends a response. (We
; can't re-entrantly call the packet driver to ARP it's own request because
; it wasn't designed to be re-entrant. Thus we lose the thread once the ARP
; packet is received & never get a chance to ACK it unless we poll somehow.)

; Packet size is passed on the stack as a word.

	ASSUME	es:BUFFER

ippkttx	PROC	NEAR
	push	bp
	mov	bp,sp
	push	cx
	push	di
	push	es
	push	si
	push	ds

	test	[arpsiz],0ffh
	jnz	neednoarp
	call	SNDARP
neednoarp:
	les	di,[IPtxad]	; zero out the fields in the header.
	lea	di,[di]+size ETHER
	push	es
	push	di
	call	clrpkt
;
;------ Init the IP header fields
;
	push	es			; save es:di for checksum computation.
	push	di
	mov	es:[di].ipproto,PROTUDP
	mov	ax,[bp+4]
	add	ax,size UDP		; UDP + data size only!
	mov	[bp+4],ax
	push	ax			; save for later call to checksum
	xchg	ah,al			; convert to network order.
	mov	es:[di].ipsum,ax
	lea	si,[cannedip]		; for now just copy over the statically
	mov	cx,4			; defined Internet addresses
	lea	di,[di].ipsrc
	rep movsw			; exit pointing to start of UDP packet
	mov	es:[di].udplen,ax	; finally note UDP's length

	mov	ax,[bp+6]		; Source port
	xchg	ah,al			; into network order
	mov	es:[di].srcport,ax
	mov	ax,[bp+8]		; Destination port
	xchg	ah,al			; network order
	mov	es:[di].dstport,ax
;
;------ need to compute the checksum for the IP + UDP + data
;
	pop	ax		; checksum will expect the far address and THEN
	add	ax,size IP + 1	; the amount of data IN WORDS. (round up)
	shr	ax,1		; --the address was saved long ago.
	push	ax
	call	headsum		; returns result in network order.
	mov	es:[di].udpsum,ax
	call	doiptx

	pop	ds
	pop	si
	pop	es
	pop	di
	pop	cx
	pop	bp
	ret	6
ippkttx	ENDP

; This routine will send the packet under IP protocol, fragmenting as needed. It
; expects the packet to begin at the buffer start and will exit with CY set if
; unable to send any fragment. Typically, the first call points to a UDP header.

;------ Packet driver will allow only IPMTU bytes for ENTIRE packet! Trim this
;	packet down if needed but be sure we stop modulo 8 otherwise the recipient
;	won't get the right fragment offsets upon reassembly.
PKTMTU = (IPMTU - size IP) AND 0fff8h 

doiptx	PROC	NEAR
	push	cx
	push	si
txfrg:	lea	ax,es:[TXpktb-size UDP]	; compute offset into packet
	sub	ax,di			; easier to go negative than swap regs.
	jz	atomic
	inc	[sndfrgs]		; subsequent fragments send count.
atomic:	neg	ax
	mov	cl,3			; Frags size is given divided by 8
	shr	ax,cl
	cmp	[bp+4],PKTMTU+1		; Is this big enough to need frags?
	jb	frflgn
	or	ax,IPFRAGM		; yeah, set packet's flag, clear CY
	test	ax,NOT IPFRAGM		; only 1st frag will have offset = 0
	jnz	frflgn
	inc	[sndfrgs]		; special case for 1st frag sent
frflgn:	xchg	ah,al
	mov	es:[di-size IP].ipfrag,ax	; WRITE: flag & offset ELSE zero
	mov	ax,[bp+4]			; recover the length
;
;------ Note here that CY is set from CMP [BP+4],... already. Else OR clears it
;
	jb	dntfrg			; (Could do IPMTU + UDP header
	mov	ax,PKTMTU		;  but that complicates algorithm.)
dntfrg:	mov	cx,ax
	lea	di,[di-size IP]		; point back to start of IP packet
	add	cx,di			; Make a pointer to the following
	push	cx			; packet we will send and save it.
	add	ax,size IP		; include IP header's length now.
	mov	cx,ax
	xchg	ah,al
	mov	es:[di].iplen,ax
	mov	es:[di].iphlen,IPVER	; static for our purposes
	mov	es:[di].ipttl,TTLUDP
	mov	ax,[ipseq]
	xchg	ah,al
	mov	es:[di].ipid,ax
	mov	ax,(size IP) / 2	; sum is based on word size.
	push	es			; where the data is...
	push	di
	push	ax			; how much data....
	mov	es:[di].ipsum,0
	call	headsum			; sum comes back in network order
	mov	es:[di].ipsum,ax	; save the checksum!

	mov	ax,PKTSND	; to send a packet we only need the pointer
	push	ds
	mov	si,es			; packet driver wants data in DS:SI
	mov	ds,si
	lea	si,[di-size ETHER]	; Back up to ether header.
	add	cx,size ETHER		; Include header in byte count
	cmp	cx,RUNT			; Round up info to minimum size.
	jge	txrnnt			; (Fragmenting could possibly leave
	mov	cx,RUNT			; a remainder < RUNT)
txrnnt:	push	dx
	call	PKTCALL
	pop	dx
	pop	ds
	pop	di			; Pointer to DATA just sent
	jc	iptxbd

	sub	[bp+4],PKTMTU		; update bytes remaining count
	jbe	iptxnd
	push	ds
	lds	si,[IPtxad]			; Now prepare next fragment.
	lea	di,[di-size ETHER]
	mov	cx,(size IP + size ETHER) / 2	; Copy over ether & IP headers
	cli					; (Previous data overwritten)
	rep movs word ptr [di],word ptr es:[si]
	sti
	pop	ds
;xor	cx,cx			; We seem to overrun the packet
;dec	cx			; driver buffer. Just kill some
;loop	$			; time in this loop.
	call	txdly
txszok:	jmp	txfrg			; Returns pointing to next data byte.
iptxnd:	clc
	inc	[sndIP]
iptxit:	inc	[ipseq]			; changes with each packet we send.
	pop	si
	pop	cx
	ret
iptxbd:	inc	[txfails]
	jmp	iptxit
doiptx	ENDP

; Clear out the header part of the packet so that checksums may be computed
; Far pointer to buffer is on the stack
; Size is fixed to IP+UDP header lengths.

clrpkt	PROC	NEAR
	push	bp
	mov	bp,sp
	push	cx
	push	di
	push	es

	les	di,[bp+4]			; Packet address
	mov	cx,(size IP + size UDP) / 2	; Assumes UDP/IP protocol.
	xor	ax,ax
	rep stosw

	pop	es
	pop	di
	pop	cx
	pop	bp
	ret	4
clrpkt	ENDP

; Check for a packet. Entry: ES:DI points to field showing size of
; packet. Exit: CY if timeout, else no CY & ES:[di] = byte count

pktwait	proc	near
	mov	[timerem],IPTIMLIM	; timeout based on 18.2 ticks/second
	mov	[lastval],0		; clear counter tracking register
pollip:	test	word ptr es:[di],0ffffh	; Any bytes received?
	jz	kpwtg
	ret
kpwtg:	call far ptr ticktoc		; don't sit here forever, timeout if
	jz	timeup			; necessary
	test	[arpsiz],0ffffh
	jz	pollip			; nothing to do but wait some more...
	call	SNDARP
	jmp	pollip
timeup:	mov	word ptr es:[di],0
	inc	[rxfails]
	stc
	ret
pktwait	endp

; Here we poll both for the IP packet we want, and any ARP packets that may
; come up from our original IP tx. If we don't answer the ARP we're never going
; to get the IP reply. If a bum packet was received, return with an error and
; let the caller decided if it wants to wait some more. We will time out in
; this loop as well. Return with ES:DI pointing to data AFTER UDP protocol header,
; AX = # bytes specified in UDP protocol size field, HOST order.

ippktrx	PROC	NEAR
	mov	di,ds		; Initially, check field in DS, but
	mov	es,di		; tester expects ES to have segment
	lea	di,[ipsiz]
	call	pktwait
	les	di,[IPrxad]		; provide a pointer to the packet
	jnc	gotone
	ret
gotone:	push	di			; Save zeroed buffer ptr on stack
	call	frghack
	jc	rxxit			; oops, bad packet.
	call	validudp		; UDP will adjust pointers after checksum
ifdef DEBUG
	jc	rxxit
	mov	word ptr [pktdup],0
endif
	inc	[udptot]
rxxit:	mov	[bufsiz],HWPKTL+MPKTSIZ
	mov	[ipsiz],0
	pop	word ptr [IPrxpt]	; reset buffer pointer to zero
	ret
ippktrx	ENDP

; We really can't handle fragments out of sequence. This is just to be sure we don't
; start with the trailing fragment of a previous IP packet that was re-sent. (It happens)
; There's not a whole lot we can do for multiple packets from other sources or packets
; that overlap or are out of sequence without consuming more memory.
; If the first packet in the buffer has an offset, skip over subsequent packets until one
; with no offset is found.

frghack	PROC	NEAR
	push	cx
	mov	cx,[ipsiz]	; can only search amount received...
newip:	test	es:[di + 2 + size ETHER].ipfrag,NOT 20h	; N/W vs. Host
	jz	okrxip		; If offset > 0 it's not right
	sub	cx,es:[di]	; so step over it but don't waste time
	add	di,es:[di]	; looking through parts of buffer holding
	add	di,2		; nothing.
	jcxz	njrxip
	jmp	newip
okrxip:	pop	cx
	call	validip
	ret
njrxip:	stc
	pop	cx
	ret
frghack	ENDP

; Here we check the packet to make sure it is a valid IP packet. First the
; necessary protocol fields and IP addresses are checked. Then we make sure
; the other data is consistent and perform a checksum. If all passes the
; IP header is cleared out so that the UDP header may be validated. Entry:
; ES:DI->byte count of next packet. Exit: ES:DI->IP header, AX = UDP size
; in NETWORK order.

validip	PROC	NEAR
	mov	ax,es:[di]		; Get # bytes RX'ed, this packet
	and	ax,ax
	jnz	iphere
	call	pktwait
	jnc	iphere
	ret
iphere:	lea	di,[di+2+size ETHER]
	push	si
	push	di
	lea	si,[cannedip]		; be sure we're the intended target
	lea	di,[di].ipdst
	cmpsw
	jnz	iphose
	cmpsw
iphose:	pop	di
	jnz	ipquit

	cmp	es:[di].iphlen,IPVER	; Make sure we're dealing with IP
	jnz	ipquit
	cmp	es:[di].iptos,0		; and no options...
	jnz	ipquit			; RX len = packet length + IP +
	push	es
	push	di
	mov	ax,(size IP) / 2	; remember, routine wants words...
	push	ax
	call	headsum			; checksum over packet with *old*
	and	ax,ax			; checksum will produce a zero result.
	jnz	ipquit
	mov	ax,es:[di].iplen	; protocol bytes shipped, NW to
	xchg	ah,al			; HOST order + ethernet overhead
	add	ax,size ETHER			; - hardware bytes rec.
	sub	ax,es:[di - 2 - size ETHER]	; should be less or = 0.
	ja	ipbdle

	mov	ax,es:[di].ipfrag	; Fragment? Round up all IP frags into
	xchg	ah,al			; big packet, then treat it as UDP
	and	ax,FRGTST
	jz	nofrag
	inc	[ipfrgs]
	cmp	ax,IPFRAGM		; 2000h -> more fragments
	jz	frstfr
	pushf				; save frag test flags.
	sub	[ipsiz],size ETHER+size IP
	call	unite			; append this frag to our root
	popf				; recover flags to see if this is end.
	jl	frgout			; Yes, last frag so quit.
	call	validip			; check it out. (recursively)
	jc	ipquit			; propagate any failures upward...
	jmp short frgout
frstfr:	push	di
	lea	di,[di-size ETHER]	; look back to packet start
	add	di,es:[di-2]		; offset to next IP packet
	call	validip			; check it out. (recursively)
	pop	di
	jc	ipquit			; propagate any failures upward...
	xor	ax,ax
	mov	es:[di].ipfrag,ax

nofrag:	mov	es:[di].iplen,ax	; Now zero out the IP fields and
	mov	es:[di].iphlen,al	; set up for checking the
	mov	es:[di].ipid,ax		; UDP sum.
	mov	es:[di].ipttl,al
	mov	ax,es:[di+size IP].udplen
	mov	es:[di].ipsum,ax
frgout:	clc
	pop	si
	ret
ipquit:	inc	[ipbdsum]
	stc
	pop	si
	ret
ipbdle:	inc	[ipbdlen]
	stc
	pop	si
	ret
validip	ENDP

; This function copies a given IP fragment to the offset specified in its
; header. ES:DI is expected to point to the IP header in the current frag.
; AX is supposed to contain the ipfrag field of the packet, HOST order.
; UPON EXIT ES:DI will be pointing to the start of the next packet in the
; buffer.

unite	PROC	NEAR
	push	cx		; preserve registers
	push	si
	mov	cl,3		; convert back to bytes (Frag flag rolls off)
	shl	ax,cl			; (already in host order)
	add	ax,2+size ETHER+size IP	; Offset to data start
	lea	si,[di+size IP]		; Now append data frag to previous one.
	lea	di,[di-size ETHER]
	mov	cx,di
	add	cx,es:[di-2]		; recover RX'ed length
	push	cx			; save as ptr to next frag (packet)
	mov	cx,es:[di-2]		; get original byte count
	sub	cx,size IP + size ETHER	; don't copy frag's IP stuff
	shr	cx,1			; convert to word count
	mov	di,word ptr [IPrxad]
	add	di,ax			; back to parent's ending...
	cli
	rep movs word ptr [di],word ptr es:[si]
	sti
	pop	di			; recover next frag ptr...
	pop	si
	pop	cx
	ret
unite	ENDP

; This function has a pretty simple job. It computes a checksum on the
; UDP packet and ensures the length is meaningful. The port fields are
; left alone for the next level in the protocol stack. It returns the
; # data bytes *AFTER* the UDP header in AX. Entry: ES:DI-> IP header.

validudp	PROC	NEAR
	push	es
	push	di
	lea	di,[di+size IP]		; skip over the old IP part
	mov	ax,es:[di].udplen
	xchg	ah,al			; convet to host order
	add	ax,size IP		; checksum is over both packets
	shr	ax,1			; computed in words...
	push	ax
	call	headsum			; again, we seek a zero sum....
	and	ax,ax
	jnz	bdudpc

	mov	ax,[ipsiz]		; Check UDP size & come back with
	sub	ax,HWPKTL				; packet size in
	xchg	ax,es:[di].udplen	; Host order for txed size...
	xchg	ah,al			; back to network order...
	sub	ax,size UDP		; less protocol's overhead
	cmp	ax,es:[di].udplen
	ja	bdudpt			; again, allow for RUNT or greater.

	lea	di,[di+size UDP]	; return pointing to actual packet data
	clc
	ret
bdudpc:	inc	[udpbsum]		; update defective checksum count
	stc
	ret
bdudpt:	inc	[udpblen]		; update defective length count
	stc
	ret
validudp	ENDP

; Compute the UDP/IP style checksum for the # of bytes at the Far address
; on the stack. (# bytes first, address second)

headsum	PROC	NEAR
	push	bp
	mov	bp,sp
	push	bx
	push	cx
	push	dx
	push	si
	push	ds

	xor	bx,bx		; dx:bx hold the longword sum so clear them
	mov	dx,bx
	lds	si,[bp+6]	; Where is the data?
	mov	cx,[bp+4]	; how much?
	and	cx,0fffch	; deal with longwords first

lwdsum:	lodsw
	xchg	ah,al		; sum is computed on NETWORK bytes
	add	bx,ax		; accumulate
	adc	dx,0		; upper word of short = 0 explicitly.
	loop	lwdsum		; => unsigned math. Do while longwords...
	add	bx,dx		; fold upper word (short carries) back.
	adc	bx,0
	xor	dx,dx		; clear high word again
	mov	cx,[bp+4]	; what was that count again...
	and	cx,3
	jcxz	sumdun

wrdsum:	lodsw
	xchg	ah,al		; now finish up for the remaining words
	add	bx,ax
	adc	dx,0
	loop	wrdsum
	add	bx,dx
	adc	bx,0

sumdun:	not	bx		; return complement in network order
	mov  	al,bh
	mov	ah,bl

	pop	ds
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	bp
	ret	6
headsum	ENDP

; This function provides a way to time out by just observing counter 0
; rather than taking over one of the timer chip's count registers explicitly.
; It comes from Clarkson University's packet drivers, Russell Nelson. I have
; modified the code somewhat.

ticktoc	PROC	FAR
	xor	al,al		; Latch the count in timer 0
	out	TIMCH,al
	in	al,TIMR0	; now bring in the values.
	mov	ah,al
	in	al,TIMR0
	xchg	ah,al
	cmp	ax,[lastval]	; If the count's less than present we've
	mov	[lastval],ax	; rolled over. In that case decrement the
	jbe	tickoff		; time remaining.
	dec	[timerem]
	ret
tickoff:
	or	sp,sp		; don't send back 0 unless we've timed out!
	ret
ticktoc	ENDP

; Checks ticks off of DOS' clock to be processor speed independent. This
; is to allow the packet driver to send a packet before we attempt to send
; another.

txdly	PROC	NEAR
	push	ax
	push	cx
	push	di
	push	ds
	mov     ax,DOSDS
	mov     ds,ax
	mov     di,DOSTMR
	mov     ax,[di]
	add     ax,3
chkmre:	mov	cx,-1
	loop    $
	cmp	ax,[di]
	ja	chkmre
	pop	ds
	pop	di
	pop	cx
	pop	ax
	ret
txdly	ENDP

; The "central" calling routine for the packet driver. Since packet driver IRQ's won't
; be known until run-time we call this function who calls the packet driver. Thus we need
; only fix up 1 routine at run time. Since we have the resident code and data segments in
; a common group, we should be able to use the group selector to alias over the code
; segment and write in the actual IRQ once it is known.

PKTCALL	PROC	NEAR
	int	PKTINTR
	ret
PKTCALL	ENDP

; Another hook function so that far (transient) routines can call the near packet request
; function. Should be less overhead than having the resident & transients make far calls
; all the time.

farpkt	PROC	FAR
	call	PKTCALL
	ret
farpkt	ENDP

; Emit the Address Resolution Protocol (ARP) response. We shouldn't have to
; rewrite our address, IP address or the other fields like opcode because this
; packet is a static template in memory. Just use the incoming ARP request to
; get the requestor's hardware & IP addresses. If it's not an ARP to us or it's
; not an ARP request then ignore it but clear RX flag so we can receive more
; ARPs.

SNDARP	PROC	NEAR
	push	cx
	push	si
	push	ds
	push	di
	push	es

	les	di,[ARPrxad]	; be sure the ARP was directed towards us
	lea	di,[di].unkip
	lea	si,[cannedip]
	cmpsw
	jnz	txarpf
	cmpsw
	jz	txarp1
txarpf:	mov	[arpsiz],0	; Clear out size of last ARP request
	pop	es
	pop	di
	pop	ds
	pop	si
	pop	cx
	ret
txarp1:	mov	di,word ptr [ARPrxad]
	cmp	es:[di].arpopcd,100h	; Request for ARP too? (Network order)
	jnz	txarpf
	les	di,[ARPtxad]	; Fill our destination address
	push	ds
	push	di		; we'll want this address again below...
	lds	si,[ARPrxad]	; address of last ARP sender we noticed.
	lea	si,[si].arptxet
	push	si		; we'll want this address later too.
	mov	cx,3
	rep movsw		; ARP back to requestor, don't broadcast it!
	pop	si		; recover sender's address location
	pop	di		; recover TX packet's base
	lea	di,[di].unktxet	; Now copy sender into his own field in ARP
	mov	cx,5		; rather than leave it zero. But continue on
	rep movsw		; and copy over the IP field sent too.
	pop	ds
	mov	[arpsiz],0	; Clear out size of last ARP request to enable
	pop	es		; receiving more ARP packets.
	pop	di

	mov	ax,PKTSND	; usual stuff, must have
	mov	cx,size ARPTX	; how many bytes and where they
	lds	si,[ARPtxad]	; may be found.
	push	dx
	call	PKTCALL
	pop	dx

	pop	ds
	pop	si
	pop	cx
	ret
SNDARP	ENDP

ifdef	BLINDALLEY
; Here is a timer vector hook. I didn't want to do this but is seems the only
; way we can free a thread to acknowledge an ARP request.
; We check ARPSPIN to keep the # of ARP responses down and as a semaphore while
; the NFS program is active. If ARPSPIN < 0 we don't do anything. The program
; checks for ARPs itself so that's no big deal. Otherwise we only ARP if the
; counter reaches 0.

	PUBLIC	tmrarp

tmrarp	PROC	FAR
	push	ds
	push	ax
	pushf
	mov	ax,ResGroup
	mov	ds,ax
	cmp	[arpspin],0
	jl	nxthk
	dec	[arpspin]	; Hold down ARP responses to about 1 per second.
	jg	nxthk		; was jnz but this could hang us up.
	mov	[arpspin],18	; based on 18.2 calls/second
	call	SNDARP
nxthk:	popf
	pop	ax
	pop	ds
	jmp	cs:[OldVect8]
tmrarp	ENDP
endif

; Automaton for ARP requests. The packet driver calls this routine to get
; a buffer. If we return a buffer then the (presumed) ARP packet is copied
; into the buffer on the subsequent call. We fill a field with the number of
; bytes received. This will queue a polling routine that there is a request
; pending. IF THERE IS NO POLLING (As in no pending NFS request/replies) then
; the ARP won't be serviced until one does occur.

; NOTE: We now tell the assembler to assume ES points to ResGroup so any refs
; we make to variables it knows are in ResGroup will be AUTOMATICALLY prefixed
; with ES overrides. At the end of these procs we need to set ES to
; give the packet driver what it wants. DON'T ADD ANY ResGroup REFERENCES
; AFTER AN *les di,[whatever]* OR YOUR PROGRAM WILL BE HOSED!

ARPRXI	PROC	FAR

	ASSUME	ds:NOTHING
	ASSUME	cs:ResGroup,es:ResGroup

	test	ax,0ffffh		; Is this the first call?
	jz	newarp			; Yup, save the data.
	ret
newarp:	cmp	cx,size ARPTX		; Will the packet fit?
	ja	arpfl
	mov	di,ResGroup		; First call we return es:di w/
	mov	es,di			; buffer address.
	test	[arpsiz],0ffffh		; Don't get a packet if one's
	jnz	arpfl			; still there.
	mov	[arpsiz],cx	
	les	di,[ARPrxad]		; supply requested buffer ptr
	ret

arpfl:	xor	di,di			; Can't store packet so return
	mov	es,di			; null buffer & it will be dropped.
	ret
ARPRXI	ENDP

; This function stows the incoming IP packet if there's room in the buffer. If
; not it is forced to just drop the packet.

IPRXI	PROC	FAR
	test	ax,0ffffh		; First call?
	jz	newipr			; Yup, see if we can copy data.
	push	si			; Save SI & zero next word AFTER
	add	si,ds:[si - 2]		; this packet. If somebody is
	mov	word ptr ds:[si],0	; polling this word. RX is blocked
	pop	si			; on THIS pkt, thus no race condition

ifdef	DEBUG
	mov	di,ResGroup
	mov	es,di
	test	word ptr [pktdup + 2],0ffffh
	jz	blowoff
	push	di
	push	si
	push	cx
	push	es
	les	di,[pktdup]
	mov	cx,ds:[si - 2]
	ror	cx,1
	rep movsw
	jnc	wrdsok
	movsb
wrdsok:	pop	es
	mov	word ptr [pktdup],di
	pop	cx
	pop	si
	pop	di
blowoff:
endif
	ret
newipr:	mov	di,ResGroup		; First call we return es:di w/
	mov	es,di			; buffer address.
	cmp	cx,[bufsiz]		; Will the packet fit?
	ja	ipfl
	add	[ipsiz],cx
	add	cx,2			; Store # bytes this packet too.
	sub	[bufsiz],cx		; update space remaining
	mov	di,word ptr [IPrxpt]	; Load DI w/ *present* buffer ptr
	add	word ptr [IPrxpt],cx	; Update buffer ptr for next call...
	sub	cx,2			; adjust count back to normal
	mov	es,word ptr [IPrxpt+2]	; BEWARE: we've changed ES!!!!
	mov	es:[di],cx		; save # bytes @ packet start
	add	di,2
	ret

ipfl:	xor	di,di			; Can't store packet so return
	mov	es,di			; null buffer & it will be dropped.
	ret
IPRXI	ENDP
Restext	ENDS

SUBTTL	PKT Driver Locate/setup
PAGE+

COMMENT	@

	Agricon International Inc	David C. Brown IV	April 24,1993

	The function of this module is to look for a valid packet driver in the
	spirit of the other routines such as NCSA telnet: by eaxaming eligible
	interrupt vectors until the packet driver's ID string is found. Once that
	happens we patch in the interrupt vector we found. (We are using a GROUP
	for the resident code & data thus the write will be done with a DATA
	segment that happens to reference the same memory as the CODE segment.
	In this way the write should be legal, EVEN in protected mode environments.
	Although our TSR won't have too much applicability there this is the way
	we should write code.)

	So then, we get info about the driver and query our hardware address from
	it. Once we know that we can use an ARP request to find out what address
	our NFS host uses. Those should be all the things necessary.

	In the future we can RARP to get our own IP address, and knowing this we
	could then query a name server for our host's IP address. Those are
	reserved for the future.

	These routines are not needed after startup and are thus in the transient
	portion of the code. ES is managed as necessary by each routine, DS is
	set to ResGroup upon entry to pktinit and should remain that way until exit.

	@

;***** Equates *****************************************************************
;*******************************************************************************

DOSPRT	equ	9
DOSCALL	equ	21h

;***** Packet Driver Equates ***************************************************

PKTMIN	equ	60h	; packet driver interrupt request starting value.
PKTMAX	equ	80h	; Maximum value for driver interrupt requests

SUBTTL Segments
PAGE+

;**** Segments ***************************************************************
;*****************************************************************************

Resdata	SEGMENT	WORD	PUBLIC	'STATIC'

ARPetyp	dw	608h	; Network ordered ARP packet type
IPetyp	dw	8	; Network ordered IP packet type

pdrvtyp	dw	0dh		; Used by packet driver to distinguish itself
pdrvcls	db	1		; The physical interface used.
pktirq	db	PKTMIN		; The IRQ used to reach the packet driver.

pktstrg	db	'PKT DRVR',0
PKTSTGL	equ	$ - offset pktstrg - 1
Resdata	ENDS

SUBTTL Transient code
PAGE+

; Strings and other things not needed after startup go here.

data	SEGMENT	WORD	PUBLIC	'DATA'
nodrv	db	'No packet driver found, exiting.',cr,lf,'$'
data	ENDS

;	Every effort has been made to keep segments "pure" --in other words,
;	segments that are to be executed are not written in this code. This
;	is to allow compatibility with protected environments.

trans	SEGMENT	PARA	PUBLIC	'CODE'

	PUBLIC	pktinit
	ASSUME	cs:trans,ds:ResGroup,ss:ResStack,es:BUFFER

; Initialize the packet driver and get handles for given packet types.

pktinit	PROC	FAR
	push	bx
	push	cx
	push	dx
	push	si
	push	ds
	push	di
	push	es

	mov	di,ResGroup
	mov	ds,di
	call	pktvrfy		; Ensure there's a packet driver out there!
	jnc	setpktd		; Can't find "PKT DRVR"?
	jmp short inxit		; Guess it isn't there. Glad we checked...

setpktd:
	call	qwrydrv		; first get class & type of driver present.
	mov	ah,PKTOPN	; First we want to open an IP packet handle
	mov	al,[pdrvcls]
	mov	bx,[pdrvtyp]	; driver specific info...
	mov	cx,2		; Ethertype field size
	xor	dx,dx		; driver specific
	lea	si,[IPetyp]	; Pointer to the desired ethertype
	mov	di,Restext	; Received packet's interrupt service routine
	mov	es,di
	lea	di,ResGroup:IPRXI
	call	farpkt
	jc	inxit		; If it failed bail out, else save handle
	mov	[iphndl],ax	; for close so others can use it later.

;------ This should be the default but we do it to be sure
;
	mov	bx,ax		; Request IP broadcasts too. If this fails we
	mov	ax,PKTTYP	; only must give back the original handle.
	mov	cx,3		; Requesting IP broadcasts...
	call	farpkt
	jc	rlsf

;------ This & any other handles will also receive broadcasts because of above
;
	mov	ah,PKTOPN	; Now register for ARP services. If this fails
	mov	al,[pdrvcls]
	mov	bx,[pdrvtyp]	; give back the IP handle we got above & return
	mov	cx,2		; with failure code.
	xor	dx,dx
	lea	si,[ARPetyp]
	mov	di,Restext
	mov	es,di
	lea	di,ResGroup:ARPRXI
	call	farpkt
	mov	[arphndl],ax		
	jnc	ippktf

rlsf:	mov	ax,PKTCLS	; Somebody after IP blew it. Give back IP handle
	mov	bx,[iphndl]
	call	farpkt
	stc
	jmp short inxit

ippktf:	call	seteadr
	call	fndhost		; Now resolve the hosts IP address
	call	tsthost		; and copy it in.
inxit:	pop	es
	pop	di
	pop	ds
	pop	si
	pop	dx
	pop	cx
	pop	bx
	ret
pktinit	ENDP

; This function uses ARP to figure out the ethernet address of our NFS server.

fndhost	PROC	NEAR
	push	si
	push	ds
	push	di
	push	es
	push	cx

	mov	[arpsiz],2 * size ARPTX	; Shut up further ARP recieves for now.
	les	di,[ARPrxad]
	push	di			; save base of structure for later
	xor	ax,ax			; Get -1 into AX
	dec	ax
	mov	cx,3			; Now fill dest. addr with broadcast (ff)
	rep stosw
	mov	si,word ptr [IPtxad]		; get ready to copy OUR e-net
	lea	si,[si].esrc			; (We're source in TX packet)
	push	si
	mov	cx,3
	cli					; x86 family prefix bugs...
	rep movs word ptr [di],word ptr es:[si]
	pop	si				; recover e-net address ptr
	pop	di				; point back to base of struct
	mov	es:[di].arpopcd,100h		; stick in ARP request opcode.
	lea	di,[di].arptxet			; offset to our TX's field
	mov	cx,3				; copy over our e-net address
	rep movs word ptr [di],word ptr es:[si]
	sti
	lea	si,[cannedip]			; Now tack in the IP fields
	movsw					; copy over our IP first.
	movsw
	xor	ax,ax				; Zero out target (Unknown tx)
	mov	cx,3				; hardware field
	rep stosw
	movsw					; Lastly, copy over host's IP
	movsw

	mov	ax,PKTSND			; Do our request!
	mov	cx,size ARPTX
	push	dx
	lds	si,[ARPrxad]
	call	farpkt
	pop	dx

	pop	cx
	pop	es
	pop	di
	pop	ds
	mov	[arpsiz],0			; re-enable arp receives
	pop	si
	ret
fndhost	ENDP

; This function will wait for a response from the host to the ARP RX queue.
; Then it extracts the host's e-net address & places them in the destination
; fields for the TX packets. A copy is saved in memory too.

tsthost	PROC	NEAR
	push	si
	push	di
	push	es
	push	cx

	mov	[arpspin],1		; enable auto ARP responses
	mov	[timerem],IPTIMLIM	; How long shall we wait for ARP?
hostlp:	test	[arpsiz],0ffffh
	jnz	hstrpl			; Got one!
	call	ticktoc
	jnz	hostlp			; Keep waiting...
	jmp short hostxt		; timed out.

hstrpl:	les	di,[ARPrxad]
	cmp	es:[di].arpopcd,200h	; Valid ARP reply packet?
	jz	hstchk
hstbd:	mov	[arpsiz],0		; enable receiving more...
	jmp	hostlp			; keep looking.
hstchk:	lea	si,[cannedip+4]
	lea	di,[di].arpip
	cmpsw
	jnz	hstbd
	cmpsw
	jnz	hstbd
	mov	si,word ptr [ARPrxad]	; Now copy over that host
	lea	si,[si].arptxet
	push	es			; Must reverse segments first.
	mov	di,ds
	mov	es,di
	lea	di,[hosteadr]		; Save a copy here
	pop	ds
	mov	cx,3			; copy over the host e-net address
	rep movsw
	mov	di,es			; restore data segment
	mov	ds,di
	les	di,[IPtxad]		; Now copy into our tx template
	lea	si,[hosteadr]
	mov	cx,3
	rep movsw
hostxt:	mov	[arpsiz],0
	pop	cx
	pop	es
	pop	di
	pop	si
	ret
tsthost	ENDP

; Look at the presumed packet driver's interrupt address. We must find
; "PKT DRVR" --zero terminated, for a valid packet driver. This will be at
; the ISR + 3 bytes. Search 60h to 80h for this value.

pktvrfy	PROC	NEAR
	push	di
	push	es
	push	si
	push	cx
pktfnd:	lea	si,[pktstrg]	; our template for testing the packet...
	mov	cx,PKTSTGL / 2	; how long is it? (in words...)
	xor	di,di
	mov	es,di		; segment 0 for vectors...
	mov	ax,di
	mov	al,[pktirq]	; Look at the interrupt vector. It will be
	shl	ax,1		; at the interrup # * 4, in segment 0
	shl	ax,1
	mov	di,ax
	les	di,es:[di]	; load the vector as a pointer...
	lea	di,[di+3]	; point past the jump instruction
	rep cmpsw		; does it match our template?
	clc
	jcxz	vrfxit
	inc	[pktirq]
	cmp	[pktirq],PKTMAX		; Less or = 80hex?
	jbe	pktfnd
	push	dx
	push	ds
	mov	dx,data
	mov	ds,dx
	lea	dx,[nodrv]
	mov	ah,DOSPRT
	int	DOSCALL
	pop	ds
	pop	dx
	stc
vrfxit:	pop	cx
	pop	si
	pop	es
	lea	di,ResGroup:[PKTCALL+1]	; Patch INT instruction to the actual packet
	mov	al,[pktirq]		; driver's IRQ. Using Group selector should
	mov	[di],al			; be legal because we've ALIASED it w/ CS & DS.
	pop	di
	ret
pktvrfy	ENDP

; We know where the driver is but must get its interface type and class. These
; could be used for checking later, for now we just blindly accept them and
; fill them in on calls.

qwrydrv	PROC	NEAR
	push	bx
	push	cx
	push	dx
	push	si		; packet driver will change these so save 'em
	push	ds
	mov	ax,PKTINF
	call	farpkt
	pop	ds
	pop	si
	mov	[pdrvtyp],dx	; Used as BX argument later for obtaining handles
	mov	[pdrvcls],ch	; Particular interface class. Used in AL for TX.
	pop	dx
	pop	cx
	pop	bx
	ret
qwrydrv	ENDP

; Set up the hardware address in the ethernet packet headers. The ARP and IP
; tx prototypes are all that need adjustment but the reciever packets are done
; for completeness.

seteadr	PROC	NEAR
	push	bx
	push	cx
	push	di
	push	es
	push	si
	mov	ah,PKTADR		; Get the real hardware address from
	mov	bx,[iphndl]		; the packet driver and copy it into
	mov	cx,EADRS		; the IP handle's tx template. Any of
	les	di,[IPtxad]		; the packet type handles we have will
	lea	di,[di].esrc		; work.
	push	dx
	call	farpkt
	pop	dx
	mov	si,word ptr [IPtxad]
	lea	si,[si].esrc
	mov	bx,si			; a copy for handy re-initialization
	mov	cx,EADRS / 2
	mov	ax,cx			; again, keep copies handy...
	les	di,[ARPtxad]		; Now update the outbound ARP template
	lea	di,[di].arpsrc
	cli					; until '386 if an intr. came
	rep movs word ptr [di],word ptr es:[si]	; only 1st prefix was restored
	mov	di,word ptr [IPrxad]		; b/c we use 2 prefixes
	lea	di,[di].edest			; upon ret(!) Don't allow ints.
	mov	si,bx				; Update ARP tx packet template
	mov	cx,ax				; and the IP rx packets
	rep movs word ptr [di],word ptr es:[si]
	mov	di,word ptr [ARPrxad]		; The ARP rx template gets done.
	lea	di,[di].unktxet
	mov	si,bx
	mov	cx,ax
	rep movs word ptr [di],word ptr es:[si]
	mov	di,word ptr [ARPtxad]		; ARPs have the hardware address
	lea	di,[di].arptxet			; twice inside them so get both
	mov	si,bx				; copies updated.
	mov	cx,ax
	rep movs word ptr [di],word ptr es:[si]
	sti
	lea	si,[cannedip]			; We must also have the correct IP
	movsw					; address in our ARP packet!
	movsw

	pop	si
	pop	es
	pop	di
	pop	cx
	pop	bx
	ret
seteadr	ENDP

trans	ENDS
	END
