TITLE Network Drive Interface
SUBTTL Resident Module
NAME   DCBNFS
PAGE   55,132

COMMENT @

	Agricon International Inc.  	October 12, 1992	David C. Brown IV

	See files COPYING and COPYWRT for copyright information. Copyright (C)
	Agricon International Inc. 1993

	This program is based on the PASCAL program PHANTOM.PAS as described in
	the book "Undocumented DOS". It is intended to provide an interface to
	NFS via TCP/IP using DOS' network redirector function. There is nothing
	in this design which would exclude Novell NetWare from working, however
	the TCP/IP driver may cause some limitations.

	Communication is via the multiplex interrupt. The program is a TSR that is
	removed from memory by calling the multiplex interrupt with a function code
	that is not presently used by DOS. Any other communication should be managed
	by DOS itself.

	@

;***** Equates *****************************************************************
;*******************************************************************************

ETN	equ	11		; Standard DOS name of 8 chars. + 3 char ext.
MAXPTH	equ	67		; Maximum path allowed, with 3 chars. overhead
TRIAL	equ	3		; Number of network retries attempted.

MPX2F	equ	2fh		; DOS' Multiplex interrupt
FCBOWN	equ	120ch		; Function request to set SFT/FCB owner

BADPTH	equ	3		; DOS' "Path not found" error code
BADACC	equ	5		; DOS' "Access denied" error code
NOFILE	equ	2		; DOS' "File not found" error code (DIR searches)
DIRBUSY	equ	16		; DOS' "Directory in use" error code (DIR remove)
ENDFILE	equ	18		; DOS' "No more files" error code (DIR searches)

BADF		equ	0feh	; Bad function code
FNDNXTCMD	equ	1ch

; Other manifest constants

NEWLINE		equ	0ah	; standard UNIX newline char.
RETCHAR		equ	0dh	; the carriage return for our DOS console
cr		equ	RETCHAR
lf		equ	NEWLINE
NFSHANDLEN	equ	32

;***** NFS Equates *************************************************************

VDIR	equ	2	; Directory flag
VREG	equ	1	; file flag

NFS_NULL	equ	0
NFS_GETATTR	equ	1
NFS_SETATTR	equ	2
NFS_LOOK	equ	4
NFS_READ	equ	6
NFS_WRITE	equ	8
NFS_CREATE	equ	9
NFS_REMOVE	equ	10
NFS_RENAME	equ	11
NFS_MKDIR	equ	14
NFS_RMDIR	equ	15
NFS_RDDIR	equ	16
NFS_STAT	equ	17

;***** Structures **************************************************************
;*******************************************************************************

include DOSSTRUC.INC

;***** NFS data structures *****************************************************

; Beware: the real NFS dirent structure is variable length. Copy them into
; this fixed-length structure to simplify location of the NFS_COOKIE field.
; if the name is > 11 characters we truncate it.

NFS_DIRENT	STRUC
inode		dd	0deadbeefh
nfsnamelen	db	13
nfsname		db	'hosehead.',cr,lf,'$',0
nfscookie	dw	20
NFS_DIRENT	ENDS

NFS_ATTRH	STRUC
NFS_HANDLE	db	NFSHANDLEN dup (?)
nfspad1		db	0,0,0		; network order data
nfstype		db	VDIR
nfspad2		dw	0		; network order high part of mode
nfsmode		dw	0
nfslinks	dd	1		; unsed
nfsids		dd	0,0		; uid/gid unused
nfssize		dd	0		; file size in bytes
nfsbsize	dd	0		; blocksize unused
nfsrdev		dd	0		; raw device id unused
nfsfsid		dd	0,0		; file system quadword unused
nfsinode	dd	0		; inode unused
nfsatime	dd	63c0190fh	; Access time unused.
		dd	0
nfsmtime	dd	63c0190fh	; August 15, 1992 @ 12:30pm
		dd	0
nfsctime	dd	63c0190fh	; creation time unused
		dd	0
NFS_ATTRH	ENDS

NFS_SATTR	STRUC
sa_mode		dd	0
sa_uid		dd	0
sa_gid		dd	0
sa_size		dd	0
sa_atime	dd	0,0		; actually nfsv2_time instance but hey
sa_mtime	dd	0,0
NFS_SATTR	ENDS

	NFSATLN	= (size NFS_ATTRH) - NFSHANDLEN

; To dissect the NFS file status (storage) reply we need to get to certain
; words and parts of words easily. Hence this dummy structure.

NFSSTAT	STRUC
blkszh	dw	?	; How many bytes/block --Need high/low addressability
blkszl	dw	?	; for pre-386 families. We'll divide it by bytes per
fragsiz	dw	?,?	; frag to get frags/block. Pretend frags = sectors &
totalblk dw	?,?	; blocks = clusters. Total blocks isn't used, just
freeblk	dw	?,?	; free and available blocks.
avlblk	dw	?,?
NFSSTAT	ENDS

;***** Records *****************************************************************
;*******************************************************************************

; Manage the flag bits in the current directory structure

flag	RECORD	NW:1,PH:1,JO:1,SB:1,SPARE:12=0	; Network, physical, join, subst

fleatr	RECORD	RES:2,ARC:1,DIR:1,VOL:1,SYS:1,HID:1,RO:1	; Flags for file

;***** Macros ******************************************************************
;*******************************************************************************

; Copy the DS into the ES register

dses	MACRO
	push	es
	push	ds
	pop	es
	ENDM

; Simply force the character in AL to be lower case if the proper
; flag is set. CS-relative addressing into our data group is used
; because all the segment registers are busy.

ucase	MACRO
	local	nofix

	test	cs:[lowmask],0ffh	;; need we bother?
	jz	nofix
	cmp	al,'a'		;; anything below this is OK
	jb	nofix
	cmp	al,'z'		;; anything above this is OK
	ja	nofix
	and	al,NOT 20h	;; force to uppercase.
nofix:
	ENDM

; Rather than manually type this code for the 6 or so functions that use it,
; have a macro expand it inline. Should be better than writing a subroutine
; because we don't pass parameters, stack frames and save registers => inline
; code is faster. EXPECTS: stack space allocated, and ES:DI point to the name
; that shall be located. RETURNS: handle in stack local varaible if success,
; AX = 0 if failure.

nfs_loc	MACRO
	lea	ax,[bp-size NFS_ATTRH]	;; Get local variable from caller's
	push	ss			;; &dirhand))
	push	ax			;; /* Save destination of attributes */
	push	es			;; &findname,
	push	di			;; /* SEG:OFFSET of name we seek */
	push	word ptr [OurCDS+2]	;; &curpath,
	mov	ax,word ptr [OurCDS]	;; /* ASCIIZ current context */
	add	ax,offset path
	push	ax			;; /* Simulated C syntax: finding handle */
	call	CDdescend
	and	ax,ax			;; if (!CDdescend(
	ENDM

;***** External Data/Functions *************************************************
;*******************************************************************************

	; You can sneak external fct. declarations out here because the
	; assembler is told they are near calls. Data doesn't always work
	; that well because the assembler may make the wrong decision in
	; computing the offsets or selecting a segment register. Only a
	; problem with multiple segments and GROUP directives.

	extrn	nfs_lookup:near,descend:near,CDdescend:near
	extrn	backup:near,nfs_readdir:near
	extrn	openmn:far,closemn:near,htonds:near
	extrn	nfsheader:near,nfsreqst:near
	extrn	DT2time:near, time2DT:near

	extrn	TXFAIL:abs,RXFAIL:abs,RPCVFL:abs,RPCSIZ:abs
	extrn	UDPMAX:abs

	extrn	tolower:near

SUBTTL Segments
PAGE+

;**** Segments resident after TSR ********************************************
;*****************************************************************************

;**** Resident code **********************************************************
;	This segment is declared first so the linker will load it first. It is
;	where the routines that service the interrupts go. The TSR is started
;	only in the event of an interrupt. The startup code comes later and
;	is jettisoned.


Restext	SEGMENT	PARA	PUBLIC	'TEXT'
Restext	ENDS

ResStack	SEGMENT	WORD	'RSTACK'
	extrn	ResBase:word,Ftop:word
ResStack	ENDS

BUFFER	SEGMENT	PARA	PUBLIC	'BUFR'
extrn	TXpktb:byte
BUFFER	ENDS

Resdata	SEGMENT	WORD	PUBLIC	'STATIC'
	extrn	RootOffs:word,drive_no:byte,vlabel:byte
	extrn	fnm1ea:word,fcb1ea:word
	extrn	lowmask:byte
Resdata	ENDS

	ResGroup	GROUP	Restext,ResStack,BUFFER,Resdata

;**** Resident data **********************************************************
;	This segment contains only the minimum data needed while the TSR is
;	active. This is to prevent hogging memory that is needed by other
;	programs. Since the TSR's typically are in low memory it is a valuable
;	commodity.

Resdata	SEGMENT	WORD	PUBLIC	'STATIC'

;	Declare the external data in the segment it is know locally. This way
;	the Assembler will automaticaly generate segment overrides if needed.
;	It also ensures that the offsets are computed correctly based on the
;	segment register selected and whether that seg. reg. is associated
;	with a default segment or group based on the ASSUME directive.

	PUBLIC	mounthandle,curdirh,OurSDA,OurCDS,OurOS
	PUBLIC	nfsrsiz,nfswsiz
	PUBLIC	workflg

OurSDA		dd	0	; DOS' Swappable Data Area
OurCDS		dd	0	; Our file system's Current Directory Struct

		even
mounthandle	NFS_ATTRH	<>
curdirh		NFS_ATTRH	<>
ffdirh		NFS_ATTRH	<>

; Allocate a cache of NFS handles and initialize a linked list of pointers
; with them.

listhead	dw	ResGroup:$ + 2
	REPT	30
	dw	ResGroup:$ + 2 + NFSHANDLEN
	db	NFSHANDLEN dup (0)
	ENDM
	dw	0			; terminate the list
	db	NFSHANDLEN dup (0)

nfsrsiz	dw	UDPMAX		; Software limits on protocol read/write
nfswsiz	dw	UDPMAX		; sizes. Presently IPUDP.ASM hard limits.

TMPFCB		db	ETN DUP (' ')

reqtry	db	TRIAL
OurOS	db	3		; Major version of DOS...

workflg	db	0		; Do we fail unsupported functions or not?
Resdata	ENDS

SUBTTL Resident code
PAGE+

;**** Resident interrupt handling code **************************************
;****************************************************************************

;	Every effort has been made to keep segments "pure" --in other words,
;	segments that are to be executed are not written in this code. This
;	is to allow compatibility with protected environments. It is unlikely
;	that this TSR will be ported to such an environment, but portions of
;	it may be adaptable to other uses. The overhead of writing with this
;	in mind will pay off in those cases.

Restext	SEGMENT	PARA	PUBLIC	'TEXT'

	public	dispatch,FCTTAB,MAXFCT
	ASSUME	cs:ResGroup,ds:ResGroup,ss:ResStack,es:NOTHING

ifdef	TESTMODE
; Hooked an unused function to allow mounting the file system.

termfct	PROC	NEAR
	call	openmn		; open packet driver too
	ret
termfct	ENDP

; Just hook an unused function so that we may unmount the file system.

qulfct	PROC	NEAR
	call	closemn		; unmount the file system, hangup the packet
	ret
qulfct	ENDP
endif

; Call the function specified by the request code. The codes were verified
; earlier. Then set the return value for this function into display level 0
; and end.

dispatch	PROC	NEAR
	xor	ah,ah			; clear upper part for table lookup
	mov	bx,offset jumptab	; load in table's base
	shl	ax,1			; convert into word offset
	add	bx,ax			; index the desired function
	call	cs:[bx]
	mov	[bp]+2,ax	; Set our return value in display 0
;	mov	[bp+4],bx	; save any alternate return value
	ret
dispatch	ENDP

; Provide a do-nothing function to fill out our table. If this function is
; called it sets an error to report that nothing was done.

nullfct	PROC	NEAR
lckfct:
unlkfct:
trncfct:

redrfct:
prnfct:
flsafct:
prnmfct:
ifndef	TESTMODE
termfct:
endif

	mov	ax,0		; Either we return with carry set or not for non-
	cmp	[workflg],1	; supported functions. Chosen at load time.
	ret
nullfct	ENDP

; This table indicates whether a given function is supported.

FCTTAB	db	0,1,2,3,4,5,6,7,8,9,0ah,0bh,0ch,BADF,0eh,0fh,BADF,11h
	db	BADF,13h,BADF,BADF,16h,17h,18h,BADF,BADF,1bh,1ch,1dh
	db	1eh,1fh,20h,21h,22h,23h,BADF,25h,7 dup (BADF),2dh,2eh
MAXFCT	equ	$-FCTTAB

	even
jumptab	dw	nullfct		; (Actually used for installation check.)
	dw	rdfct		; 01h	Delete directory	(DOS 3.1+)
	dw	rdfct		; 02h   Delete directory	(DOS 4.x)
	dw	mdfct		; 03h	Make directory		(DOS 3.1+)
	dw	mdfct		; 04h	Make directory		(DOS 4.x)
	dw	cdfct		; 05h	Change directory	(DOS 3.1+)
	dw	clsfct		; 06h	Close file
	dw	cmmtfct		; 07h	Commit file
	dw	readfct		; 08h	Read file
	dw	wrtfct		; 09h	Write file
	dw	lckfct		; 0ah	Lock file
	dw	unlkfct		; 0bh	Unlock file
	dw	spcefct		; 0ch	Report space available
	dw	nullfct
	dw	setafct		; 0eh	Set file's attributes
	dw	getafct		; 0fh	Get file's attributes
	dw	nullfct
	dw	renfct		; 11h	Rename file
	dw	nullfct
	dw	dltfct		; 13h	Delete file
	dw	nullfct
	dw	nullfct
	dw	opnfct		; 16h	Open file
	dw	crtfct		; 17h	Create file
	dw	trncfct		; 18h	Open & truncate file
	dw	nullfct
	dw	nullfct
	dw	fndfst		; 1bh	Find first entry
	dw	fndnxt		; 1ch	Find next entry
	dw	clsafct		; 1dh	Close all files
	dw	redrfct		; 1eh	Redirect stream
	dw	prnfct		; 1fh	Printer functions (redirected)
	dw	flsafct		; 20h	Flush all files
	dw	skenfct		; 21h	Seek from end
	dw	termfct		; 22h	Process termination hook.
	dw	qulfct		; 23h	Qualify path/file name
	dw	nullfct
	dw	prnmfct		; 25h	Network printer function
	REPT	7
	dw	nullfct
	ENDM
	dw	nullfct		; 2dh	Unknown but passively ack'ed
	dw	xtnded		; 2eh	Extended open	(DOS 4.x+)

; This function installs the fully-qualified file name taken from DOS' SDA into
; the packet. It returns with ES:DI pointing to the next byte free in the packet.
; If there was an error looking up the last part of the name -1 is returned with
; the carry flag set. Otherwise 0 is returned and the carry is cleared.
; Two entry points are provided to share this routine with the rename function
; and not hose up the stack.

nmins2	PROC	NEAR
	push	cx			; save general registers
	push	si
	push	di			; save packet pointer...
	push	es
	les	di,[OurSDA]
	mov	ax,[fnm1ea]		; locate fully-qualified name
	add	ax,fname2 - fname1	; hack to fix DOS versionitis
	jmp short nmtrk
nmins:	push	cx			; save general registers
	push	si
	push	di			; save packet pointer...
	push	es
	les	di,[OurSDA]
	mov	ax,[fnm1ea]		; locate fully-qualified name
nmtrk:	push	es
	push	ax
	call	backup			; isolate name from path
	jnc	nminok			; if bogus don't bother
	pop	es			; restore stack, get out.
	pop	di
	pop	si
	pop	cx
	sbb	ax,ax			; return -1 and CY set.
	ret
nminok:	mov	si,ax			; save a copy of this pointer
	mov	di,ax
	xor	cx,cx			; CX = -1, count down & invert
	mov	ax,cx			; to get string length. Trick is to
	dec	cx			; clear AX, & load CX w/ minimum code.
	repnz scasb			; Quickly gives string length + 1...
	not	cx			; math trick: convert sign AND decrement!
	pop	es			; (used since SCASB gave len + 1 above)
	pop	di			; Recovered pointer into packet...
	dec	cx			; remove \0 from length count
	stosw				; And put string length in N/W order
	mov	ah,cl			; as a longword
	stosw
	push	es			; Copy over string, converting to
	push	word ptr [OurSDA+2]	; lower case *IF* the user reqested.
	call	tolower
	mov	cl,4			; (Upper byte already 0 from rep opcode)
	and	ah,3			; Mask off longword modulus
	jz	nmxit			; Longword aligned already, don't pad.
	sub	cl,ah			; Get bytes needed for mod = 0
	rep stosb			; And AL = 0 still. Neat!
nmxit:	xor	ax,ax			; Zero AX and clear CY for success!
	pop	si
	pop	cx
	ret
nmins2	ENDP

; Rename the given file. Look up the name and then request the rename.

renfct	PROC	NEAR
	push	bx			; save general registers
	push	cx
	push	si
	push	ds
	push	di
	push	es
	push	bp
	mov	bp,sp			; Allocate a stack frame
	sub	sp,14+size NFS_ATTRH
;
;------ Find the handle for the directory of the file to be renamed.
;
	les	di,[OurSDA]
	mov	di,[fnm1ea]
	call	dopth			; Expects local space on stack.
	jz	noren			; BADPTH returned if failure.
;
;------ Build the request header & append name to it
;
	mov	ax,NFS_RENAME
	push	ax
	call	dirfin
	jc	noren			; BADPTH already loaded.
;
;------ We can duplicate handle for destination b/c DOS doesn't have mv(1)
;
	lea	si,[bp-size NFS_ATTRH]		; find our handle on stack
	mov	cx,NFSHANDLEN / 2
	cli					; usual caution w/ 2 prefixes
	rep movs word ptr [di],word ptr ss:[si]	; copy over handle again.
	sti
;
;------ Now affix new name to end of packet
;
	call	nmins2			; jump into the name insertion fct.
	jc	noren			; Again, test for trouble...
	xor	ax,ax			; We're not interested in NFS returns
	push	ax
	call	nfsreqst		; Do it!
	mov	ax,es:[di-2]		; recover NFS result, HIGH WORD ASSUMED
	xchg	ah,al			; ZERO. Convert to host order...
	cmp	es:[di-4],ax		; Set CY if AX != 0.
	jnc	noren			; All worked?
	mov	ax,BADACC		; else return with DOS error code
noren:	mov	sp,bp			; Since an SFT doesn't exist we just
	pop	bp			; return.
	pop	es
	pop	di
	pop	ds
	pop	si
	pop	cx
	pop	bx
	ret
renfct	ENDP

; Delete the file given by the name in DOS' SDA. Registers are preserved, CY
; clear if successful.

dltfct	PROC	NEAR
	push	bp
	mov	bp,sp
	sub	sp,14+size NFS_ATTRH
	push	cx		; Save other registers
	push	si
	push	di
	push	es

	les	di,[OurSDA]	; Pass the far address of the name
	mov	di,[fnm1ea]	; who's handle we seek.
	call	dopth		; ASSUMES LOCAL HANDLE ALLOCATED ON STACK!!!
	jz	dltout		; If error the flag will be BADPTH
;
;------ Prepare the request header
;
	mov	ax,NFS_REMOVE	; handle & all is already on stack.
	push	ax
	call	dirfin		; Copy handle, build header.
	jc	dltout		; BADPTH already set if error.
;
;------ Prepare for reply and make request
;
	xor	ax,ax		; No return size, just CY clear if success.
	push	ax
	call	nfsreqst	; Do the deed!
	mov	ax,es:[di-2]	; recover NFS result, HIGH WORD ASSUMED
	xchg	ah,al		; ZERO. Convert to host order...
	cmp	es:[di-4],ax	; Set CY if AX != 0.
	jnc	dltout		; All OK. Go on
	mov	ax,BADACC	; else return DOS error code
dltout:	pop	es
	pop	di
	pop	si
	pop	cx
	mov	sp,bp
	pop	bp
	ret
dltfct	ENDP

; Delete the directory given. The fully-qualified path is taken from
; DOS' swappable data area. Calls to the NFS host will furnish the
; handle we require. We then return that handle with the node name to delete it.

rdfct	PROC	NEAR
	push	di
	push	es
	push	si
	push	ds
	push	bp
	mov	bp,sp
	sub	sp,14+size NFS_ATTRH	; allocate a local handle/attrib
;
;------ Track down the handle from the name given.
;
	les	di,[OurSDA]	; Pass the far address of the name
	mov	di,[fnm1ea]	; who's handle we seek.
	call	dopth		; ASSUMES LOCAL HANDLE ALLOCATED ON STACK!!!
	jz	rmdxit		; BADPTH alread set if error.
;
;------ Prepare header, copy name to packet
;
	mov	ax,NFS_RMDIR	; handle & all is already on stack.
	push	ax
	call	dirfin		; that will do it.
	jc	rmdxit		; BADPTH alread set if error.
;
;------ Do the request
;
	xor	ax,ax		; no return data, just flag status.
	push	ax
	call	nfsreqst	; cut it free.
	mov	ax,es:[di-2]	; recover NFS result, HIGH WORD ASSUMED
	xchg	ah,al		; ZERO. Convert to host order...
	cmp	es:[di-4],ax	; Set CY if AX != 0.
	jnc	rmdxit		; If OK get out
	mov	ax,DIRBUSY	; Else return suitable error code.
rmdxit:	mov	sp,bp		; deallocate varaibles
	pop	bp		; retrieve frame pointer
	pop	ds
	pop	si
	pop	es
	pop	di
	ret
rdfct	ENDP

; Create the directory given. Again, the SDA is consulted to get the name.
; We call the server to look up the proper parent handle, then create the
; directory. Since DOS doesn't provide an SFT for our directory's attributes
; we don't bother to translate them.

mdfct	PROC	NEAR
	push	di
	push	es
	push	si
	push	ds
	push	cx
	push	bp
	mov	bp,sp
	sub	sp,14+size NFS_ATTRH	; allocate a local handle/attrib

	call	dopth			; resolve path to parent node...
	jz	mdxit			; BADPTH already set if error.
	mov	ax,NFS_MKDIR
	push	ax			; parent already on stack, set up
	call	dirfin			; name & fct header...
	jc	mdxit			; something went wrong with name...
;
;------ Now we must init & install SATTR to finish job & call nfsreqst
;
	xor	ax,ax			; clear AX.
	stosw				; Access modes set below
	dec	ax			; was 0 on success from dirfin fct.
	mov	cx,(size NFS_SATTR) / 2 - 1
	rep stosw			; write defaults in all fields
;
;------ override dir mode to world access
;
	mov	byte ptr es:[di-(size NFS_SATTR)],81h

	mov	ax,size NFS_ATTRH	; We ignore return handle but need size
	push	ax			; to see if call worked.
	call	nfsreqst		; CY if error or bytes in AX if OK. If
	sbb	ax,ax			; OK clear AX & CY, else AX = -1 w/ CY
	jnc	mdxit			; signal all OK
	mov	ax,BADACC		; Else return suitable DOS error code.
mdxit:	mov	sp,bp			; deallocate variables
	pop	bp			; retrieve registers
	pop	cx
	pop	ds
	pop	si
	pop	es
	pop	di
	ret
mdfct	ENDP

; Trace down the requested path. This function expects the stack to have an
; allocated NFS_ATTRH structure. ES:DI MODIFIED AND NOT SAVED.
;
; It is useful for tracing to the PARENT of a file or for a file that doesn't
; exist yet. Use nfs_lookup() to get the handle of a file that already exists
; in one call. Always returns BADPTH, Zero flag set if failure occurred.

dopth	PROC	NEAR
	push	di
	push	es
;
;------ Many fcts call us to save space, descend() does real work.
;
	push	ss			; destination for parent's
	lea	ax,[bp-size NFS_ATTRH]	; handle...
	push	ax
	les	di,[OurSDA]
	push	es
	push	[fnm1ea]		; the path we want to check
	push	word ptr [OurCDS+2]	; the current context
	mov	ax,word ptr [OurCDS]
	add	ax,offset path
	push	ax
	call	descend			; the action.
	and	ax,ax			; set flags for return
	mov	ax,BADPTH	; Hack: Always have DOS' error code
	pop	es		; so caller need only check flags &
	pop	di		; exit if error. Overwrite AX if success.
	ret
dopth	ENDP

; This function collects the tasks of copying over the parent handle, node
; name to be created/deleted, and preps the NFS request. The desired request
; is passed on the stack. It is assumed that the parent node handle is on the
; stack in the caller's display (frame.) Always return BATPTH, if successful,
; CY clear, else CY set. Also ES:DI point to next available byte in the packet.

dirfin	PROC	NEAR
	push	bp
	mov	bp,sp
	push	bx
	push	cx
	push	si
;
;------ First build the request header
;
	push	ss
	push	bp			; save current display
	mov	bp,[bp]			; get caller's display
	lea	ax,[bp-size NFS_ATTRH]	; get the address of the variable
	pop	bp			; recover display
	push	ax			; pass the pointer
	xor	ax,ax
	push	ax
	push	[bp+4]			; recover the desired function
	mov	di,BUFFER		; point to the packet's buffer
	mov	es,di
	mov	di,offset BUFFER:TXpktb
	call	nfsheader
;
;------ Now build the name into the packet
;
	call	nmins			; failure if AX = -1, CY set.
	mov	ax,BADPTH		; Return this regardless. Saves many
	pop	si			; callers having to do it.
	pop	cx
	pop	bx
	pop	bp
	ret	2
dirfin	ENDP

; Change directories on the server and update the CDS to show where
; we are.

cdfct	PROC	NEAR
	push	bp		; allocate a stack frame for our local
	mov	bp,sp		; NFS handle variable
	sub	sp,size NFS_ATTRH
	push	di
	push	es
	push	cx
	push	si

	les	di,[OurSDA]
	mov	di,[fnm1ea]
	add	di,[RootOffs]
	mov	al,'\'		; If current dir ends with '\' *AND* 0
	cbw
	cmp	ax,es:[di]	; then just copy over mount handle.
	jnz	CDothr
	mov	di,ds
	mov	es,di
	lea	di,[curdirh]
	lea	si,[mounthandle]
	mov	cx,(size NFS_ATTRH) / 2
	rep movsw
	xor	ax,ax
	jmp short CDxit

;
;------ Trace back from current directory to new path via nfs calls
;
CDothr:	sub	di,[RootOffs]	; Point back to FULL name
	push	ss
	lea	ax,[bp-size NFS_ATTRH]	; We need handle for nfs dir.
	push	ax
	push	es			; Now push far address of new path
	push	di
	push	word ptr [OurCDS+2]	; and far address of current path
	mov	ax,word ptr [OurCDS]
	add	ax,offset path
	push	ax
	call	CDdescend		; descend the tree...
	and	ax,ax
	jz	CDFAIL			; Must have been an invalid path...
	cmp	[bp-size NFS_ATTRH].nfspad1,VDIR
	jnz	CDFAIL			; CD only to directory handles please!
;
;------ Now update the handle for the current NFS host directory
;
	lea	si,[bp-size NFS_ATTRH]
	mov	di,ds
	mov	es,di			; ES is micro-coded destination
	lea	di,[curdirh]
	mov	cx,(size NFS_ATTRH) / 2
	cli
	rep movs word ptr [di],word ptr ss:[si]
	sti

	xor	ax,ax		; Signal a successful CD
CDxit:	pop	si
	pop	cx
	pop	es		; saved at the outset...
	pop	di
	mov	sp,bp		; deallocate parameters & get out.
	pop	bp
	ret

CDFAIL:	stc
	mov	ax,BADPTH
	jmp	CDxit
cdfct	ENDP

; Close the specified file. ES:DI point to an SFT that describes
; the file to be closed.

clsfct	PROC	NEAR
; Beware, the documentation for these functions say
; Don't do the step below! DOS wants to do this itself!
; but I have discovered that this doesn't work. Change it
; if errors come back.
	dec	es:[di].handles		; Decrement open count
	test	es:[di].fo_mod,3	; Opened for write or read/write?
	jz	keepdt			; Nope, don't change date/time.
; Should we try to set the NFS attributes here? In particular, take the
; date/time given in the SFT? (At least for write-access files.)
	push	bp
	mov	bp,sp
	sub	sp,size NFS_ATTRH
	mov	dl,es:[di].sf_attr	; File modes requested...
	push	di
	push	es
	push	es:[di].sf_time	; We'll want these for a later
	push	es:[di].sf_date	; fct call & must change ES:DI
	push	ds
	push	es:[di].dirsect	; Pass handle
	xor	ax,ax
	push	ax
	mov	ax,NFS_SETATTR
	push	ax
	mov	di,BUFFER
	mov	es,di
	mov	di,offset BUFFER:TXpktb
	call	nfsheader
	call	trnsatt
	dec	word ptr es:[di-size NFS_SATTR].sa_size
	dec	word ptr es:[di-size NFS_SATTR+2].sa_size
	call	DT2time		; time & date already on stack...
	xchg	dh,dl		; convert to Network order
	xchg	ah,al
	mov	word ptr es:[di-size NFS_SATTR+2].sa_mtime,ax	; MSWs to
	mov	word ptr es:[di-size NFS_SATTR+2].sa_atime,ax	; Network
	mov	word ptr es:[di-size NFS_SATTR].sa_mtime,dx	; order too
	mov	word ptr es:[di-size NFS_SATTR].sa_atime,dx
	mov	ax,NFSATLN
	push	ax
	call	nfsreqst
	pop	es
	pop	di
	mov	sp,bp
	pop	bp
keepdt:	mov	ax,es:[di].dirsect	; retrieve server's handle
	call	repl_ent
	xor	ax,ax
	mov	es:[di].dev,ax		; clear the device flag.
	ret
clsfct	ENDP

; Flush the buffer for the given file. ES:DI point to the proper
; SFT to find the file.

cmmtfct	PROC	NEAR
;
;------ Should call NFS_SYNC here?
;
	xor	ax,ax
	ret
cmmtfct	ENDP

; Transfer a specified block of data (up to 65535 bytes)
; ES:DI point to the SFT to be used. The handle and requested
; bytes are sent to the server for processing. The rdzero proc is
; a simple way to keep an exit within 128 bytes for x86 processor.

rdzero	PROC	NEAR
	mov	cx,[bp-4]
	mov	sp,bp	; Deallocate frame
	pop	bp
	xor	ax,ax
	ret		; Return is here to be in range of JCXZ.
rdzero	ENDP
readfct	PROC	NEAR
;
;------ Validate the requested transfer size
;
	mov	[reqtry],TRIAL
	push	dx
	mov	ax,word ptr es:[di].sf_size	; Ensure request isn't
	mov	dx,word ptr es:[di].sf_size+2	; bigger than the file
	sub	ax,word ptr es:[di].pos
	sbb	dx,word ptr es:[di].pos+2
	sub	ax,cx
	sbb	dx,0
	pop	dx
	jae	go4it	; too greedy! Round request to what's available
	add	cx,ax
;
;------ Set up local variables
;
go4it:	push	bp
	mov	bp,sp
	sub	sp,0ch		; local ptrs, byte counters
	mov	[bp-2],cx	; bytes requested
	push	si
	push	ds
	lds	si,[OurSDA]	; get long pointer to disk trans. area
	cmp	[OurOS],3
	jg	rd4
	lds	si,[si].curdta
	jmp short rdvi
rd4:	lds	si,[si].curdta4
rdvi:	mov	[bp-6],ds	; save as destination pointer
	mov	[bp-8],si
	mov	[bp-0ah],es	; Save SFT pointer
	mov	[bp-0ch],di
	pop	ds		; recover local data seg. addressability
	pop	si
	mov	word ptr [bp-4],0	; clear bytes moved counter
;
;------ Build the request header
;
rdmore:	jcxz	rdzero		; Don't EVEN waste time for 0 bytes...
	cmp	cx,[nfsrsiz]	; Ensure we don't exceed UDP max size
	jbe	rdszok
	mov	cx,[nfsrsiz]
rdszok:	push	si		; Save copy of Resgroup for later
	push	ds
	push	ds		; Long address for file handle
	push	es:[di].dirsect	; (offset for Resgroup saved in SFT)
	xor	ax,ax		; NFS requests are longwords
	push	ax
	mov	ax,NFS_READ
	push	ax
	mov	di,BUFFER		; Now ES:DI addresses packet buffer
	mov	es,di
	mov	di,offset BUFFER:TXpktb
	call	nfsheader
;
;------ Include the offset & quantity in packet
;
	lds	si,[bp-0ch]		; Copy offset & quantity into buffer
	mov	ax,word ptr [si].pos+2	; need to get SFT pointer for this.
	xchg	ah,al			; Convert longword to network order
	stosw				; as always
	mov	ax,word ptr [si].pos
	xchg	ah,al
	stosw
	xor	ax,ax			; Request is longword for alignment but
	stosw				; must be < UDP packet size.
	mov	ax,cx			; and the # bytes to read
	xchg	ah,al
	stosw
	xor	ax,ax			; Finally a longword of nulls.
	stosw
	stosw
	mov	ax,4 + NFSATLN		; minimum expected reply size
	pop	ds			; Routines below need global data
	pop	si
	push	ax
	call	nfsreqst
	jnc	rdwrkd	
;
;------ Attempt retries on RPC or network failures
;
	les	di,[bp-0ch]	; restore state: point to SFT
	cmp	ax,TXFAIL	; Don't bother re-trying TX failures
	stc			; re-set carry in case we bail out...
	jz	noread
	dec	[reqtry]	; We won't re-try forever.
	jnz	rdszok		; Back to top of loop and do it again.
	jmp	noread
;
;------ Now transfer the received packet to the current DTA
;
rdwrkd:	add	di,4 + NFSATLN	; skip over attrib data & longword
	mov	ax,es:[di-2]	; Index 1/2 longword back for bytes sent.
	xchg	ah,al		; convert bytes read to host order
	mov	cx,ax		; get received character count
	jcxz	rdovr		; Don't loop forever with 0 data!

	push	si
	push	ds
	mov	si,es		; Get DS:SI to point to incoming data
	mov	ds,si
	mov	si,di
	les	di,[bp-8]	; point to the destination (inside DOS)
	add	[bp-4],cx	; update bytes moved count
	shr	cx,1		; Perform word moves. Odd count to CY.
	rep movsw		; Packet has SI always even, is DI?
	jnc	rdevn		; CY was preserved. Move odd byte?
	movsb			; 5 bytes more code but more efficient
rdevn:	mov	[bp-8],di	; save pointer for next time. (ES always OK)
	pop	ds
	pop	si

	les	di,[bp-0ch]			; Restore SFT addressability
	add	word ptr es:[di].pos,ax		; update the file pointer
	adc	word ptr es:[di].pos+2,0	; if the read was successful
	mov	cx,[bp-2]			; Get # bytes requested
	sub	cx,[bp-4]			; <= # bytes moved?
	jbe	rdovr
	mov	[reqtry],TRIAL	; reset retry counter.
	jmp	rdmore
rdovr:	mov	cx,[bp-2]	; return this in case NFS host rounded up.
	mov	ax,cx		; superstition! Put bytes moved into AX.
	clc
finre:	mov	sp,bp		; Deallocate frame
	pop	bp
	ret
noread:	mov	cx,[bp-4]	; get # of bytes moved
	jmp	finre
readfct	ENDP

; Transfer a specified block of data (up to 65535 bytes)
; ES:DI point to the SFT to be used. The handle and requested
; bytes are sent to the server for processing.

wtzero	PROC	NEAR	; See readfct for reasoning behind this...
	mov	cx,[bp-4]
	mov	sp,bp	; Deallocate frame
	pop	bp
	xor	ax,ax
	ret		; Return is here to be in range of JCXZ.
wtzero	ENDP
wrtfct	PROC	NEAR
	push	bp
	mov	bp,sp
	sub	sp,0ch		; local ptrs, byte counters
	mov	[bp-2],cx	; bytes presented
	push	si
	push	ds
	lds	si,[OurSDA]	; get long pointer to disk trans. area
	cmp	[OurOS],3
	jg	wr4
	lds	si,[si].curdta
	jmp short wrvi
wr4:	lds	si,[si].curdta4
wrvi:	mov	[bp-6],ds	; save as destination pointer
	mov	[bp-8],si
	mov	[bp-0ah],es	; Save SFT pointer
	mov	[bp-0ch],di
	pop	ds		; recover local data seg. addressability
	pop	si
	mov	word ptr [bp-4],0	; clear bytes moved counter
	mov	[reqtry],TRIAL		; initialize retry counter
;
;------ Build the request header
;
wtmore:	jcxz	wtzero		; bail out for bogus write sizes!
	cmp	cx,[nfswsiz]	; Ensure we don't exceed UDP max size
	jbe	wtszok
	mov	cx,[nfswsiz]
wtszok:	push	cx
	push	si		; Save copy of Resgroup for later
	push	ds
	push	ds		; Long address for file handle
	push	es:[di].dirsect	; (offset for Resgroup saved in SFT)
	xor	ax,ax		; NFS requests are longwords
	push	ax
	mov	ax,NFS_WRITE
	push	ax
	mov	di,BUFFER	; Now ES:DI addresses packet buffer
	mov	es,di
	mov	di,offset BUFFER:TXpktb
	call	nfsheader
;
;------ Include the offset & quantity in packet
;
	xor	ax,ax
	stosw				; There's a dword of zeros, why?
	stosw
	lds	si,[bp-0ch]		; Copy offset & quantity into buffer
	mov	ax,word ptr [si].pos+2	; need to get SFT pointer for this.
	xchg	ah,al			; Convert longword to network order
	stosw				; as always
	mov	ax,word ptr [si].pos
	xchg	ah,al
	stosw
	xor	ax,ax
	stosw
	stosw			; Also, dword of zero between offset
	stosw			; and length. (NFS version 2)
	mov	ax,cx		; and the # bytes to write
	xchg	ah,al		; save in network order...
	stosw
	lds	si,[bp-8]	; point to the source (inside DOS)
	add	[bp-4],cx	; update bytes moved count
	shr	cx,1		; Perform word moves. Odd count to CY.
	rep movsw		; Packet has SI always even, is DI?
	jnc	wtevn		; CY was preserved. Move odd byte?
	movsb			; 5 bytes more code but more efficient
wtevn:	mov	[bp-8],si	; save pointer for next time. (DS always OK)
	and	ax,300h		; Mask count's longword modulus & clear AL
	jz	wtisln
	mov	cl,4		; Now ensure packet ends on a longword.
	sub	cl,ah		; Calculate bytes needed
	rep stosb		; Pads with zeros since AL was cleared above.

wtisln:	mov	ax,NFSATLN	; minimum expected reply size
	pop	ds		; Routines below need global data
	pop	si
	push	ax
	call	nfsreqst
;
;------ Again, we do some network retries
;
	jnc	wrwrkd
	les	di,[bp-0ch]	; Recover SFT ptr in case we re-try
	cmp	ax,TXFAIL	; But TX failures aren't retried.
	stc			; CMP cleared CY, re-set in case we don't retry
	jz	nowrit
	dec	[reqtry]	; Don't do it indefinitely
	jz	nowrit
	pop	cx		; How many bytes to move...
	jmp	wtszok
;
;------ Now transfer the current DTA to the received packet
;
wrwrkd:	add	di,size NFS_ATTRH	; skip over attrib data & 1/2 longword
	mov	cx,dx				; preserve DX register
	push	word ptr es:[di+2].nfsmtime	; Convert to DOS format time
	push	word ptr es:[di].nfsmtime
	call	time2DT
	les	di,[bp-0ch]			; Restore SFT addressability
	mov	es:[di].sf_date,dx		; Update the file's status.
	mov	es:[di].sf_time,ax
	mov	dx,cx				; restore DX register
	pop	ax				; bytes moved this call...
	add	word ptr es:[di].pos,ax		; update the file pointer
	adc	word ptr es:[di+2].pos,0	; if the read was successful
	add	word ptr es:[di].sf_size,ax	; And file size...
	adc	word ptr es:[di+2].sf_size,0
	or	byte ptr es:[di].dev,040h	; flag file as written.
	mov	cx,[bp-2]			; Get # bytes requested
	sub	cx,[bp-4]			; <= # bytes moved?
	jbe	wtovr
	mov	[reqtry],TRIAL	; allow more retries for next request
	jmp	wtmore
wtovr:	mov	cx,[bp-2]	; return this in case NFS host rounded up.
	mov	ax,cx		; superstition! Put bytes moved into AX.
	clc
finwt:	mov	sp,bp		; Deallocate frame
	pop	bp
	ret
nowrit:	pop	cx		; be nice, clean off stack.
	mov	cx,[bp-4]	; get # of bytes moved
	jmp	finwt
wrtfct	ENDP

; Set the file's attributes. The file is located via the path given in the SDA.
; Not all NFS values are tracked by DOS so some defaults are used. The attributes
; were passed on the stack before DOS called us with an interrupt. dirfin() not
; used because we don't want the file's name appended to the packet.

setafct	PROC	NEAR
	push	bx		; Save general registers
	push	cx
	push	dx
;
	push	di		; Get original stack so that we can
	push	es		; retrieve parameters
	les	di,[bp]+6
	mov	dl,es:[di]+10	; Now locate parameter out of mess.
	pop	es
	pop	di
;
;------ Now save registers & allocate local storage
;
	push	si
	push	ds
	push	bp
	mov	bp,sp
	sub	sp,14+size NFS_ATTRH
	push	di
	push	es

	les	di,[OurSDA]	; Pass the far address of the name
	mov	di,[fnm1ea]	; who's handle we seek.
	nfs_loc			; ASSUMES LOCAL HANDLE ALLOCATED ON STACK!!!
	jnz	setaok
	mov	ax,BADPTH	; Show the path was wrong...
	stc
	jmp short setabd
;
;------ Now prepare the request packet
;
setaok:	push	ss
	push	ax			; ONLY the handle gets used
	xor	ax,ax
	push	ax
	mov	ax,NFS_SETATTR		; and the NFS op desired
	push	ax
	mov	di,BUFFER		; nfsrqst expects ES:DI to point to the
	mov	es,di			; packet buffer.
	mov	di,offset BUFFER:TXpktb
	call	nfsheader		; build up an NFS/RPC request
;
	call	trnsatt			; convert SFT to NFS attributes
	dec	word ptr es:[di-size NFS_SATTR].sa_size	; choose -1 rather than 0 size
	dec	word ptr es:[di-size NFS_SATTR].sa_size+2
	mov	ax,NFSATLN		; Minimum reply size
	push	ax
	call	nfsreqst
	jnc	setaok
	mov	ax,BADACC		; Access denied to file
;
;------ Since we weren't given an SFT there's nowhere to put result. Trash it.
;
setabd:	pop	es
	pop	di
	mov	sp,bp		; deallocate variables
	pop	bp
	pop	ds
	pop	si
	pop	dx
	pop	cx
	pop	bx
	ret
setafct	ENDP

; Use the name given in DOS' SFT to look up a file's handle. Return the
; attributes found in AX. dirfin() not used because we don't want the file
; name appended to the request.

getafct	PROC	NEAR
	push	bp			; allocate a frame & local storage
	mov	bp,sp
	sub	sp,14+size NFS_ATTRH
	push	cx			; Save other registers
	push	si
	push	di
	push	es

	les	di,[OurSDA]		; Pass the far address of the name
	mov	di,[fnm1ea]		; who's handle we seek.
	nfs_loc			; ASSUMES LOCAL HANDLE ALLOCATED ON STACK!!!
	jz	gafail
;
;------ Clear AX to normal attributes & then adjust as appropriate
;
	xor	ax,ax
	mov	cx,[bp-size NFS_ATTRH].nfspad2
	test	cx,300o				; test for read-only attributes
	jnz	gahat				; no flag => normal
	or	al,MASK RO			; flag read-only
gahat:	test	cx,777o				; any attributes?
	jnz	gaout
	or	al,MASK HID			; no NFS attribs => hidden file
gaout:	pop	es				; Remove SFT pointer from stack.
	pop	di
	pop	si
	pop	cx
	mov	sp,bp
	pop	bp
	ret
gafail:	mov	al,BADPTH		; Else report failure.
	cbw				; won't affect CY flag!
	jmp	gaout
getafct	ENDP

; Return the space available to DOS

spcefct	PROC	NEAR
	push	di
	push	es
	mov	di,BUFFER		; Lower functions assume ES:DI
	mov	es,di			; set already since they return
	mov	di,offset BUFFER:TXpktb	; ES:DI affected by their efforts.
	lea	ax,[mounthandle]	; The file system we're using
	push	ds
	push	ax
	xor	ax,ax		; the longword for the function we want
	push	ax
	mov	ax,NFS_STAT
	push	ax
	call	nfsheader
	mov	ax,size NFSSTAT	; how many words do we expect back?
	push	ax
	call	nfsreqst	; do the tx/rx/verify stuff.
	jnc	spcok
	pop	es
	pop	di
	ret
;
;------ Convert the result from network order & decrypt.
;
spcok:	push	es
	push	di
	push	es
	push	di
	mov	ax,(size NFSSTAT) / 4	; count of LONGWORDS! Not bytes!
	push	ax
	call	htonds
;
;------ Try to interpolate UNIX file system units into DOS' cluster FAT units
;
	xor	cx,cx			; Only 3 bytes loads -1 in CX
	dec	cx			; Allows loop/count but gives
	mov	ax,es:[di].totalblk	; bias of 1 unit. Keeps loop
	mov	dx,es:[di+2].totalblk	; fast & that's done more...
	and	dx,dx
	jz	sztiny			; Exactly 1KB in file system?
	rcl	ax,1			; Compensate for loop entry,
lglp3:	rcr	ax,1			; scale number, do DX last so
	shr	dx,1			; we know when MSW = 0.
	loopnz	lglp3			; Ends biased by 1 => tot. bias = 2.
	rcr	ax,1			; Exited before we got this...
sztiny:	not	cx			; Math trick: convert sign & add 1
	mov	bx,cx			; save a copy of "sectors/cluster"
	mov	[bp+4],ax		; Scaled "Total clusters" (blocks)

	mov	ax,es:[di+2].avlblk	; Now convert available blocks.
	mov	dx,es:[di].avlblk	; Result to remain in DX for DOS.
lgdiv3:	shr	ax,1
	rcr	dx,1
	loop	lgdiv3			; "available clusters"
	mov	ax,cx			; still 0 from loop
	inc	ax			; may save 1 byte over MOV AX,1...
	mov	cx,bx
	shl	ax,cl			; "sectors/cluster" not log2 of it.
	mov	cx,es:[di].fragsiz	; recover bytes/"sector" (frag)
	clc				; BX set upon exit from dispatch
	pop	es
	pop	di
	ret
spcefct	ENDP

; This function provides support for the unified interface to create/opens
; provided starting with DOS 4.0. It allows creation/open/relace of a file.
; The result is supplied in the CX register. Note that as usual, the SERVER
; provides many of the logical consistance checks: attempting to write a
; read-only file, etc. CX has the following meanings:
;		1 - file opened, 2 - file created, 3 - file truncated

xtnded	PROC	NEAR
	push	si
	push	ds
	push	ax
	push	ds			; save DS, we'll want it later

	lds	si,[OurSDA]
	mov	al,byte ptr [si].spopnmo	; copy special mode to older
	and	al,7fh			; DOS mode & remove inheritance flags
	mov	[si].fo_mod4,al
SRCHMSK	equ	(MASK DIR) OR (MASK SYS) OR (MASK HID) OR (MASK RO)
	mov	[si].lookatt4,SRCHMSK	; find system/hidden/read-only/normal

	pop	ds			; need to recover DS for fct entry...
	call	fndfst			; see if file exists
	lds	si,[OurSDA]		; Need addressability to SDA again.
	jnz	flnew				; If file's new forget DIR test
	test	[si+ffile4].fouattr,MASK DIR	; Don't munge directories, exit.
	jnz	xtdfls
	test	byte ptr [si].spopnop,0fh	; file exists. See if fct. says
	jz	xtdfls				; we want it to.
	pop	ax
	test	byte ptr [si].spopnop,2	; Do we allow creation ops?
	jz	xflrd			; No, must mean open existing.
	mov	cx,3			; Yes, signal we truncated file.
	jmp short xflwr
xflrd:	mov	cx,1			; otherwise signal open operation.
	pop	ds
	pop	si
	jmp	opnfct				; Go open existing file.
flnew:	test	byte ptr [si].spopnop,0f2h	; we have a new file.
	jz	xtdfls
	pop	ax
	mov	cx,2			; indicate we *CREATED* a file
xflwr:	pop	ds
	pop	si
	jmp	crtfct			; From here we act as though DOS called these.
xtdfls:	stc
	pop	ax
	mov	ax,BADACC
	pop	ds
	pop	si
	ret
xtnded	ENDP

; Open or create/truncate a given file. ES:DI point to an SFT to be
; initialized. The file name is passed to the server. Attributes are
; retrieved from the original stack. Open mode is taken from the SDA.
; The server will return a handle which is saved in the directory
; cluster offset since that field is unused otherwise.

crtfct	PROC	NEAR
	push	si		; For CREATE | TRUNCATE there isn't a
	push	ds		; mode specified. To keep the code general
	lds	si,[OurSDA]	; we force one in there that allows writes
	cmp	[OurOS],3
	jg	crtv4
	or	[si].fo_mod,1	; and then retrieve it later.
	jmp short crtvi
crtv4:	or	[si].fo_mod4,1	; flag, DOS version 4 and later
crtvi:	pop	ds		; Don't confuse this with setting attributes
	pop	si		; (SFT) which *IS* handled by preopn.
;
;------ Save general registers and allocate storage
;
	push	bx
	push	cx
	push	dx
	push	si
	push	ds
	push	bp
	mov	bp,sp
	sub	sp,14+size NFS_ATTRH	; allocate a copy of file's name too
	call	initSFT			; Set up SFT default values
	mov	dl,es:[di].sf_attr	; Keep attributes handy
	call	dopth			; Trace to nearest parent's handle
	jnz	crton
	jmp	cntopn			; BADPTH error code already loaded.
;
;------ Build request header & copy over parent's handle. Append name to packet
;
crton:	push	di			; save SFT pointer for later
	push	es
	mov	ax,NFS_CREATE
	push	ax
	call	dirfin
	jc	crtfld			; BADPTH error code already loaded
;
;------ Convert to NFS attributes & append to packet
;
	call	trnsatt
;
;------ Now do the request
;
	mov	ax,NFSHANDLEN		; How much should come back?
	push	ax			; Pass expected reply size word
	call	nfsreqst		; Get it
	jc	crtfld
;
;------	Interpret what we got back. Straight from packet buffer.
;	(A handle (32 bytes, host ordered) and the attributes, NW ordered)
;
	call	get_ent			; Ask for a handle entry from the
	and	ax,ax			; cache.
	jz	crtfld
	mov	bx,ax			; save a copy of handle for later
	mov	si,di
	push	es
	mov	di,ds			; Destination is in data segment
	mov	es,di
	pop	ds			; Source is packet buffer
	mov	di,ax			; Offset of our cache entry...
	mov	cx,NFSHANDLEN / 2
	rep	movsw			; copy it over
	push	ds
	mov	di,es			; Restore data segment addressability
	mov	ds,di
	push	si			; Now convert to host order the
	push	ss			; SFT. Give source and Destination
	lea	ax,[bp-NFSATLN]		; BUFFER segment pushed earlier
	push	ax
	mov	ax,NFSATLN / 4
	push	ax
	call	htonds
;
;------ Now convert attributes to DOS formats & save what applies
;
	mov	ax,bx			; 2 bytes extra code to use
	lea	bx,[bp-size NFS_ATTRH]	; but it lets us jump to open()
	jmp	opnatt			; for common completion.

crtfld:	mov	ax,BADACC		; Set return code & bail out. Intel
	jmp	cntopn			; cond. jumps < 128 bytes thus this.
crtfct	ENDP

; Translate the DOS attributes into meaningful NFS attributes. Initialize
; the NFS SATTR structure with default values & those specified in the SFT
; given. On entry ES:DI assumed pointing into packet, THEY WILL BE MODIFIED
; upon return. BX, CX, DX, SI MODIFIED AND NOT SAVED. The attributes are in
; DX upon entry.

trnsatt	PROC	NEAR
;
;------ Initialize the new attribute block to -1 for default fields
;
	push	es
	mov	bx,ss			; block is on stack segment so
	mov	es,bx			; get registers set up
	lea	bx,[bp-NFSATLN]		; Clear just the attributes!
	xchg	di,bx
	mov	cx,(size NFS_SATTR) / 2 - 2	; Set all but file mode
	xor	ax,ax				; work with SATTR
	stosw					; Clear file mode
	stosw
	dec	ax			; loads AX with -1 (1 byte vs. 4)
	rep stosw
	lea	di,[bp-NFSATLN].sa_size
	inc	ax			; put AX = 0 again (1 byte!)
	stosw				; initial file size = 0L.
	stosw
	pop	es			; recover packet data pointer
	mov	di,bx
;
; Here we use SS:BX for addressing our local variable (the attribute structure)
; while we preserve ES:DI as the packet buffer and DS for other data.
;
	and	dl,07fh			; and convert to BSD format
	lea	bx,[bp-NFSATLN]		; drop to just above mount
;
;------ Consider DOS open and sharing modes. Sharing is transitory and isn't
;	saved. We can't do much about it but NFS is atomic/stateless anyway
;	and really doesn't need locking. The open modes themselves are handled
;	by the type of call: create/truncate vs. seek/write. The only thing
;	we need to preserve is the read only attribute. We can fake hidden.
;
	mov	ax,100744o 			; assume default flags
	test	dl,MASK RO
	jz	crthid
	mov	ax,100444o 			; revoke write/exec priv.
crthid:	test	dl,MASK HID
	jz	crtnrm
	mov	ax,8000h			; clear all flags
crtnrm:	mov	word ptr ss:[bx].sa_mode,ax
	push	ss				; convert to network order
	push	bx				; source
	push	es				; destination
	push	di
	mov	ax,(size NFS_SATTR) / 4		; size
	push	ax
	call	htonds
	add	di,size NFS_SATTR	; point back to packet end
	ret
trnsatt	ENDP

; Open a given file. ES:DI point to an SFT to be initialized. The file name
; taken from DOS' SDA. Attributes are retrieved from the original stack.
; Open mode is taken from the SDA. The server will return a handle which
; is saved in the directory cluster offset since that field is unused otherwise.

opnfct	PROC	NEAR
	push	bx			; Save general registers & allocate
	push	cx			; storage.
	push	dx
	push	si
	push	ds
	push	bp
	mov	bp,sp
	sub	sp,14+size NFS_ATTRH	; allocate a copy of file's name too
	call	initSFT			; As in create, setup code is common.
	push	di			; preserve registers...
	push	es

	les	di,[OurSDA]		; Reference the desired string
	mov	di,[fnm1ea]
	nfs_loc
	mov	ax,BADPTH
	jz	opnfpt			; Was the path valid? (File exists)
	cmp	[bp-size NFS_ATTRH].nfspad1,VDIR
	jz	opnfld			; disallow UNIX style directory opens
;
;------ Interpret what we got back.
; 	(It will be a handle (32 bytes, host ordered) and the attributes,
;	NW ordered) Save the original in es:di copy & of attribs into cache.
;	(attrib cache code doesn't exist yet.)
;
	call	get_ent			; Ask for a handle entry from the
	and	ax,ax			; cache.
	jz	opnfld
	mov	di,ds			; Handle cache is addressed from
	mov	es,di			; data segment. Switch around
	mov	di,ax			; segment registers to do the job.
	lea	si,[bp-size NFS_ATTRH]
	mov	bx,si
	mov	cx,NFSHANDLEN / 2
	cli					; interrupts cause 1st prefix to
	rep movs word ptr [di],word ptr ss:[si]	; be ignored upon return so CYA
	sti
;
;------ Convert to DOS attributes
;
opnatt:	pop	es				; restore SFT pointer we had
	pop	di				; upon entry.
	mov	es:[di].dirsect,ax		; handle
	xor	ax,ax				; initially, no attributes
	mov	cx,word ptr ss:[bx].nfspad2	; was ss:[bx].nfsmode
	test	cx,300o				; test for read-only attributes
	jnz	opnhat				; no flag => normal
	or	al,MASK RO			; flag read-only
opnhat:	test	cx,777o				; any attributes?
	jnz	opnnat
	or	al,MASK HID			; no NFS attribs => hidden file
opnnat:	mov	es:[di].sf_attr,al		; set DOS' attributes
;
;------ Copy size, convert elapsed seconds to DOS date/time
;
	mov	ax,word ptr ss:[bx+2].nfssize
	mov	word ptr es:[di+2].sf_size,ax	; size
	mov	ax,word ptr ss:[bx].nfssize
	mov	word ptr es:[di].sf_size,ax
	push	word ptr ss:[bx+2].nfsmtime
	push	word ptr ss:[bx].nfsmtime
	call	time2DT
	mov	es:[di].sf_date,dx		; creation date
	mov	es:[di].sf_time,ax		; creation time
;
; ---- Caution: avoid this step below because DOS does that itself!
;	inc	es:[di].handles		; Update the # of opens for this
	xor	ax,ax			; clear AX & carry flag

opnxit:	mov	sp,bp		; deallocate stack frame & GET OUT!
	pop	bp
	pop	ds
	pop	si
	pop	dx
	pop	cx
	pop	bx
	ret
opnfld:	mov	ax,BADACC
opnfpt:	pop	es
	pop	di
cntopn:	stc			; Put -1 into AX and set carry flag
	jmp	opnxit
opnfct	ENDP

; This function links common stack frame setup code for open and create. It's
; job is to save room and prevent mistakes from differences in code that would
; appear twice. BEWARE: It uses the caller's frame so that had better exist!
; It assumes the space is *AT LEAST* NFS_ATTRH long and is IMMEDIATELY below BP

initSFT	PROC	NEAR
	push	di
	push	si
	push	ds
	push	di		; Get original stack so that we can
	push	es		; retrieve parameters (attributes)
	push	bp		; back to caller's caller's frame.
	mov	bp,[bp]
	les	di,[bp]+6
	mov	al,es:[di]+10	; Now locate parameter out of mess.
	pop	bp
	pop	es
	pop	di
	mov	es:[di].sf_attr,al	; Set specified attributes
;
;------ Back to processing the request...
;
	mov	ax,8040h	; Update the device in the SFT for a
	or	al,[drive_no]	; network file before we loose DS
	mov	es:[di].dev,ax	; addressability below.

	xor	ax,ax				; Null out this pointer
	mov	word ptr es:[di].devdrv,ax
	mov	word ptr es:[di].devdrv+2,ax
	mov	word ptr es:[di].pos,ax		; Initailize zero offset
	mov	word ptr es:[di].pos+2,ax	; into file.
	mov	es:[di].sf_dir,al		; fake offset into dir.

	lds	si,[OurSDA]		; SDA has other file parameters
	mov	al,[si].fo_mod		; Open mode
	cmp	[OurOS],3
	jbe	init3
	mov	al,[si].fo_mod4
init3:	and	al,07fh			; Clear inheritance flags
;*** older circumstance: mov	es:[di].o_mode,ax
	mov	byte ptr es:[di].o_mode,al
	and	byte ptr es:[di+1].o_mode,0c0h
;
;------ Set up SFT with FCB style name
;
	push	cx			; I have this theory that DOS keeps
	lea	di,[di].sf_name		; ASCIIZ in SDA->fname1 & FCB in
	mov	si,[fcb1ea]		; SDA->nm1fcb & guess what... YES!!
	mov	cx,ETN			; FCB names have a fixed length
	rep movsb
	pop	cx
	pop	ds
	pop	si
	pop	di
	test	byte ptr es:[di+1].o_mode,80h
	jz	regSFT
	call	entDOS
regSFT:	clc			; make sure we return with carry flag clear
	ret
initSFT	ENDP

; 	This fucntion switches to DOS' data segment and stack, makes
; ES:BX point to the SFT in the SDA, and saves all other registers. Only the flags
; register will reflect the result from DOS upon return. It is most useful for setting
; SFT owners (open/create requests) and checking for existing character devices.

entDOS	PROC	NEAR
	push	bx
	push	cx
	push	dx
	push	bp
	push	si
	push	di
	push	es
	push	di		; We'll need two copies of this.
	push	es

	mov	bp,[bp]		; recover caller's frame again.
	les	bx,[bp+6]	; get DOS' stack into ES:BX

	mov	ax,es:[bx]	; Recover DOS' DS from program entry...
	mov	ds,ax

	pop	es:[bx-4]	; save ES:DI on DOS' stack
	pop	es:[bx-6]	; This is a FAR pointer to the SFT
	mov	es:[bx-2],sp	; Save the SP for later...
	sub	bx,6

	mov	bp,ds		; Now the request wants SS/DS = DOS' DS
	mov	ss,bp		; and we want DOS' old SP so switch 'em
	mov	sp,bx
	pop	es
	pop	bx
	mov	ax,FCBOWN
	int	MPX2F

	mov	bx,ResGroup
	mov	ds,bx		; restore our data segment first
	pop	bp		; Now work on getting our stack back.
	mov	ss,bx
	mov	sp,bp		; We're done!

	pop	es
	pop	di
	pop	si
	pop	bp
	pop	dx
	pop	cx
	pop	bx
	ret
entDOS	ENDP

; Return the first matching file/directory name when given an path name,
; search template & search attributes. This is somewhat odd looking. We
; have the fully qualified path & global name as ASCIIZ in the swappable
; data area. Use that to validate the path & template & nothing more. Now
; use the FCB style expansion to verify the candidate.
;
; Find next is called by the user until 18h is returned --no more files, or
; until the user is satisfied. Either directory or files can be found, NOT
; volume labels. The SDB set up globally in the DOS SDA from the findfirst or
; previous findnext call is used to figure out the next entry we want from
; the server. An nfsreaddir/nfslookup call sequence tells us if this file
; matches the search pattern/attributes. If so it is returned, otherwise the
; search continues.

fndfst	PROC	NEAR
fndnxt:	push	bp		; set up stack frame & local stg.
	mov	bp,sp

nfsde = size NFS_DIRENT				; struct nfs_dirent nfsde;
nfsatt= size NFS_DIRENT + size NFS_ATTRH	; struct nfs_attrib nfsatt;

	sub	sp,nfsatt + 4		; Space for attribute as well...
	push	si
	push	di
	push	es
	push	bx
	push	cx

	les	di,[OurSDA]		; init pointer to DOS' SDA
	cmp	[OurOS],3
	jg	safe4
	mov	bl,es:[di].lookatt	; save search attrib for later
	mov	[bp - (nfsatt + 1)],bl
	mov	bx,offset fffn		; pointer to sys data struct
	jmp short did34
safe4:	mov	bl,es:[di].lookatt4	; save search attrib from DOS 4.x
	mov	[bp - (nfsatt + 1)],bl	; (and later versions) for later.
	mov	bx,offset fffn4
;
;------	Which call is this? We share code to cut down on bugs & bytes
;
did34:	cmp	ax,2 * FNDNXTCMD
	jnz	nflord
	mov	word ptr [bp - (nfsatt + 4)],ENDFILE	
	jmp	flord1			; must be find next, go to it!
nflord:	lea	ax,[ffdirh]		; keep using this to find files, the
	push	ds			; FILE'S attrib.s go into local nfs_attrh
	push	ax			; &attrib);
	push	es
	push	[fnm1ea]		; /* pointer to the new file */ name,
	push	word ptr [OurCDS+2]	; Our current context
	mov	ax,word ptr [OurCDS]
	add	ax,offset path
	push	ax			; CDSstr,
	call	descend			; get the dope on this directory.
	and	ax,ax			; if (!descend(
	jnz	f1ok			; exit(3);
	mov	ax,BADPTH
	stc
	jmp	f1out

f1ok:	mov	word ptr es:[di+bx].dirent,0	; Dir entry #1
	mov	word ptr [bp - (nfsatt + 4)],NOFILE	; special return value...
	mov	ax,di				; save ptr to OurSDA
	mov	si,[fcb1ea]
	lea	di,[di+bx].patt			; copy search template
	mov	cx,ETN
	cli					; Don't interrupt 2nd prefix
	rep movs byte ptr [di],byte ptr es:[si]
	sti
	mov	di,ax				; recover OurSDA ptr
	mov	al,[bp - (nfsatt + 1)]		; save search attribute
	mov	es:[di+bx].fndattr,al

	cmp	al,MASK VOL
	jz	getvol				; if (OurSDA.lookatt == VOL
	test	al,MASK VOL
	jz	flord				;     || ((OurSDA.lookatt & VOL)
	mov	si,[fnm1ea]
	mov	al,'\'
	xchg	si,di
	add	di,[RootOffs]	; Skip past initial stuff
	inc	di
	mov	cx,13
tstroo:	test	byte ptr es:[di],0ffh
	jz	getv1				; check until name ends. If
	scasb					; '\' occurs we weren't @ root.
	loopnz	tstroo
	xor	cx,cx				; Force jump, (CX = 0 || '\' means !root)
getv1:	xchg	di,si				; deal with SCASB using di only.
	jcxz	flord				; && isroot()))
getvol:	lea	ax,[vlabel]
	push	ds
	push	ax				; candidate))
	push	es
	push	[fcb1ea]			; fcbname,
	call	match				; if (!match(
	jnz	flord
	call	lblde				; fill in volume label
	xor	ax,ax
	jmp short f1out    			;     exit(0);

flord:	call	quicklk				; first see if exact name exists
	jz	flord2				; Yeah. We saved much time!
				; DOS returns 2 if no files exit for FIND FIRST
				; BUT our NFS host will always have a parent &
				; current node, even at the mount point.
flord1:	call	fndent				; search for a matching entry
	mov	ax,[bp - (nfsatt + 4)]		; return value
	jnz	f1out				; /* Copy over matching entry */
flord2:	lea	ax,[bp-nfsde]			; dirent);
	push	ax
	lea	ax,[bp-nfsatt]			; handle,
	push	ss
	push	ax
	call	setsysfld			; setsysfld(
	xor	ax,ax
f1out:	pop	cx
	pop	bx
	pop	es
	pop	di
	pop	si
	mov	sp,bp
	pop	bp
	ret					; return(ENOMOREFILES);
fndfst	ENDP

; First check and see if the name given was complete. We can spare looking
; through the entire directory for a name that has no wildcards. This should
; speed up things considerably.

quicklk	PROC	NEAR
	push	word ptr [OurSDA+2]	; Potential fully-qualified
	push	[fnm1ea]		; name to search
	call	backup			; Isolate name from path.
	mov	si,ax			; Save pointer if good,
	mov	ax,BADPTH		; Else just bail out.
	jc	qkdie

	push	di			; Save DI
	push	cx			; Save CX
	push	ss			; Copy string to stack, fold
	push	es			; case if appropriate.
	lea	di,[bp-nfsde].nfsname
	mov	cx,ETN+2		; Heck, just move 13 bytes...
	call	tolower
	pop	cx
	pop	di

	lea	ax,[bp-nfsatt]			; &attrib);
	push	ss
	push	ax
	lea	ax,[ffdirh]			; &ffdirh,
	push	ds
	push	ax
	lea	ax,[bp-nfsde].nfsname		; dirent.name,
	push	ax
	call	nfs_lookup			; nfslookup(
	mov	ax,NOFILE
	jbe	qkdie				; if CY or Z flag set, give up.
;
;------ if (!(OurSDA.lookatt & DIR) && attrib.type == VDIR)
;
	test	byte ptr es:[di+bx].fndattr,MASK DIR
	jnz	qkchk
	cmp	[bp-nfsatt].nfstype-3,VDIR
	mov	ax,NOFILE
	jz	qkdie				; return(-1);

qkchk:	mov	ax,word ptr [bp-nfsatt].nfsinode	; Simulate a readdir()
	mov	word ptr [bp-nfsde].inode,ax
	lea	ax,[bp-nfsde].nfsname		; candidate))
	push	ss
	push	ax
	lea	ax,[di+bx].patt			; nm1fcb,
	push	es
	push	ax
	call	match				; if (!match(
	and	ax,ax
;
;**** HACK: We don't want to report the file for BOTH findfirst & findNext but don't
;     know the NFS_COOKIE (directory entry number) so we contrive a value.
	mov	word ptr es:[di+bx].dirent,7fffh	; Trouble if > 32767 DIR entries
	ret
qkdie:	and	ax,ax
	stc
	ret
quicklk	ENDP

; Handles the repetitive lookup/match portion of global pattern searches.
; Can be called by fndfst/fndnxt, or glob. It expects BP to have a structure
; allocated to hold the entries it needs. ES:DI should point to either a real
; DOS structure to hold the found entry, or a local copy. BX points to an
; offset inside that structure. AX = 12h if no more entries exist, 0 if a match
; was found, non-zero otherwise.

fndent	PROC	NEAR
	cmp	es:[di+bx].dirent,7fffh		; prevent repeated lookups for files
	jz	entend				; found with explicit name on first try.
	lea	ax,[bp-nfsde]			; &dirent);
	push	ss
	push	ax
	push	word ptr es:[di+bx].dirent	; NFS_COOKIE,
	lea	ax,[ffdirh]			; &ffdirh,
	push	ds
	push	ax
	call	nfs_readdir			; nfs_readdir(
	and	ax,ax
	jz	entend
	lea	ax,[bp-nfsatt]			; &attrib);
	push	ss
	push	ax
	lea	ax,[ffdirh]			; &ffdirh,
	push	ds
	push	ax
	lea	ax,[bp-nfsde].nfsname		; dirent.name,
	push	ax
	call	nfs_lookup			; nfslookup(
	mov	ax,[bp-nfsde].nfscookie
	mov	word ptr es:[di+bx].dirent,ax
;
;------ if (!(OurSDA.lookatt & DIR) && attrib.type == VDIR)
;
	test	byte ptr es:[di+bx].fndattr,MASK DIR
	jnz	chkfl
	cmp	[bp-nfsatt].nfstype-3,VDIR
	jz	fndent				; continue;

chkfl:	lea	ax,[bp-nfsde].nfsname		; candidate))
	push	ss
	push	ax
	lea	ax,[di+bx].patt			; nm1fcb,
	push	es
	push	ax
	call	match				; if (!match(
	and	ax,ax
	jnz	fndent
	jmp short fqout				;     exit(0);
entend:	mov	ax,ENDFILE
	cmp	al,0ffh				; Set CY and NZ flags
fqout:	ret
fndent	ENDP

; Set up the registers for extended error info if a name was wrong.

nmerr	PROC	NEAR
	mov	ax,3		; path not found
	mov	bx,83h		; file not found, get corrected info
	mov	cx,300h		; Error locus = network
	stc
	ret
nmerr	ENDP

; Convert the candidate ASCIIZ name to an FCB style name. The long address
; for the name is taken off the stack. Only AX is modified upon return. The
; ASCIIZ name is converted to *at most* 11 characters. Provisions are made
; to recognize '.' and '..' as directory names. Otherwise the name is split
; upon encountering the first '.' and the '.' is not placed explictily in
; the name. Parameters deallocated upon return. Returns 0 in AX if successful.

; Called as match(char FAR *nm1, char FAR *candidate);

match	PROC	NEAR
	push	bp		; make up a stack frame for us
	mov	bp,sp
	push	cx
	push	ds
	push	si
	push	di
	mov	cx,ETN
	mov	al,' '		; Clear FCB template to blanks
	mov	di,offset ds:TMPFCB
	dses
	rep stosb

	mov	cx,ETN
	lds	si,[bp+8]	; copy candidate into FCB style name
	mov	di,offset ds:TMPFCB
	lodsb			; The '.'/'..' need special care. Get
	ucase			; 1st char into AH & also move it.
	stosb
	mov	ah,al
whl011:	lodsb			; rest are moved as encountered.
	and	al,al
	jz	strend
	cmp	al,'\'		; Copy up to but not including \ or 0
	jz	strend
	cmp	al,'.'		; and skip over any . encountered
	jnz	cpychr		; except .. which is parent entry in subdir.
	cmp	ah,al
	jz	cpychr
	mov	di,offset ds:TMPFCB+8	; Jump to extension field in case of '.'
	loop	whl011			; ONLY move name + extension chars...
	jmp short strend
cpychr:	ucase
	stosb			; Ordinary character, copy it over.
	loop	whl011		; ONLY move name + extension chars...

strend:	lds	si,[bp+4]	; compare against search pattern
	lea	di,ds:[TMPFCB]	; now they're both in FCB format
	mov	cx,ETN+1	; TEST/DECR algorithm requires this.
chrchk:	rep cmpsb		; Do a straight comparison. Allow ?'s to
	jcxz	allcool		; be global. If CX = 0 all chars. matched.
	cmp	byte ptr [si]-1,'?'
	jz	chrchk
	cmp	byte ptr [si]-1,' '	; Spaces in the template are *implicit*
	jnz	notcool			; wild cards. BUT ONLY in the extension.
	cmp	cx,2		; If candidate < template then both have spaces
	jg	notcool		; if > then fail unless we're on the last 3, NOTE
	jmp	chrchk		; that template wildcard < 8 must fail too.

allcool:
	clc
	pop	es
	pop	di
	pop	si
	pop	ds
	pop	cx
	xor	ax,ax	; signal success
	pop	bp
	ret	8
notcool:
	stc
	mov	ax,ENDFILE	; flag no more files if match fails.
	pop	es
	pop	di
	pop	si
	pop	ds
	pop	cx
	pop	bp
	ret	8
match	ENDP

; Here we must initialize two regions of DOS' swappable data area. One is
; for subsequent calls to find next. It will expect the template & search
; attributes. The other is a region DOS will return with the first match
; for the given name.

; setsysfld(struct NFS_ATTRH FAR *nfsatt, struct NFS_DE *nfsde);

setsysfld	PROC	NEAR
	push	bp
	mov	bp,sp
	push	es
	push	di
	push	si
	push	cx
	push	bx
	les	di,[OurSDA]
	cmp	[OurOS],3
	jg	sys4d
	lea	di,[di].fffn		; the result
	jmp short sys4a
sys4d:	lea	di,[di].fffn4
;
; ---- First set up/update the SDB for future searches.
;
sys4a:	mov	bx,[bp+8]		; pointer to nfs info struct
	mov	ax,word ptr ss:[bx].inode	; What inode? (presently
	mov	es:[di].clutr,ax		; unused but supplied anyway)

	mov	al,[drive_no]		; which drive?
	or	al,80h
	mov	es:[di].drive,al
;
; ---- Now set up the "directory entry" for the found file
;
	mov	di,word ptr [OurSDA]	; Recover index to SDA
	cmp	[OurOS],3
	jg	sys4d2
	lea	di,[di].ffile
	jmp short sys4a2
sys4d2:	lea	di,[di].ffile4

sys4a2:	push	ds
	lds	si,[bp+4]

	mov	ax,word ptr [si].nfssize + 2
	mov	word ptr es:[di].fousize + 2,ax	; size is a longword so
	mov	ax,word ptr [si].nfssize
	mov	word ptr es:[di].fousize,ax	; fix it accordingly

	xor	ax,ax
	cmp	byte ptr [si].nfstype - 3,VDIR	; keep it simple. Only dir
	jnz	sysreg				; or files are distinguished.
	mov	word ptr [si].nfssize + 2,ax	; DIR's have zero size in DOS
	mov	word ptr [si].nfssize,ax
	mov	al,MASK DIR			; No system, hidden or archive
sysreg:	mov	byte ptr es:[di].fouattr,al	; testing done.

	mov	cx,dx				; preserve DX contents...
	push	word ptr [si].nfsmtime + 2	; get the UNIX time_t and
	push	word ptr [si].nfsmtime		; convert it into DOS' format
	call	time2DT
	mov	word ptr es:[di].m_date,dx	; 190fh = August 15, 1992
	mov	word ptr es:[di].m_time,ax	; 63c0h = 12:30pm
	mov	dx,cx				; restore DX register

	mov	ax,word ptr [si].nfsinode	; recover cluster saved before
	mov	es:[di].clust,ax

	pop	ds
	mov	si,offset ds:TMPFCB	; Now copy over the found file's
	mov	cx,ETN			; name. The search template was
	lea	di,[di].founame		; saved above. Don't be confused!
	rep	movsb

	pop	bx
	pop	cx
	pop	si
	pop	di
	pop	es
	pop	bp
	clc
	ret	6
setsysfld	ENDP

; Volume labels require somewhat different treatment. Be sure the attribute
; is given and then get the name over. Zero out all the other fields
; ON ENTRY: ES:DI MUST point to DOS' SDA, DS to ResGroup.

lblde	PROC	NEAR
	push	cx
	push	di
	push	si

	cmp	[OurOS],3	; Offsets to the found file data structure
	jg	lbl4a		; are different for DOS 3 vs. DOS 4 & later
	lea	ax,[di].fffn	; compute & save location of SDB too.
	push	ax
	lea	di,[di].ffile	; Humm. Was DOS 3
	jmp short lbl4b
lbl4a:	lea	ax,[di].fffn4	; Was a DOS 4 SDB...
	push	ax
	lea	di,[di].ffile4	; Nope, was DOS 4
lbl4b:	lea	si,[TMPFCB]	; FCB formatted name
	mov	cx,ETN
	mov	ax,di		; Save the address DI had.
	lea	di,[di].founame
	rep movsb
	mov	di,ax

	mov	es:[di].fouattr,MASK VOL	; Fix attribute of found item.
	xor	ax,ax				; Zero the other unused fields
	mov	cx,5
	lea	di,[di].m_time
	rep stosw

	pop	di		; Now we're ready to address SDB
	mov	al,[drive_no]	; saved above. Find first did
	or	al,80h		; most work, just put in drive #.
	mov	es:[di].drive,al
	pop	si
	pop	di
	pop	cx
	ret
lblde	ENDP

; Seek the specified number of bytes from the end of file.

skenfct	PROC	NEAR
	push	bx
	mov	bx,word ptr es:[di].sf_size+2	; find out where EOF is.
	mov	ax,word ptr es:[di].sf_size
	sub	ax,dx				; try seeking to that
	sbb	bx,cx				; offset from file's end
	jae	vlidsk
	xor	ax,ax				; too far, seek to file's start.
	mov	bx,ax
vlidsk:	mov	word ptr es:[di].pos+2,bx	; Save it in the SFT
	mov	word ptr es:[di].pos,ax
	clc
	pop	bx
	ret
skenfct	ENDP

; Qualify the path/file name given by DS:SI into the string of ES:DI

ifndef	TESTMODE
qulfct	PROC	NEAR
	mov	ax,3
	stc
	ret
qulfct	ENDP

endif

; Called when DOS wants to close all open files for a given process.

clsafct	PROC	NEAR
	xor	ax,ax
	ret
clsafct	ENDP

; Get a handle cache entry. Return it in AX if successful.

get_ent	PROC	NEAR
	push	bx
	mov	bx,[listhead]	; if successful. Get next avail
	and	bx,bx		; entry. Is there one?
	jz	outget
	mov	bx,[bx]		; track along the trail of pointers
	xchg	bx,[listhead]
	add	bx,2		; Step over pointer & return start
outget:	mov	ax,bx		; of structure
	pop	bx
	ret
get_ent	ENDP

; Release a handle and replace it in the cache. Handle is passed in AX

repl_ent	PROC	NEAR
	push	bx
	sub	ax,2		; Use pointer to the linked list.
	mov	bx,ax
	xchg	ax,[listhead]	; Install this entry at the list head.
	mov	[bx],ax		; Prev. list head is next after us.
	pop	bx
	ret
repl_ent	ENDP

; glob: GLOBalize name.
;
; This function is intended to expand the global characters * and ? into
; appropriate names. It takes a given name and attempts to look up a handle
; for the parent of that name. If successful, the parent's handle is returned.
; If not then 0 is returned with the CY flag set. This allows functions such
; as DELETE, RENAME, GET_ATTRIBUTE, SET_ATTRIBUTE to function with global
; arguments but still be independent of the network representation.

ifdef	NOTYET
glob	PROC	NEAR
	push	bp
	mov	bp,sp
	sub	sp,nfsatt
	push	si
	push	es
	push	di
	push	bx
	push	cx
	lea	ax,[ffdirh]	; /* directory search handle */
	push	ds
	push	ax
	push	es		; /* Name we wish to globalize */
	push	di
	push	word ptr [OurCDS+2]	; /* our current context */
	mov	ax,word ptr [OurCDS]
	add	ax,offset path
	push	ax
	call	descend
	and	ax,ax
	jz	globnd

; Need to set up bogus system area, copy over globalized search pattern
; and then call the find-first/next engine. If it succeeds just return the
; [ffdirh] because that's the parent we want.
	sub	sp,size SDB	; allocate bogus system data block
	mov	bx,ss		; Fake access to DOS data area.
	mov	es,bx
	mov	bx,2		; offset from DI into bogus SDB
	xor	ax,ax		; Regular file look attribute
	push	ax
	mov	di,sp		; Point to the look attribute
; What to do about the name? It will need global replacement of * with ?'s
; and it needs to go into our es:[di+bx].patt
; The name exists in [OurSDA].fname1.
globnd:	pop	cx
	pop	bx
	pop	di
	pop	es
	pop	si
	mov	sp,bp
	pop	bp
	ret
glob	endp
endif

Restext	ENDS

ifdef	TESTMODE

; Make the internal DOS to NFS mapping functions visible

	public	opnfct, crtfct, readfct, wrtfct, fndfst
	public	getafct, setafct, rdfct, cdfct, mdfct
	public	dltfct, renfct, clsfct, skenfct

endif
	END

