/*
 * $Author: djh $ $Date: 1992/03/07 11:04:41 $
 * $Header: /mac/src/cap60/lib/cap/RCS/abkas.c,v 2.1 1992/03/07 11:04:41 djh Rel djh $
 * $Revision: 2.1 $
 */

/*
 * abkas.c - use UNIX Kernel AppleTalk Support via AF_APPLETALK sockets
 *
 * AppleTalk package for UNIX (4.2 BSD).
 *
 * Copyright (c) 1986, 1987, 1988 by The Trustees of Columbia University 
 *  in the City of New York.
 *
 * Copyright (c) 1992 The University of Melbourne.
 *
 * Edit History:
 *
 *  February 1992	djh	Created
 *
 */

/*
 * The following list of exported routines is provided so you'll know what
 * have to be done to do another interface type (ethertalk, etc)
 *
 * EXPORTED ROUTINES:
 *
 *  OSErr GetNodeAddress(int *mynode, int *mynet)
 *     Return node addresses
 *  OSErr GetBridgeAddress(AddrBlock *addr)
 *     Return bridge addresses
 *  OSErr SetBridgeAddress(AddrBlock *addr)
 *     Set bridge addresses
 *  OSErr SetNetRange(u_short range_start, u_short range_end)
 *     Set Network Range (Phase 2)
 *  int abInit(boolean dispay_message)
 *     Initialize AppleTalk
 *  int abOpen(int *returnsocket, int wantsocket, struct iovec iov[], iovlen)
 *     Open a DDP socket
 *  int abClose(int socket)
 *     Close a DDP socket
 *  void abnet_cacheit(word srcNet, byte srcNode)
 *     Call in DDP protocol layer to tell the lower layer that
 *      the last packet that came in was from srcNet, srcNode
 *  int routeddp(struct iovec *iov, int iovlen)
 *     This is the DDP incursion.  With a full AppleTalk implementation,
 *     this would be part of DDP (abddp2).  This routes the DDP packet:
 *     normally would decide where to send and then send via lap, with KIP
 *     decides where and sends via UDP.
 *
 */

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <netinet/in.h>
#include <sys/uio.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/file.h>
#include <sys/types.h>
#include <net/if.h>
#include <netdb.h>
#include <netat/appletalk.h>
#include <netat/compat.h>	/* to cover difference between bsd systems */
#include <netat/abnbp.h>

#define ATADDR_ANYNET	(u_short)0x0000
#define ATADDR_LATENET	(u_short)0xffff
#define ATADDR_ANYNODE	(u_char)0x00
#define ATADDR_ANYPORT	(u_char)0x00
#define ATADDR_BCAST	(u_char)0xff

#ifdef s_net
#undef s_net
#endif s_net

/* AppleTalk Address */

struct at_addr {
  u_short s_net;
  u_char s_node;
};

/* Socket Address for AppleTalk */

struct sockaddr_at {
  short sat_family;
  u_char sat_port;
  struct at_addr sat_addr;
  char sat_zero[8];
};

/* imported network information */

extern word	this_net,	bridge_net,	nis_net,	async_net;
extern byte	this_node,	bridge_node,	nis_node;
extern char	this_zone[34],	async_zone[34],	interface[50];

extern struct in_addr bridge_addr;

#ifdef PHASE2
extern word net_range_start, net_range_end;
#endif PHASE2

short lap_proto = LAP_KERNEL;	/* kernel appletalk support */

/*
 * Configuration defines
 *
 * NORECVMSG - no recvmsg()
 * NOSENDMSG - no sendmsg()
 * MEEDMSGHDR - no msghdr in sockets.h - define our own
 *
 */

#ifdef NORECVMSG
# ifndef NEEDNETBUF
#  define NEEDNETBUF
# endif  NEEDNETBUF
#endif NORECVMSG
#ifdef NOSENDMSG
# ifndef NEEDNETBUF
#  define NEEDNETBUF
# endif  NEEDNETBUF
#endif NOSENDMSG

import int ddp_protocol();	/* DDP protocol handler */

private int ddp_get();		/* kernel packet listener */
private int kip_get();		/* loopback packet listener */

private struct sockaddr_at addr; /* local appletalk socket addr */
private struct sockaddr_at raddr;/* input appletalk socket addr */
private struct sockaddr_in from; /* loopback receive address */
private struct sockaddr_in send; /* send packet to loopback */

private int skt2fd[ddpMaxSkt+1]; /* translate socket to file descriptor */
private int skt2kip[ddpMaxSkt+1];/* loopback socket to file descriptor */
private int ddpskt2udpport[ddpMaxWKS+1]; /* ddp "wks" skt to udp port */

private LAP laph;

export DBUG dbug;

/*
 * initialize
 *
 */

export
abInit(disp)
int disp;
{
  int i;
  static int here_before = 0;

  for (i = 0; i < ddpMaxSkt+1; i++) {
    skt2fd[i] = -1;		/* mark all these as unused */
    skt2kip[i] = -1;		/* mark all these as unused */
  }

  for (i = 0; i < ddpMaxWKS+1; i++)
    ddpskt2udpport[i] = -1;	/* mark as unused */

  if (!here_before) {
    openetalkdb(NULL);		/* set up variables */
    here_before = 1;
  }

  init_fdlistening();

  if (disp) {
      printf("abInit: [ddp: %3d.%02d, %d]",
	     ntohs(this_net) >> 8, ntohs(this_net) & 0xff, this_node);
      if (this_net != nis_net || this_node != nis_node)
	printf(", [NBP (atis) Server: %3d.%02d, %d]",
	       ntohs(nis_net) >> 8, ntohs(nis_net) & 0xff, nis_node);
      if (bridge_node)
	printf(", [GW: %3d.%02d, %d]",
	       ntohs(this_net) >> 8, ntohs(this_net) & 0xff, bridge_node);
      printf(" starting\n");
  }

  DDPInit();
  return(0);
}

/*
 * int abOpen(int *skt,rskt, iov, iovlen)
 *
 * abOpen opens the ddp socket in "skt" or if "skt" is zero allocates
 * and opens a new socket.  Upon return "skt" contains the socket number
 * and the returned value is >=0 if no error, < 0 if error.
 *
 * iov should be an array of type "struct iov" of length at least
 * IOV_LAP_SIZE+1.  Levels after IOV_LAP_LVL are assume to filled.
 *
 */

int abOpen(skt,rskt, iov, iovlen)
int *skt;
int rskt;
struct iovec *iov;
int iovlen;
{
  int len, fd, err;

  if (iov == NULL || iovlen < IOV_LAP_SIZE+1 || iovlen > IOV_READ_MAX)
    return(-1);

  if ((fd = socket(AF_APPLETALK, SOCK_DGRAM, 0)) < 0) {
    perror("socket()");
    return(fd);
  }

  bzero(&addr, sizeof(struct sockaddr_at));
  addr.sat_family = AF_APPLETALK;
  addr.sat_addr.s_net = ATADDR_LATENET;
  addr.sat_addr.s_node = ATADDR_ANYNODE;
  addr.sat_port = (rskt == 0) ? ATADDR_ANYPORT : rskt;

  if ((err = bind(fd, &addr, sizeof(struct sockaddr_at))) < 0) {
    perror("bind()");
    close(fd);
    return(err);
  }

  len = sizeof(struct sockaddr_at);
  if ((err = getsockname(fd, &addr, &len)) < 0) {
    perror("getsockname()");
    close(fd);
    return(err);
  }

  *skt = addr.sat_port;

  iov[IOV_LAP_LVL].iov_base = (caddr_t)&laph;	/* remember this */
  iov[IOV_LAP_LVL].iov_len = lapSize;		/* and this */

  fdlistener(fd, ddp_get, iov, iovlen);		/* remember for later */

  skt2fd[*skt] = fd;		/* remember file descriptor for socket */

  /* open KIP loopback socket */

  if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    perror("socket()");
    fdunlisten(skt2fd[*skt]);
    close(skt2fd[*skt]);
    return(fd);
  }

  send.sin_family = AF_INET;
  send.sin_addr.s_addr = INADDR_ANY;
  if ((send.sin_port = ddp2ipskt(*skt)) == 0) {
    fdunlisten(skt2fd[*skt]);
    close(skt2fd[*skt]);
    close(fd);
    return(ddpSktErr);
  }

  if ((err = bind(fd, (struct sockaddr *)&send, sizeof(struct sockaddr))) < 0) {
    perror("bind()");
    fdunlisten(skt2fd[*skt]);
    close(skt2fd[*skt]);
    close(fd);
    return(ddpSktErr);
  }

  fdlistener(fd, kip_get, iov, iovlen);

  skt2kip[*skt] = fd;

  return(noErr);
}

/*
 * close off socket opened by abOpen()
 *
 */

export int
abClose(skt)
int skt;
{
  int fd;

  if (skt < 0 || skt > ddpMaxSkt) {
    fprintf(stderr,"abClose: skt out of range\n");
    exit(0);
  }

  /* close AppleTalk skt */
  if ((fd = skt2fd[skt]) >= 0) {
    fdunlisten(fd);
    close(fd);
  }

  /* close KIP loopback */
  if ((fd = skt2kip[skt]) >= 0) {
    fdunlisten(fd);
    close(fd);
  }
    
  skt2fd[skt] = -1;		/* mark as unused */
  skt2kip[skt] = -1;		/* mark as unused */

  return(0);
}

int
abnet_cacheit()
{
}

/*
 * get a packet from the network
 *
 * construct appropriate DDP header.
 *
 */

private int
ddp_get(fd, iov, iovlen)
int fd;
struct iovec *iov;
int iovlen;
{
  struct msghdr msg;
  int len, err;
  LAP *lap;
  DDP *ddp;

  /* want DDP type in byte 0 */
  iov[IOV_DDP_LVL].iov_len = 1;

  msg.msg_name = (caddr_t) &raddr;
  msg.msg_namelen = sizeof(struct sockaddr_at);
  msg.msg_iov = iov+1;
  msg.msg_iovlen = iovlen-1;
  msg.msg_accrights = 0;
  msg.msg_accrightslen = 0;

  len = sizeof(struct sockaddr_at);
  if ((err = getsockname(fd, &addr, &len)) < 0) {
    perror("getsockname()");
    return(err);
  }

  if ((len = recvmsg(fd, &msg, 0)) < 0) {
    perror("recvmsg()");
    return(len);
  }

  /* make up some dummy headers */

  lap = (LAP *)iov[IOV_LAP_LVL].iov_base;
  lap->type = lapDDP;
  lap->dst = addr.sat_addr.s_node;
  lap->src = raddr.sat_addr.s_node;
  iov[IOV_LAP_LVL].iov_len = lapSize;

  ddp = (DDP *)iov[IOV_DDP_LVL].iov_base;
  ddp->type = iov[IOV_DDP_LVL].iov_base[0];
  ddp->length = len + ddpSize - 1;
  ddp->checksum = 0;
  ddp->dstNet = addr.sat_addr.s_net;
  ddp->srcNet = raddr.sat_addr.s_net;
  ddp->dstNode = addr.sat_addr.s_node;
  ddp->srcNode = raddr.sat_addr.s_node;
  ddp->dstSkt = addr.sat_port;
  ddp->srcSkt = raddr.sat_port;
  iov[IOV_DDP_LVL].iov_len = ddpSize;

  return(ddp_protocol(iov+1, iovlen-1, len+ddpSize));
}

/*
 * read a KIP packet via loopback
 *
 */

private int
kip_get(fd, iov, iovlen)
int fd;
struct iovec *iov;
int iovlen;
{
  struct msghdr msg;
  int len;
  LAP *lap;

  msg.msg_name = (caddr_t) &from;
  msg.msg_namelen = sizeof(from);
  msg.msg_iov = iov;
  msg.msg_iovlen = iovlen;
  msg.msg_accrights = 0;
  msg.msg_accrightslen = 0;

  if ((len = recvmsg(fd, &msg, 0)) < 0) {
    perror("recvmsg()");
    return(len);
  }
  if (iov->iov_len != lapSize) /* check */
    return(-1);

  lap = (LAP *)iov->iov_base;

  if (lap->type == lapDDP)
    return(ddp_protocol(iov+1, iovlen-1, len-lapSize));

  return(-1);
}

export int
routeddp(iov, iovlen)
struct iovec *iov;
int iovlen;
{
  struct sockaddr_at taddr;
  struct sockaddr_in tokip;
  struct msghdr msg;
  DDP *ddp;
  LAP lap;
  int err;
  int fd;

  ddp = (DDP *)iov[IOV_DDP_LVL].iov_base; /* pick out ddp header */

  /* check ddp socket(s) for validity */
  if ( ddp->srcSkt == 0 || ddp->srcSkt == ddpMaxSkt ||
       ddp->dstSkt == 0 || ddp->dstSkt == ddpMaxSkt ||
       skt2fd[ddp->srcSkt] == -1 || skt2kip[ddp->srcSkt] == -1)
    return(ddpSktErr);

  /* check for loopback */
  if ((ddp->dstNet == this_net)
   && (ddp->dstNode == this_node || ddp->dstNode == 0xff)) {
    lap.type = lapDDP;
    lap.dst = ddp->dstNode;
    lap.src = this_node;
    iov[IOV_LAP_LVL].iov_base = (caddr_t)&lap;
    iov[IOV_LAP_LVL].iov_len = lapSize;

    tokip.sin_family = AF_INET;
    tokip.sin_addr.s_addr = bridge_addr.s_addr;
    if ((tokip.sin_port = ddp2ipskt(ddp->dstSkt)) == 0)
      return(ddpSktErr);

    msg.msg_name = (caddr_t)&tokip;
    msg.msg_namelen = sizeof(tokip);
    msg.msg_iov = iov;
    msg.msg_iovlen = iovlen;
    msg.msg_accrights = 0;
    msg.msg_accrightslen = 0;
    fd = skt2kip[ddp->srcSkt];

    if ((err = sendmsg(fd, &msg, 0)) < 0)
      perror("sendmsg()");

    if (ddp->dstNode != 0xff)
      return(noErr);
  }

  /* send via kernel EtherTalk */

  bzero(&taddr, sizeof(struct sockaddr_at));
  taddr.sat_family = AF_APPLETALK;
  taddr.sat_addr.s_net = ddp->dstNet;
  taddr.sat_addr.s_node = ddp->dstNode;
  taddr.sat_port = ddp->dstSkt;

  /* collapse header, except for DDP Type */
  iov[IOV_DDP_LVL].iov_base += (ddpSize-1);
  iov[IOV_DDP_LVL].iov_len = 1; /* ddp type */

  msg.msg_name = (caddr_t)&taddr;
  msg.msg_namelen = sizeof(struct sockaddr_at);
  msg.msg_iov = iov+1;
  msg.msg_iovlen = iovlen-1;
  msg.msg_accrights = 0;
  msg.msg_accrightslen = 0;
  fd = skt2fd[ddp->srcSkt];

  if ((err = sendmsg(fd, &msg, 0)) < 0) {
    perror("sendmsg()");
    return(err);
  }

  return(noErr);
}

/*
 * configure the network interface
 *
 */

int
ifconfig(net, node)
int *net, *node;
{
  struct sockaddr_at addr;
  struct sockaddr_at *sat;
  struct ifreq ifreq;
  int len, err;
  int fd;

  if ((fd = socket(AF_APPLETALK, SOCK_DGRAM, 0)) < 0) {
    perror("socket()");
    return(fd);
  }
  
  strncpy(ifreq.ifr_name, interface, sizeof(ifreq.ifr_name));
  sat = (struct sockaddr_at *) &ifreq.ifr_addr;
  bzero(sat, sizeof(struct sockaddr_at));
  sat->sat_family = AF_APPLETALK;
  sat->sat_addr.s_net = *net;
  sat->sat_addr.s_node = *node;
  sat->sat_port = ATADDR_ANYPORT;

  if ((err = ioctl(fd, SIOCSIFADDR, &ifreq)) < 0) {
    perror("SIOCSIFADDR");
    return(err);
  }

  bzero(&addr, sizeof(struct sockaddr_at));
  addr.sat_family = AF_APPLETALK;
  addr.sat_addr.s_net = ATADDR_ANYNET;
  addr.sat_addr.s_node = ATADDR_ANYNODE;
  addr.sat_port = ATADDR_ANYPORT;

  if ((err = bind(fd, &addr, sizeof(struct sockaddr_at))) < 0) {
    perror("bind()");
    return(err);
  }

  len = sizeof(struct sockaddr_at);
  if ((err = getsockname(fd, &addr, &len)) < 0) {
    perror("getsockname()");
    return(err);
  }

  *net = addr.sat_addr.s_net;
  *node = addr.sat_addr.s_node;

  close(fd);
  return(0);
}

/*
 * ddp to udp wks translate table
 *
 */

struct wks {
  char *name;			/* name of wks (as in /etc/services) */
  int ddpport;			/* ddp port to map from */
  int udpport;			/* udp port to map to */
  int notinited;		/* tried /etc/services? */
};

/*
 * udpport is initially set to the old
 * (768) values for compatibility
 *
 */

#define WKS_entry(name, ddpsock) {(name), (ddpsock), ddpWKSUnix+(ddpsock), 1}

private struct wks wks[] = {
  WKS_entry("at-rtmp", rtmpSkt),
  WKS_entry("at-nbp", nbpNIS),
  WKS_entry("at-echo", echoSkt),
  WKS_entry("at-zis", zipZIS),
  WKS_entry(NULL, 0)
};

/*
 * Translate ddp socket to UDP port: returns 0 if no mapping
 *
 */

ddp2ipskt(ddpskt)
int ddpskt;
{
  struct wks *wksp;
  struct servent *serv;

  if (ddpskt < 0 || ddpskt > ddpMaxSkt)
    return(0);

  if (ddpskt & ddpWKS)		/* 128+x means non-wks */
    return(ddpskt + ddpNWKSUnix);

  if (ddpskt2udpport[ddpskt] < 0) {
    for (wksp = wks; wksp->name != NULL; wksp++)
      if (wksp->ddpport == ddpskt) {
	if ((serv = getservbyname(wksp->name, "udp")) != NULL)
	  wksp->udpport = ntohs(serv->s_port); /*  replace with new */
	if (dbug.db_ini)
	  fprintf(stderr, "port for %s is %d\n",wksp->name,wksp->udpport);
	endservent();
	ddpskt2udpport[ddpskt] = wksp->udpport;
	return(wksp->udpport);
      }
    ddpskt2udpport[ddpskt] = 0;
  }
  return(ddpskt2udpport[ddpskt]);
}

#ifdef NEEDNETBUF
#ifdef NEEDMSGHDR
struct msghdr {
  caddr_t msg_name;		/* name to send to */
  int msg_namelen;		/* size of name */
  struct iovec *msg_iov;	/* io vec */
  int msg_iovlen;		/* length */
  int msg_accrights;		/* dummy */
  int msg_accrightslen;
};
#endif NEEDMSGHDR

/*
 * buffer larger than maximum ddp pkt by far
 *
 */

private char net_buffer[ddpMaxData*2];

#ifdef NOSENDMSG
/*
 * limited sendmsg - limits to sizeof(net_buffer)
 *
 */

sendmsg(fd, msg, flags)
int fd;
struct msghdr *msg;
int flags;
{
  int err;
  int i, pos, len;
  struct iovec *iov;

  iov = msg->msg_iov;
  for (i=0, pos=0; i < msg->msg_iovlen; i++, iov++) {
    len = iov->iov_len;
    if (len+pos > sizeof(net_buffer)) /* if overflow */
      len = sizeof(net_buffer)-pos; /* then limit */
    bcopy(iov->iov_base, net_buffer+pos, len);
    pos+= len;
    if (len != iov->iov_len)	/* we don't have any more space */
      break;
  }
  len = pos;
  if ((err=sendto(fd,net_buffer,len,0,msg->msg_name,msg->msg_namelen)) < 0)
    perror("sendmsg()");
  return(err);
}
#endif NOSENDMSG

#ifdef NORECVMSG
recvmsg(fd, msg, flags)
int fd;
struct msghdr *msg;
int flags;
{
  int err;
  int i, pos, len, blen;
  struct iovec *iov;

  err = recvfrom(fd, net_buffer, sizeof(net_buffer), 0,
		 msg->msg_name, &msg->msg_namelen);
  if (err < 0)
    perror("recvfrom()");
  for (blen=err,pos=0,i=0,iov=msg->msg_iov; i < msg->msg_iovlen; i++, iov++) {
    len = min(iov->iov_len, blen);
    if ((pos + len) > sizeof(net_buffer)) /* if asking for too much */
      len = sizeof(net_buffer) - pos; /* then limit */
    bcopy(net_buffer+pos, iov->iov_base, len);
    pos += len;
    blen -= len;
    /* either no more room or no more data */
    if (len != iov->iov_len)
      break;
  }
  return(err);  
}
#endif NORECVMSG
#endif NEEDNETBUF

/*
 * OSErr GetNodeAddress(int *myNode,*myNet)
 *
 * GetNodeAddress returns the net and node numbers for the current host.
 *
 */

export OSErr
GetNodeAddress(myNode,myNet)
int *myNode,*myNet;
{
  *myNode = this_node;
  *myNet = this_net;
  return(noErr);			   /* is ok */
}

/*
 * Set Node Address
 *
 */

OSErr
SetNodeAddress(myNet, myNode)
int myNet, myNode;
{
      if (ifconfig(&myNet, &myNode) == 0) {
        this_net = nis_net = myNet;
        this_node = nis_node = myNode;
        etalkdbupdate(NULL);
      }
      return(noErr);
}

/*
 * Get Bridge Address
 *
 */

OSErr
GetBridgeAddress(baddr)
AddrBlock *baddr;
{
      baddr->net = bridge_net;
      baddr->node = bridge_node;
      return(noErr);
}

/*
 * Set Bridge Address
 *
 */

OSErr
SetBridgeAddress(baddr)
AddrBlock *baddr;
{
      bridge_net = baddr->net;
      bridge_node = baddr->node;
      if (this_net == 0)
	SetNodeAddress(bridge_net, 0);
      etalkdbupdate(NULL);
      return(noErr);
}

#ifdef PHASE2
/*
 * Set Network Range
 *
 */

OSErr
SetNetRange(range_start, range_end)
u_short range_start, range_end;
{
      this_net = nis_net = net_range_start = range_start;
      net_range_end = range_end;
      etalkdbupdate(NULL);
      return(noErr);
}
#endif PHASE2
