/*
 *	Copyright (c) 1993 The CAD lab of the
 *	Novosibirsk Institute of Broadcasting and Telecommunication
 *
 *	BPFT $Id: trafshow.c,v 1.4 1994/01/14 11:38:39 bob Exp $
 *
 *	$Log: trafshow.c,v $
 * Revision 1.4  1994/01/14  11:38:39  bob
 * Bug fixed in display tcp/udp fragmented datagramms
 *
 * Revision 1.3  1994/01/06  15:34:35  bob
 * Full modification of the visual information
 *
 * Revision 1.2  1993/12/05  15:51:15  bob
 * Added ICMP messages
 *
 * Revision 1.1  1993/11/20  12:15:59  bob
 * Initial revision
 *
 *
 * Redistribution and use in source forms, with and without modification,
 * are permitted provided that this entire comment appears intact.
 * Redistribution in binary form may occur without any restrictions.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT ANY WARRANTIES OF ANY KIND.
 */

/*	trafshow.c - show tcp/udp network data traffic	*/

static char copyright[] = "*** Copyright (c) 1993 CAD lab, Institute of Telecommunication, Novosibirsk ***";

#include <curses.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/udp.h>
#include <netinet/udp_var.h>
#include <netinet/tcp.h>
#include <netinet/tcpip.h>
#include <netinet/ip_icmp.h>
#include <netdb.h>

#include "../include/addrtoname.h"
#include "../include/traffic.h"
#include "trafshow.h"

static void onterm();
static void onalarm();

static int iplen, dlen;
static u_char s_port_big[IPPORT_RESERVED];
struct timeval time_act;
WINDOW *win;
int alarm_flag;
u_long tcp_pdata, udp_pdata, all_bytes;

/*
 * Full initialization.
 */
int
traf_init(fflag)
	int fflag;
{
	register i;
	struct servent *sv;

	entries = (struct t_entry *) calloc(MAX_ENTRIES + 1,
					    sizeof(struct t_entry));
	entries1 = (struct t_entry *) calloc(MAX_ENTRIES + 1,
					    sizeof(struct t_entry));
	atable = (struct t_addr *) calloc(ADDR_TABLE, sizeof(struct t_addr));
	if (entries == NULL || entries1 == NULL || atable == NULL)
		return 1;

	for (i = 1; i < ADDR_TABLE; i++)
		atable[i].prev = &(atable[i-1]);
	atable[0].prev = &(atable[ADDR_TABLE-1]);

	for(i = 0; i < IPPORT_RESERVED; i++)
		s_port_big[i] = 0;
	setservent(1);
	while (sv = getservent()) {
		NTOHS(sv->s_port);
		if (sv->s_port > IPPORT_RESERVED) {
			i = sv->s_port & (IPPORT_RESERVED-1);
			if (strcmp(sv->s_proto, "tcp") == 0)
				s_port_big[i] |= IPPROTO_TCP;
			else if (strcmp(sv->s_proto, "udp") == 0)
				s_port_big[i] |= IPPROTO_UDP;
		}
	}
	endservent();
	n_entry=alarm_flag=tcp_pdata=udp_pdata=all_bytes=0;
	init_addrtoname(device_name, fflag);
	signal(SIGHUP, onterm);
	signal(SIGINT, onterm);
	signal(SIGQUIT, onterm);
	signal(SIGTERM, onterm);
	signal(SIGALRM, onalarm);
	win = initscr();
	printw("\
 From Address                 To Address                 Proto     Bytes  CPS");
	move(0, 79);
	refresh();
	alarm(REFRESH);
	gettimeofday(&time_act, (struct timezone *)NULL);
	return 0;
}

/*
 * Clean return, the routine endwin() should be called before exit.
 */
static void
onterm()
{
	clear();
	refresh();
	endwin();
	exit(0);
}

/*
 * SIGALRM interrupt handler, should be ran each REFRESH seconds.
 */
static void
onalarm()
{
	if (alarm_flag) deleteold();
	else alarm(REFRESH);
	alarm_flag++;
}

/*
 * Cache function for speed up domain resolving.
 */
char *
addrhash(in)
	struct in_addr in;
{
	register i;
	register struct t_addr *p = atable;

	for (i = 0; i < ADDR_TABLE; i++)
		if (p->ina.s_addr == in.s_addr)
			return p->astr;
		else
			p = p->prev;
	atable = atable->prev;
	atable->ina.s_addr = in.s_addr;
	alarm(0);
	signal(SIGALRM, SIG_IGN);
	strcpy(atable->astr, ipaddr_string(&in));
	signal(SIGALRM, onalarm);
	return atable->astr;
}

/*
 * Pretty print an Internet address (net address + port).
 */
char *
inet_print(in, port, proto)
	struct in_addr in;
	u_short port;
	u_char proto;
{
	static char aline[29];
	char pline[11], *cp;
	int alen = 28, plen;

	sprintf(aline, "%-28.28s", addrhash(in));
	if (!port)
		return aline;
	plen = sprintf(pline, "..%.8s", proto == IPPROTO_TCP ?
		       tcpport_string(port) : udpport_string(port));
	if ((cp = strchr(aline, ' ')) != NULL)
		alen = cp - aline;
	if ((alen + plen) >= 28)
		bcopy(pline, &aline[28 - plen], plen);
	else
		bcopy(pline, &aline[alen], plen);
	return aline;
}

void
main_print(i, j)
	register i;
	int j;
{
	char *proto;

	if (entries[i].who_srv & 1)
		printw("%s ", inet_print(entries[i].in_ip, entries[i].p_port,
					 entries[i].ip_protocol));
	else
		printw("%-28.28s ", addrhash(entries[i].in_ip));
	if (entries[i].who_srv & 2)
		printw("%s ", inet_print(entries[i].out_ip, entries[i].p_port,
					 entries[i].ip_protocol));
	else
		printw("%-28.28s ", addrhash(entries[i].out_ip));
	switch(entries[i].ip_protocol) {
		case IPPROTO_TCP:
			proto = "tcp";
			break;
		case IPPROTO_UDP:
			proto = "udp";
			break;
		case IPPROTO_ICMP:
			proto = "icmp";
			break;
		case IPPROTO_EGP:
			proto = "egp";
			break;
		case IPPROTO_OSPF:
			proto = "ospf";
			break;
		case IPPROTO_IGMP:
			proto = "igmp";
			break;
		default:
			proto = "unkn";
			break;
	}	
	if (j)	printw("%-4.4s %10ld %-6d", proto, entries[i].n_psize, j);
	else	printw("%-4.4s %10ld       ", proto, entries[i].n_psize);
}

/*
 * If REFRESH time any entry wasn't changed then purge it and redraw screen.
 */
deleteold()
{
	register i, j = 0;
	struct timeval time_now;
	int cps[MAX_ENTRIES];

	alarm(0);
	gettimeofday(&time_now, (struct timezone *)NULL);
	for (i = 0; i < n_entry; i++) {
		if (entries[i].n_psize != entries1[i].n_psize) {
			cps[j] = (entries[i].n_psize - entries1[i].n_psize)/REFRESH;
			entries1[j] = entries[i];
			j++;
		}
	}
	n_entry = j;
	bzero(entries1+j, (MAX_ENTRIES+1 - n_entry) * sizeof(struct t_entry));
	bcopy(entries1, entries, n_entry * sizeof(struct t_entry));
	move(SCR_OFFS, 0);
	for (i = 0; i < n_entry; i++)
		main_print(i, cps[i]);
	if (!i)
		switch (alarm_flag) {
		case 1:
			addstr("Where is the tcp or udp packets contained data? It is exactly what I want! Now!");
			break;
		case 2:
			addstr("Nothing to show, this interface sleeping or broken. blah-blah-gluk-gluk-wait...");
			break;
		default:
			addstr(copyright);
			alarm_flag = 1;
		}
	clrtobot();
	move(24, 0);
	printw("(%s) TCP pkts: %-8ld UDP pkts: %-8ld KBytes: %-8ld Inactivity: %d",
	       device_name, tcp_pdata, udp_pdata, all_bytes/1024,
	       time_now.tv_sec - time_act.tv_sec);
	if (!i) {
		move(0,79);
		refresh();
	}
	alarm(REFRESH);
}

/*
 * Find entry in table.
 */
static inline int
findentry( e )
	register p_entry e;
{
	register i;

	for (i = 0; i < n_entry; i++) {
		if (!entries[i].in_ip.s_addr)
			continue;
		if (e->in_ip.s_addr == entries[i].in_ip.s_addr &&
		    e->out_ip.s_addr == entries[i].out_ip.s_addr &&
		    e->ip_protocol == entries[i].ip_protocol &&
		    e->who_srv == entries[i].who_srv &&
		    e->p_port == entries[i].p_port)
			return i;
	}
	return -1;
}

/*
 * Insert entry.
 */
int
insertentry( e )
	register p_entry e;
{
	register int i;

	if ((i = findentry(e)) != -1)
		return i;
	entries[n_entry] = *e;
	return -1;
}

/*
 * Add new entry or add bytes to existed record.
 */
int
traf_add(i, bytes)
	register int i;
	register u_long bytes;
{

	all_bytes += bytes;
	gettimeofday(&time_act, (struct timezone *)NULL);
	if (i >= 0) {
		entries[i].n_psize += bytes;
		move(i + SCR_OFFS, 63);
		printw("%10ld", entries[i].n_psize);
	} else {
		if (++n_entry <= MAX_ENTRIES) {
			i = n_entry - 1;
			move(i + SCR_OFFS, 0);
			main_print(i, 0);
			move(i + SCR_OFFS + 1, 0);
			clrtoeol();
			alarm(REFRESH);
		} else {
			--n_entry;
			move(24, strlen(device_name) + 3);
			printw("screen overflow: lose sight of %d bytes", bytes);
			clrtoeol();
		}
	}
	move(0, 79);
	refresh();
	return 1;
}

void
traf_tcp(tp, ip)
	register struct tcphdr *tp;
	register struct ip *ip;
{
	struct t_entry t;

	if ((u_char *)(tp + 1) > snapend || dlen < sizeof(struct tcphdr))
		return;

	NTOHS(tp->th_sport);
	NTOHS(tp->th_dport);

	t.in_ip.s_addr = ip->ip_src.s_addr;
	t.out_ip.s_addr = ip->ip_dst.s_addr;
	t.ip_protocol = IPPROTO_TCP;
	t.n_psize = (u_long)iplen;

	if (tp->th_sport < tp->th_dport)
		t.p_port = tp->th_sport, t.who_srv = 1;
	else if (tp->th_sport > tp->th_dport)
		t.p_port = tp->th_dport, t.who_srv = 2;
	else if (tp->th_sport = tp->th_dport)
		t.p_port = tp->th_sport, t.who_srv = 3;
	if (t.p_port > IPPORT_RESERVED) {
		if (s_port_big[tp->th_sport & (IPPORT_RESERVED-1)] & IPPROTO_TCP) {
			t.p_port = tp->th_sport;
			t.who_srv = 1;
		} else if (s_port_big[tp->th_dport & (IPPORT_RESERVED-1)] & IPPROTO_TCP) {
				t.p_port = tp->th_dport;
				t.who_srv = 2;
			}
		if (tp->th_sport = tp->th_dport) t.who_srv = 3;
	}
	if (traf_add(insertentry(&t), t.n_psize))
		tcp_pdata++;

	HTONS(tp->th_sport);
	HTONS(tp->th_dport);
	return;
}

void
traf_udp(up, ip)
	register struct udphdr *up;
	register struct ip *ip;
{
	register u_char *cp = (u_char *)(up + 1);
	struct t_entry t;

	if (cp > snapend || dlen < sizeof(struct udphdr))
		return;

	NTOHS(up->uh_sport);
	NTOHS(up->uh_dport);

	t.in_ip.s_addr = ip->ip_src.s_addr;
	t.out_ip.s_addr = ip->ip_dst.s_addr;
	t.ip_protocol = IPPROTO_UDP;
	t.n_psize = (u_long)iplen;

	if (up->uh_sport < up->uh_dport)
		t.p_port = up->uh_sport, t.who_srv = 1;
	else if (up->uh_sport > up->uh_dport)
		t.p_port = up->uh_dport, t.who_srv = 2;
	else if (up->uh_sport = up->uh_dport)
		t.p_port = up->uh_sport, t.who_srv = 3;
	if (t.p_port > IPPORT_RESERVED) {
		if (s_port_big[up->uh_sport & (IPPORT_RESERVED-1)] & IPPROTO_UDP) {
			t.p_port = up->uh_sport;
			t.who_srv = 1;
		} else if (s_port_big[up->uh_dport & (IPPORT_RESERVED-1)] & IPPROTO_UDP) {
				t.p_port = up->uh_dport;
				t.who_srv = 2;
			}
		if (up->uh_sport = up->uh_dport) t.who_srv = 3;
	}
	if (traf_add(insertentry(&t), t.n_psize))
		udp_pdata++;

	HTONS(up->uh_sport);
	HTONS(up->uh_dport);
	return;
}

void
traf_ip(ip)
	register struct ip *ip;
{
	struct t_entry t;

	t.in_ip.s_addr = ip->ip_src.s_addr;
	t.out_ip.s_addr = ip->ip_dst.s_addr;
	t.ip_protocol = ip->ip_p;
	t.n_psize = (u_long)iplen;
	t.p_port = t.who_srv = 0;
	traf_add(insertentry(&t), t.n_psize);
}

void
processing_ip(ip, length)
	register struct ip *ip;
	register int length;
{
	register int hlen;
	register unsigned char *cp;

	if ((u_char *)(ip + 1) > snapend)
		return;
	if (length < sizeof (struct ip)) {
		move(24, strlen(device_name) + 3);
		printw("truncated-ip: discard %d bytes", length);
		clrtoeol();
		move(0,79);
		refresh();
		return;
	}
	iplen = (int)ntohs(ip->ip_len);
	if (length < iplen) {
		move(24, strlen(device_name) + 3);
		printw("truncated-ip: %d bytes missing", iplen - length);
		clrtoeol();
		move(0,79);
		refresh();
	}
	hlen = ip->ip_hl * 4;
	dlen = iplen - hlen;
	/*
	 * If this is fragment zero, hand it to the next higher level protocol,
	 * else it is fragmented datagram.
	 */
	if (!(ntohs(ip->ip_off) & 0x1fff)) {
		cp = (unsigned char *)ip + hlen;
		switch (ip->ip_p) {

		case IPPROTO_TCP:
			traf_tcp((struct tcphdr *)cp, ip);
			break;
		case IPPROTO_UDP:
			traf_udp((struct udphdr *)cp, ip);
			break;
		case IPPROTO_ICMP:
			print_icmp((struct icmp *)cp, ip);
			break;
		default:
			traf_ip(ip);
			break;
		}
	} else	traf_ip(ip);

	if (alarm_flag) {
		alarm_flag = 0;
		deleteold();
	}
	return;
}

/*
 * Print all ICMP messages on the last line of screen activity.
 */
print_icmp(dp, ip)
	register struct icmp *dp;
	register struct ip *ip;
{
	char buf[256];
	register char *str = buf;
	register struct ip *oip;
	register struct udphdr *ouh;
	register int hlen;
	u_char *ep = (u_char *)snapend;

#define TCHECK(var, l)	if ((u_char *)&(var) > ep - l) goto trunc

	move(SCR_OFFS + n_entry, 0);
	printw("%-28.28s %-28.28s ", addrhash(ip->ip_src), addrhash(ip->ip_dst));
	strcpy(str, "unknown icmp");

	TCHECK(dp->icmp_code, sizeof(dp->icmp_code));
	switch (dp->icmp_type) {
	case ICMP_ECHOREPLY:
		str = "echo reply";
		break;
	case ICMP_UNREACH:
		TCHECK(dp->icmp_ip.ip_dst, sizeof(dp->icmp_ip.ip_dst));
		switch (dp->icmp_code) {
		case ICMP_UNREACH_NET:
			sprintf(buf,"net %s unreach",addrhash(dp->icmp_ip.ip_dst));
			break;
		case ICMP_UNREACH_HOST:
			sprintf(buf,"host %s unreach",addrhash(dp->icmp_ip.ip_dst));
			break;
		case ICMP_UNREACH_PROTOCOL:
			TCHECK(dp->icmp_ip.ip_p, sizeof(dp->icmp_ip.ip_p));
			sprintf(buf,"%s proto %d unreach",
				addrhash(dp->icmp_ip.ip_dst), dp->icmp_ip.ip_p);
			break;
		case ICMP_UNREACH_PORT:
			TCHECK(dp->icmp_ip.ip_p, sizeof(dp->icmp_ip.ip_p));
			oip = &dp->icmp_ip;
			hlen = oip->ip_hl * 4;
			ouh = (struct udphdr *)(((u_char *)oip) + hlen);
			switch (oip->ip_p) {
			case IPPROTO_TCP:
				sprintf(buf,"%s %s/tcp unreach",
					addrhash(oip->ip_dst),
					tcpport_string(ntohs(ouh->uh_dport)));
				break;
			case IPPROTO_UDP:
				sprintf(buf,"%s %s/udp unreach",
					addrhash(oip->ip_dst),
					udpport_string(ntohs(ouh->uh_dport)));
				break;
			default:
				sprintf(buf,"%s %d/%d unreach",
					addrhash(oip->ip_dst),
					ntohs(ouh->uh_dport), oip->ip_p);
				break;
			}
			break;
		case ICMP_UNREACH_NEEDFRAG:
			sprintf(buf, "%s need to frag", addrhash(dp->icmp_ip.ip_dst));
			break;
		case ICMP_UNREACH_SRCFAIL:
			sprintf(buf,"%s src route failed", addrhash(dp->icmp_ip.ip_dst));
			break;
		}
		break;
	case ICMP_SOURCEQUENCH:
		str = "source quench";
		break;
	case ICMP_REDIRECT:
		TCHECK(dp->icmp_ip.ip_dst, sizeof(dp->icmp_ip.ip_dst));
		switch (dp->icmp_code) {
		case ICMP_REDIRECT_NET:
			sprintf(buf,"redir %s to net %s",
				addrhash(dp->icmp_ip.ip_dst),
				addrhash(dp->icmp_gwaddr));
			break;
		case ICMP_REDIRECT_HOST:
			sprintf(buf, "redir %s to host %s",
				addrhash(dp->icmp_ip.ip_dst),
				addrhash(dp->icmp_gwaddr));
			break;
		case ICMP_REDIRECT_TOSNET:
			sprintf(buf, "redir-tos %s to net %s",
				addrhash(dp->icmp_ip.ip_dst),
				addrhash(dp->icmp_gwaddr));
			break;
		case ICMP_REDIRECT_TOSHOST:
			sprintf(buf, "redir-tos %s to host %s",
				addrhash(dp->icmp_ip.ip_dst),
				addrhash(dp->icmp_gwaddr));
			break;
		}
		break;
	case ICMP_ECHO:
		str = "echo request";
		break;
	case ICMP_TIMXCEED:
		TCHECK(dp->icmp_ip.ip_dst, sizeof(dp->icmp_ip.ip_dst));
		switch (dp->icmp_code) {
		case ICMP_TIMXCEED_INTRANS:
			str = "time exceeded in-transit";
			break;
		case ICMP_TIMXCEED_REASS:
			str = "ip reassembly time exceeded";
			break;
		}
		break;
	case ICMP_PARAMPROB:
		if (dp->icmp_code)
			sprintf(buf,"param problem, code %d",dp->icmp_code);
		else {
			TCHECK(dp->icmp_pptr, sizeof(dp->icmp_pptr));
			sprintf(buf,"param problem, octet %d",dp->icmp_pptr);
		}
		break;
	case ICMP_TSTAMP:
		str = "time stamp request";
		break;
	case ICMP_TSTAMPREPLY:
		str = "time stamp reply";
		break;
	case ICMP_IREQ:
		str = "information request";
		break;
	case ICMP_IREQREPLY:
		str = "information reply";
		break;
	case ICMP_MASKREQ:
		str = "address mask request";
		break;
	case ICMP_MASKREPLY:
		TCHECK(dp->icmp_mask, sizeof(dp->icmp_mask));
		sprintf(buf, "address mask is 0x%08x", dp->icmp_mask);
		break;
	}
print:
	alarm(REFRESH);
        addstr(str);
	clrtoeol();
	move(0, 79);
	refresh();
	return;
trunc:
	str = "truncated-icmp";
	goto print;
#undef TCHECK
}
