/*
 * support.c
 * Native language support.
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

#include <assert.h>
#include <stdarg.h>
#include "constants.h"
#include "baseClasses.h"
#include "classMethod.h"
#include "lookup.h"
#include "errors.h"
#include "md.h"

#define	MAXEXCEPTIONLEN		200
#define	CONSTRUCTOR_NAME	"<init>"
#define	ERROR_SIGNATURE		"(Ljava/lang/String;)V"

void classname2pathname(char*, char*);

/*
 * Convert an Java String to a C string.
 */
char*
javaString2CString(stringClass* js, char* cs, int len)
{
	if (len > js->count) {
		len = js->count;
	}
	strncpy(cs, STRING_OBJ2DATA(js->value), len);
	cs[len] = 0;	/* Is this necessary ? */
	return (cs);
}

/*
 * Convert a C string into a Java String.
 */
stringClass*
makeJavaString(char* cs, int len)
{
	return (getString(STRING_DATA2BASE(addString(cs))));
}

/*
 * Call a Java method from native code.
 */
long
do_execute_java_method(void* ee, object* obj, char* method_name, char* signature, methods* mb, bool isStaticCall, ...)
{
	void* func;
	char* sig;
	int args;
	va_list ap;

	if (mb != 0 && mb->ncode != 0) {
		func = mb->ncode;
	}
	else {
		func = findMethod(obj->mtable->class, addStringPair(method_name, signature));
		if (func == 0) {
			throwException(NoSuchMethodError);
		}
	}

	/* Calculate number of arguments */
	sig = signature;
	args = sizeofSig(&sig);

	/* Make the call */
	va_start(ap, isStaticCall);
 	CALL_KAFFE_FUNCTION_VARARGS(func, obj, args, ap);
	va_end(ap);

	return (0);
}

/*
 * Call a Java static method on a class from native code.
 */
long
do_execute_java_class_method(char* cname, char* method_name, char* signature, ...)
{
	void* func;
	char* sig;
	int args;
	va_list ap;
	classes* class;

	/* Convert "." to "/" */
	classname2pathname(cname, cname);

	class = lookupClass(addString(cname));
	assert(class != 0);
	func = findMethod(class, addStringPair(method_name, signature));
	if (func == 0) {
		throwException(NoSuchMethodError);
	}

	/* Calculate number of arguments */
	sig = signature;
	args = sizeofSig(&sig);

	/* Make the call */
	va_start(ap, signature);
 	CALL_KAFFE_FUNCTION_VARARGS(func, 0, args, ap);
	va_end(ap);

	return (0);
}

/*
 * Allocate and object and execute the constructor.
 */
object*
execute_java_constructor(void* ee, char* cname, classes* cc, char* signature, ...)
{
	int args;
	object* obj;
	void* func;
	char* sig;
	va_list ap;
	char buf[MAXEXCEPTIONLEN];
	int i;

	if (cc == 0) {
		/* Convert "." to "/" */
		classname2pathname(cname, buf);

		cc = lookupClass(addString(buf));
		assert(cc != 0);
	}

	func = findMethod(cc, addStringPair(CONSTRUCTOR_NAME, signature));
	if (func == 0) {
		throwException(NoSuchMethodError);
	}

	obj = alloc_object(cc, false);
	assert(obj != 0);

	/* Calculate number of arguments */
	sig = signature;
	args = sizeofSig(&sig);

	/* Make the call */
	va_start(ap, signature);
 	CALL_KAFFE_FUNCTION_VARARGS(func, obj, args, ap);
	va_end(ap);

	return (obj);
}

 
/*
 * Signal an error by creating the object and throwing the exception.
 */
void
SignalError(void* ee, char* cname, char* str)
{
	object* obj;

	obj = execute_java_constructor(ee, cname, 0, ERROR_SIGNATURE, makeJavaString(str, strlen(str)));
	throwException(obj);
}

/*
 * Convert a class name to a path name.
 */
void
classname2pathname(char* from, char* to)
{
	int i;

	/* Convert any '.' in name to '/' */
	for (i = 0; from[i] != 0; i++) {
		if (from[i] == '.') {
			to[i] = '/';
		}
		else {
			to[i] = from[i];
		}
	}
	to[i] = 0;
}
