
/*
 * pwrfaild:	daemon for monitoring a Best Power Technology
 *		``Patriot'' model UPS and APS Back-UPS and PowerChute UPS.
 *
 *    For Best Power Technology ``Patriot'' model UPS's:
 *
 *	The ``inverter normally open''
 *	contact should be wired to an inverter and connected
 *	to DCD of the incoming serial port.  The ``Low Runtime
 *	normally open contact'' should be wired to an inverter
 * 	and connected to the RD line of the serial port.  The 
 * 	TD line of the serial port should be connected to the 
 *	``RS232 Shutdown'' line on the UPS and the signal ground
 *	for the serial line should be wired to the common on 
 *	the UPS.  I use the DTR signal from the serial port to 
 *	provide power for the inverters.
 *
 *   For APC Back-UPS and Smart-UPS UPS's:
 *	($Id: power.c,v 1.5 1996/03/10 01:12:46 wdp Exp $)
 *	(contributed by - Walter Poxon <wdp@NetStar.com>)
 *
 *      This description covers APS Smart-UPS 600RM, 900XLRM, 1250RM, 2000RM
 *	and APS Back-UPS 400, 450, and 600 VA models.
 *	(probably other APS models as well - check the wiring diagram
 *	in your APC manual).
 *
 *	For interfacing with the APS power supplies, you'll need a cable
 *	built as follows:
 *
 *	Pin 2 on the UPS (normally low) should be wired to DCD (CAR) on the
 *	BSD system serial port (pin 1).
 *
 *	Pin 9 on the UPS should be connected to the pin 5 (ground) on the BSD
 *	system.
 *
 *	(the rest of the wires should be cut).
 *
 *	The UPS has a DB-9 female connector		The PC has a DB-9 male
 *
 *	so the cable needs to look like:
 *
 *	DB-9 male						DB-9 female
 *	 UPS							    BSD/OS
 *	=====							    =====
 *	Pin 2 ----- (pwr fail) ------------------------ (DCD) ----- Pin 1
 *	Pin 9 ------ (ground) ----------------------- (ground) ---- Pin 5
 *
 */

/* 
 * Copyright (c) 1993 Jeff Polk
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/fcntl.h>
#include <termios.h>
#include <sys/signal.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <syslog.h>
#include <sys/reboot.h>

#define CLOSE_FDS		10 		/* how many fds to close? */
#define NUMBRKS			20		/* 12.5 == 5sec in theory */

#define PWR_TTY			"/dev/tty00"
#define LOG_PWR			LOG_LOCAL1
#define SHUTDOWN_COMMAND	"/etc/pwrdown"
#define DEFAULT_SHUTDOWN	"/sbin/shutdown"
int fd;

void
lowpower()
{
	int i, pid, status;

	syslog(LOG_EMERG, "*** LOW POWER -- system going down IMMEDIATELY");

	if (access(SHUTDOWN_COMMAND, X_OK) >= 0) {

		syslog(LOG_INFO, "executing `%s'", SHUTDOWN_COMMAND);

		switch (pid = fork()) {
		case -1:
			syslog(LOG_ERR, "fork for `%s' failed: %m",
				SHUTDOWN_COMMAND);
			/* fall thru ... */
		case 0:
			execl(SHUTDOWN_COMMAND, SHUTDOWN_COMMAND, NULL);
			syslog(LOG_ERR, "error running %s: %m - trying %s",
				SHUTDOWN_COMMAND, DEFAULT_SHUTDOWN);
			execl(DEFAULT_SHUTDOWN, DEFAULT_SHUTDOWN, "-h",
			      "now", NULL);
			syslog(LOG_ERR, "error running %s: %m",
			       DEFAULT_SHUTDOWN);
			break;
		default:
			syslog(LOG_INFO, "waiting for `%s' to finish",
				SHUTDOWN_COMMAND);
			waitpid(pid, &status, 0);
			syslog(LOG_INFO, "`%s' returned status %d", 
				SHUTDOWN_COMMAND, status);
		}
	}

	syslog(LOG_INFO, "scheduling shutdown of UPS");

	for (i=0; i<NUMBRKS; i++)
		tcsendbreak(fd, 0);
	sleep(5);	/* wait for most of the breaks to be sent */

	syslog(LOG_INFO, "halting local system");
	reboot(RB_HALT);
}

void 
sighandler(int sig)
{
	switch (sig) {
	case SIGHUP:
		syslog(LOG_INFO, "received SIGHUP");
		break;
	case SIGINT:
		syslog(LOG_INFO, "received SIGINT");
		lowpower();
		break;
	default:
		syslog(LOG_ERR, "received unknown signal (%d)", sig);
		exit(1);
		break;
	}
	return;
}

void
usage()
{
	fprintf(stderr, "Usage: pwrfaild [-D] [-d] [-f port] [-t timeout]\n");
	exit(1);
}

main(int argc, char *argv[])
{
	int i, n, ch;
	int debug=0;
	int detach=1;
	int timeout=30;
	char *tty=PWR_TTY;
	struct sigaction act;
	struct termios tparams;
	struct itimerval itv;
	int port_settings = 0;

	while ((ch=getopt(argc, argv, "Ddf:t:")) != EOF)
		switch(ch) {
		case 'D':
			detach = 0; /* for running under a debugger */
			break;
		case 'd':
			debug = 1;
			break;
		case 'f':
			tty = optarg;
			break;
		case 't':
			timeout = atoi(optarg);
			break;
		case '?':
		default:
			usage();
		}

	if ((fd = open(tty, O_RDWR|O_NONBLOCK)) < 0) {
		perror("pwrfaild: open");
		exit(1);
	}

	if (tcgetattr(fd, &tparams) < 0) {
		perror("pwrfaild: tcgetattr");
		exit(1);
	}

	tparams.c_cflag &= ~CLOCAL;
	tparams.c_iflag &= ~IGNBRK;
	tparams.c_iflag |= BRKINT;

	if (tcsetattr(fd, TCSAFLUSH, &tparams) < 0) {
		perror("pwrfaild: tcsetattr");
		exit(1);
	}

	for (i=0; i<CLOSE_FDS; i++)
		close(i);

	act.sa_handler = sighandler;
	act.sa_mask = SIGHUP|SIGINT;
	act.sa_flags = 0;
	sigaction(SIGHUP, &act, NULL);
	sigaction(SIGINT, &act, NULL);

	for (;;) {

		if (detach) {
			if ((n = fork()) < 0) {
				syslog(LOG_PWR|LOG_ERR, "pwrfaild: fork failed: %m");
				exit(1);
			}
			if (n)
				exit(0);
		}

		openlog("pwrfaild", LOG_CONS|LOG_PID, LOG_PWR);

		if (!debug)
			setlogmask(LOG_UPTO(LOG_ERR));

		if (detach) {
			if (setsid() < 0) {
				syslog(LOG_ERR, "setsid() failed: %m");
                		exit(1);
			}
		}

loop:
		if ((fd = open(tty, O_RDWR)) < 0) {
			syslog(LOG_ERR, "open failed?!?: %m");
			exit(1);
		}

		syslog(LOG_EMERG, "*** POWER HAS FAILED! ***");

		if (detach) {
			if (ioctl(fd, TIOCSCTTY, (char *)NULL) < 0) {
				syslog(LOG_ERR, "TIOCSCTTY failed: %m");
                		exit(1);
			}
		}

		syslog(LOG_INFO, "shutdown will occur in %d seconds",
		       timeout);

		for (i = 0; i < timeout; i++) {

			port_settings = 0;

			if (ioctl(fd, TIOCMGET, &port_settings) < 0) {
				syslog(LOG_INFO,
				       "unable to get %s settings", tty);
				/*
				 * assume the worst and shut down now
				 */
				lowpower();
				exit(1);
			}

			if (port_settings & TIOCM_CAR) {
				/*
				 * low power signal still present
				 */
				sleep(1);
				continue;
			} else {

				/*
				 * main power restored
				 */
				syslog(LOG_EMERG,
				       "*** Power has been restored ***");

				close(fd);
				goto loop;
			}
		}
		
		/*
		 * power failed and didn't come back!
		 */
		lowpower();
	}
}
