/*
 * Berkeley specific routines for manipulating the TTY
 */

#include <stdio.h>
#include <termios.h>
#include <fcntl.h>
#include "dial_dir.h"
#include "modem.h"
#include "param.h"

static struct termios hold;

/*
 * Change the communication line settings to the new values.
 */

void
line_set()
{
	static int first = 1;
	extern int fd;
 	struct termios tbuf;
 	struct termios traw;
	unsigned int baud;
					/* nothing to do! */
	if (fd == -1)
		return;

	if (first) {
		tcgetattr(fd, &hold);
		first = 0;
	}
	set_bsd_local(fd);
					/* get the current settings */
	tcgetattr(fd, &tbuf);
					/* set some beginning values */
	tty_cbreak(fd);
	tcgetattr(fd, &traw);
	tcsetattr(fd, TCSADRAIN, &traw);

	if (*param->flow_ctrl == 'X')
		tbuf.c_iflag |= IXON;
	/*
	 * If the DTE speed is locked, then ignore all request to change
	 * the speed.
	 */
	baud = modem->lock_sp[modem->t_cur];
	if (baud == 0)
		baud = dir->baud[0];
					/* the baud rate */
	switch (baud) {
		case 1200:
			tbuf.c_ispeed = B1200;
			tbuf.c_ospeed = B1200;
			break;
		case 2400:
			tbuf.c_ispeed = B2400;
			tbuf.c_ospeed = B2400;
			break;
		case 4800:
			tbuf.c_ispeed = B4800;
			tbuf.c_ospeed = B4800;
			break;
		case 9600:
			tbuf.c_ispeed = B9600;
			tbuf.c_ospeed = B9600;
			break;
		case 19200:
#ifdef B19200
			tbuf.c_ispeed = B19200;
			tbuf.c_ospeed = B19200;
#else /* B19200 */
#ifdef EXTA
			tbuf.c_ispeed = EXTA;
			tbuf.c_ospeed = EXTA;
#endif /* EXTA */
#endif /* B19200 */
			break;
		case 38400:
#ifdef B38400
			tbuf.c_ispeed = B38400;
			tbuf.c_ospeed = B38400;
#else /* B38400 */
#ifdef EXTB
			tbuf.c_ispeed = EXTB;
			tbuf.c_ospeed = EXTB;
#endif /* EXTB */
#endif /* B38400 */
			break;
		case 57600:
			tbuf.c_ispeed = B57600;
			tbuf.c_ospeed = B57600;
			break;
	}
					/* the parity */
	switch (dir->parity[0]) {
		case 'N':
			tbuf.c_cflag &= ~PARENB;
			break;
		case 'O':
			tbuf.c_cflag |= PARODD;
			break;
		case 'E':
			tbuf.c_cflag &= ~PARODD;
			break;
	}
					/* now set 'em! */
	tbuf.c_cflag |= HUPCL;
	tcsetattr(fd, TCSADRAIN, &tbuf);
	return;
}

/*
 * Put things back the way they were.
 */

void
reset_line()
{
	extern int fd;

	tcsetattr(fd, TCSADRAIN, &hold);
 	set_bsd_nolocal(fd);
	return;
}

/*
 * Put the stdin/stdout in terminal mode.  We've divided up the
 * responsibility for the line settings options between the serial port
 * and the stdin and stdout.
 */

void
term_mode()
{
	struct termios tbuf;

	tty_raw(0);
	tcgetattr(0, &tbuf);

	tbuf.c_oflag &= ~(ONLCR|ECHO);
	tbuf.c_lflag &= ~(ECHO);

	if (dir->duplex[0] == 'H')
		tbuf.c_lflag |= ECHO;

	tcsetattr(0, TCSADRAIN, &tbuf);
	return;
}

/*
 * Put the TTY driver in the mode suitable for xmodem transfers.
 */

void
xmodem_mode(fds)
int fds;
{
	struct termios tbuf;

	tty_raw(fds);
	tcgetattr(fds, &tbuf);

#ifdef OLD
	/*
	 * Turn off the XON/XOFF flow control, turn off echoing, and
	 * switch to 8 bit no parity.
	 */
	tbuf.c_cflag |= (RAW|ANYP);
#endif
	tbuf.c_lflag &= ~ECHO;
	tcsetattr(fds, TCSADRAIN, &tbuf);

	return;
}

/*
 * Put the TTY line in a mode suitable for the ASCII transfer.
 */

void
ascii_mode(up)
int up;
{
	extern int fd;
	struct termios tbuf;

	tty_cbreak(fd);
	tcgetattr(fd, &tbuf);
	tbuf.c_iflag |= IXON;
	tbuf.c_lflag &= ~(ECHO);

#ifdef OLD
	tbuf.c_cflag |= (CBREAK|TANDEM);
	tbuf.c_cflag &= ~(RAW|CRMOD|ECHO|CRDELAY);

	if (up) {
					/* CR delay times */
		switch (param->cr_delay) {
			case 0:
				break;
			case 100:
				tbuf.c_cflag |= CR1;
				break;
			case 150:
				tbuf.c_cflag |= CR2;
				break;
		}
	}

#endif
	tcsetattr(fd, TCSADRAIN, &tbuf);
	return;
}

/*
 * Flush the file descriptor.  Very messy... flushing the input causes a
 * wait for the output to drain, and there is no output flushing.
 */

int
tty_flush(fds, mode)
int fds, mode;
{
	int ret_code = 0;
	struct termios tbuf;

	switch(mode) {
		case 0:			/* flush input queue */
			tcgetattr(fds, &tbuf);
			tcsetattr(fds, TCSADRAIN, &tbuf);
			break;
		case 1:			/* flush output queue */
			/* sorry! */
			break;
		case 2:			/* flush both input and output */
			tcflush(fds, TIOCFLUSH);
			break;
		default:
			ret_code++;
			break;
	}
	return(ret_code);
}

/*
 * Wait for the output to drain
 */

int
tty_drain(fds)
int fds;
{
	struct termios tbuf;
					/* this flushes the input too */
	tcgetattr(fds, &tbuf);
/*	tcflush(fds, &tbuf);*/
	return(tcsetattr(fds, TCSADRAIN, &tbuf));
}

/*
 * Send a modem break
 */

int
tty_break(fds)
int fds;
{
	unsigned int sleep();

	return (tcsendbreak(fds,0));
#ifdef OLD
	ioctl(fds, TIOCSBRK, (struct termios *) 0);
	return(ioctl(fds, TIOCCBRK, (struct termios *) 0));
#endif
}

/*
 * Fix the file descriptor so that a read is satisfied immediately.  When
 * read() is called it returns the character in the queue, or an error if
 * no key was pressed.
 */

int
tty_noblock(fds, on)
int fds, on;
{
	int current;

	current = fcntl(fds, F_GETFL, 0);
	if (on)
		return(fcntl(fds, F_SETFL, current | FNDELAY));
	else
		return(fcntl(fds, F_SETFL, current & ~FNDELAY));
}

/*
 * Get the current baud rate of the terminal
 */

int
my_speed()
{
	struct termios tbuf;

	tcgetattr(0, &tbuf);
	return(cfgetispeed(&tbuf));
}

/*
 * Restart any XON/XOFF flow control that may have stopped the tty 
 */

void
tty_restart()
{
	extern int fd;

	if (fd != -1 && *param->flow_ctrl == 'X')
		tcflush(fd, TIOCFLUSH);
	return;
}
