/**********************************************************************
 * 
 * FILENAME:          xoj.c
 *
 * COPYRIGHT: Copyright 1994.  J. Eric Bracken <bracken@performance.com>
 * ALL RIGHTS RESERVED.  
 *
 * THIS SOFTWARE MAY BE FREELY DISTRIBUTED SO LONG AS IT RETAINS THE
 * AUTHOR'S NAME AND COPYRIGHT NOTICE LISTED ABOVE.
 *
 * THIS SOFTWARE IS PROVIDED AS IS--NO WARRANTEE OF FITNESS FOR ANY
 * PURPOSE IS IMPLIED.  THIS SOFTWARE IS DISTRIBUTED IN THE FORM OF
 * SOURCE CODE.  BY COMPILING AND EXECUTING THIS SOFTWARE, YOU TAKE 
 * RESPONSIBILITY FOR ANY DAMAGES, CONSEQUENTIAL OR INCIDENTAL,
 * THAT MIGHT BE CAUSED.
 *
 * DESCRIPTION:       OJ Simpson simulator
 *
 * NOTES:             Has one command line switch:  -display, which
 *                    can be used to set the X display.
 *
 * CREDITS:
 *                    This code is derived from the "xsnow" program
 *                    written by Rick Jansen <rick@sara.nl>.
 *
 * NECESSARY FILES:   
 *
 * REVISION HISTORY:
 * $Log: xoj.c,v $
 *   Revision 0.0
 *   20 June 1994     J. Eric Bracken <bracken@performance.com>
 *                    Created
 *
 **********************************************************************/

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>

typedef unsigned long Pixel;
typedef int ErrorHandler();

 
#include "vroot.h"  /* Use this if vroot.h installed in current directory */
 
#include <stdio.h>
#include <math.h>
#ifndef bsdi
#include <malloc.h>
#endif
#include <signal.h>
#include <limits.h>
#ifdef bsdi
#include <unistd.h>
#endif

#include "ojmap.h"



Display *display;
int screen;
Window rootWin;
int display_width, display_height;
int center_x, center_y;
GC gc;
char *display_name = NULL;
Pixel black, white;

int done = 0;
int eventBlock = 0;
int errorVal = 0;

unsigned int borderWidth = 0;

Region SubR;
Region WindowTops = NULL;
int ClearX, ClearY;

char *snowColor = "snow";
char *slcColor = "black";
char *trColor = "chartreuse";

char *blackColor = "black";
char *redColor = "red";
char *whiteColor = "white";
char *bgColor = "none";
char *greenColor = "chartreuse"; 

Pixel redPix;
Pixel whitePix;
Pixel greenPix;
Pixel blackPix;
Pixel snowcPix;
Pixel bgPix;
Pixel trPix;
Pixel slcPix;

#ifndef bsdi
int theDelay = 25000; /* microseconds */
#else
u_int theDelay = 25000; /* microseconds */
#endif
int BroncoSpeed = 2;
Bronco theBronco;

int n_cops = 6;
Cop *copcars;


/* Forward declarations */
void SigHandler();
void InitBronco();
void DrawBronco();
void EraseBronco();
void InitCop();
void UpdateBronco();
void UpdateCop();
void DrawCop();
void EraseCop();
#ifndef bsdi
void usleep();
#endif
Pixel AllocNamedColor();

void
main(ac, av)
int ac;
char *av[];
{
    XGCValues xgcv;
    int ax;
    char *arg;
    CopMap *cp;
    CopMap *cwp;
    CopMap *lp;
    BroncoMap *bp;
    BroncoMap *bwp;
    XEvent ev;
    int needCalc;
    int i; 

    /* Seed random */
    srand((int)time((long *)NULL));
    
    /*
       Catch some signals so we can erase any visible snowflakes.
    */
    signal(SIGKILL, SigHandler);
    signal(SIGINT, SigHandler);
    signal(SIGTERM, SigHandler);
    signal(SIGHUP, SigHandler);

    /*
       Process command line options.
    */
    for (ax=1; ax<ac; ax++){
      arg = av[ax];
      if (strcmp(arg, "-display") == 0){
	display_name = av[++ax];
      }
    }

    /* Open X */
    display = XOpenDisplay(display_name);
    if (display == NULL) {
	if (display_name == NULL) display_name = (char *)getenv("DISPLAY");
	(void) fprintf(stderr, "%s: cannot connect to X server %s\n", av[0],
	    display_name ? display_name : "(default)");
	exit(1);
    }

    screen = DefaultScreen(display);
    rootWin = RootWindow(display, screen);
    black = BlackPixel(display, screen);
    white = WhitePixel(display, screen);

    display_width = DisplayWidth(display, screen);
    display_height = DisplayHeight(display, screen);

    center_x = display_width / 2;
    center_y = display_height / 2;


    /*
        P I X M A P S 
    */


    /* Create the cop car pixmap */
    cp = &copPix;
    cp->pixmap = XCreateBitmapFromData( display, rootWin, cp->copBits,
				       cp->width, cp->height );
    
    cwp = &copwhitePix;
    cwp->pixmap = XCreateBitmapFromData( display, rootWin, cwp->copBits,
				       cwp->width, cwp->height );

    lp = &lightPix;
    lp->pixmap = XCreateBitmapFromData( display, rootWin, lp->copBits,
				       lp->width, lp->height );

    /* Allocate structures for cop cars */
    copcars = (Cop *)malloc( sizeof(Cop) * n_cops );
	       
    /* Pixmap for bronco */
    bp = &broncoPix;
    bp->pixmap = XCreateBitmapFromData( display, rootWin, bp->broncoBits,
				       bp->width, bp->height );
    
    bwp = &broncoblackPix;
    bwp->pixmap = XCreateBitmapFromData( display, rootWin, bwp->broncoBits,
				       bwp->width, bwp->height );


    /* Allocate colors just once */
    redPix =   AllocNamedColor(redColor, black);
    whitePix = AllocNamedColor(whiteColor, black);
    greenPix = AllocNamedColor(greenColor, black);
    blackPix = AllocNamedColor(blackColor, black);
    snowcPix = AllocNamedColor(snowColor, white);   
    trPix = AllocNamedColor(trColor, black);
    slcPix = AllocNamedColor(slcColor, black); 

    gc = XCreateGC(display, rootWin, 0L, &xgcv);
    XSetForeground(display, gc, blackPix);
    XSetFillStyle(display, gc, FillStippled);

    /* Set the background color, if specified */
    if(strcmp(bgColor,"none") != 0) {
      bgPix = AllocNamedColor(bgColor, white);

      XSetWindowBackground(display, rootWin, bgPix);
      XClearWindow(display, rootWin);
      XFlush(display);
    }
    
    
    /* Initialize all cop cars */
    for (i=0; i< n_cops; i++) InitCop(i);

    InitBronco();   

    /* Notify if part of the root window changed */
    XSelectInput(display, rootWin, ExposureMask | SubstructureNotifyMask);

    needCalc = 0;
/*    if (!NoKeepSnow) needCalc = CalcWindowTops(); */


    /*
     *  M A I N   L O O P
     */
    while (!done) {

/*      SnowTicks--;   
      if (SnowTicks == 0) SnowTicks = ULONG_MAX; */

      /* X event ? */
      /* Handle all the expose events and redo CalcWindowTops after */
      while (XPending(display)) {
        XNextEvent(display, &ev);
       
      }  /* while Xpending */

      /* If things have changed while we were away... */
/*      if (needCalc) needCalc = CalcWindowTops(); */


      /* 
       *  Update     
       */

      for (i=0; i< n_cops; i++) UpdateCop(i);

      /* The bronco */
      XSetForeground(display, gc, slcPix);
      UpdateBronco();

      /* Sleep a bit */
      usleep(theDelay);
    }
    
    XClearWindow(display, rootWin);

    XCloseDisplay(display);

    exit(0);
}
  

/* Initialize the i'th cop car */
void InitCop( i )
int i;
{
  copcars[i].x = 0;
  copcars[i].y = RandInt( display_height / 2 / n_cops - 32 ) + 
                     i * (display_height/ 2 / n_cops ) + 40;

  copcars[i].xStep = BroncoSpeed;
  copcars[i].yStep = 0;

  /* Color of flashing light */ 
  copcars[i].color = i % 2;
}


/* Update position of cop i */
void UpdateCop( i )
int i;
{
  EraseCop(i);

  /* Move forward */
  copcars[i].x = copcars[i].x + copcars[i].xStep;
 
  /* Cop cars stay at the same height */

  if (copcars[i].x >= display_width) InitCop(i); 

  copcars[i].color = 1 - copcars[i].color;

  DrawCop(i);
}  


/* Redraw cop number i */
void
DrawCop( i )
int i;
{
  XSetForeground( display, gc, blackPix );
  XSetStipple(display, gc, copPix.pixmap);
  XSetTSOrigin(display, gc, copcars[i].x, copcars[i].y);
  XFillRectangle(display, rootWin, gc,
         copcars[i].x, copcars[i].y,
         copPix.width, copPix.height );

  XSetForeground( display, gc, whitePix );
  XSetStipple(display, gc, copwhitePix.pixmap);
  XSetTSOrigin(display, gc, copcars[i].x, copcars[i].y);
  XFillRectangle(display, rootWin, gc,
         copcars[i].x, copcars[i].y,
         copwhitePix.width, copwhitePix.height );

  /* Flash the lights red & white, alternatingly */
  if ( copcars[i].color == 0 ){
    XSetForeground( display, gc, redPix );
  }
  else{
    XSetForeground( display, gc, whitePix );
  }

  XSetStipple(display, gc, lightPix.pixmap);
  XSetTSOrigin(display, gc, copcars[i].x, copcars[i].y);
  XFillRectangle(display, rootWin, gc,
         copcars[i].x, copcars[i].y,
         lightPix.width, lightPix.height );


}


/* Erase cop i */
void
EraseCop( i )
{
  if (copcars[i].x >= 0) 
      XClearArea(display, rootWin,
                 copcars[i].x , copcars[i].y,     
                 copPix.width, copPix.height, False);
}



/*
   Give birth to a Bronco. (What a conception)
*/
void
InitBronco()      
{
  theBronco.x = 100;
  theBronco.y = RandInt(display_height / 3)+80;
  theBronco.xStep = BroncoSpeed;
  theBronco.yStep = 1;
}



/*
  Update Bronco
*/
void
UpdateBronco()
{
  EraseBronco();

  /* Move forward */
  theBronco.x = theBronco.x + theBronco.xStep;
 
  /* Move down */
  if (RandInt(10) > 3) theBronco.y = theBronco.y + theBronco.yStep; 
  if (theBronco.y < 0) theBronco.y = 0;
  if (RandInt(100) > 80) theBronco.yStep = -theBronco.yStep;

  if (theBronco.x >= display_width) theBronco.x = 0;

  DrawBronco();
}




/*
  Draw Bronco
*/
void
DrawBronco()
{
  XSetForeground( display, gc, whitePix );
  XSetStipple(display, gc, broncoPix.pixmap);
  XSetTSOrigin(display, gc, theBronco.x, theBronco.y);
  XFillRectangle(display, rootWin, gc,
         theBronco.x, theBronco.y,
         broncoPix.width, broncoPix.height );

  XSetForeground( display, gc, blackPix );
  XSetStipple(display, gc, broncoblackPix.pixmap);
  XSetTSOrigin(display, gc, theBronco.x, theBronco.y);
  XFillRectangle(display, rootWin, gc,
         theBronco.x, theBronco.y,
         broncoblackPix.width, broncoblackPix.height );

}

/*
  Erase Bronco
*/
void
EraseBronco()
{
  if (theBronco.x >= 0) 
      XClearArea(display, rootWin,
                 theBronco.x , theBronco.y,     
                 broncoPix.width, broncoPix.height, False);
}



void
SigHandler()
{
  done = 1;
}


/*
   Generate random integer between 0 and maxVal-1.
*/
int
RandInt(maxVal)
int maxVal;
{
	return rand() % maxVal;
}


#ifndef bsdi
/*
 * sleep for a number of micro-seconds
 */
void usleep(usec) 
unsigned long usec;
{
#ifdef SVR3
    poll((struct poll *)0, (size_t)0, usec/1000);   /* ms resolution */
#else
    struct timeval t;
    t.tv_usec = usec%(unsigned long)1000000;
    t.tv_sec = usec/(unsigned long)1000000;
    select(0, (void *)0, (void *)0, (void *)0, &t);
#endif
}
#endif




/*
   Allocate a color by name.
*/
Pixel
AllocNamedColor(colorName, dfltPix)
char *colorName;
Pixel dfltPix;
{
	Pixel pix;
	XColor scrncolor;
	XColor exactcolor;

	if (XAllocNamedColor(display, DefaultColormap(display, screen),
		colorName, &scrncolor, &exactcolor)) {
		pix = scrncolor.pixel;
	}
	else {
		pix = dfltPix;
	}

	return pix;
}
