/*************************************************************************
 * PROGRAM:
 *  xgetftp
 *
 * USAGE:
 *  xgetftp [-q] [hostname]
 *
 * AUTHOR:
 *  Salim Alam
 *  University of Colorado, Boulder
 *
 * MODIFICATION LOG:
 *  93.04.19 S.A. - Added new open and cache info stuff
 *  93.04.01 S.A. - Allow Prospero-based archie server
 *  93.03.17 S.A. - Bug fix in get_view_data and get_archie_data
 *  93.02.15 S.A. - Mods for new ctrl_delete_cache
 *  93.02.03 S.A. - Better error-handling: ctrl_start_session,ctrl_down_dir
 *  93.01.26 S.A. - Now gets local hostname correctly
 *
 **************************************************************************/

#include <X11/Xos.h>
#include <Xm/Xm.h>
#include <Xm/PushBG.h>
#include <Xm/Text.h>
#include <Xm/List.h>
#include <sys/stat.h>
#include <stdio.h>
#include <signal.h>
#include <pwd.h>
#include <sys/socket.h>
#include <netdb.h>
#include "prefs.h"
#include "control.h"
#include "ui_funcs.h"
#include "config.h"
#include "patchlevel.h"

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif


/*
 * Imported vars
 */
extern int x_data_done;
extern FILE *response_stream;

/*
 * Global variables
 */
#define INBUFSIZE	1024		/* size of data input buffer	*/
#define MAX_SELECTIONS 	100		/* max# of user selections	*/


XmString * list_items = NULL;		/* list of items in curr. dir. 	*/
int        item_count = 0;		/* count of items in curr dir.  */

int	   curr_sel_cnt = 0;		/* count of selected items	*/
int 	   curr_selections[MAX_SELECTIONS]; /* items selected by user   */

Prefs 	   user_prefs;			/* User preferences		*/

int 	   fConnected = FALSE;		/* Flag to indicate server conn */

char 	   retrieve_dir[255];		/* Directory for file retrieval */

char	   apassword[100];		/* Anonymous passwd=User@hostname */

void set_currdir(void);


/************************** directory list functions *********************/

void alloc_list_mem(int x)
/* 
 * allocates memory for the list
 */
{
    void free_list_mem(XmString *);

    if (list_items)
	free_list_mem(list_items);

    list_items = (XmString *) XtMalloc(x * sizeof(XmString));
}


void add_list_item(char *str)
/*
 * adds an item to the list
 */
{
    list_items[item_count++] = XmStringCreateLtoR(str,XmSTRING_DEFAULT_CHARSET);
}


void free_list_mem(XmString *list)
/*
 * free all the items in the list
 */
{
    int i;

    for (i=0; i < item_count; i++)
	XmStringFree(list_items[i]);

    item_count = 0;
}

/************************* user preference callback ********************/

void userprefCB(Widget w, int item, XmToggleButtonCallbackStruct *cb)
{
    switch (item)
    {
      case prefCACHE_VIEW:
	user_prefs.cache_view = cb->set;
	break;

      case prefREUSE_VIEW:
	user_prefs.reuse_view = cb->set;
	break;

      case prefREUSE_DIR:
	user_prefs.reuse_dir = cb->set;

      case prefAUTOSAVE_DIR:
	user_prefs.autosave_dir = cb->set;
	break;

      case prefCONFIRM:
	user_prefs.confirm = cb->set;
	break;
      
      default:
	fprintf(stderr, "pref_cb: unknown item '%d'\n", item);
    }
}


/************************* retreive directory callback *****************/

void setretrievedirCB(Widget w, caddr_t data, XmSelectionBoxCallbackStruct *cb)
{
    XmStringContext context;
    char *text;
    XmStringCharSet charset;
    XmStringDirection dir;
    Boolean seperator;
    int len;
    struct stat stat_buf;

    XmStringInitContext(&context, cb->value);

    /* look only at first segment */
    if (XmStringGetNextSegment(context, &text, &charset, &dir,
	&seperator))
    {
	len = strlen(text);
	if (text[len-1] == '/')
	    text[len-1] = '\0';

	if (stat(text, &stat_buf) == 0)    
	{
	    if (!S_ISDIR(stat_buf.st_mode))
	    {
		fprintf(stderr, "'%s' not a valid directory.\n", text);
		ui_beep(0);
	    }
	    else
	    {
		strcpy(retrieve_dir, text);
#ifdef DEBUG
		fprintf(stderr, "New retreive dir: '%s'\n", retrieve_dir);
#endif
	    }
	}
	else
	{
	    fprintf(stderr, "'%s' not a valid directory.\n", text);
	    ui_beep(0);
	}
    }

    XmStringFreeContext(context);
}


/************************* archie widget function **********************/

/*
 * archie data callback
 */
void get_archie_data(int *ppos, int *fid, XtInputId *id)
{
    unsigned char buf[INBUFSIZE+1];
    int nbytes;

    nbytes = read(*fid, buf, INBUFSIZE);

    if (nbytes)
    {
	int i, len;

	/*
	 * add text in buffer to text widget, and clear any
	 * unwritable characters (just in case this is a binary
	 * file.  also get rid of "\r" characters, just in case
	 * we got this thru an ASCII connection.
	 *
	 */
	for (i=0, len=0; i < nbytes; i++)
	{
	    if ( (isprint(buf[i]) || isspace(buf[i])) && (buf[i]!='\r') )
		buf[len++] = buf[i];
	}
	buf[len] = '\0';
	XmTextInsert(archietextW, (XmTextPosition)*ppos, (char *)buf);
	*ppos += len;
    }
    else
    {
	char filename[50];

	XtRemoveInput(*id);
	close (*fid);
	ui_set_cursor_normal(toplevelW);
   	sprintf(filename, "/tmp/xgetftpARCHIE%ld", getpid());
   	unlink(filename);
    }
}


/*
 * archie search widget callback
 */
void archiefindCB(Widget w, caddr_t call_data, XmPushButtonCallbackStruct *cb)
{
   char *find_str;
   static int pos;
   FILE *fp;
   char dummy[256];
   char filename[50];

   ui_set_cursor_busy(archiebbW);

   find_str = (char *) XmTextGetString(archiefindtextW);
#ifdef DEBUG
   fprintf(stderr, "Archie find: '%s'\n", find_str);
#endif

   sprintf(filename, "/tmp/xgetftpARCHIE%ld", getpid());

   sprintf(dummy, "%s %s %s %s > %s", ARCHIEBIN, ARCHIESRVRARG, ARCHIESERVER,
	find_str, filename);
   system(dummy);
   XtFree(find_str);

   ui_set_cursor_normal(archiebbW);

   if ((fp = fopen(filename, "r")) == NULL)
   {
	fprintf(stderr, "Can't open file '%s' !\n", filename);
	return;
   }

   /* XtUnmanageChild(archietextW); */
   XmTextSetString(archietextW, " ");

   pos = 0;

   ui_set_cursor_busy(toplevelW);
   XtAppAddInput(app_context, fileno(fp), (XtPointer) XtInputReadMask,
	(XtInputCallbackProc) get_archie_data, (XtPointer) &pos);
}


/******************************* menu callback *************************/

/*
 * menu button callback
 */
void MenuButtonCB(Widget w, int menuitm, caddr_t call_data)
{
    switch (menuitm)
    {
      case menuFILE_ABOUT:
	XtManageChild(aboutdialogW);
	break;

      case menuFILE_SET_PREFS:
	XtManageChild(prefdialogW);
	break;

      case menuFILE_EXIT:
	if (fConnected && user_prefs.autosave_dir)
	    ctrl_save_cache();
	exit(1);
	break;



      case menuFTP_OPEN:
	XtManageChild(opendlgW);
	break;

      case menuFTP_CLOSE:
	if (fConnected)
	{
	    if (user_prefs.autosave_dir)
		ctrl_save_cache();
	    ctrl_logout();
	    fConnected = FALSE;
	    curr_sel_cnt = 0;
	    XmListDeleteAllItems(filelistW);
	    ui_set_status_line(" ");
	    ui_set_currdir(" ");
	}
	break;

      case menuFTP_SET_DIR:
	XtManageChild(retrievedirdlgW);
	break;



      case menuCACHE_PURGE_DIR:
	if (fConnected)
	{
	    curr_sel_cnt = 0;
	    XmListDeleteAllItems(filelistW);
	    if (!ctrl_delete_cache())
	    {
	    	ctrl_logout();
	    	fConnected = FALSE;
	        ui_set_status_line("Unrecoverable Error! Disconnecting.");
	    	ui_set_currdir(" ");
	    }
	    else
	    {
	    	XmListAddItems(filelistW, list_items, item_count, 0);
	    	set_currdir();
	    	ui_set_status_line("Directory cache deleted. Moving to root.");
	    }
	}
	else
	    ui_beep(0);
	break;

      case menuCACHE_PURGE_VIEW:
	if (fConnected)
	{
	    ctrl_delete_file_cache();
	    ui_set_status_line("Viewed file cache deleted.");
	}
	else
	    ui_beep(0);
	break;

      case menuCACHE_SAVE_DIR:
	if (fConnected)
	{
	    ctrl_save_cache();
	    ui_set_status_line("Directory cache saved.");
	}
	else
	    ui_beep(0);
	break;



      case menuARCHIE_SEARCH:
	XtManageChild(archiebbW);
	break;

      default:
	fprintf(stderr, "Unknown menu item: %d\n", menuitm);
	break;
    }
}


/************************** file list callbacks ************************/

void doubleclickCB(Widget w, caddr_t client_data, XmListCallbackStruct *cb)
{
    if (!x_data_done)
    {
	ui_beep(0);
	fprintf(stderr,"Data connection busy\n");
    };

    ui_set_cursor_busy(toplevelW);

    if (ctrl_down_dir(cb->item_position-1)==TRUE)
    /* successful traversal */
    {
	XmListDeleteAllItems(w);
	XmListAddItems(w, list_items, item_count, 0);
	curr_sel_cnt = 0;
	set_currdir();
    }
    else
    /* was not a directory, or we had an error! */
	ui_beep(0);

    ui_set_cursor_normal(toplevelW);
}


void selectionCB(Widget w, caddr_t client_data, XmListCallbackStruct *cb)
{
    int i;

#ifdef DEBUG
    printf("Selected items: ");
    for (i=0; i < cb->selected_item_count; i++)
	printf("%d ", cb->selected_item_positions[i]);
    printf("\n");
#endif

    curr_sel_cnt = cb->selected_item_count;
    for (i=0; i < cb->selected_item_count; i++)
	curr_selections[i] = cb->selected_item_positions[i];
}


/************************** command callbacks **************************/

void cd_up(void)
{
    if (!fConnected) return;

    if (ctrl_up_dir()==TRUE)
    /* successfully went up */
    {
	XmListDeleteAllItems(filelistW);
	XmListAddItems(filelistW, list_items, item_count, 0);
	curr_sel_cnt = 0;
	set_currdir();
    }
    else
    /* was already at root */
    {
	ui_beep(0);
    }
}


/*
 * retrieve callback
 */
void retrieve_item(void)
{
    int i;

    if (!fConnected) return;

    if (!x_data_done)
    {
	ui_beep(0);
	fprintf(stderr,"Data connection busy\n");
	return;
    }

    ui_set_cursor_busy(toplevelW);

    for (i=0; i < curr_sel_cnt; i++)
    {
	char *name;
	char line[100];

#ifdef DEBUG
	printf("Retrieving file index %d into %s\n", curr_selections[i]-1,
	    retrieve_dir);
#endif
	name = ctrl_get_item_name(curr_selections[i]-1);
	sprintf(line, "Getting %s", name);
	ui_set_status_line(line);
	free(name);

	if (!ctrl_get_selection(retrieve_dir, curr_selections[i]-1, NULL))
	    fprintf(stderr, "Retrieve FAILED!\n");

	ui_set_status_line("Selection retrieved.");
    }

    ui_set_cursor_normal(toplevelW);
}


/*
 * view callback
 */
void get_view_data(int *ppos, int *fid, XtInputId *id)
{
    unsigned char buf[INBUFSIZE+1];
    int nbytes, len;

    nbytes = read(*fid, buf, INBUFSIZE);

    if (nbytes)
    {
	int i;

	/*
	 * add text in buffer to text widget, and clear any
	 * unwritable characters (just in case this is a binary
	 * file.  also get rid of "\r" characters, just in case
	 * we got this thru an ASCII connection.
	 *
	 */
	for (i=0, len=0; i < nbytes; i++)
	{
	    if ( (isprint(buf[i]) || isspace(buf[i])) && (buf[i]!='\r') )
		buf[len++] = buf[i];
	}
	buf[len] = '\0';
	XmTextInsert(viewtextW, (XmTextPosition)*ppos, (char *)buf);
	*ppos += len;
    }
    else
    {
	XtRemoveInput(*id);
	close (*fid);
	ui_set_cursor_normal(toplevelW);
    }
}


void view_text_file(void)
{
    static int pos;
    int len;
    char line[100];
    FILE *fp;
    void str_to_ascii(char *);

    if (!fConnected) return;

    if (!x_data_done)
    {
	ui_beep(0);
	fprintf(stderr,"Data connection busy\n");
	return;
    }

    /*
     * read in the file
     */
    if (curr_sel_cnt <= 0)
    {
	ui_beep(0);
	fprintf(stderr, "No file SELECTED!\n");
	return;
    }

    ui_set_cursor_busy(toplevelW);
    if ((fp=ctrl_view_file(curr_selections[0]-1)) == NULL)
    {
	ui_beep(0);
	fprintf(stderr, "Error trying to get file!\n");
	ui_set_cursor_normal(toplevelW);
	return;
    }

    /*
     * pop up the dialog
     *
     * we need to do this before putting in the text due to a bug
     * in Motif 1.1.xxx . This bug has been corrected in later
     * versions of Motif.
     *
     */
    XtManageChild(viewbbW);


    /*
     * Hand over the data getting to X
     */
    pos = 0;
    XmTextSetString(viewtextW, " ");

    ui_set_cursor_busy(toplevelW);
    XtAppAddInput(app_context, fileno(fp), (XtPointer) XtInputReadMask, 
	(XtInputCallbackProc) get_view_data, (XtPointer) &pos);
}



void commandCB(Widget w, int cmd, XmPushButtonCallbackStruct *cb)
{
    switch (cmd)
    {
      case cmdVIEW:
	view_text_file();
	break;

      case cmdRETRIEVE:
	retrieve_item();
	break;

      case cmdCD_UP:
	cd_up();
	break;

      default:
	fprintf(stderr, "commandCB: Unknown command '%s'\n", cmd);
    }
}


/************************** command callbacks **************************/

/*
 * type toggle callback
 */
void typetoggleCB(Widget w, int type, XmToggleButtonCallbackStruct *cb)
{
    if (!fConnected) return;

    if ((type==cmdASCII) && (cb->set==True))
	ctrl_set_type(typASCII);
    else if ((type==cmdIMAGE) && (cb->set==True))
	ctrl_set_type(typIMAGE);
}


/************************** new connection callback ********************/

/*
 * open connection "ok" callback
 */
void openconnokCB(Widget w, caddr_t calldata, XmSelectionBoxCallbackStruct *cb)
{
    char *text;
    char *utext;
    char *pword;
    char *buf = NULL;
    char line[100];
    time_t mtime;

    text = XmTextGetString(hosttextW);
    utext = XmTextGetString(usertextW);


#ifdef DEBUG
    fprintf(stderr, "Connecting to: %s\n", text);
#endif

    if (fConnected)
    {
	/* TO DO: Add "confirm" requester */
	if (user_prefs.autosave_dir)
	    ctrl_save_cache();
	ctrl_logout();
	curr_sel_cnt = 0;
	XmListDeleteAllItems(filelistW);
	ui_set_status_line(" ");
	ui_set_currdir(" ");
	fConnected = FALSE;
    }

    if (strcmp(utext, "anonymous")==0)
	pword = apassword;
    else
	pword = user_passwd;

    if (ctrl_login(text, utext, pword))
    {    
	fConnected = TRUE;
	if (ctrl_start_session(alloc_list_mem, add_list_item, &mtime))
	{
	    /* add the items to the list */
	    XmListAddItems(filelistW, list_items, item_count, 0);

	    /* Set status line */
	    sprintf(line, "Connected to: %s", text);
	    ui_set_status_line(line);
	    set_currdir();

	    if ((user_prefs.confirm) && (mtime != -1))
	    /* pop up cache notice */
	    {
		char mstr[256];

		strcpy(mstr,"A directory cache exists for host\n    ");
		strcat(mstr,text);
		strcat(mstr,"\ndated ");
		strcat(mstr,asctime(localtime(&mtime)));

		ui_set_cache_dialog_string(mstr);
		ui_show_cache_dialog();
	    }
	}
	else
	    ui_beep(0);
    }
    else
    	ui_beep(0);

   XtFree(text);
   XtUnmanageChild(opendlgW);
}

/************************** misc functions *****************************/


void set_currdir(void)
{
    char *path;

    path = ctrl_get_item_name(-1);
    ui_set_currdir(path);

    free(path);
}


void str_to_ascii(char *str)
{
    int i, len;

    len = strlen(str);
    for (i=0; i < len; i++)
    {
	if (!isprint(str[i]) && !isspace(str[i])) str[i] = ' ';
    }
}

void handle_sigint(void)
{
    char filepath[50];

    /*
     * Save cache, if needed
     */
    if (user_prefs.autosave_dir && fConnected)
	ctrl_save_cache();

    /*
     * Heinous hack to clean up /tmp.  This stuff should really 
     * be done by the control module
     */
    sprintf(filepath, "/tmp/xgetftpVIEW%ld", getpid());
    unlink(filepath);

    exit(1);
}


/************************** main program *******************************/

main(int argc, char *argv[])
{
    Arg args[4];
    char *hostname = NULL;
    struct passwd *pwd;
    char myhostname[100];
    char mydomainname[100];
    char username[50];
    struct hostent *hent;

    /*
     * init toolkit & create toplevel widgets 
     */
    ui_initialize(&argc, argv);


    /*
     * Check & get args
     */
    if (argc == 2)
    {
	if (strcmp(argv[1], "-q") == 0)
	    response_stream = fopen("/dev/null", "w");
	else
	    hostname = argv[1];
    }
    else if (argc == 3)
    {
	if (strcmp(argv[1], "-q") == 0)
	    response_stream = fopen("/dev/null", "w");
	else
	    fprintf(stderr, "usage: %s [-q] [hostname]\n", argv[0]);

	hostname = argv[2];
    }
    else if (argc > 3)
    {
	fprintf(stderr, "usage: %s [-q] [hostname]\n", argv[0]);
	exit(1);
    }


    /*
     * create menus (ie, menubarW)
     */
    ui_create_menus(mainwindowW, (XtCallbackProc) MenuButtonCB);


    /*
     * create work window (file list, etc)
     */
    ui_create_work_window(mainwindowW, (XtCallbackProc) doubleclickCB,
	(XtCallbackProc) selectionCB);


    /*
     * create commands
     */
    ui_create_commands(mainwindowW, (XtCallbackProc) commandCB,
	(XtCallbackProc) typetoggleCB);


    /*
     * create info window (label)
     */
    ui_create_status_line(mainwindowW);

 
    /*
     * put together all the components
     */
    XmMainWindowSetAreas(mainwindowW, menubarW, cmdrowcolW, NULL, NULL,
	workframeW);
    XtSetArg(args[0], XmNmessageWindow, statuslineW);
    XtSetValues(mainwindowW, args, 1);


    /*
     * create popup text viewing widget
     */
    ui_create_view_dialog(toplevelW);


    /*
     * create popup archie widget
     */
    ui_create_archie_dialog(toplevelW, (XtCallbackProc) archiefindCB);


    /*
     * create popup "About.." dialog
     */
    ui_create_about_dialog(toplevelW);			


    /*
     * create cache info dialog
     */
    ui_create_cache_dialog(toplevelW);


    /*
     * create popup host prompt dialog
     */
    ui_create_open_dialog(toplevelW, (XtCallbackProc) openconnokCB);


    /*
     * check user preferences
     */
    user_prefs.cache_view = 1;
    user_prefs.reuse_view = 1;
    user_prefs.reuse_dir = 1;
    user_prefs.autosave_dir = 1;
    user_prefs.confirm = 1;

    ui_create_pref_dialog(toplevelW, (XtCallbackProc) userprefCB, &user_prefs);


    /*
     * set retrieve directory
     */
    if (getcwd(retrieve_dir, 253) == NULL)
	retrieve_dir[0] = '\0';

    ui_create_retrieve_dir_dialog(toplevelW, (XtCallbackProc) setretrievedirCB,
	retrieve_dir);


     /*
      * Figure out a password for anonymous login
      */
     if (pwd = getpwuid(getuid()))
	strcpy(username, pwd->pw_name);
     else
	strcpy(username, "nobody");

     if (gethostname(myhostname, 100) != 0)
	strcpy(myhostname, "unknown");

#ifdef NEVER
     if (getdomainname(mydomainname, 100) != 0)
	strcpy(mydomainname, "unknown");
#endif

     if ((hent = gethostbyname(myhostname)) == NULL)
     	sprintf(apassword, "%s@%s.%s", username, myhostname);
     else
	sprintf(apassword, "%s@%s", username, hent->h_name);

#ifdef DEBUG
    printf("Anonymous password = %s\n", apassword);
#endif


    /*
     * connect to ftp server, if needed
     *
     * WARNING: We _MUST_ have the correct user_preferences before we
     *          give any ctrl commands!
     */
    if (hostname && ctrl_login(hostname, "anonymous", apassword))
    {
    	char line[100];
	time_t mtime;

	fConnected = TRUE;
    	ctrl_start_session(alloc_list_mem, add_list_item, &mtime);

    	XmListAddItems(filelistW, list_items, item_count, 0);
	sprintf(line, "Connected to: %s", hostname);
	ui_set_status_line(line);
	set_currdir();

	if ((user_prefs.confirm) && (mtime != -1))
	/* pop up cache notice */
	{
	    char mstr[256];

	    strcpy(mstr,"A directory cache exists for host\n    ");
	    strcat(mstr,hostname);
	    strcat(mstr,"\ndated ");
	    strcat(mstr,asctime(localtime(&mtime)));

	    ui_set_cache_dialog_string(mstr);
	    ui_show_cache_dialog();
	}
    }
    else
    {
	ui_set_status_line(" ");
    }


    /*
     * Handle ^C
     */
    signal(SIGINT, handle_sigint);


    /*
     * get & dispatch events 
     */
    XtRealizeWidget(toplevelW);
    XtAppMainLoop(app_context);
}

