/*=========================================================================
 * MODULE:
 *  "ftp"
 *
 * DESCRIPTION:
 *  Provides a FTP interface for a client program.  
 *
 * AUTHOR:
 *  Salim Alam
 *  University of Colorado, Boulder
 *
 * MODIFICATION LOG:
 *  93.04.19 S.A. - added A Lunn's patch for allowing IP numbers
 *  93.03.16 S.A. - yet another bug fix (DEBUG mode)
 *  93.03.10 S.A. - better support for variable arguments
 *  93.01.14 S.A. - fixed return bug in ftp_init_conn
 *  92.12.29 S.A. - ftp_init_conn now handles some errors more gracefully.
 *		    also, better use of "reponse_stream".
 *  92.12.01 S.A. - fixed response bug by making buffer larger
 *  92.11.24 S.A. - adding functionality to allow X-based I/O
 *  92.10.28 S.A. - ftp_get_response : better multi-line response handling.
 *=======================================================================*/

#include <X11/Xos.h>
#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#ifdef USE_PROTOTYPES
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#ifndef INADDR_NONE
#define INADDR_NONE     0xffffffff
#endif 

/* Special stuff for X-based I/O */
#define XGETFTP

#ifdef XGETFTP
#include <X11/Intrinsic.h>
extern void x_get_data(caddr_t, int *, XtInputId *);
extern XtAppContext app_context;
#endif


/* telnet codes */
#define tcSE 	240
#define tcNOP	241
#define tcDM	242
#define tcBRK	243
#define tcIP	244
#define tcAO	245
#define tcAYT	246
#define tcEC	247
#define tcEL	248
#define tcGA	249
#define tcSB	250
#define tcWILL	251
#define tcWONT	252
#define tcDO	253
#define tcDONT	254
#define tcIAC	255

/* other misc defines */
#define typeASCII		0
#define typeIMAGE		1
#define FTP_SERVER_PORTNUM	21	/* hardcoded, for now 		*/

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

#define PRIVATE static
#define UC(b) (((int)b)&0xff)
#define err_die(z) {perror(z); exit(1);}

#define RESP_BUFSIZE		256	/* buffer size, for a response line */

/*
 * Exported Variables
 */
FILE *response_stream = stdout;


/*
 * Global Variables
 */
PRIVATE FILE *responsefp;	/* response stream			*/
PRIVATE FILE *commandfp;	/* command stream			*/
PRIVATE FILE *datainfp;		/* data input stream			*/
PRIVATE FILE *dataoutfp;	/* data output stream			*/
PRIVATE int sd_ctrl;		/* socket descr. for control connection */
PRIVATE int sd_data;		/* socket descr. for initial data conn.	*/
PRIVATE struct sockaddr_in ctrl_addr; /* address for server		*/
PRIVATE struct sockaddr_in my_addr;   /* address for client		*/
PRIVATE int  type;		/* Image or Ascii			*/


int ftp_init_conn(char *hostname)
/*
 * Attempts a connection to the FTP port on the remote server "hostname".
 *
 * Dies if any serious errors encountered.
 *
 * Returns TRUE on success, FALSE otherwise.
 */
{
    int len;
    struct hostent *hp;
    unsigned long inaddr;           /* address returned by inet_addr */

    /*
     * open primary socket connection 
     */
    if ((sd_ctrl = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	err_die("ftp_init_conn ctrl socket");

    /*
     * find description of host & set up it's address
     */

    if ((inaddr = inet_addr(hostname)) != INADDR_NONE) {
					/* Is it an IP number address? */	
	bcopy((char *) &inaddr, (char *) &ctrl_addr.sin_addr, sizeof(inaddr));
    } else {				/* Try hostname as a real name */
	    if ((hp = gethostbyname(hostname)) == 0)
    	{				/* Cannot work address out */
		close(sd_ctrl);
		return FALSE;
    	} else {
		bzero((char *)&ctrl_addr, sizeof(ctrl_addr));
		bcopy(hp->h_addr, &ctrl_addr.sin_addr, hp->h_length);
	}
    }
    ctrl_addr.sin_family = AF_INET;
    ctrl_addr.sin_port = htons(FTP_SERVER_PORTNUM);

    /*
     * establish a connection
     */
    if (connect(sd_ctrl, (struct sockaddr *)&ctrl_addr, sizeof(ctrl_addr)) < 0)
    {
#ifdef DEBUG
	perror("connect");
#endif
	close(sd_ctrl);
	return FALSE;
    }

    /*
     * open stream descriptors, to make variable arg handling
     * easy.
     */
    responsefp = fdopen(sd_ctrl, "r");
    commandfp  = fdopen(sd_ctrl, "w");


    /*
     * find out about client's port
     */
    len = sizeof(my_addr);
    if (getsockname(sd_ctrl, (struct sockaddr *)&my_addr, &len) < 0)
	err_die("ftp_init_data getsockname my_addr");

    return TRUE;
}


#ifdef USE_PROTOTYPES
int ftp_send_command(char *fmt, ...)
#else
int ftp_send_command(va_alist)
va_dcl
#endif
/*
 * Sends a command to the server.  The input parameters are in a
 * "printf" form --- ie, a format specification string, followed
 * by a variable number of arguments.
 *
 * After sending command, gets the appropriate number of responses
 * from the server.  Gets a SINGLE response from the server and
 * returns the response code.
 *
 * For multiple-response commands, further responses need to be
 * checked for by the server.
 */ 
{
#ifndef USE_PROTOTYPES
    char *fmt;
#endif
    va_list ap;

#ifdef DEBUG
    fprintf(stderr,"Sending: ");
#ifdef USE_PROTOTYPES
    va_start(ap, fmt);
#else
    va_start(ap);
    fmt = (char *)va_arg(ap, char *);
#endif
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    fprintf(stderr, "\n");
#endif

#ifdef USE_PROTOTYPES
    va_start(ap, format);
#else
    va_start(ap);
    fmt = (char *)va_arg(ap, char *);
#endif

    vfprintf(commandfp, fmt, ap);
    va_end(ap);
    fprintf(commandfp, "\r\n");
    (void) fflush(commandfp);

    return (ftp_get_response(response_stream));
}



int ftp_get_response(FILE *fdout)
/*
 * Gets a single response from the server thru the command connection.
 * Multiple-line responses are handled.  The complete text of the
 * responses are written to the file fdout, which should be open and
 * ready for writing.  ALL TELNET CONTROLS ARE IGNORED.
 *
 * The response code is returned.
 */
{
    int code, test_code = 0;
    char continue_flag;
    char line[RESP_BUFSIZE];

    fgets(line, RESP_BUFSIZE-2, responsefp);
    fprintf(fdout, line);
#ifdef DEBUG
    if (strchr(line,tcIAC) != NULL)
	fprintf(stderr,"<ignoring Telnet IAC>\n");
#endif

    continue_flag = line[3];
    line[3] = '\0';
    code = atoi(line);

    if (continue_flag == '-')
    {
	while (test_code != code)
	{
	    fgets(line, RESP_BUFSIZE-2, responsefp);
	    fprintf(fdout, line);
	    if (line[3] == '-') continue;
	    line[3] = '\0';
	    test_code = atoi(line);
	}
    }

    fflush(fdout);
    return code;
}



int ftp_get_next_response(int oldcode, FILE *fdout)
/*
 * Given the previous response code, gets any expected responses
 * from the server.  If no more expected responses, returns -1,
 * else returns the code for the response it got.
 */
{
    if ( ((int) (oldcode/100.0))  == 1 )
	return (ftp_get_response(fdout));
    else
	return -1;
}



void ftp_get_all_responses(int code, FILE *fdout)
/*
 * Gets all further remaining responses from the server.  If
 * code is equal to 0, then an initial response will get gotten
 * from the server;  otherwise, code will be assumed to be the
 * result of a previous ftp_get_response command.
 */
{
    int nextcode = code;

    if (code == 0)
	nextcode = ftp_get_response(fdout);

    while ( (nextcode=ftp_get_next_response(nextcode,fdout)) > 0 ) ;
}



int ftp_init_dataconn(void)
/*
 * Creates a new port for data connections, and sends a PORT command
 * to the server. Starts listening on the newly created port.
 *
 * All errors are fatal!  Return is positive integer.
 */
{
    int len, result;
    struct sockaddr_in data_addr;
    char *p, *a;


    /*
     * create a new port & start listening
     */
    data_addr = my_addr;
    data_addr.sin_port = 0;	/* system picks a port */

    if ( (sd_data = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	err_die("ftp_init_dataconn data socket");
  
    if (bind(sd_data, (struct sockaddr *)&data_addr, sizeof(data_addr)) < 0)
	err_die("ftp_init_dataconn bind");

    len = sizeof(data_addr);
    if (getsockname(sd_data, (struct sockaddr *)&data_addr, &len) < 0)
	err_die("ftp_init_dataconn getsockname");

    listen(sd_data, 5);

    /*
     * send PORT command to server
     */
    p = (char *)&data_addr.sin_port;
    a = (char *)&data_addr.sin_addr;

    result = ftp_send_command("PORT %d,%d,%d,%d,%d,%d",
	UC(a[0]),UC(a[1]),UC(a[2]),UC(a[3]),UC(p[0]),UC(p[1]) );

    return result;
}



int ftp_get_data(FILE *fdout)
/*
 * Gets a file from the data connection.  The data port should be listening
 * for an incoming connection.  The data file is written to the stream fdout,
 * which should already be open and ready for writing. A stream data
 * connection mode is assumed.
 *
 * Returns void.
 */
{
    int fromlen = 0;
    int c, s;

    /*
     * Establish a connection with server data port
     */
    s = accept(sd_data, (struct sockaddr *)NULL, &fromlen);
    datainfp = fdopen(s, "r");

#ifdef XGETFTP
    /*
     * Set up Xt routines for getting input.  x_get_data is
     * an external function that must be provided.
     */
    XtAppAddInput(app_context, s, (XtPointer) XtInputReadMask, 
	(XtInputCallbackProc) x_get_data, (XtPointer) fdout);
#else
    /*
     * Read in & echo the input
     */
    while ((c = getc(datainfp)) != EOF)
	(void) putc(c, fdout);

    close(s);
#endif

}



int ftp_put_data(FILE *fdout)
/* stub */
{
    fprintf(stderr, "STUB ftp_put_data called!\n");
}



void ftp_close_conn(void)
/*
 * Close the connection to the ftp server
 */
{
    close(sd_ctrl);
    close(sd_data);
}
