/***************************************************************************
 *
 * MODULE:
 *  file cache
 *
 * DESCRIPTION
 *  Provides functions used to maintain a cache of files "Viewed" by
 *  xgetftp.
 *
 * AUTHOR:
 *  Salim Alam
 *  University of Colorado, Boulder
 *
 * MODIFICATION LOG:
 *  93.03.16 S.A. - Fixed bug in fc_mkdir_path, added auto cache dir
 *		    creating in fc_set_root_dir.
 *  92.12.29 S.A. - Added file cache delete function
 *
 **************************************************************************/

#include <stdio.h>
#include <X11/Xos.h>
#include <sys/stat.h>
#ifndef __386BSD__
#include <malloc.h>
#endif
#include <stdlib.h> 
#include <dirent.h>

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

#define PRIVATE static


/*
 * Private global variables
 */

#define DEFAULT_CACHE_DIR ".xgetftpcachedir"

PRIVATE char * cache_root;



void fc_set_root_dir(char *hostname)
/*
 * This function determines the directory in the user's filespace
 * where the cache hierarchy should be organized.  First the env
 * variable XGETFTPCACHEDIR is checked --- if available, it will be
 * used.  Otherwise, the env var HOME will be looked up and the directory
 * $HOME/.xgetftpcachedir will be used.  If the HOME var cannot be found,
 * the current working directory will be used.
 *
 * After determining the cache directory, this function saves the name
 * of the host and appends it to the cache directory path.  All files
 * cached in this session will now be cached here.
 *
 */
{
    int len;
    char *tmp_str;

  
    /*
     * Create cache root
     */
    if (tmp_str = getenv("XGETFTPCACHEDIR"))
    {
	len = strlen(tmp_str);
	cache_root = malloc(len + strlen(hostname) + 2);
	if (tmp_str[len] == '/')
	    sprintf(cache_root, "%s%s", tmp_str, hostname);
	else
	    sprintf(cache_root, "%s/%s", tmp_str, hostname);
    }
    else if (tmp_str = getenv("HOME"))
    {
        len = strlen(tmp_str) + strlen(DEFAULT_CACHE_DIR) + 2;
        cache_root = malloc ( len + strlen(hostname) );
        sprintf(cache_root, "%s/%s/%s", tmp_str, DEFAULT_CACHE_DIR, hostname);
    }
    else
    {
	tmp_str = (char *) getcwd(NULL, 256);
        len = strlen(tmp_str) + strlen(DEFAULT_CACHE_DIR) + 2;
        cache_root = malloc ( len + strlen(hostname) );
        sprintf(cache_root, "%s/%s/%s", tmp_str, DEFAULT_CACHE_DIR, hostname);
    }


    /*
     * If directory doesn't exist, create it.
     */
    tmp_str = malloc(strlen(cache_root) + 2); 	/* we need to muck around
						   a little bit here... */
    sprintf(tmp_str,"%s/", cache_root);

    if (!fc_mkdir_path(tmp_str))
    {
	fprintf(stderr, "fc_set_root_dir: can't make root path '%s'\n",
		cache_root);
    }

    free(tmp_str);


#ifdef DEBUG
        fprintf(stderr, "Cache root = '%s'\n", cache_root);
#endif

}




char * fc_make_cache_path(char *rem_path)
/*
 * Given the remote path for a file (NOT including the file name), this
 * function appends the remote path to the cache_root and returns a pointer
 * to the complete path.  Memory is allocated, so the pointer returned should
 * be eventually freed by the caller.
 *
 */
{
    char *complete_path;

    complete_path = malloc(strlen(rem_path) + strlen(cache_root) + 2);

    if (rem_path[0] == '/')
    	sprintf(complete_path, "%s%s", cache_root, rem_path);
    else
    	sprintf(complete_path, "%s/%s", cache_root, rem_path);

    return complete_path;
}



int fc_mkdir_path(char *path)
/*
 * Given a complete path in the user's file system, this function makes
 * sure that all the directories in the path are present.  This function
 * will call "mkdir" to create any needed directories.
 *
 * Note that the "path" can contain a file at the end.  If the last 
 * component of the path does not end with a "/" it is assumed to be a
 * file and is _not_ checked!
 *
 * Returns TRUE for success, FALSE otherwise.
 *
 */
{
    char *s;
    struct stat stat_buf;

    /*
     * First, check the complete path at once.  If the complete path
     * exists, check to see if it is a directory.
     */
    if (stat(path, &stat_buf) == 0)    
    {
	if (!S_ISDIR(stat_buf.st_mode))
	{
	    fprintf(stderr, "fc_mkdir_path: '%s' is not a dir.\n",
		path);
	    return FALSE;
	}
	return TRUE;
    }


    /*
     * Otherwise check component-wise, directory by directory
     */
    s = path;
    if ((*s) && (*s == '/')) s++; 	/* skip the first component, since
					   stat on some systems can't handle
					   empty paths */

    while (*s)
    {
	while ((*s) && (*s != '/')) s++; /* skip to the next component */

	if (*s == '/')
	/* check directory component */
	{
	    *s = '\0';
#ifdef DEBUG
	    fprintf(stderr,"fc_mkdir_path: Checking '%s'\n", path);
#endif
	    if (stat(path, &stat_buf) < 0)
	    /* assume file didn't exist */
	    {
#ifdef DEBUG
		fprintf(stderr,"fc_mkdir_path: Making '%s'\n", path);
#endif
		if (mkdir(path, 0700) < 0)
		{
		     fprintf(stderr, "fc_mkdir_path: couldn't make `%s`\n",
			path);
		     return FALSE;
		}
	    }
	    else
	    /* check to make sure this is a dir */
	    {
		if (!S_ISDIR(stat_buf.st_mode))
		{
		    fprintf(stderr, "ctrl_get_directory: '%s' is not a dir.\n",
			path);
		    return FALSE;
		}
	    }
	    *s = '/';
	    s++;
	} /* if */

    } /* while */
}


int fc_delete_cache(void)
/*
 * Deletes the file viewed cache
 */
{
    DIR *dir;
    struct dirent * entry;
    char command_str[255];

    if ( (dir = opendir(cache_root)) == NULL )
	return FALSE;

    while ( (entry = readdir(dir)) != NULL )
    {
	if ( (strcmp(entry->d_name, ".")==0) ||
	     (strcmp(entry->d_name, "..")==0) ||
	     (strcmp(entry->d_name, "DIRCACHE")==0) )
	    continue;

	sprintf(command_str, "/bin/rm -rf %s/%s", cache_root, entry->d_name);

	system(command_str);
    }

    closedir(dir);
}

