/**************************************************************************
 * MODULE:
 *  control
 *
 * DESCRIPTION:
 *  Provides a high-level interface to the ftp and cache modules.
 *  Basically, this is the module that will provide all the functionality
 *  to the user-interface.
 *
 * AUTHOR:
 *  Salim Alam
 *  University of Colorado, Boulder
 * 
 * MODIFICATION LOG:
 *  93.04.19 S.A. - Added Peter Brandstrom's patch to allow viewing of
 *		    compressed text files.. this is a hack that should
 *		    be made more robust later.  Also modified interface
 *		    for ctrl_start_session so that it returns the last-
 *		    modified-time for the cache, if we are using it.
 *  93.02.15 S.A. - ctrl_delete_cache now reloads directory info from the
 *                  server.  also slightly rewrote ctrl_start_session to
 *		    allow code reuse.
 *  93.02.03 S.A. - Better error handling: ctrl_start_session, ctrl_down_dir
 *		    and read_directory.
 *  92.12.29 S.A. - Better handling of login.  Added ctrl_delete_file_cache.
 *		    Better handling of "response_stream" -- allows main prog.
 *		    to arbirarily specify where most responses go.
 *  92.12.16 S.A. - Fixed file view bug.  Better cleanup of /tmp.
 *		    Added ctrl_delete_cache function.
 *  92.12.11 S.A. - Added function to query name of item in cache.  
 *		    Now also check user prefs to see how much caching
 *		    should be done.
 *  92.12.09 S.A. - Using "/tmp" for temporary files now :)  Added dir
 *		    caching support.
 *  92.12.01 S.A. - Added directory retrieval and new file view routine.  
 *  92.11.24 S.A. - Added mods to handle X-based I/O.
 *  92.10.27 S.A. - Changed use of InitFunc & CreateFunc. Added function
 *		    ftp_set_type.  Also now use the type field in struct
 *		    fileinfo to check for correctness of commands.
 *
 **************************************************************************/

#include <X11/Xos.h>
#include <stdio.h>
#include <sys/stat.h>
#ifndef __386BSD__
#include <malloc.h>
#endif
#include "config.h"
#include "ftp.h"
#include "cache.h"
#include "file_cache.h"
#include "control.h"
#include "prefs.h"

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

#define XGETFTP
#ifdef XGETFTP
#define INBUFSIZE 1024
#include <X11/Intrinsic.h>
extern XtAppContext app_context;
#endif


/*
 * exported variables
 */
int ftp_curr_type = -1;
int x_data_done = 1;


/*
 * global vars
 */
static InitFunc   F_init_str;
static CreateFunc F_add_str;



/*
 * useful internal functions
 */
int read_directory(char *path, cacheinfo_ptr *plevel, cacheinfo_ptr prev);
void x_get_data(FILE *, int *, XtInputId *);
char *server_get_root(void);


/********************** login/logout functions ***************************/

int ctrl_login(char *host, char *user, char *pass)
/*
 * Connect to the server whose name is "host" and log in.  If
 * user and pass are specified, they will be used to log into a
 * user account, else the user "anonymous" will be used.
 *
 * Success returns TRUE, failure returns FALSE.  
 * Fatal errors may occur, in which case there is no return.
 */
{
    /*
     * Connect to the server.
     */
    if (!ftp_init_conn(host))
    {
	fprintf(stderr, "ctrl_login: cannot connect to '%s'\n", host);
	return FALSE;
    }

    if (ftp_get_response(response_stream) >= 300)
    {
	fprintf(stderr, "ctrl_login: cannot connect to '%s'\n", host);
	return FALSE;
    }

    /*
     * Try to log in
     */
    if (!user)
    {
	user = "anonymous";
	pass = "guest";
    }

    if (ftp_send_command("USER %s", user) >= 400)
    {
	fprintf(stderr, "ctrl_login: cannot login as '%s'\n", user);
	return FALSE;
    }

    if (ftp_send_command("PASS %s", pass) >= 400)
    {
	fprintf(stderr, "ctrl_login: invalid password\n");
	return FALSE;
    }


    /*
     * Initialize File Cache
     */
    fc_set_root_dir(host);

    return TRUE;
}



void ctrl_logout(void)
{
    ftp_close_conn();
    ftp_curr_type = -1;
    x_data_done = 1;
    cache_free_levels(&toplevel);
}


int ctrl_start_session(InitFunc init_str, CreateFunc add_str, time_t *ptime)
{
    char *s;
    int i;

    /*
     * save values for init_str and add_str
     */
    F_init_str = init_str;
    F_add_str  = add_str;

    /* set type to IMAGE */
    ctrl_set_type(typIMAGE);

    /*
     * If we're using previous directory cache, load it
     */
    cache_set_root("");
    *ptime = -1;  /* flag = no time available */
    if (user_prefs.reuse_dir && ctrl_load_cache(ptime))
    {
#ifdef DEBUG
	fprintf(stderr, "Reusing directory cache.\n");
#endif
	goto done;
    }


    /*
     * Otherwise, query server for info....
     */
    if ( (s=server_get_root()) == NULL )
	return FALSE;

    cache_set_root(s);


    /*
     * get listing of root dir
     */
    if (!read_directory(s, &toplevel, NULL))
    {
	fprintf(stderr, "ctrl_start_session: can't read root dir!\n");
	return FALSE;
    }

done:
    currlevel = toplevel;

    /*
     * create outgoing data using add_str
     */
    init_str(toplevel->num_entries);

    for (i=0; i < toplevel->num_entries; i++)
    {
	add_str(toplevel->entry_arr[i].line);
    }

    return TRUE;
}


/********************** directory movement *******************************/

int ctrl_down_dir(int index)
/*
 * This function attempts to traverse the directory tree at the entry
 * specified by "index".  If that directory has been cached, it will
 * use the cached version, otherwise it will bring in the contents of
 * the directory remotely and cache it.  F_init_str is called once,
 * and then F_add_str is called for each entry.
 *
 * If "index" does not specify a directory then the function will
 * immediately return FALSE, otherwise it will return TRUE after
 * processing the directory as described above.  The function may also
 * return FALSE if it cannot read a new directory from the server.
 */
{
    int i;
    char *path;

    /*
     * check to see if it is a directory
     */
    if (currlevel->entry_arr[index].type != fitypDIRECTORY)
	return FALSE;

    /*
     * check for cached directory
     */
    if (currlevel->entry_arr[index].next_level)
    {
	cache_next_dir(currlevel->entry_arr[index].name);
	currlevel = currlevel->entry_arr[index].next_level;
    }
    else
    /*
     * otherwise, load in the needed level
     */
    {
	path = cache_make_filename(currlevel->entry_arr[index].name);
	cache_next_dir(currlevel->entry_arr[index].name);
	if (!read_directory(path, &(currlevel->entry_arr[index].next_level),
		currlevel))
	{
	    fprintf(stderr, "ctrl_down_dir: can't read directory!\n");
	    free(path);
    	    cache_prev_dir();
	    return FALSE;
	}
	currlevel = currlevel->entry_arr[index].next_level;
	free(path);
    }

    /*
     * create outgoing data 
     */
    F_init_str(currlevel->num_entries);
    for (i=0; i < currlevel->num_entries; i++)
    {
	F_add_str(currlevel->entry_arr[i].line);
    }

    return TRUE;
}



int ctrl_up_dir(void)
{
    int i;

    /*
     * check current level
     */
    if (!currlevel->prev_level)
    {
	fprintf(stderr,"ctrl_down_dir: already at top level.\n");
	return FALSE;
    }

    /*
     * create outgoing data
     */
    cache_prev_dir();
    currlevel = currlevel->prev_level;

    F_init_str(currlevel->num_entries);
    for (i=0; i < currlevel->num_entries; i++)
    {
	F_add_str(currlevel->entry_arr[i].line);
    }

    return TRUE;
}


/********************** ftp control **************************************/

void ctrl_set_type(int type)
/*
 * Set type to either ascii or binary
 */
{
    int code;
    char typec;

    if (ftp_curr_type == type)
	return;

    typec = (type == typASCII)? 'A' : 'I';

    if ((code=ftp_send_command("TYPE %c", typec)) >= 300)
    {
	fprintf(stderr, "set_type: cant set type.\n");
    }

    ftp_get_all_responses(code, response_stream);
    ftp_curr_type = type;
}



int ctrl_get_selection(char *path, int index, char *newname)
/*
 * This function gets the file or directory specfied by "index".
 * "path" is the retrieval path in the local file system, or 
 * the current directory if NULL.  If "index" specifies a normal
 * file and newname is not NULL, then the incoming file is renamed
 * to "newname".
 *
 * Returns TRUE for success, FALSE otherwise.
 */
{
    switch (currlevel->entry_arr[index].type)
    {
      case fitypFILE:
	return ctrl_get_file(path, index, newname);
	break;

      case fitypDIRECTORY:
	return ctrl_get_directory(path, index);
	break;

      default:
	return FALSE;
	break;
    }
}


int ctrl_get_directory(char *path, int index)
/*
 * This function gets the directory indicated by "index".  "Path" is
 * the path in the local file system where the incoming directory 
 * should be retrieved into.  If "Path" is NULL, it is assumed to be
 * the current directory.
 *
 * Returns TRUE if success, FALSE otherwise.
 */
{
    int i;
    int len;
    char *local_path;
    char *name;
    struct stat stat_buf;

    /*
     * check file type
     */
    if (currlevel->entry_arr[index].type != fitypDIRECTORY)
	return FALSE;

    /*
     * create a local path
     */
    name = currlevel->entry_arr[index].name;
    len = (path? strlen(path)+1 : 0) + strlen(name) + 1 ;
    local_path = (char *) malloc(len+1);

    if (path)
    {
    	strcpy(local_path, path);
	if (path[strlen(path)-1] != '/')
	    strcat(local_path, "/");
    }
    else
	local_path[0] = '\0';

    strcat(local_path, name);

   
    /*
     * create local directory, if needed
     */
     if (stat(local_path, &stat_buf) < 0)
     /* assume that directory does not exist */
     {
	if (mkdir(local_path, 0700) < 0)
	{
	    fprintf(stderr, "ctrl_get_directory: couldn't make `%s`\n",
		local_path);
	    return FALSE;
	}
     }
     else
     /* check to make sure it is a directory */
     {
	if (!S_ISDIR(stat_buf.st_mode))
	{
	    fprintf(stderr, "ctrl_get_directory: '%s' is not a dir.\n",
		local_path);
	    return FALSE;
	}
     }

#ifdef DEBUG
    printf("ctrl_get_directory: getting '%s' -> '%s'\n", 
	currlevel->entry_arr[index].name, local_path);
#endif


    /*
     * go down directory and get everything recursively
     */
    if (!ctrl_down_dir(index))
    {
	ctrl_up_dir();
	return FALSE;
    }

    for (i=0; i <  currlevel->num_entries; i++)
    {
	ctrl_get_selection(local_path, i, NULL);
    }

    ctrl_up_dir();

    return TRUE;
}



int  ctrl_get_file(char *path, int index, char *newname)
/*
 * This function gets the file indicated by "index".  "Path" is
 * the path for the directory that the file is to be retreived in,
 * or current directory if NULL.  "newname", if not NULL, indicates
 * a different name for the incoming file.
 *
 * Returns TRUE if success, FALSE otherwise.
 */
{
    int len;
    int code;
    char *local_path;
    char *rem_path;
    char *name;
    FILE *fp;

    /*
     * check file type
     */
    if (currlevel->entry_arr[index].type != fitypFILE)
	return FALSE;

    /*
     * create fully-qualified remote name
     */
    rem_path = cache_make_filename(currlevel->entry_arr[index].name);

    /*
     * create a local filename (with optional path)
     */
    name = (newname? newname : currlevel->entry_arr[index].name);

    len = (path? strlen(path)+1 : 0) + strlen(name) + 1 ;
    local_path = (char *) malloc(len+1);

    if (path)
    {
    	strcpy(local_path, path);
	if (path[strlen(path)-1] != '/')
	    strcat(local_path, "/");
    }
    else
	local_path[0] = '\0';

    strcat(local_path, name);

#ifdef DEBUG
    printf("ctrl_get_file: getting '%s' -> '%s'\n", 
	currlevel->entry_arr[index].name, local_path);
#endif

    /*
     * if local file already exists.... 
     */

    /*     ------- to do -------         */


    /*
     * open local file, get data from remote file
     */
    if ( (fp = fopen(local_path, "w")) == NULL )
    {
	fprintf(stderr, "ctrl_get_file: can't open '%s' for writing.\n",
	    local_path);
	return FALSE;
    }

    ftp_init_dataconn();

    if ((code=ftp_send_command("RETR %s", rem_path)) >= 300)
    {
	fprintf(stderr, "ctrl_get_file: error in RETR.\n");
    	ftp_get_all_responses(code, response_stream);
    	free(rem_path);
	return FALSE;
    }
    else
    {
#ifdef XGETFTP
	if (!x_data_done)
	    fprintf(stderr, "x_data_done already being used!\n");
	x_data_done = 0;
#endif
    	ftp_get_data(fp);
    }

#ifdef XGETFTP
    while (!x_data_done)
	XtAppProcessEvent(app_context, XtIMAll);
#endif

    ftp_get_all_responses(code, response_stream);
    free(rem_path);
  
    fflush(fp);
    fclose(fp);

    return TRUE;
}

/************************** file cache interaction ***********************/

FILE * ctrl_view_file(int index)
/*
 * If the file specified by index is a normal file, this function will
 * either retrieve the file from the remote host or from the file cache,
 * open it for reading, and return a pointer to the open file.  If the
 * file is not cached, the file will be cached.  
 *
 * Returns a pointer to an open file on success, or NULL otherwise.
 */
{
    FILE *fp = NULL;
    char *rem_path = NULL;
    char *cache_path = NULL;
    char *new_name = NULL;
    char *complete_name = NULL;
    struct stat stat_buf;

    /*
     * check file type
     */
    if (currlevel->entry_arr[index].type != fitypFILE)
	return NULL;

    /*
     * Create path and name of the local file, depending on
     * whether the user wants to cache it or not.
     */
    rem_path = cache_make_filename("");

    if (user_prefs.cache_view)
    {
    	cache_path = fc_make_cache_path(rem_path);

    	complete_name = malloc( strlen(cache_path) + 
	    strlen(currlevel->entry_arr[index].name) + 2);

    	sprintf(complete_name, "%s/%s", cache_path, 
	    currlevel->entry_arr[index].name);
    }
    else
    {
	cache_path = (char *) malloc(10);
	strcpy(cache_path, "/tmp");
	new_name = (char *) malloc(50);
	sprintf(new_name, "xgetftpVIEW%ld", getpid());
	complete_name = malloc(strlen(cache_path) + strlen(new_name) + 2);
	sprintf(complete_name, "%s/%s", cache_path, new_name);
    }

    free ( rem_path );


    /*
     * Check actual cache to see if file present.  This only happens
     * if the user preferences "cache_view" and "reuse_view" are set.
     *
     */
    if (user_prefs.cache_view && user_prefs.reuse_view &&
	(stat(complete_name, &stat_buf)==0))
    {
	if (S_ISREG(stat_buf.st_mode))
	{
#ifdef DEBUG
	    fprintf(stderr, "Reusing cached file\n");
#endif
	    currlevel->entry_arr[index].fCached = 1;
	}
    }


    /*
     * Get and cache the file, depending on user prefs
     *
     */
    if (user_prefs.cache_view && currlevel->entry_arr[index].fCached == 0)
    {
	fc_mkdir_path(cache_path);

	if (!ctrl_get_file(cache_path, index, NULL))
	    goto done;

	currlevel->entry_arr[index].fCached = 1;
    } 
    else if (!user_prefs.cache_view)
    {
	if (!ctrl_get_file(cache_path, index, new_name))
	    goto done;
    }

    /*
     * open the file and return pointer
     */
    {
      int n;
      char *command = NULL;

      n = strlen(complete_name);
      if (n > 2) {
      /*
       * WARNING:
       *
       * This is a hack that will allow viewing of compressed files.
       * It will try to uncompress the file every time, even though an
       * uncompressed copy might be present.  Also, it might overwrite
       * another file. Finally, this will not work unless we are cacheing
       * viewed files. Strange things may happen if uncompress is not 
       * present or fails.
       *
       */
        if (strcmp(&complete_name[n-2], ".Z") == 0) {
          command = XtMalloc(strlen(complete_name)+strlen("uncompress") + 5);
          sprintf(command, "%s %s\n", UNCOMPRESSBIN, complete_name);
          system(command);
	  XtFree(command);
          complete_name[n-2]  = '\0';
        }
      }

      fp = fopen(complete_name, "r");
    }

done:
    if (cache_path)    free (cache_path);
    if (complete_name) free (complete_name);
    if (new_name)      free (new_name);

    return fp;
}

void ctrl_delete_file_cache(void)
{
    fc_delete_cache();
}


/************************** dir cache interaction ************************/

void ctrl_save_cache(void)
{
    char *cache_path;

    cache_path = fc_make_cache_path("DIRCACHE");
    cache_save_cache(cache_path);
    free(cache_path);
}

int ctrl_load_cache(time_t *ptime)
{
    char *cache_path;
    int res;

    *ptime = -1; /* flag = no time available */

    cache_path = fc_make_cache_path("DIRCACHE");

    if (res = cache_load_cache(cache_path))
    /* get last modified time and put it in *ptime */
    {
    	struct stat buf;

	if (stat(cache_path, &buf) == 0)
	{
	    *ptime = buf.st_mtime;
	}
    }

    free(cache_path);
    return res;
}

int ctrl_delete_cache(void)
/*
 * Deletes all cached directory information and attempts to reload
 * info from the server.  Returns TRUE if successful, FALSE otherwise.
 *
 * NOTE: The caller should disconnect upon a FALSE return since currently
 *       there is no easy recovery.... should allow recovery in future
 *       versions.
 */
{
    int i;
    char *s;
    char *cache_path;

    /*
     * Get rid of all cached data
     */
    cache_path = fc_make_cache_path("DIRCACHE");
    unlink(cache_path);
    free(cache_path);
    cache_free_levels(&toplevel);


    /*
     * Reload top level directory info from server
     *
     * NOTE: Most of this is cut-n-pasted directly from ctrl_start_session
     *
     */
    if ( (s=server_get_root()) == NULL )
	return FALSE;

    cache_set_root(s);

    if (!read_directory(s, &toplevel, NULL))
	return FALSE;

    currlevel = toplevel;

    F_init_str(toplevel->num_entries);
    for (i=0; i < toplevel->num_entries; i++)
    {
	F_add_str(toplevel->entry_arr[i].line);
    }

    return TRUE;
}


/************************** misc functions *******************************/

char *server_get_root(void)
/*
 * This function issues a "PWD" command to the ftp server and receives
 * and parses the response to get the root directory on the server.
 * Returns a pointer to the string with the root if successful, or NULL
 * if an error occurs.
 */
{
    FILE *tmpfile;
    FILE *old_stream;
    char *s, *t;
    char filename[100];
    static char root[100];

    sprintf(filename, "/tmp/xgetftpROOT%ld", getpid());
    if ( (tmpfile = fopen(filename, "w+")) == NULL )
    {
	fprintf(stderr, "server_get_root: can't open '%s'.\n", filename);
	return NULL;
    }

    old_stream = response_stream;
    response_stream = tmpfile;

    if (ftp_send_command("PWD") >= 300)
    {
	fprintf(stderr, "server_get_root: cant get PWD\n");
	unlink(filename);
	response_stream = old_stream;
	return NULL;
    }

    response_stream = old_stream;

    fseek(tmpfile, 0, 0);
    fgets(root, 98, tmpfile);

    s = root;
    while (*s && *s++ !='"');
    t = s;
    while (*t && *++t !='"');
    *t = '\0';
    fclose(tmpfile);

#ifdef DEBUG
    printf("server_get_root on: root = '%s'\n", s);
#endif

    return s;
}


char *ctrl_get_item_name(int item)
/*
 * This function returns a string that specifies the complete path of
 * the selected item.   If item < 0, then the complete path of the
 * current directory is returned.
 *
 * Returns a string (which should be freed later by caller), or NULL
 * if the item number is too large.
 *
 */
{
    char * curr_path = NULL;

    if (item < 0)
    	curr_path = cache_make_filename("");
    else if (item < currlevel->num_entries)
	curr_path = cache_make_filename(currlevel->entry_arr[item].name);

    return curr_path;
}



void x_get_data(FILE *fdout, int *fid, XtInputId *id)
/* data input callback */
{
    unsigned char buf[INBUFSIZE];
    int nbytes;

    nbytes = read(*fid, buf, INBUFSIZE);

    if (nbytes)
    {
	write(fileno(fdout), buf, nbytes);	
    }
    else
    /* EOF reached */
    {
	x_data_done = 1;
	XtRemoveInput(*id);
	close(*fid);
    }
}


int read_directory(char *path, cacheinfo_ptr *plevel, cacheinfo_ptr prev)
{
    char filename[100];
    FILE *tmpfile;
    int code;

#ifdef DEBUG
    printf("read_directory: Reading from path: '%s'\n", path);
#endif

    ftp_init_dataconn();
    sprintf(filename, "/tmp/xgetftpLISTING%ld", getpid());
    if ( (tmpfile = fopen(filename, "w")) == NULL )
    {
	fprintf(stderr, "read_directory: can't open '%s'.\n", filename);
	return FALSE;
    }

    if ((code=ftp_send_command("LIST %s", path)) >= 300)
    {
	fprintf(stderr, "read_directory: can't get LIST.\n");
	unlink(filename);
	return FALSE;
    }

#ifdef XGETFTP
    if (!x_data_done)
	fprintf(stderr, "x_data_done already being used!\n");
    x_data_done = 0;
#endif

    ftp_get_data(tmpfile);

#ifdef XGETFTP
    while (!x_data_done)
	XtAppProcessEvent(app_context, XtIMAll);
#endif

    ftp_get_all_responses(code, response_stream);
    fclose(tmpfile);

    cache_init_level(plevel, prev, filename);
#ifdef DEBUG
    cache_print_level(*plevel);
#endif

    unlink(filename);
    return TRUE;
}

