/***************************************************************************
 * MODULE:
 *  cache.c
 *
 * DESCRIPTION:
 *  Provides functions to maintain a directory cache.
 *
 * AUTHOR:
 *  Salim Alam
 *  University of Colorado, Boulder
 *
 * MODIFICATION LOG:
 *  93.03.16 S.A. Made cache_set_root more robust.
 *  93.01.27 S.A. Now closing cache saved/loaded files correctly
 *  92.12.15 S.A. Fixed bug, made cache_load_levels more robust
 *  92.12.09 S.A. Added dir caching functions.
 *  92.12.01 S.A. Added fCached flag.
 *  92.10.27 S.A. Fixed bugs in cache_init_level & cache_set_root, added
 *		  file types.
 *
 **************************************************************************/

#include <stdio.h>
#include "cache.h"

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

/************************* public module variables ************************/

cacheinfo_ptr toplevel  = NULL;
cacheinfo_ptr currlevel = NULL;



/************************* private module variables ***********************/

/*
 * curr_path holds pointers to names of directories leading to the
 * current path.  For example, curr_path[0] would be the names of
 * the root directory, etc.  The concatenation of all the names
 * (with the appropriate "/"s appended) yield the current directory.
 *
 * curr_path_index is the index to the most recent directory name.
 *
 * Note that memory only needs to be allocated for the root.  All other
 * entries will be pointers into the fileinfo structure.
 */
#define MAX_PATH_LEN	20
static int curr_path_index = -1;
static char *curr_path[MAX_PATH_LEN];

#define MAX_LINESIZE	256	/* Max. size of a line in a dir listing */
#define MAX_LINKS	256	/* Max. # of traversed entries / level */

/************************* path handling functions ************************/

void cache_set_root(char *root)
/*
 * saves the root name.
 */
{
    int len = strlen(root);
    
    /*
     * save complete pathname of the root, without the "/" postfix
     */
    curr_path_index = 0;
    if (*root && (root[len-1] == '/')) len--;	
    curr_path[0] = (char *) malloc(len + 1);
    if (len) strncpy(curr_path[0], root, len);
    curr_path[0][len] = '\0';
#ifdef DEBUG
    printf("cache_set_root: Saving root as: '%s'\n", curr_path[0]);
#endif
}



void cache_next_dir(char *dir)
/*
 * saves the pointer to the dir name.
 */
{
    if (++curr_path_index >= MAX_PATH_LEN)
    {
	fprintf(stderr, "WARNING: cache_next_dir: out of stack space.\n");
	return;
    }

    curr_path[curr_path_index] = dir;
}



void cache_prev_dir(void)
/*
 * tells us to cd up.
 */
{
    if (curr_path_index <= 0)
    {
	fprintf(stderr, "WARNING: cache_prev_dir: stack underflow.\n");
	return;
    }

    curr_path_index--;
}



char *cache_make_filename(char *filename)
/*
 * Creates a string with the complete path for the given file.
 * The address of the string is returned.  This string should be
 * freed later.
 */
{
    char *s;
    int i, path_len = 0, total_len;

    for (i=0; i <= curr_path_index; i++)
	path_len += strlen(curr_path[i]) + 1;

    total_len = path_len + strlen(filename);

    s = (char *) malloc( total_len + 1);
    sprintf(s, "%s/", curr_path[0]);

    for (i = 1; i <= curr_path_index; i++)
    {
	strcat(s, curr_path[i]);
	strcat(s, "/");
    }

    strcat(s, filename);

    return s;
}



/************************* cache maintainance functions *******************/

int cache_init_level(cacheinfo_ptr *level, cacheinfo_ptr prev, char *filename)
/*
 * Given a ftp directory listing in file "filename" (which should not be
 * open), this function saves the names of the  files and directories in the
 * listing into the parameter *level which should be initially NULL.  The
 * parameter "prev" is a pointer to the cache item for the parent directory
 * -- if the root directory is being cached then "prev" should be NULL.
 *
 * The format of the file "filename" should be _exactly_ as produced by
 * an FTP data connection when given a "LIST" command.  This includes
 * the first line that gives the total.
 *
 * Memory will be allocated for the cacheinfo entry *level, as well
 * as for the fileinfo entries within.
 *
 * Returns TRUE if success and FALSE if file is unreadable.
 *
 */
{
    FILE *fp;
    char line[100];
    int c, i;
    int entries = -1;
    int len;
    fileinfo *fi;
    char *s;

    /*
     * open file and ready it for reading lines of info.
     */
    if ((fp=fopen(filename,"r")) == NULL)
    {
	fprintf(stderr,"cache_init_level: can't open file '%s'\n", filename);
	return FALSE;
    }

    while ( (c=fgetc(fp)) != EOF )
	if (c=='\n') entries++;		/* count # of lines */

    fseek(fp, 0, 0);
#ifdef DEBUG
    printf("cache_init_level: %d entries.\n", entries);
#endif
    fgets(line, 98, fp);		/* skip the first line */
	
    /*
     * initialize level
     */
    if (*level != NULL)
	fprintf(stderr, "WARNING: cache_init_level: level re-init.\n");

    *level = (cacheinfo *) malloc(sizeof(cacheinfo));
    (*level)->prev_level = prev;
    (*level)->num_entries = entries;
    (*level)->entry_arr = (fileinfo *) malloc(entries * sizeof(fileinfo));


    /*
     * read in entries
     */
    for (i=0; i < entries; i++)
    {
	fgets(line, 98, fp);
	len = strlen(line);

	/* check for newline */
	if (line[len-1] == '\n')
	    line[(len--) - 1] = '\0';
	else
	    fprintf(stderr, "WARNING: cache_init_level: incomplete entry.\n");

	/* If we got this in ASCII mode, there will be a linefeed as well */
	if (line[len-1]=='\r')
	    line[(len--) - 1] = '\0';

	fi = & ( (*level)->entry_arr[i] );
	fi->line = (char *) malloc(len+1);
	strcpy(fi->line, line);
	fi->next_level = NULL;

	fi->type = (line[0]=='-')? fitypFILE :
		   (line[0]=='d')? fitypDIRECTORY :
		   fitypLINK;

	fi->fCached = 0;

	/* point to name */
	s = fi->line + len - 1;
	while (*s && (*s != ' ')) s--;
	fi->name = s+1;
    }

    fclose(fp);
    return TRUE;
}


void cache_free_levels(cacheinfo_ptr *plevel)
/*
 * Recursively deallocates the given level and all levels below it.
 */
{
    int i;
    fileinfo *fi;

    for (i=0; i < (*plevel)->num_entries; i++)
    {
	fi = & ( (*plevel)->entry_arr[i] );
	free ( fi->line );
	if (fi->next_level)
	{
	    cache_free_levels(&(fi->next_level));
	}
    }

    free( (*plevel)->entry_arr );
    free( *plevel );
    *plevel = NULL;
}


/*************************** saving/loading caches **********************/

int cache_save_cache(char *filepath)
/*
 * Given the complete pathname of the directory cache file, this
 * function saves the directory cache to that file.
 *
 * Returns TRUE for success, FALSE otherwise.
 */
{
    FILE *fp;
    int stat;

    if ( (fp = fopen(filepath, "w")) == NULL )
    {
	fprintf(stderr,"cache_save_cache: can't open '%s'.\n", filepath);
	return FALSE;
    }

    /*
     * Save the 'root' of remote hierarchy, with "/" appended
     */
    fprintf(fp, "%s/\n", curr_path[0]);

    /*
     * Save the rest of the directory tree
     */
    stat = cache_save_levels(toplevel, fp);

    /*
     * Close file and exit
     */
    fflush(fp);
    fclose(fp);

    return stat;
}



int cache_save_levels(cacheinfo_ptr level, FILE *fp)
/*
 * Given a level "level", and file pointer to an open, writable
 * file, this function writes out the current level and all
 * child levels to the file.  The cache is output in using inorder
 * traversal.
 *
 * Returns TRUE if successful, FALSE otherwise.
 */
{
    fileinfo *fi;
    int i, nlinks = 0;

    /*
     * Print out number of items 
     */
    fprintf(fp, "%d\n", level->num_entries);

    /*
     * Print out all entries for this level
     */
    for (i=0; i < level->num_entries; i++)
    {
	fi = & (level->entry_arr[i]);
	fprintf(fp, "%s\n", fi->line);
	if (fi->next_level) nlinks++;
    }

    /*
     * Print out the links 
     */
    fprintf(fp, "%d\n", nlinks);
    for (i=0; i < level->num_entries; i++)
    {
	if (level->entry_arr[i].next_level)
	    fprintf(fp, "%d\n", i);
    }

   /*
    * Traverse and save other levels
    */
    for (i=0; i < level->num_entries; i++)
    {
	if (level->entry_arr[i].next_level)
	    cache_save_levels(level->entry_arr[i].next_level, fp);
    }

    return TRUE;
}



int cache_load_levels(cacheinfo_ptr *plevel, cacheinfo_ptr prev, FILE *fp)
/*
 * Given a double-pointer to a level "plevel", and a file pointer to an
 * open readable file, this function loads in the data for "plevel" and
 * all its children levels.  "prev" specifies a pointer to the parent
 * level, or NULL "plevel" is the root.  Memory is dynamically allocated
 * for "plevel" and all its children.
 *
 * Returns TRUE on success, FALSE otherwise.
 */
{
    int n_entries;
    int n_links;
    int i, len;
    int links[MAX_LINKS];
    char line[MAX_LINESIZE];
    int remove_newline(char *);
    fileinfo *fi;

    /*
     * Read #entries & alloc memory for this level
     */
    if (fscanf(fp, "%d\n", &n_entries)==EOF)
    {
	fprintf(stderr, "cache_load_levels: no entries!\n");
	return FALSE;
    }

    *plevel = (cacheinfo *) malloc(sizeof(cacheinfo));
    (*plevel)->prev_level = prev;
    (*plevel)->num_entries = n_entries;
    (*plevel)->entry_arr = (fileinfo *) malloc(n_entries * sizeof(fileinfo));

    /*
     * Read in entries
     */
    for (i=0; i < n_entries; i++)
    {
	char *s;

	fgets(line, MAX_LINESIZE-1, fp);
	len = remove_newline(line);
	fi = & ( (*plevel)->entry_arr[i] );
	fi->line = (char *) malloc(len+1);
	strcpy(fi->line, line);
	fi->next_level = NULL;
	fi->type = (line[0]=='-')? fitypFILE :
		   (line[0]=='d')? fitypDIRECTORY :
		   fitypLINK;
	s = fi->line + len - 1;
	while (*s && (*s != ' ')) s--;
	fi->name = s+1;
    }


    /*
     * Read in links
     */
    fscanf(fp, "%d\n", &n_links);
    if (n_links > MAX_LINKS)
    {
	fprintf(stderr, "cache_load_levels: #links > %d! Aborting!\n",
	    MAX_LINKS);
	return FALSE;
    }

    for (i=0; i < n_links; i++)
	fscanf(fp, "%d\n", &(links[i]));


    /*
     * Traverse the necessary levels
     */
    for (i=0; i < n_links; i++)
    {
	fi = & ( (*plevel)->entry_arr[links[i]] );
	cache_load_levels(& (fi->next_level), *plevel, fp);
    }

    return TRUE;
}



int cache_load_cache(char *filepath)
/*
 * Given the complete path to the cache file, this function loads in
 * the cached directory hierarchy into "toplevel".
 *
 * WARNING: This function will attempt to free the root stored in
 *          curr_path as well as toplevel.  These should either be
 *          NULL or point to valid data.
 *
 * Returns TRUE on success, FALSE otherwise.
 */
{
    FILE *fp;
    int len, stat;
    char line[MAX_LINESIZE];
    int remove_newline(char *);


    /*
     * Open file
     */
    if ( (fp=fopen(filepath,"r")) == NULL )
    {
	/* fprintf(stderr, "cache_load_cache: can't open '%s'.\n", filepath); */
	return FALSE;
    }

    /*
     * Free everything
     */
    if (curr_path[0])
	free(curr_path[0]);

    if (toplevel)
	cache_free_levels(&toplevel);


    /*
     * Read in root
     */
    fgets(line, MAX_LINESIZE-1, fp);
    remove_newline(line);
    cache_set_root(line);
     

    /*
     * Read in everything else
     */
    stat = cache_load_levels(&toplevel, NULL, fp);

    /*
     * Close file and exit
     */
    fclose(fp);
    return stat;
}


/*************************** misc functions ******************************/

int remove_newline(char *s)
/*
 * Given a string s, removes the newline at the end of s.
 * Returns length of modified string.
 */
{
    int len = strlen(s);

    if (s[len-1] == '\n')
	s[--len] = '\0';

    return len;
}


/*************************** debugging functions *************************/

void cache_print_level(cacheinfo_ptr level)
{
    fileinfo *fi;
    int ent;

    for (ent=0; ent < level->num_entries; ent++)
    {
	printf("%s:", level->entry_arr[ent].name);
	printf("'%s'\n", level->entry_arr[ent].line);
    }
}
