/* file_rw.c  -  consist of ReadFile and WriteFile procedures to be used
		 together with Motif applications.
		 (C) A. Stochniol,  1991 - 1993
		 Last revision: 9.03.1993
*/
/*
 * Copyright 1991 - 1993,  Andrzej Stochniol, London, UK
 *
 * ASEDIT text editor, both binary and source (hereafter, Software) is
 * copyrighted by Andrzej Stochniol (hereafter, AS) and ownership remains
 * with AS.
 *
 * AS grants you (hereafter, Licensee) a license to use the Software
 * for academic, research and internal business purposes only, without a
 * fee.  Licensee may distribute the binary and source code (if released)
 * to third parties provided that the copyright notice and this statement
 * appears on all copies and that no charge is associated with such copies.
 *
 * Licensee may make derivative works.  However, if Licensee distributes
 * any derivative work based on or derived from the Software, then
 * Licensee will:
 * (1) notify AS regarding its distribution of the derivative work, and
 * (2) clearly notify users that such derivative work is a modified version
 *      and not the original ASEDIT distributed by AS.
 *
 * Any Licensee wishing to make commercial use of the Software should
 * contact AS to negotiate an appropriate license for such commercial use.
 * Commercial use includes:
 * (1) integration of all or part of the source code into a product for sale
 *     or license by or on behalf of Licensee to third parties, or 
 * (2) distribution of the binary code or source code to third parties that
 *     need it to utilize a commercial product sold or licensed by or on 
 *     behalf of Licensee.
 *
 * A. STOCHNIOL MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS
 * SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR
 * IMPLIED WARRANTY.  IN NO EVENT SHALL A. STOCHNIOL BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *
 * 	Andrzej Stochniol	(A.Stochniol@ic.ac.uk)
 * 	30 Hatch Road
 * 	London SW16 4PN
 * 	UK
 */


#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <X11/Xatom.h>

#include <Xm/Xm.h>


#include "asedit.h"		/* the only reason to include that is to get access
				   to lstr.backupFileSuffix */

/* declare make_tmpnam function implemeted in version 1.2 of asedit */
#ifdef _NO_PROTO
extern char *make_tmpnam ();
#else  /* ! _NO_PROTO */

extern char *make_tmpnam (void);
#endif



/****************************  ReadFile  ***********************************
**
**		Open the specified file and reads its contents.
**		Returns a pointer to the file contents  if file exists and
**		open is sucessful (otherwise returns NULL).
**		If the file was opened in the read only mode
**		*read_only is True.
*/
#ifdef _NO_PROTO
char * ReadFile(file_name, read_only)
   char *file_name;
   Boolean *read_only;
#else  /* _NO_PROTO */

char * ReadFile(char *file_name, Boolean *read_only)
#endif
{
   struct stat statbuf;		/* Information on a file. */
   long file_length;		/* Length of file. 	  */
   FILE *fp = NULL;		/* Pointer to open file   */
   static char *file_string; /* pointer to the file contents */
   long n_read_items;		/* how many items (here bytes) were read in */

   *read_only = False;

   /* make sure the file is a regular text file and open it */
   if(stat(file_name, &statbuf) == 0  &&
	(statbuf.st_mode & S_IFMT) != S_IFREG )
   {
	fprintf(stderr, "\n%s: not a regular file.", file_name);
	return(NULL);
   }
   if ((fp = fopen(file_name, "r+")) == NULL)
	if ((fp = fopen(file_name, "r")) != NULL)
	{
		    /** fprintf(stderr, "Warning: file opened read only.\n"); we show that visually */
		    *read_only = True;
	}
	else    return(NULL);


   if (stat(file_name, &statbuf) == 0)
	 file_length = statbuf.st_size;
   else
	 file_length = 500000L; /* arbitrary file length */

   /* read the file string */
   file_string = XtMalloc((unsigned)(file_length+1));	/* +1 for NULL character */

   n_read_items = fread(file_string, sizeof(char), file_length, fp);
   file_string[n_read_items]=NULL;	/* adding the NULL character to the end
					   of the file_string */
   /* n_read_items was specially introduced because the file string should be
      null terminated. For DOS file file_length takes into account both line feed
      and carriege return, but during read operation they are translated into
      a single line feed character!!! (so n_read_items is smaller than file_length
      for the DOS file !!!) */

   /* close up the file */
   if (fclose(fp) != NULL) fprintf(stderr, "Warning: unable to close file.\n");

   return(file_string);

}  /* ReadFile */


/****************************  WriteFile  **********************************
	WriteFile  - writes a text pointed to by *file_string to the file
	file_name. To preserve hard links no intermediate temporary file is used for save result.
	If the file already exist it is overwritten without any message
	(this procedure is used mainly for saving option. If successful
	returns True.)
*/
#ifdef _NO_PROTO
Boolean WriteFile(file_name, file_string)
    char *file_name;
    char *file_string;
#else  /* _NO_PROTO */

Boolean WriteFile(char *file_name, char *file_string)
#endif
{
    FILE *fp;				/* Pointer to an open file. */
    FILE *tfp;           		/* Pointer to open temporary file. */
    Boolean result;
    int status;
    char system_call_buf[BUFSIZ];

    char *tempname;	/* Temporary file name .*/

    tempname = make_tmpnam();	/* use tmpnam function and consider tmpDir X resource
				   or TMPDIR environmental variable (if set) */

    /* Note: before asedit 1.2 we used mktemp procedure and a piece of code below; since
       asedit 1.2 we use make_tmpnam procedure that takes account of user specified
       tmdDir recource;
       Old piece of code follows (note that in the old code we didn't have to release
       memory allocated for tempname!):
	char tempname[25];
	strcpy(tempname,"/tmp/aseditXXXXXX");
	mktemp(tempname);
    **/

    /* to preserve hard links we are not using rename command any more; instead
       of that we are using copy command 
    */

    if ((tfp = fopen(tempname, "w")) == NULL) {
       fprintf(stderr, "Warning: unable to open temp file '%s', text not saved.\n", tempname);
       XtFree(tempname);
       return(False);
    }


    /* check if the file_name already existed, if yes make a backup of it (make the backup when 
       the user requested it by defining BACKUP_FILE_SUFFIX or backupFileSuffix X resource) */
    if ((fp = fopen(file_name, "r")) != NULL)
    {
	char *backupFileSuffix=NULL;		/* asedit v. 1.2 */	
        char *backname;

        /* if a user set backupFileSuffix recource use it, otherwise use
	   BACKUP_FILE_SUFFIX defined in the Imakefile/Makefile file
	*/
	backupFileSuffix = (char *)lstr.backupFileSuffix;
	if(!backupFileSuffix) 	/* it is NULL, use the install default */
		    backupFileSuffix = (char *)BACKUP_FILE_SUFFIX;
	/* Backup  file name.    */
	backname = XtMalloc(strlen(file_name) + strlen(backupFileSuffix) +1);

	fclose(fp);
	strcpy(backname, file_name);
	if(strlen(backupFileSuffix))
        {    /* make the backup of the file */
             strcat(backname, backupFileSuffix);
	     /* to preserve any possible hard link we copy the old file onto
	        the backup (instead of moving it)  */
	     /* unfortunately we lose the old date/time but we might set
		that properly in the future if needed
	     */
  	     sprintf (system_call_buf, "cp %s %s\0", file_name , backname);
  	     status = system(system_call_buf);
  	     if (status != 0)
	     {
	        fprintf(stderr, "Warning: unable to create backup file '%s'.\nFollowing file not safed: '%s'.\n", 
			backname, file_name);
	        XtFree(backname);
		XtFree(tempname);
	        return(False);
	     }
	     XtFree(backname);
	}
    }

    /* write to a temp file */
    fwrite(file_string, sizeof(char), strlen(file_string) , tfp);

    /* because asedit might be used to edit programs we are checking
       if the last line is a complete line (if it is not we add the line
       feed there to avoid any problems during compilation of the program,
       especially true for FORTRAN programs;
       it is the same what vi probably does)
    */
    if(file_string[strlen(file_string)-1] != '\n')
	fwrite("\n", sizeof(char), 1, tfp);

    result = True;
    /* flush and close the file */
    if (fflush(tfp) != NULL)
	{ result=False;  fprintf(stderr,"Warning: unable to flush file.\n"); }
    if (fclose(tfp) != NULL)
	{ result=False;  fprintf(stderr,"Warning: unable to close file.\n"); }

    if(!result) { XtFree(tempname); return(result); }

    /* Move the tempname to the saved file, but do it independent
     of filesystem boundaries; on some modern operating systems the move/rename command is
     not supported accross different file systems!! (e.g. IBM RS/6000) (/tmp probably lies
     in a different file system than the user data files)  */

    sprintf (system_call_buf, "cp %s %s\0", tempname, file_name);
    status = system(system_call_buf);
    unlink (tempname);
    if (status == 0) 	result = True;
    else
    {
        fprintf(stderr, "Warning: unable to save file.\n");
	result = False;
    }
    XtFree(tempname);
    return(result);

}   /* WriteFile */
