/*	$NetBSD: chacha_selftest.c,v 1.1 2020/07/25 22:46:34 riastradh Exp $	*/

/*-
 * Copyright (c) 2020 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "chacha_impl.h"

#ifdef _KERNEL

#include <lib/libkern/libkern.h>

#else

#include <stdio.h>
#include <string.h>

static void
hexdump(int (*prf)(const char *, ...) __printflike(1,2), const char *prefix,
    const void *buf, size_t len)
{
	const uint8_t *p = buf;
	size_t i;

	(*prf)("%s (%zu bytes @ %p)\n", prefix, len, buf);
	for (i = 0; i < len; i++) {
		if (i % 16 == 8)
			(*prf)("  ");
		else
			(*prf)(" ");
		(*prf)("%02hhx", p[i]);
		if ((i + 1) % 16 == 0)
			(*prf)("\n");
	}
	if (i % 16)
		(*prf)("\n");
}

#endif

/* https://tools.ietf.org/html/draft-strombergson-chacha-test-vectors-00 */
static int
chacha_core_selftest(const struct chacha_impl *ci)
{
	/* TC1, 32-byte key, rounds=12, keystream block 1 */
	static const uint8_t zero[32];
	static const uint8_t expected0[64] = {
		0x9b,0xf4,0x9a,0x6a, 0x07,0x55,0xf9,0x53,
		0x81,0x1f,0xce,0x12, 0x5f,0x26,0x83,0xd5,
		0x04,0x29,0xc3,0xbb, 0x49,0xe0,0x74,0x14,
		0x7e,0x00,0x89,0xa5, 0x2e,0xae,0x15,0x5f,
		0x05,0x64,0xf8,0x79, 0xd2,0x7a,0xe3,0xc0,
		0x2c,0xe8,0x28,0x34, 0xac,0xfa,0x8c,0x79,
		0x3a,0x62,0x9f,0x2c, 0xa0,0xde,0x69,0x19,
		0x61,0x0b,0xe8,0x2f, 0x41,0x13,0x26,0xbe,
	};
	/* TC7, 32-byte key, rounds=12, keystream block 2 */
	static const uint8_t k1[32] = {
		0x00,0x11,0x22,0x33, 0x44,0x55,0x66,0x77,
		0x88,0x99,0xaa,0xbb, 0xcc,0xdd,0xee,0xff,
		0xff,0xee,0xdd,0xcc, 0xbb,0xaa,0x99,0x88,
		0x77,0x66,0x55,0x44, 0x33,0x22,0x11,0x00,
	};
	static const uint8_t in1[16] = {
		0x01,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0x0f,0x1e,0x2d,0x3c, 0x4b,0x59,0x68,0x77,
	};
	static const uint8_t expected1[64] = {
		0xcd,0x9a,0x2a,0xa9, 0xea,0x93,0xc2,0x67,
		0x5e,0x82,0x88,0x14, 0x08,0xde,0x85,0x2c,
		0x62,0xfa,0x74,0x6a, 0x30,0xe5,0x2b,0x45,
		0xa2,0x69,0x62,0xcf, 0x43,0x51,0xe3,0x04,
		0xd3,0x13,0x20,0xbb, 0xd6,0xaa,0x6c,0xc8,
		0xf3,0x26,0x37,0xf9, 0x59,0x34,0xe4,0xc1,
		0x45,0xef,0xd5,0x62, 0x31,0xef,0x31,0x61,
		0x03,0x28,0x36,0xf4, 0x96,0x71,0x83,0x3e,
	};
	uint8_t out[64];
	int result = 0;

	(*ci->ci_chacha_core)(out, zero, zero, chacha_const32, 12);
	if (memcmp(out, expected0, 64)) {
		hexdump(printf, "chacha core 1", out, sizeof out);
		result = -1;
	}

	(*ci->ci_chacha_core)(out, in1, k1, chacha_const32, 12);
	if (memcmp(out, expected1, 64)) {
		hexdump(printf, "chacha core 2", out, sizeof out);
		result = -1;
	}

	return result;
}

static int
hchacha_selftest(const struct chacha_impl *ci)
{
	/* https://tools.ietf.org/html/draft-irtf-cfrg-xchacha-03, §2.2.1 */
	static const uint8_t k[32] = {
		0x00,0x01,0x02,0x03, 0x04,0x05,0x06,0x07,
		0x08,0x09,0x0a,0x0b, 0x0c,0x0d,0x0e,0x0f,
		0x10,0x11,0x12,0x13, 0x14,0x15,0x16,0x17,
		0x18,0x19,0x1a,0x1b, 0x1c,0x1d,0x1e,0x1f,
	};
	static const uint8_t in[16] = {
		0x00,0x00,0x00,0x09, 0x00,0x00,0x00,0x4a,
		0x00,0x00,0x00,0x00, 0x31,0x41,0x59,0x27,
	};
	static const uint8_t expected[32] = {
		0x82,0x41,0x3b,0x42, 0x27,0xb2,0x7b,0xfe,
		0xd3,0x0e,0x42,0x50, 0x8a,0x87,0x7d,0x73,
		0xa0,0xf9,0xe4,0xd5, 0x8a,0x74,0xa8,0x53,
		0xc1,0x2e,0xc4,0x13, 0x26,0xd3,0xec,0xdc,
	};
	uint8_t out[32];
	int result = 0;

	(*ci->ci_hchacha)(out, in, k, chacha_const32, 20);
	if (memcmp(out, expected, 32)) {
		hexdump(printf, "hchacha", out, sizeof out);
		result = -1;
	}

	return result;
}

static int
chacha_stream_selftest(const struct chacha_impl *ci)
{

	/* XXX */
	return 0;
}

static int
xchacha_stream_selftest(const struct chacha_impl *ci)
{
	/* https://tools.ietf.org/html/draft-irtf-cfrg-xchacha-03, A.2.1 */
	static const uint8_t k[32] = {
		0x80,0x81,0x82,0x83, 0x84,0x85,0x86,0x87,
		0x88,0x89,0x8a,0x8b, 0x8c,0x8d,0x8e,0x8f,
		0x90,0x91,0x92,0x93, 0x94,0x95,0x96,0x97,
		0x98,0x99,0x9a,0x9b, 0x9c,0x9d,0x9e,0x9f,
	};
	static const uint8_t nonce[24] = {
		0x40,0x41,0x42,0x43, 0x44,0x45,0x46,0x47,
		0x48,0x49,0x4a,0x4b, 0x4c,0x4d,0x4e,0x4f,
		0x50,0x51,0x52,0x53, 0x54,0x55,0x56,0x58,
	};
	static const uint8_t p[608] = {
		0x54,0x68,0x65,0x20, 0x64,0x68,0x6f,0x6c,
		0x65,0x20,0x28,0x70, 0x72,0x6f,0x6e,0x6f,
		0x75,0x6e,0x63,0x65, 0x64,0x20,0x22,0x64,
		0x6f,0x6c,0x65,0x22, 0x29,0x20,0x69,0x73,
		0x20,0x61,0x6c,0x73, 0x6f,0x20,0x6b,0x6e,
		0x6f,0x77,0x6e,0x20, 0x61,0x73,0x20,0x74,
		0x68,0x65,0x20,0x41, 0x73,0x69,0x61,0x74,
		0x69,0x63,0x20,0x77, 0x69,0x6c,0x64,0x20,
		0x64,0x6f,0x67,0x2c, 0x20,0x72,0x65,0x64,
		0x20,0x64,0x6f,0x67, 0x2c,0x20,0x61,0x6e,
		0x64,0x20,0x77,0x68, 0x69,0x73,0x74,0x6c,
		0x69,0x6e,0x67,0x20, 0x64,0x6f,0x67,0x2e,
		0x20,0x49,0x74,0x20, 0x69,0x73,0x20,0x61,
		0x62,0x6f,0x75,0x74, 0x20,0x74,0x68,0x65,
		0x20,0x73,0x69,0x7a, 0x65,0x20,0x6f,0x66,
		0x20,0x61,0x20,0x47, 0x65,0x72,0x6d,0x61,
		0x6e,0x20,0x73,0x68, 0x65,0x70,0x68,0x65,
		0x72,0x64,0x20,0x62, 0x75,0x74,0x20,0x6c,
		0x6f,0x6f,0x6b,0x73, 0x20,0x6d,0x6f,0x72,
		0x65,0x20,0x6c,0x69, 0x6b,0x65,0x20,0x61,
		0x20,0x6c,0x6f,0x6e, 0x67,0x2d,0x6c,0x65,
		0x67,0x67,0x65,0x64, 0x20,0x66,0x6f,0x78,
		0x2e,0x20,0x54,0x68, 0x69,0x73,0x20,0x68,
		0x69,0x67,0x68,0x6c, 0x79,0x20,0x65,0x6c,
		0x75,0x73,0x69,0x76, 0x65,0x20,0x61,0x6e,
		0x64,0x20,0x73,0x6b, 0x69,0x6c,0x6c,0x65,
		0x64,0x20,0x6a,0x75, 0x6d,0x70,0x65,0x72,
		0x20,0x69,0x73,0x20, 0x63,0x6c,0x61,0x73,
		0x73,0x69,0x66,0x69, 0x65,0x64,0x20,0x77,
		0x69,0x74,0x68,0x20, 0x77,0x6f,0x6c,0x76,
		0x65,0x73,0x2c,0x20, 0x63,0x6f,0x79,0x6f,
		0x74,0x65,0x73,0x2c, 0x20,0x6a,0x61,0x63,
		0x6b,0x61,0x6c,0x73, 0x2c,0x20,0x61,0x6e,
		0x64,0x20,0x66,0x6f, 0x78,0x65,0x73,0x20,
		0x69,0x6e,0x20,0x74, 0x68,0x65,0x20,0x74,
		0x61,0x78,0x6f,0x6e, 0x6f,0x6d,0x69,0x63,
		0x20,0x66,0x61,0x6d, 0x69,0x6c,0x79,0x20,
		0x43,0x61,0x6e,0x69, 0x64,0x61,0x65,0x2e,

		0x54,0x68,0x65,0x20, 0x64,0x68,0x6f,0x6c,
		0x65,0x20,0x28,0x70, 0x72,0x6f,0x6e,0x6f,
		0x75,0x6e,0x63,0x65, 0x64,0x20,0x22,0x64,
		0x6f,0x6c,0x65,0x22, 0x29,0x20,0x69,0x73,
		0x20,0x61,0x6c,0x73, 0x6f,0x20,0x6b,0x6e,
		0x6f,0x77,0x6e,0x20, 0x61,0x73,0x20,0x74,
		0x68,0x65,0x20,0x41, 0x73,0x69,0x61,0x74,
		0x69,0x63,0x20,0x77, 0x69,0x6c,0x64,0x20,
		0x64,0x6f,0x67,0x2c, 0x20,0x72,0x65,0x64,
		0x20,0x64,0x6f,0x67, 0x2c,0x20,0x61,0x6e,
		0x64,0x20,0x77,0x68, 0x69,0x73,0x74,0x6c,
		0x69,0x6e,0x67,0x20, 0x64,0x6f,0x67,0x2e,
		0x20,0x49,0x74,0x20, 0x69,0x73,0x20,0x61,
		0x62,0x6f,0x75,0x74, 0x20,0x74,0x68,0x65,
		0x20,0x73,0x69,0x7a, 0x65,0x20,0x6f,0x66,
		0x20,0x61,0x20,0x47, 0x65,0x72,0x6d,0x61,
		0x6e,0x20,0x73,0x68, 0x65,0x70,0x68,0x65,
		0x72,0x64,0x20,0x62, 0x75,0x74,0x20,0x6c,
		0x6f,0x6f,0x6b,0x73, 0x20,0x6d,0x6f,0x72,
		0x65,0x20,0x6c,0x69, 0x6b,0x65,0x20,0x61,
		0x20,0x6c,0x6f,0x6e, 0x67,0x2d,0x6c,0x65,
		0x67,0x67,0x65,0x64, 0x20,0x66,0x6f,0x78,
		0x2e,0x20,0x54,0x68, 0x69,0x73,0x20,0x68,
		0x69,0x67,0x68,0x6c, 0x79,0x20,0x65,0x6c,
		0x75,0x73,0x69,0x76, 0x65,0x20,0x61,0x6e,
		0x64,0x20,0x73,0x6b, 0x69,0x6c,0x6c,0x65,
		0x64,0x20,0x6a,0x75, 0x6d,0x70,0x65,0x72,
		0x20,0x69,0x73,0x20, 0x63,0x6c,0x61,0x73,
		0x73,0x69,0x66,0x69, 0x65,0x64,0x20,0x77,
		0x69,0x74,0x68,0x20, 0x77,0x6f,0x6c,0x76,
		0x65,0x73,0x2c,0x20, 0x63,0x6f,0x79,0x6f,
		0x74,0x65,0x73,0x2c, 0x20,0x6a,0x61,0x63,
		0x6b,0x61,0x6c,0x73, 0x2c,0x20,0x61,0x6e,
		0x64,0x20,0x66,0x6f, 0x78,0x65,0x73,0x20,
		0x69,0x6e,0x20,0x74, 0x68,0x65,0x20,0x74,
		0x61,0x78,0x6f,0x6e, 0x6f,0x6d,0x69,0x63,
		0x20,0x66,0x61,0x6d, 0x69,0x6c,0x79,0x20,
		0x43,0x61,0x6e,0x69, 0x64,0x61,0x65,0x2e,
	};
	static const uint8_t expected[608] = {
		0x45,0x59,0xab,0xba, 0x4e,0x48,0xc1,0x61,
		0x02,0xe8,0xbb,0x2c, 0x05,0xe6,0x94,0x7f,
		0x50,0xa7,0x86,0xde, 0x16,0x2f,0x9b,0x0b,
		0x7e,0x59,0x2a,0x9b, 0x53,0xd0,0xd4,0xe9,
		0x8d,0x8d,0x64,0x10, 0xd5,0x40,0xa1,0xa6,
		0x37,0x5b,0x26,0xd8, 0x0d,0xac,0xe4,0xfa,
		0xb5,0x23,0x84,0xc7, 0x31,0xac,0xbf,0x16,
		0xa5,0x92,0x3c,0x0c, 0x48,0xd3,0x57,0x5d,
		0x4d,0x0d,0x2c,0x67, 0x3b,0x66,0x6f,0xaa,
		0x73,0x10,0x61,0x27, 0x77,0x01,0x09,0x3a,
		0x6b,0xf7,0xa1,0x58, 0xa8,0x86,0x42,0x92,
		0xa4,0x1c,0x48,0xe3, 0xa9,0xb4,0xc0,0xda,
		0xec,0xe0,0xf8,0xd9, 0x8d,0x0d,0x7e,0x05,
		0xb3,0x7a,0x30,0x7b, 0xbb,0x66,0x33,0x31,
		0x64,0xec,0x9e,0x1b, 0x24,0xea,0x0d,0x6c,
		0x3f,0xfd,0xdc,0xec, 0x4f,0x68,0xe7,0x44,
		0x30,0x56,0x19,0x3a, 0x03,0xc8,0x10,0xe1,
		0x13,0x44,0xca,0x06, 0xd8,0xed,0x8a,0x2b,
		0xfb,0x1e,0x8d,0x48, 0xcf,0xa6,0xbc,0x0e,
		0xb4,0xe2,0x46,0x4b, 0x74,0x81,0x42,0x40,
		0x7c,0x9f,0x43,0x1a, 0xee,0x76,0x99,0x60,
		0xe1,0x5b,0xa8,0xb9, 0x68,0x90,0x46,0x6e,
		0xf2,0x45,0x75,0x99, 0x85,0x23,0x85,0xc6,
		0x61,0xf7,0x52,0xce, 0x20,0xf9,0xda,0x0c,
		0x09,0xab,0x6b,0x19, 0xdf,0x74,0xe7,0x6a,
		0x95,0x96,0x74,0x46, 0xf8,0xd0,0xfd,0x41,
		0x5e,0x7b,0xee,0x2a, 0x12,0xa1,0x14,0xc2,
		0x0e,0xb5,0x29,0x2a, 0xe7,0xa3,0x49,0xae,
		0x57,0x78,0x20,0xd5, 0x52,0x0a,0x1f,0x3f,
		0xb6,0x2a,0x17,0xce, 0x6a,0x7e,0x68,0xfa,
		0x7c,0x79,0x11,0x1d, 0x88,0x60,0x92,0x0b,
		0xc0,0x48,0xef,0x43, 0xfe,0x84,0x48,0x6c,
		0xcb,0x87,0xc2,0x5f, 0x0a,0xe0,0x45,0xf0,
		0xcc,0xe1,0xe7,0x98, 0x9a,0x9a,0xa2,0x20,
		0xa2,0x8b,0xdd,0x48, 0x27,0xe7,0x51,0xa2,
		0x4a,0x6d,0x5c,0x62, 0xd7,0x90,0xa6,0x63,
		0x93,0xb9,0x31,0x11, 0xc1,0xa5,0x5d,0xd7,
		0x42,0x1a,0x10,0x18, 0x49,0x74,0xc7,0xc5,

		0x08,0x38,0x2d,0x64, 0x35,0x8d,0x21,0x77,
		0x2e,0xb9,0x73,0xa8, 0x8f,0xb6,0x2b,0xf8,
		0xce,0xfa,0xb4,0xca, 0x6f,0x0c,0x26,0xbb,
		0x7f,0xd6,0x6d,0xb2, 0xa0,0xbe,0xb0,0x5a,
		0x1a,0x6e,0x39,0xcb, 0xd5,0xda,0xf2,0xfc,
		0x0b,0x74,0x31,0x3d, 0x2e,0xcd,0x5f,0x94,
		0xc2,0x9f,0x30,0xdb, 0x11,0x5e,0x41,0x53,
		0x8c,0x6d,0x30,0xba, 0x97,0xa0,0xc5,0x07,
		0x70,0x78,0x02,0x5a, 0xc1,0x69,0x70,0x8f,
		0x22,0x85,0xcb,0x98, 0xbc,0x6a,0x51,0xfb,
		0xc6,0xa7,0xc3,0x3d, 0x76,0xe4,0x93,0x9a,
		0x21,0xe2,0xc6,0x12, 0xe1,0x3a,0xcc,0xfb,
		0x6f,0xa6,0x57,0xc0, 0x09,0x8c,0x6f,0xf3,
		0x8d,0x83,0x21,0x1b, 0x71,0xa9,0xc1,0x93,
		0x88,0x35,0xfc,0x18, 0x1f,0x94,0xa2,0x57,
		0x3a,0x4e,0xd0,0xc0, 0xbc,0x92,0xa7,0x9c,
		0x52,0x8a,0x82,0x9d, 0x44,0x75,0x7b,0xa0,
		0xcf,0x3d,0x2d,0xbf, 0xf9,0x6f,0x71,0x56,
		0x38,0xb0,0x63,0x5e, 0x55,0xcd,0x28,0x12,
		0xc5,0xea,0x52,0xf4, 0xdc,0xf7,0xdc,0x3d,
		0xd8,0x96,0x09,0xe8, 0x2a,0xcc,0x00,0x16,
		0x88,0x77,0x82,0x10, 0xed,0x7d,0xd8,0x8b,
		0xf5,0xd3,0xe1,0xfc, 0x49,0x66,0x36,0x8d,
		0x55,0xd2,0x33,0xb8, 0x6d,0xff,0xe3,0xd3,
		0x55,0x80,0x0e,0xd8, 0x95,0x32,0x32,0x55,
		0x83,0xe7,0x58,0x6f, 0xec,0xc3,0x8c,0xf8,
		0x52,0x16,0xdc,0x0d, 0x29,0x02,0xe5,0x27,
		0x35,0xc2,0xbb,0xe2, 0xe2,0x3b,0xf5,0x19,
		0xcd,0x44,0x83,0xe8, 0x21,0x55,0xd0,0x10,
		0x15,0x68,0x8e,0x46, 0xa3,0x2f,0xa5,0x7c,
		0xa8,0x2c,0xc6,0x8f, 0x14,0xcd,0xb3,0x79,
		0x92,0x32,0x71,0xac, 0xd9,0xaf,0x9c,0x4d,
		0x00,0x88,0xd1,0x42, 0xd5,0x23,0xfa,0xe6,
		0x7f,0x38,0xa2,0x56, 0x99,0xbe,0x6f,0xcf,
		0xe0,0xaa,0x44,0x11, 0x8a,0xc8,0x3a,0x99,
		0x48,0x6d,0x33,0x0e, 0x94,0xf2,0xb9,0x87,
		0xed,0x4f,0x6a,0x9c, 0x33,0x93,0x6d,0xe4,
		0x92,0x76,0xab,0xfa, 0xce,0x5b,0x17,0x14,
	};
	uint8_t c[608];
	unsigned i;
	int result = 0;

	/*
	 * 608 = 96 (mod 256)
	 * 607 = 95 (mod 256), = 7 (mod 8)
	 * 543 = 31 (mod 256), = 7 (mod 8)
	 * 511 = 255 (mod 256), = 7 (mod 8)
	 *
	 * This exercises several branches when there are special cases
	 * for integral numbers of 4-byte words, integral numbers of
	 * 64-byte blocks, and integral numbers of 256-byte chunks.
	 */

	(*ci->ci_xchacha_stream)(c, 608, 0, nonce, k, 20);
	for (i = 0; i < 608; i++)
		c[i] ^= p[i];
	if (memcmp(c, expected, 608)) {
		for (i = 0; i < 608; i++)
			c[i] ^= p[i];
		hexdump(printf, "xchacha_stream", c, 608);
		for (i = 0; i < 608; i++)
			c[i] = expected[i] ^ p[i];
		hexdump(printf, "expected", c, 608);
		result = -1;
	}

	(*ci->ci_xchacha_stream)(c, 607, 0, nonce, k, 20);
	for (i = 0; i < 607; i++)
		c[i] ^= p[i];
	if (memcmp(c, expected, 607)) {
		for (i = 0; i < 607; i++)
			c[i] ^= p[i];
		hexdump(printf, "xchacha_stream", c, 607);
		for (i = 0; i < 607; i++)
			c[i] = expected[i] ^ p[i];
		hexdump(printf, "expected", c, 607);
		result = -1;
	}

	(*ci->ci_xchacha_stream)(c, 543, 0, nonce, k, 20);
	for (i = 0; i < 543; i++)
		c[i] ^= p[i];
	if (memcmp(c, expected, 543)) {
		for (i = 0; i < 543; i++)
			c[i] ^= p[i];
		hexdump(printf, "xchacha_stream", c, 543);
		for (i = 0; i < 543; i++)
			c[i] = expected[i] ^ p[i];
		hexdump(printf, "expected", c, 543);
		result = -1;
	}

	(*ci->ci_xchacha_stream)(c, 511, 0, nonce, k, 20);
	for (i = 0; i < 511; i++)
		c[i] ^= p[i];
	if (memcmp(c, expected, 511)) {
		for (i = 0; i < 511; i++)
			c[i] ^= p[i];
		hexdump(printf, "xchacha_stream", c, 511);
		for (i = 0; i < 511; i++)
			c[i] = expected[i] ^ p[i];
		hexdump(printf, "expected", c, 511);
		result = -1;
	}

	(*ci->ci_xchacha_stream)(c, 63, 0, nonce, k, 20);
	for (i = 0; i < 63; i++)
		c[i] ^= p[i];
	if (memcmp(c, expected, 63)) {
		for (i = 0; i < 63; i++)
			c[i] ^= p[i];
		hexdump(printf, "xchacha_stream", c, 63);
		for (i = 0; i < 63; i++)
			c[i] = expected[i] ^ p[i];
		hexdump(printf, "expected", c, 63);
		result = -1;
	}

	(*ci->ci_xchacha_stream_xor)(c, p, 608, 0, nonce, k, 20);
	if (memcmp(c, expected, 608)) {
		hexdump(printf, "xchacha_stream_xor", c, 608);
		hexdump(printf, "expected", expected, 608);
		result = -1;
	}

	memset(c, 0, sizeof c);
	(*ci->ci_xchacha_stream_xor)(c, p, 607, 0, nonce, k, 20);
	if (memcmp(c, expected, 607)) {
		hexdump(printf, "xchacha_stream_xor", c, 607);
		hexdump(printf, "expected", expected, 607);
		result = -1;
	}

	memset(c, 0, sizeof c);
	(*ci->ci_xchacha_stream_xor)(c, p, 543, 0, nonce, k, 20);
	if (memcmp(c, expected, 543)) {
		hexdump(printf, "xchacha_stream_xor", c, 543);
		hexdump(printf, "expected", expected, 543);
		result = -1;
	}

	memset(c, 0, sizeof c);
	(*ci->ci_xchacha_stream_xor)(c, p, 511, 0, nonce, k, 20);
	if (memcmp(c, expected, 511)) {
		hexdump(printf, "xchacha_stream_xor", c, 511);
		hexdump(printf, "expected", expected, 511);
		result = -1;
	}

	memset(c, 0, sizeof c);
	(*ci->ci_xchacha_stream_xor)(c, p, 63, 0, nonce, k, 20);
	if (memcmp(c, expected, 63)) {
		hexdump(printf, "xchacha_stream_xor", c, 63);
		hexdump(printf, "expected", expected, 63);
		result = -1;
	}

	return result;
}

int
chacha_selftest(const struct chacha_impl *ci)
{
	int result = 0;

	result |= chacha_core_selftest(ci);
	result |= chacha_stream_selftest(ci);
	result |= hchacha_selftest(ci);
	result |= xchacha_stream_selftest(ci);

	return result;
}
