/*	$NetBSD: t_cgd_3des.c,v 1.2 2017/01/13 21:30:39 christos Exp $	*/
/*-
 * Copyright (c) 2016 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Alexander Nasonov.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/sysctl.h>

#include <atf-c.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <util.h>

#include <dev/cgdvar.h>

#include <rump/rump.h>
#include <rump/rump_syscalls.h>

#include "h_macros.h"

#define SECSIZE 512

struct testvec {
	unsigned int blkno;
	const uint8_t *ptxt;	/* PlainText  */
	const uint8_t *ctxt;	/* CipherText */
};

/*
 * 192 bits CBC key, NUL terminated.
 */
static const char c3des_cbc_192_key[25] = {
	0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, /* ABCDEFGH */
	0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, /* IJKLMNOP */
	0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, /* QRSTUVWX */
	0
};

static const uint8_t c3des_cbc_ptxt[SECSIZE] =
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop";

/*
 * IV method encblkno1, blkno 0.
 */
static const uint8_t c3des_cbc_192_encblkno1_vec0_ctxt[SECSIZE] = {
	0x19, 0x92, 0xc8, 0xce, 0xdf, 0xa3, 0x14, 0xef,
	0xff, 0x88, 0x9f, 0x01, 0xfa, 0x6f, 0xfa, 0xa6,
	0xdd, 0x2b, 0x43, 0x67, 0xfa, 0xce, 0x37, 0x95,
	0x73, 0x4d, 0x18, 0x33, 0x0c, 0x29, 0xb6, 0xbb,
	0x37, 0x77, 0x31, 0x74, 0xf6, 0x62, 0x03, 0xd2,
	0x78, 0x13, 0x55, 0xf6, 0x58, 0x49, 0xaf, 0x2a,
	0x15, 0x4c, 0xc2, 0x4a, 0x55, 0x99, 0x82, 0xb9,
	0xfb, 0x8b, 0x4f, 0x92, 0xe3, 0xbc, 0x9b, 0x09,
	0x42, 0x7b, 0x5f, 0x54, 0xed, 0xf0, 0xcb, 0x5d,
	0x93, 0xba, 0x09, 0x4b, 0x20, 0xf3, 0xe6, 0x44,
	0x30, 0x5e, 0x9e, 0xfc, 0x7a, 0x3c, 0x7d, 0x11,
	0x63, 0xea, 0x40, 0x94, 0xaa, 0xd0, 0xa9, 0xf6,
	0xc7, 0x1e, 0x8f, 0xc8, 0xa6, 0x2c, 0xf7, 0xeb,
	0x51, 0x26, 0xdc, 0xf8, 0x73, 0xf9, 0xb4, 0xa8,
	0x80, 0x4a, 0xe5, 0x6f, 0xb6, 0x33, 0x13, 0x6b,
	0x1b, 0x7d, 0x00, 0xde, 0x44, 0x7e, 0x26, 0xa2,
	0x82, 0xa7, 0x80, 0x16, 0x34, 0xde, 0xb9, 0x82,
	0x4c, 0x42, 0x8e, 0x0d, 0x48, 0x7e, 0x38, 0xbd,
	0x1d, 0x7d, 0x98, 0xbb, 0x11, 0x8a, 0x72, 0x14,
	0x4e, 0xaa, 0xd0, 0xef, 0x4d, 0x7f, 0xa3, 0xa6,
	0xfc, 0x85, 0x9d, 0x74, 0x63, 0x9d, 0xe4, 0x5c,
	0xf7, 0xa8, 0xd0, 0xd7, 0x95, 0xb4, 0x28, 0x64,
	0x41, 0x2d, 0x5d, 0xd9, 0xba, 0x79, 0xa7, 0xb3,
	0x9c, 0x16, 0xfa, 0xb8, 0x10, 0x5d, 0x1d, 0xd4,
	0xce, 0xad, 0x67, 0x27, 0x91, 0x8a, 0xb3, 0xbc,
	0x37, 0x20, 0x95, 0xac, 0xf7, 0x0d, 0xe0, 0x1e,
	0x59, 0xa7, 0xe5, 0x81, 0x82, 0x6a, 0x71, 0x07,
	0x85, 0x43, 0x43, 0xdb, 0xbf, 0x56, 0xb0, 0x0a,
	0x4c, 0xf1, 0xcd, 0xcd, 0xa3, 0x9a, 0x10, 0x8e,
	0x0c, 0xe2, 0x6d, 0xf2, 0x16, 0xd0, 0x4c, 0xac,
	0xf9, 0xfc, 0xc9, 0x56, 0x1f, 0x31, 0x89, 0x1c,
	0xfa, 0xb7, 0x49, 0xea, 0x69, 0x91, 0xfe, 0x45,
	0x96, 0x5e, 0x45, 0xc3, 0x2c, 0xb1, 0x40, 0xd9,
	0x1f, 0x82, 0x3f, 0xc1, 0x45, 0x7c, 0x39, 0x72,
	0x6f, 0x52, 0xe4, 0xaf, 0x15, 0xa4, 0xe2, 0xd4,
	0xa1, 0xa4, 0xb2, 0xb5, 0x4a, 0x0b, 0xad, 0xe4,
	0x1e, 0x5c, 0x26, 0x62, 0x81, 0x78, 0x3e, 0xd3,
	0x6a, 0x98, 0x94, 0x2a, 0x00, 0xa7, 0xe4, 0x04,
	0x9d, 0x9a, 0xfc, 0xcf, 0xad, 0x2b, 0xba, 0x9b,
	0x40, 0x1e, 0x71, 0x3a, 0xb6, 0x92, 0xc4, 0xc5,
	0x56, 0x58, 0x92, 0x2a, 0x69, 0xbe, 0x0f, 0xb0,
	0x91, 0xae, 0xaa, 0x3f, 0x07, 0xe8, 0xf9, 0x71,
	0x20, 0x06, 0xed, 0xe0, 0x80, 0xec, 0xc9, 0xe7,
	0x54, 0xaa, 0xaa, 0xf4, 0x4c, 0xb2, 0x34, 0xf7,
	0x8a, 0x76, 0xc2, 0x4a, 0xae, 0x71, 0x7a, 0x07,
	0xd7, 0xec, 0x75, 0x2f, 0x8a, 0x99, 0x59, 0x13,
	0xd0, 0x8d, 0x18, 0x69, 0x0d, 0xd9, 0x39, 0x73,
	0x2b, 0xd0, 0xa3, 0xbc, 0x9e, 0x29, 0x4d, 0x88,
	0xff, 0x98, 0x02, 0xb4, 0xcf, 0xa1, 0xf9, 0x2a,
	0xa6, 0xef, 0x7c, 0x72, 0x26, 0x4e, 0xd7, 0xdf,
	0xec, 0x3a, 0xbc, 0x8e, 0xe6, 0xb3, 0x2b, 0x43,
	0xcd, 0x67, 0x8b, 0x72, 0x00, 0x6f, 0xe5, 0x85,
	0xe2, 0x2a, 0x4c, 0x8d, 0x02, 0x44, 0x6b, 0x7a,
	0x89, 0x7a, 0x18, 0x3b, 0xc8, 0x9c, 0x8d, 0x60,
	0xec, 0x79, 0x58, 0x15, 0x98, 0x71, 0x4b, 0x1a,
	0x34, 0x69, 0x96, 0xd0, 0x0f, 0x01, 0x27, 0x2e,
	0x19, 0x02, 0xf0, 0x17, 0x8c, 0x89, 0xbf, 0x05,
	0xf0, 0xfe, 0xc3, 0xe6, 0x90, 0x9d, 0xa2, 0xb1,
	0x40, 0x06, 0x7e, 0xcd, 0x20, 0x7e, 0x5f, 0x54,
	0x31, 0xfb, 0x79, 0x84, 0x47, 0x38, 0x71, 0x69,
	0xe1, 0xd5, 0x4e, 0x84, 0xa3, 0x2b, 0x4a, 0x86,
	0xc2, 0x21, 0x5b, 0x15, 0xc3, 0x63, 0xbb, 0xc5,
	0x5c, 0xc1, 0xfb, 0x31, 0x3a, 0x4d, 0xb1, 0x9e,
	0xe1, 0xd8, 0x67, 0x4b, 0x08, 0x42, 0xc4, 0xe8,
};

/*
 * IV method encblkno1, blkno 1.
 */
static const uint8_t c3des_cbc_192_encblkno1_vec1_ctxt[SECSIZE] = {
	0x1d, 0x65, 0xdf, 0x01, 0x9b, 0x24, 0xa5, 0x10,
	0x94, 0x9a, 0x5b, 0x81, 0x96, 0x4e, 0xa3, 0x42,
	0x42, 0xd5, 0x05, 0x52, 0xab, 0x3c, 0x67, 0x40,
	0x79, 0xf9, 0x4b, 0x58, 0x39, 0xf6, 0xd0, 0x97,
	0x48, 0xf4, 0x77, 0xb8, 0xac, 0xe2, 0x10, 0x66,
	0xa8, 0x04, 0x0a, 0x1e, 0xa6, 0xbb, 0x4c, 0xd9,
	0x5d, 0x0c, 0x11, 0xb5, 0xe0, 0x26, 0x84, 0x50,
	0x10, 0x80, 0xbf, 0xd6, 0xdc, 0x82, 0x53, 0x0a,
	0xcf, 0xf6, 0xd3, 0x07, 0x45, 0xb0, 0x8e, 0x36,
	0x2e, 0x60, 0x0f, 0xd0, 0xc1, 0xb9, 0xd8, 0x29,
	0x6e, 0x13, 0x8e, 0xc1, 0xa8, 0x63, 0x20, 0xe0,
	0x8d, 0x47, 0x8b, 0xf9, 0xa0, 0x60, 0x55, 0x53,
	0x1d, 0xaf, 0x43, 0x46, 0xe5, 0x10, 0xd5, 0xcd,
	0x91, 0x9e, 0x11, 0x4a, 0x6f, 0x6a, 0x13, 0xdf,
	0xee, 0x7a, 0x88, 0xbe, 0x59, 0x96, 0xdb, 0x65,
	0x25, 0x57, 0x9e, 0x82, 0xad, 0xc2, 0xd6, 0x28,
	0x96, 0xb3, 0x7f, 0x57, 0x5d, 0xb2, 0xfa, 0x60,
	0x43, 0x22, 0xa5, 0x33, 0x14, 0x99, 0x8f, 0x68,
	0x5a, 0x7f, 0xaf, 0x9e, 0xe9, 0x23, 0x57, 0x9b,
	0x52, 0xe9, 0x20, 0x59, 0x26, 0x89, 0x9b, 0x59,
	0xb0, 0xee, 0xe8, 0x6d, 0x06, 0x8c, 0x01, 0xc2,
	0xea, 0xbc, 0x7d, 0x93, 0x3f, 0x79, 0x7f, 0xeb,
	0x57, 0xc9, 0x0a, 0xca, 0x37, 0x81, 0xa7, 0x82,
	0xde, 0x37, 0x7d, 0x69, 0x01, 0xaa, 0x19, 0x98,
	0x26, 0xfe, 0x06, 0x83, 0xeb, 0x9d, 0x26, 0xdc,
	0x04, 0x5d, 0xc9, 0x05, 0xee, 0x1a, 0xd3, 0xeb,
	0x20, 0x8c, 0xb7, 0x99, 0x75, 0xe0, 0x19, 0x98,
	0xca, 0x83, 0xae, 0x94, 0x28, 0xbf, 0x47, 0x42,
	0x92, 0x05, 0x8c, 0xaa, 0xeb, 0x99, 0x0f, 0xcc,
	0x33, 0x79, 0x24, 0x62, 0xa0, 0x7a, 0x65, 0xcb,
	0x53, 0xb7, 0x86, 0x0d, 0xcb, 0x44, 0x2d, 0xbf,
	0xe8, 0x5d, 0x62, 0xeb, 0x21, 0x4d, 0x35, 0x86,
	0x56, 0x6c, 0x51, 0xff, 0xa3, 0x45, 0xcc, 0x88,
	0x09, 0x43, 0x08, 0x97, 0x13, 0x7d, 0x00, 0xd8,
	0x82, 0x2d, 0xbe, 0xbe, 0x44, 0x0c, 0x2c, 0xa4,
	0x4f, 0x84, 0x07, 0x20, 0x9c, 0x3f, 0xf6, 0x5b,
	0x9e, 0xe8, 0x68, 0x40, 0xd3, 0x64, 0x8f, 0xb4,
	0x9e, 0xac, 0xc6, 0x41, 0x11, 0xda, 0xf2, 0x60,
	0xfa, 0x29, 0x9d, 0x26, 0x68, 0x5b, 0x79, 0x3a,
	0xd1, 0x66, 0x78, 0xca, 0x80, 0x87, 0xae, 0xab,
	0x7b, 0x29, 0x3c, 0xb0, 0xe6, 0xa2, 0x6b, 0x24,
	0x81, 0xeb, 0x51, 0xf9, 0xcb, 0x4a, 0x08, 0x37,
	0x2a, 0x75, 0xb5, 0xd3, 0xb3, 0x8f, 0x3d, 0x13,
	0x11, 0x0c, 0xa9, 0xf7, 0xf6, 0x57, 0x7e, 0xb7,
	0xa6, 0x22, 0xe8, 0x13, 0xfd, 0xf1, 0x6a, 0xe9,
	0xc1, 0x94, 0xa6, 0xf5, 0xa5, 0xec, 0xfa, 0x31,
	0xd2, 0x66, 0x8f, 0xe3, 0x6e, 0x9a, 0xaa, 0xb0,
	0xe3, 0x04, 0x09, 0x00, 0x1e, 0x67, 0x3c, 0xbe,
	0x2a, 0x8c, 0xd5, 0x1f, 0x4f, 0x55, 0x2c, 0x1d,
	0x26, 0x7f, 0xc9, 0x27, 0x00, 0x88, 0x7d, 0x45,
	0x4e, 0xe1, 0x36, 0xf6, 0xf5, 0xa8, 0xd4, 0xef,
	0x8b, 0x26, 0x76, 0x41, 0x28, 0x87, 0xf4, 0x51,
	0x14, 0x36, 0xad, 0x60, 0x8d, 0xe9, 0xe2, 0x9d,
	0x3c, 0xea, 0x09, 0x51, 0x3c, 0x81, 0xdf, 0x1a,
	0xc2, 0xc2, 0xf6, 0x45, 0xe1, 0x73, 0xac, 0xae,
	0x85, 0x74, 0x83, 0x8f, 0x56, 0x3c, 0x36, 0x1c,
	0xe0, 0x07, 0xc6, 0x6a, 0x48, 0xe4, 0x34, 0xe9,
	0x81, 0x53, 0xb7, 0x53, 0x95, 0xa7, 0x94, 0x21,
	0x7e, 0x32, 0x53, 0xda, 0x83, 0xd8, 0x57, 0x92,
	0xd1, 0x15, 0x45, 0x86, 0x40, 0xac, 0xf1, 0x6f,
	0x3c, 0x29, 0xef, 0x8d, 0x12, 0xe1, 0x9d, 0x04,
	0x17, 0x3a, 0xcc, 0xa6, 0xc5, 0xe4, 0x27, 0x41,
	0xcb, 0xfb, 0x5e, 0x77, 0x73, 0x5a, 0x2c, 0x03,
	0xe9, 0x2b, 0x76, 0x4e, 0x69, 0xea, 0xcb, 0xb3,
};

/*
 * IV method encblkno1, blkno 2.
 */
static const uint8_t c3des_cbc_192_encblkno1_vec2_ctxt[SECSIZE] = {
	0x87, 0xb1, 0x3c, 0xd6, 0x60, 0xa0, 0x5a, 0x35,
	0xf7, 0xe1, 0x6b, 0x87, 0xa0, 0x90, 0x2f, 0xc7,
	0x8c, 0xed, 0x53, 0xda, 0x93, 0x32, 0x78, 0x5d,
	0x24, 0x23, 0x42, 0xdd, 0x93, 0x5b, 0x2e, 0x40,
	0xa1, 0xb1, 0x3b, 0xbb, 0xf0, 0x50, 0xb4, 0x61,
	0xea, 0x15, 0x37, 0xf3, 0x49, 0xe1, 0xa0, 0x32,
	0x88, 0x85, 0x81, 0xfd, 0xb7, 0x96, 0xd7, 0x9d,
	0xd7, 0x29, 0x4b, 0x14, 0xf9, 0x18, 0x6a, 0xf6,
	0x46, 0xae, 0x69, 0xdf, 0x63, 0x9a, 0xe0, 0x0b,
	0x2c, 0x53, 0xd7, 0x82, 0x6f, 0xe5, 0xa0, 0x95,
	0x2f, 0x61, 0x7f, 0x15, 0xff, 0xc7, 0xe8, 0x83,
	0xfc, 0xfc, 0x16, 0x1c, 0x37, 0x0f, 0x9b, 0xbb,
	0x14, 0xb2, 0xe2, 0xb7, 0x1f, 0x85, 0xb7, 0x07,
	0x8a, 0x18, 0xed, 0xf7, 0x5f, 0x27, 0xff, 0x2f,
	0x07, 0xf9, 0x9d, 0xe3, 0x79, 0x45, 0x1f, 0x51,
	0x08, 0x54, 0x0f, 0x56, 0x84, 0xee, 0x87, 0x9a,
	0xa9, 0x46, 0xb8, 0x77, 0x85, 0x40, 0x46, 0x50,
	0xc1, 0x58, 0x07, 0xfd, 0xfa, 0x2b, 0x20, 0xd6,
	0x4e, 0xba, 0x08, 0x02, 0x59, 0x3d, 0x23, 0x3b,
	0x5d, 0xf9, 0x5e, 0x2f, 0xac, 0x9e, 0xa0, 0xd7,
	0x3f, 0x9a, 0xdf, 0x50, 0x66, 0xcc, 0x28, 0xce,
	0x93, 0xc8, 0x11, 0x5c, 0x74, 0xe2, 0x4f, 0xfd,
	0xaf, 0x33, 0xbb, 0xce, 0x96, 0x1f, 0xb3, 0x46,
	0x6e, 0xcd, 0xe4, 0xef, 0xfa, 0x2f, 0x93, 0xb1,
	0xe5, 0x7c, 0x54, 0xbc, 0x17, 0x1f, 0xd5, 0x31,
	0x0e, 0x88, 0xe7, 0xcd, 0xb0, 0xb5, 0x2e, 0x1e,
	0x9e, 0x40, 0x36, 0xa5, 0xbb, 0xa7, 0x4e, 0xc8,
	0x11, 0x6c, 0xae, 0x1c, 0x2d, 0xdb, 0x55, 0xd8,
	0x14, 0x40, 0x02, 0xad, 0xaf, 0x19, 0x28, 0x59,
	0xd7, 0x4f, 0x81, 0xd0, 0xc1, 0x54, 0x63, 0x73,
	0x0e, 0xfb, 0x26, 0xf2, 0xa6, 0x80, 0xca, 0x2e,
	0xf3, 0xca, 0x1e, 0xa4, 0x62, 0x07, 0x22, 0x10,
	0x11, 0x6a, 0x57, 0x28, 0x45, 0x80, 0xdf, 0x34,
	0x88, 0xe5, 0xf1, 0x23, 0xe0, 0xb6, 0x44, 0x51,
	0x54, 0xd8, 0xb3, 0x66, 0xac, 0x46, 0x4d, 0xdf,
	0xa2, 0x8e, 0x72, 0x3a, 0x1c, 0x87, 0x2a, 0x43,
	0xfe, 0xdb, 0x00, 0xff, 0xb7, 0x1c, 0x13, 0xc3,
	0x18, 0xfc, 0x71, 0x13, 0xe3, 0xd1, 0x1f, 0xde,
	0x16, 0x63, 0x73, 0xf5, 0x0e, 0xf7, 0x18, 0xe5,
	0x48, 0x8d, 0x30, 0xd9, 0x26, 0x20, 0x6d, 0xa1,
	0xba, 0xde, 0xe8, 0x7d, 0x77, 0x02, 0x33, 0x0d,
	0x73, 0xb2, 0xab, 0x35, 0xfd, 0xa5, 0x6e, 0x4c,
	0x5c, 0x27, 0xc7, 0x7e, 0x4a, 0x28, 0xf8, 0xf5,
	0x00, 0xbe, 0x4c, 0xd7, 0x2c, 0x27, 0x83, 0x16,
	0x37, 0xda, 0x0c, 0xb1, 0xd7, 0x89, 0xd8, 0x8f,
	0x17, 0x69, 0x1b, 0x6b, 0x48, 0x2b, 0xce, 0x9c,
	0xbd, 0xf4, 0x0d, 0xb5, 0x4d, 0x12, 0x11, 0x36,
	0x49, 0xd3, 0x8b, 0x52, 0xce, 0x7e, 0x47, 0xb0,
	0xb5, 0x54, 0x77, 0xef, 0x90, 0xb8, 0x0e, 0xaf,
	0x6f, 0x97, 0x88, 0xde, 0x6b, 0x37, 0x24, 0xdd,
	0x91, 0x84, 0x00, 0x51, 0xab, 0x06, 0x96, 0x3c,
	0x82, 0x73, 0xcf, 0xae, 0x8d, 0x23, 0x86, 0x59,
	0x62, 0x5b, 0xeb, 0x2a, 0xaf, 0x40, 0x17, 0xed,
	0x2b, 0x60, 0x73, 0x7d, 0x99, 0x95, 0x3f, 0xd6,
	0x6c, 0xca, 0x1e, 0xf3, 0xb0, 0xcd, 0xd5, 0x1d,
	0x53, 0xe0, 0xd2, 0x8b, 0x57, 0x7b, 0xac, 0x67,
	0x5a, 0x5a, 0x0a, 0x64, 0x82, 0xab, 0x8f, 0x5a,
	0x36, 0xe2, 0x45, 0x50, 0xec, 0x3e, 0x14, 0x80,
	0x7c, 0xfd, 0x0c, 0xa9, 0x94, 0xfb, 0xfe, 0x72,
	0xec, 0x47, 0x71, 0x2e, 0x90, 0x97, 0xf6, 0x33,
	0xbd, 0x7d, 0x7e, 0x77, 0x8f, 0xad, 0xd4, 0x1d,
	0x1d, 0x53, 0x0f, 0x28, 0x39, 0x77, 0x06, 0x1a,
	0x75, 0xfc, 0x12, 0xe6, 0x45, 0xfc, 0x87, 0xe1,
	0x46, 0xac, 0xb0, 0x73, 0xca, 0x24, 0x7c, 0x71,
};

/*
 * IV method encblkno1, blkno 3.
 */
static const uint8_t c3des_cbc_192_encblkno1_vec3_ctxt[SECSIZE] = {
	0xb1, 0xef, 0x7c, 0xd0, 0xa0, 0x6b, 0xe4, 0x88,
	0x5c, 0xd7, 0xf1, 0xbf, 0x5f, 0xce, 0xda, 0x19,
	0x81, 0x32, 0xbb, 0x96, 0x7e, 0xb9, 0x6e, 0xa1,
	0x43, 0xde, 0x53, 0x66, 0x9c, 0x27, 0x94, 0x85,
	0xcb, 0x09, 0x4e, 0x16, 0xd8, 0x60, 0x7a, 0x38,
	0x27, 0x21, 0x4d, 0x08, 0xaa, 0xe2, 0x1e, 0x6e,
	0xa3, 0xcb, 0x9a, 0x7f, 0xd1, 0xbf, 0x18, 0x36,
	0x5a, 0x4d, 0x7a, 0x7f, 0xcf, 0x3f, 0xba, 0xa5,
	0x77, 0x5b, 0xb4, 0x79, 0xdc, 0xbf, 0x2a, 0x28,
	0x16, 0x27, 0x0f, 0x8b, 0xd7, 0x95, 0xc3, 0xcb,
	0xa1, 0x6a, 0x49, 0x53, 0xa8, 0x0c, 0x70, 0xde,
	0x90, 0x2e, 0x36, 0x74, 0x40, 0x5d, 0x81, 0x74,
	0x03, 0x11, 0xbd, 0xba, 0x40, 0x8d, 0x03, 0x86,
	0x2b, 0x17, 0x55, 0x20, 0xd8, 0x81, 0x30, 0xd2,
	0x2a, 0xbd, 0xea, 0xff, 0x5c, 0x69, 0x9b, 0xe6,
	0xe3, 0x21, 0x9a, 0x10, 0x3e, 0xb0, 0xf4, 0x7a,
	0xfc, 0x6e, 0x66, 0xec, 0x44, 0x0b, 0x95, 0x8d,
	0x13, 0xd4, 0xf6, 0x3e, 0xa1, 0xa1, 0xac, 0xb1,
	0xd8, 0x3d, 0x86, 0xaf, 0x5e, 0xef, 0x14, 0x6a,
	0x32, 0xf3, 0x13, 0x75, 0x3b, 0x64, 0x9a, 0xf4,
	0xd0, 0xf5, 0x00, 0x36, 0x9e, 0xdb, 0xfd, 0xcb,
	0xda, 0x1f, 0xed, 0x9d, 0x6d, 0x52, 0xd7, 0xb5,
	0x48, 0xce, 0x53, 0x5e, 0xdc, 0xc8, 0xe4, 0x96,
	0x04, 0x32, 0xa5, 0xcf, 0x0c, 0xba, 0xa0, 0xd0,
	0x44, 0xb3, 0xe8, 0x72, 0xc6, 0xff, 0x8f, 0xd4,
	0x4d, 0x0a, 0x22, 0x89, 0x74, 0x50, 0xaa, 0x65,
	0x15, 0xab, 0x99, 0xc8, 0xf9, 0xa4, 0x10, 0xe6,
	0xa6, 0x4b, 0x0c, 0xc8, 0xb9, 0xa7, 0x60, 0x41,
	0xe7, 0x57, 0x31, 0xfa, 0x86, 0x55, 0xdf, 0x29,
	0x49, 0xac, 0x55, 0x7b, 0x21, 0xf9, 0x3b, 0x1e,
	0x1f, 0xb4, 0x1c, 0x0b, 0x77, 0xcb, 0x88, 0xbf,
	0xa6, 0x79, 0xbf, 0x9a, 0x51, 0xc4, 0x8e, 0x59,
	0x9c, 0xb3, 0x9d, 0x9d, 0x6b, 0xb2, 0x15, 0x41,
	0x0d, 0x6c, 0xf7, 0x5e, 0xe2, 0xf9, 0xb3, 0x80,
	0x8f, 0x03, 0x67, 0x68, 0x6e, 0x4b, 0x4d, 0x52,
	0xbc, 0x9b, 0xa2, 0xd8, 0x29, 0x1e, 0x5c, 0xd7,
	0x59, 0x67, 0x94, 0x40, 0x9e, 0x08, 0x15, 0x0d,
	0x7e, 0xc9, 0x14, 0x53, 0xa8, 0x67, 0xb3, 0xb8,
	0xaa, 0x21, 0x0f, 0x79, 0x69, 0x48, 0x52, 0xea,
	0x56, 0x03, 0x7b, 0x55, 0xb7, 0xf3, 0xfe, 0xb1,
	0x8a, 0x22, 0x7d, 0x75, 0x55, 0x31, 0xad, 0x20,
	0x6a, 0xc2, 0xa4, 0xd1, 0x1e, 0xab, 0xdd, 0x29,
	0xb5, 0xf8, 0xdd, 0x9b, 0x1a, 0xb8, 0xe7, 0xde,
	0xae, 0xa1, 0xab, 0xbb, 0xf6, 0x00, 0x87, 0xc4,
	0x29, 0xee, 0x2b, 0xa1, 0xa9, 0x1a, 0x46, 0x05,
	0x5a, 0x12, 0x3f, 0x32, 0x81, 0x25, 0x20, 0x71,
	0xb6, 0xfa, 0x1f, 0x27, 0x2a, 0x33, 0x49, 0xfc,
	0x95, 0x00, 0x72, 0x6b, 0x03, 0x53, 0x94, 0x57,
	0x2f, 0x47, 0x3d, 0x2d, 0x7c, 0xb4, 0xde, 0xa7,
	0x96, 0x81, 0x12, 0xff, 0x2c, 0xec, 0x5c, 0x03,
	0x2a, 0x8c, 0x76, 0xc4, 0xed, 0x09, 0xe6, 0x00,
	0x28, 0xdb, 0x9b, 0x44, 0xb0, 0xb4, 0x7b, 0x57,
	0x3b, 0xb6, 0x4f, 0x0b, 0xff, 0xf2, 0xf5, 0x02,
	0x56, 0xcf, 0xd5, 0xbf, 0x71, 0xe6, 0x66, 0xf3,
	0x08, 0x8e, 0x8b, 0x15, 0x57, 0x07, 0x41, 0xa3,
	0x91, 0xc1, 0xe4, 0x64, 0x92, 0x89, 0xed, 0x22,
	0x88, 0x8f, 0x17, 0x91, 0xde, 0xea, 0x0c, 0xa6,
	0x86, 0x8e, 0x4c, 0xd9, 0x63, 0xc9, 0xe5, 0xdc,
	0xd6, 0xd3, 0x7b, 0x2b, 0x65, 0xfa, 0x36, 0x47,
	0x20, 0xa4, 0xe7, 0x0b, 0x52, 0xfa, 0xa6, 0xeb,
	0x1d, 0x20, 0xd0, 0x4b, 0xfd, 0x88, 0x8c, 0xbb,
	0x52, 0x9c, 0x2f, 0xb7, 0xba, 0x8b, 0xdd, 0x10,
	0x2d, 0x7d, 0x77, 0x79, 0x40, 0xa7, 0xed, 0xf9,
	0xbd, 0x2a, 0x55, 0x1f, 0x87, 0x1e, 0x3c, 0xfc,
};

const struct testvec c3des_cbc_192_1_vectors[] = {
	{
		.blkno = 0,
		.ptxt = c3des_cbc_ptxt,
		.ctxt = c3des_cbc_192_encblkno1_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = c3des_cbc_ptxt,
		.ctxt = c3des_cbc_192_encblkno1_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = c3des_cbc_ptxt,
		.ctxt = c3des_cbc_192_encblkno1_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = c3des_cbc_ptxt,
		.ctxt = c3des_cbc_192_encblkno1_vec3_ctxt,
	},
};

/*
 * IV method encblkno8, blkno 0.
 */
static const uint8_t c3des_cbc_192_encblkno8_vec0_ctxt[SECSIZE] = {
	0x9e, 0x5d, 0x35, 0x56, 0xa7, 0xcc, 0xc0, 0x1c,
	0x60, 0x4c, 0x42, 0x90, 0x35, 0xf3, 0xc1, 0x20,
	0xf2, 0x07, 0x6f, 0xf8, 0x7c, 0x33, 0x6a, 0x74,
	0xdc, 0x85, 0xbc, 0x9c, 0xa2, 0x29, 0xc6, 0x69,
	0x0e, 0xef, 0x0f, 0xa9, 0x6e, 0xec, 0xf2, 0x23,
	0x2f, 0x9a, 0xbe, 0x1a, 0x89, 0x22, 0x00, 0xc4,
	0x5a, 0xaf, 0x4a, 0xa0, 0x4f, 0x30, 0x8f, 0x99,
	0xd2, 0x93, 0x6d, 0xfa, 0xcd, 0x2f, 0xad, 0x19,
	0x10, 0x14, 0x90, 0x3a, 0x4b, 0xab, 0x17, 0x2e,
	0x2c, 0xe1, 0x26, 0xe5, 0x76, 0xf1, 0xd1, 0x1d,
	0x4c, 0x77, 0x68, 0xfb, 0x45, 0x9a, 0x3e, 0x19,
	0xe0, 0xfb, 0xdc, 0xd4, 0x0e, 0x29, 0x7c, 0x06,
	0xd3, 0x45, 0xa8, 0xf7, 0x39, 0x91, 0xe6, 0x18,
	0x0f, 0x81, 0xe6, 0x7d, 0x6c, 0x65, 0x2e, 0x16,
	0x24, 0xa4, 0x16, 0x96, 0x0a, 0x7b, 0x5f, 0x3a,
	0x0c, 0xe9, 0x0e, 0x3f, 0x34, 0x38, 0xb0, 0xe1,
	0x39, 0x23, 0x5c, 0x3c, 0x00, 0xb4, 0xa0, 0xf7,
	0x42, 0x18, 0x70, 0x25, 0x82, 0x13, 0x24, 0x49,
	0xbb, 0x3f, 0xfb, 0xef, 0xb6, 0xc6, 0x7f, 0x3d,
	0x8c, 0x17, 0x62, 0x60, 0x6f, 0xd5, 0xda, 0x2c,
	0xf8, 0x85, 0xee, 0xa7, 0xc2, 0x76, 0x5d, 0x34,
	0x4c, 0xe1, 0x0d, 0x36, 0x6e, 0x02, 0xdd, 0x08,
	0x85, 0xe4, 0x90, 0xfe, 0x1f, 0x81, 0x4a, 0x06,
	0xa6, 0x72, 0x81, 0x79, 0x47, 0xd7, 0x6d, 0x92,
	0x8f, 0xb7, 0xb2, 0xfd, 0xd0, 0x60, 0x6c, 0x06,
	0x44, 0xcd, 0x20, 0x28, 0xef, 0x16, 0xc3, 0x01,
	0x19, 0x14, 0x34, 0x39, 0xad, 0x87, 0x9f, 0xde,
	0x76, 0xb9, 0xb9, 0x87, 0x1a, 0xbd, 0x8e, 0x2c,
	0xe6, 0xb3, 0xe7, 0xb6, 0x80, 0xf8, 0xc5, 0x22,
	0x5f, 0x53, 0xed, 0x03, 0xfe, 0x09, 0x2c, 0x9d,
	0xb6, 0x61, 0x4a, 0xbb, 0x07, 0x5d, 0xbd, 0x68,
	0x74, 0xab, 0x02, 0x81, 0x64, 0x7b, 0x97, 0xa3,
	0xad, 0x15, 0x99, 0x7a, 0x04, 0x33, 0xbd, 0x50,
	0x94, 0x11, 0xcc, 0xf7, 0x8b, 0x77, 0x88, 0x78,
	0x80, 0xfe, 0x5f, 0xa1, 0x63, 0xbc, 0xb0, 0x65,
	0xcb, 0x9d, 0x4c, 0xfe, 0x66, 0x4e, 0xff, 0xe3,
	0x43, 0x61, 0x99, 0x88, 0x88, 0x4c, 0xbc, 0x8a,
	0xf1, 0x69, 0x00, 0xc2, 0xe5, 0xb9, 0x65, 0x8b,
	0x10, 0xdf, 0x38, 0x3e, 0x9e, 0x9f, 0x87, 0xed,
	0x84, 0x71, 0xe7, 0xf2, 0xb5, 0xb6, 0x11, 0xed,
	0x1e, 0xd4, 0xc0, 0x6d, 0x77, 0x08, 0x4b, 0xfd,
	0x95, 0xd5, 0xc0, 0xbe, 0xa6, 0xcc, 0x3b, 0xea,
	0x11, 0x38, 0xa5, 0x59, 0x36, 0x2a, 0xf4, 0x98,
	0x52, 0x9d, 0x3b, 0x8c, 0x8a, 0x19, 0xbd, 0xfb,
	0x49, 0xcb, 0xb0, 0x57, 0x91, 0xc7, 0xf8, 0x2a,
	0x89, 0xa8, 0x85, 0x03, 0xdf, 0x6e, 0xad, 0xf4,
	0x8a, 0x88, 0x9a, 0x2b, 0x5d, 0xe8, 0xca, 0xa9,
	0x8f, 0x18, 0xa3, 0x6a, 0x37, 0x84, 0xa9, 0x24,
	0x5b, 0xce, 0xd6, 0xbe, 0x7e, 0x40, 0x86, 0x6a,
	0xc3, 0x47, 0x28, 0x66, 0xf0, 0x8c, 0x2d, 0x69,
	0x22, 0x64, 0x61, 0x36, 0x6a, 0x0c, 0xc4, 0x18,
	0x5f, 0xd7, 0xff, 0xbc, 0xf1, 0x94, 0x16, 0xfb,
	0x26, 0xa7, 0x80, 0xa4, 0x2d, 0x72, 0xc6, 0x9d,
	0xa7, 0xed, 0x04, 0x13, 0x0f, 0xe7, 0xf8, 0x93,
	0x57, 0x6b, 0xd5, 0xa4, 0xad, 0x9a, 0x97, 0xeb,
	0x97, 0xe7, 0x60, 0x01, 0x89, 0x3f, 0x88, 0xf2,
	0xee, 0xf3, 0x79, 0xd6, 0x5a, 0x03, 0x94, 0x07,
	0xd3, 0x33, 0xc8, 0xda, 0x15, 0x17, 0x0a, 0x8f,
	0xbd, 0x58, 0x1b, 0xfe, 0x3d, 0x77, 0x5d, 0x8f,
	0x4e, 0x0e, 0x98, 0x7d, 0x02, 0x63, 0x94, 0x73,
	0x4a, 0x58, 0x47, 0xed, 0x52, 0xfc, 0x85, 0x19,
	0x5d, 0x2f, 0xfa, 0x07, 0x44, 0xbd, 0x8e, 0xcb,
	0x20, 0x63, 0x9d, 0x2b, 0x61, 0x5c, 0x19, 0x71,
	0x80, 0xe5, 0x25, 0x5b, 0x2e, 0xc5, 0xfe, 0x1a,
};

/*
 * IV method encblkno8, blkno 1.
 */
static const uint8_t c3des_cbc_192_encblkno8_vec1_ctxt[SECSIZE] = {
	0xf4, 0xb0, 0xb0, 0xcb, 0x79, 0xcc, 0x8c, 0x0a,
	0x3b, 0xc7, 0x43, 0x4e, 0x62, 0x9d, 0xde, 0xb4,
	0xab, 0xa5, 0x62, 0x63, 0x32, 0xa7, 0x18, 0x2b,
	0xe3, 0xee, 0x44, 0xc6, 0x6f, 0xb2, 0xdc, 0x21,
	0xc5, 0xc8, 0x9e, 0x32, 0x71, 0x4c, 0x7a, 0x82,
	0x8d, 0xe0, 0xad, 0x91, 0x88, 0x0c, 0x41, 0x83,
	0x28, 0x0d, 0xed, 0xa7, 0xeb, 0x48, 0xb1, 0x31,
	0xfa, 0x40, 0xd9, 0x44, 0x19, 0xee, 0x8d, 0x2c,
	0x7d, 0xe2, 0x39, 0xa0, 0x39, 0xaa, 0x86, 0xab,
	0xb5, 0x68, 0xe5, 0x83, 0x06, 0x61, 0xec, 0xe6,
	0xc2, 0x85, 0xb2, 0x46, 0xf4, 0x5b, 0x0e, 0x34,
	0x7e, 0x0c, 0xa0, 0xda, 0xef, 0x58, 0x9c, 0x39,
	0x95, 0xa2, 0xca, 0xd3, 0x3b, 0x4d, 0x76, 0xe3,
	0x34, 0x6d, 0x08, 0xa4, 0xba, 0x88, 0x58, 0x39,
	0xb4, 0xe4, 0x6b, 0xb6, 0x32, 0x50, 0x2c, 0xe2,
	0x0a, 0x37, 0xbc, 0x98, 0x38, 0x32, 0x17, 0x1b,
	0x12, 0xef, 0xdc, 0x9d, 0x91, 0x09, 0x8e, 0xd8,
	0xc3, 0xf8, 0x7b, 0x35, 0x41, 0x3b, 0xf8, 0xf5,
	0x37, 0x48, 0x04, 0xf7, 0x94, 0xbf, 0x54, 0x8d,
	0x79, 0x49, 0x8f, 0xf0, 0x3f, 0xb7, 0x90, 0x76,
	0x14, 0x09, 0xc6, 0x8c, 0xba, 0x1a, 0x30, 0x1b,
	0xbb, 0xd9, 0xe2, 0xb5, 0xe8, 0xd9, 0x9b, 0x68,
	0x60, 0x90, 0xd3, 0x4a, 0xe8, 0x65, 0x7b, 0xaa,
	0xb0, 0xda, 0x69, 0x1d, 0x45, 0x78, 0x2c, 0x3b,
	0x59, 0x29, 0x3c, 0x26, 0x9a, 0xd2, 0xa5, 0xfd,
	0xb7, 0x16, 0x59, 0x7c, 0x46, 0xea, 0x99, 0xd0,
	0x06, 0x01, 0x3f, 0xd2, 0x23, 0xcc, 0xde, 0xb8,
	0xaa, 0x88, 0x17, 0x03, 0xe1, 0x48, 0x2c, 0xdd,
	0xce, 0xd1, 0x2c, 0xce, 0x37, 0xee, 0xe6, 0xa6,
	0x47, 0x8c, 0x07, 0xe5, 0xfe, 0x01, 0xc6, 0x27,
	0xfe, 0x3f, 0x9d, 0x30, 0x18, 0x36, 0xe7, 0xa7,
	0x37, 0x1d, 0xcf, 0x6d, 0x4c, 0x82, 0xec, 0x58,
	0xa1, 0x6f, 0x56, 0xc6, 0x08, 0x25, 0x94, 0xda,
	0xae, 0x1a, 0x4f, 0xda, 0xb2, 0xf4, 0xbf, 0x94,
	0xff, 0x66, 0x6a, 0xb1, 0x1f, 0x42, 0xfe, 0x32,
	0xa4, 0x0e, 0x3d, 0x6a, 0x16, 0x44, 0xe0, 0xac,
	0xe8, 0xc1, 0xe2, 0xa8, 0x73, 0xab, 0xac, 0x58,
	0xb1, 0xbc, 0x94, 0xb2, 0x6a, 0xe4, 0x45, 0xf5,
	0x90, 0x6b, 0x82, 0xeb, 0x9e, 0x22, 0x9e, 0xb2,
	0x27, 0x3e, 0xc8, 0x55, 0xf4, 0x8f, 0xda, 0x04,
	0xa3, 0x9c, 0xa4, 0x79, 0xbd, 0x79, 0xd3, 0xbd,
	0xbe, 0x72, 0x7f, 0x90, 0xef, 0xc3, 0x34, 0x17,
	0x72, 0x6f, 0xb4, 0xfe, 0x62, 0x56, 0xc3, 0xd6,
	0x43, 0xc8, 0x4c, 0x76, 0x91, 0x04, 0x97, 0x4c,
	0x84, 0x98, 0x56, 0xb7, 0x7b, 0x4f, 0xd5, 0xcf,
	0x1b, 0x9c, 0x09, 0xe3, 0x1d, 0xdf, 0x0e, 0xfa,
	0x39, 0xc8, 0x48, 0x43, 0x84, 0xec, 0x79, 0xc8,
	0x7f, 0x4f, 0xa8, 0xc0, 0xb4, 0xde, 0x8b, 0x79,
	0xcb, 0x9c, 0x42, 0x81, 0x49, 0xdc, 0x39, 0xb5,
	0x31, 0xa6, 0x22, 0xba, 0x71, 0xb8, 0x2d, 0x1d,
	0xc8, 0x17, 0xd8, 0x9d, 0x26, 0x2b, 0xd5, 0xcf,
	0x57, 0x46, 0x0a, 0x61, 0x7e, 0xb7, 0xc3, 0x9c,
	0xa6, 0x44, 0x60, 0x2d, 0x30, 0xb8, 0x10, 0x47,
	0x7d, 0x7e, 0x87, 0x76, 0xc1, 0x4e, 0x85, 0x77,
	0xbc, 0x30, 0x32, 0x56, 0x0a, 0x5b, 0x1c, 0xd0,
	0xf6, 0x47, 0x48, 0x22, 0xf4, 0x6e, 0x38, 0xc5,
	0xab, 0xe2, 0xd0, 0x4d, 0x40, 0x27, 0xab, 0x8f,
	0x43, 0xb1, 0x60, 0x29, 0x07, 0xd0, 0xf5, 0x25,
	0xe5, 0xfa, 0xe7, 0x46, 0x32, 0x37, 0xb9, 0xae,
	0x2e, 0x02, 0x8c, 0x94, 0x15, 0x69, 0xd6, 0x74,
	0xb4, 0x36, 0xdd, 0x94, 0x70, 0xa7, 0x16, 0x7b,
	0x4c, 0xd3, 0x48, 0x83, 0xc5, 0xb2, 0xb0, 0x6a,
	0xfe, 0x7e, 0xd4, 0xe5, 0x6d, 0xa5, 0x96, 0x20,
	0x08, 0x59, 0xbd, 0x0c, 0x3d, 0x55, 0xa5, 0x03,
};

/*
 * IV method encblkno8, blkno 2.
 */
static const uint8_t c3des_cbc_192_encblkno8_vec2_ctxt[SECSIZE] = {
	0xea, 0x7c, 0x8c, 0x8e, 0x3e, 0x61, 0x34, 0x3d,
	0xe0, 0x7f, 0xd3, 0xe1, 0x3a, 0xb9, 0xc8, 0xf2,
	0x98, 0xdc, 0x59, 0x26, 0xd2, 0xd8, 0xa7, 0x7f,
	0x41, 0x98, 0x24, 0xa8, 0x28, 0x0c, 0x88, 0x55,
	0x91, 0xdb, 0x29, 0x17, 0x70, 0xd7, 0x03, 0xff,
	0xbd, 0x0e, 0xbf, 0xf8, 0x73, 0x92, 0x19, 0xe9,
	0x92, 0x67, 0xdb, 0x08, 0x94, 0x77, 0x71, 0x2d,
	0x00, 0xad, 0x26, 0x42, 0x2d, 0xac, 0x8c, 0x67,
	0x6f, 0xb3, 0x8e, 0x36, 0x22, 0xeb, 0x1f, 0x8c,
	0xd4, 0x9b, 0x9f, 0xa6, 0xa9, 0xb1, 0x52, 0x65,
	0x9a, 0xfe, 0xcc, 0x92, 0x48, 0x75, 0xf6, 0xb8,
	0x59, 0xfe, 0x0e, 0x67, 0x93, 0xce, 0x3b, 0x7e,
	0x51, 0x74, 0xe5, 0x24, 0x35, 0x08, 0x68, 0x21,
	0x6a, 0x7f, 0xdd, 0x8c, 0xfd, 0xcd, 0x6d, 0x90,
	0xc5, 0x3b, 0x26, 0x9e, 0x00, 0xf4, 0x1e, 0x70,
	0xd3, 0xe7, 0xe8, 0x2f, 0x52, 0x87, 0x76, 0x84,
	0xbb, 0x5c, 0x76, 0x5a, 0xc8, 0xea, 0x74, 0xe2,
	0x9e, 0x85, 0xf6, 0x53, 0x85, 0x1a, 0x6e, 0x02,
	0x0d, 0x32, 0x11, 0xc4, 0xec, 0xee, 0x79, 0x27,
	0xda, 0xca, 0xc0, 0x0b, 0x8e, 0x2d, 0xb7, 0x7d,
	0x8c, 0x6e, 0xfb, 0xa3, 0xa8, 0x24, 0x24, 0x62,
	0xc8, 0xdd, 0xc7, 0x16, 0x09, 0x33, 0x0f, 0xe5,
	0xc8, 0x60, 0x3d, 0xb6, 0xbf, 0x6c, 0x28, 0xd2,
	0x0b, 0x9c, 0xd9, 0xcb, 0x64, 0x49, 0xe4, 0x80,
	0x72, 0x58, 0xaa, 0xaa, 0x7e, 0x1d, 0x9f, 0xd7,
	0x29, 0x15, 0x65, 0xfc, 0xfd, 0x3f, 0xe1, 0x82,
	0x25, 0x3c, 0xd4, 0xbe, 0x59, 0x79, 0x63, 0xd1,
	0xd6, 0x0e, 0xda, 0x00, 0xf3, 0xaa, 0x13, 0xd3,
	0xed, 0xef, 0xca, 0x8b, 0x97, 0x15, 0x2d, 0x10,
	0x6f, 0xcf, 0xee, 0xc7, 0x21, 0xad, 0xe3, 0xe4,
	0xd8, 0x95, 0x21, 0x1f, 0xc0, 0x06, 0x3a, 0xbc,
	0xbb, 0x2a, 0x92, 0x78, 0x76, 0x9d, 0x1e, 0x7b,
	0xb5, 0x29, 0xaf, 0x96, 0x75, 0x2b, 0x41, 0xbd,
	0xae, 0x79, 0x28, 0x72, 0xe7, 0x54, 0xc4, 0x08,
	0xd3, 0xd2, 0xac, 0x96, 0xd0, 0x0f, 0x9b, 0x68,
	0x7d, 0x3f, 0xc2, 0xdd, 0x3d, 0xfc, 0xca, 0xcd,
	0x11, 0x71, 0xd9, 0x48, 0x53, 0x9f, 0xd3, 0x79,
	0x7d, 0x47, 0x71, 0x2a, 0x6d, 0x9e, 0xa9, 0x47,
	0xa1, 0xf7, 0x97, 0x80, 0x83, 0x70, 0x6b, 0xfe,
	0x10, 0x11, 0x6a, 0x0e, 0xdd, 0xde, 0x22, 0x3c,
	0x19, 0x30, 0x73, 0x73, 0x2e, 0x4b, 0x54, 0x17,
	0xc3, 0x2e, 0xe9, 0xce, 0xe0, 0xe3, 0xa0, 0x1a,
	0x28, 0xd1, 0x50, 0xa8, 0xd2, 0x40, 0xe2, 0x1b,
	0xfa, 0x49, 0x06, 0x49, 0x8b, 0x4b, 0xd9, 0xd5,
	0xf5, 0x50, 0xae, 0x64, 0x19, 0xe1, 0xd9, 0x4e,
	0xbb, 0x29, 0x70, 0x66, 0x46, 0xa8, 0x7e, 0x5b,
	0xdc, 0xe2, 0xd5, 0x9d, 0x56, 0x6d, 0x4c, 0xe6,
	0x0e, 0x6b, 0x71, 0x40, 0x82, 0xf7, 0xb3, 0xad,
	0x23, 0x17, 0xe3, 0x1c, 0x61, 0x1d, 0x3b, 0x71,
	0xfc, 0x06, 0x17, 0xec, 0x6c, 0x77, 0x98, 0x27,
	0xc7, 0x4b, 0x65, 0x17, 0x81, 0xe7, 0xcb, 0xce,
	0x09, 0x76, 0x82, 0x82, 0x4a, 0x53, 0x67, 0xa0,
	0x05, 0x25, 0x4c, 0xc4, 0xa7, 0xad, 0xa7, 0xaf,
	0xa0, 0x11, 0xd7, 0x73, 0x3b, 0x30, 0xbf, 0x53,
	0x50, 0x9b, 0xd8, 0xf3, 0x32, 0x15, 0xdd, 0x36,
	0x88, 0xc2, 0x39, 0x51, 0xb6, 0xb8, 0x0d, 0x5c,
	0x20, 0x4e, 0x24, 0xee, 0x95, 0x32, 0x61, 0x25,
	0xda, 0x73, 0x0d, 0x8a, 0x58, 0xe6, 0xcc, 0xad,
	0x79, 0x3d, 0xef, 0x29, 0x0c, 0x9f, 0xe1, 0xa7,
	0x22, 0x1e, 0xea, 0x7a, 0x4f, 0xfb, 0xc1, 0x1f,
	0x17, 0xca, 0x69, 0xd6, 0xa4, 0xce, 0x6e, 0xc0,
	0x70, 0xa3, 0x08, 0x32, 0x87, 0xb4, 0x6b, 0x80,
	0x5c, 0x7f, 0x88, 0x5c, 0xbf, 0x07, 0xd8, 0xe9,
	0xdd, 0xd2, 0x76, 0xa9, 0xaa, 0xd9, 0x55, 0x48,
};

/*
 * IV method encblkno8, blkno 3.
 */
static const uint8_t c3des_cbc_192_encblkno8_vec3_ctxt[SECSIZE] = {
	0xf3, 0x49, 0xda, 0x5c, 0xde, 0x9d, 0x3e, 0x9d,
	0xb9, 0xc2, 0x6e, 0x96, 0xa9, 0x93, 0x10, 0x73,
	0x0e, 0x26, 0x39, 0xd6, 0x9f, 0x04, 0x5f, 0x69,
	0x54, 0xa3, 0x7c, 0x46, 0x7b, 0x18, 0x93, 0xc0,
	0xbb, 0x0c, 0x96, 0x6f, 0xb0, 0xbf, 0xce, 0x67,
	0x33, 0x3e, 0x56, 0xe8, 0x6b, 0x4d, 0x3f, 0xc8,
	0x3c, 0xc6, 0x89, 0x2c, 0x0b, 0x95, 0x3a, 0xaf,
	0xc0, 0xf3, 0x1f, 0x0e, 0x07, 0x01, 0xa6, 0x35,
	0x19, 0x79, 0x91, 0x24, 0xaa, 0x0d, 0xf0, 0x53,
	0x27, 0x7d, 0xbb, 0xa6, 0xb6, 0x44, 0x31, 0x4b,
	0xd4, 0xcf, 0xf6, 0x6d, 0x18, 0xa2, 0x28, 0x8a,
	0xc1, 0x0a, 0xbe, 0x57, 0x0c, 0x61, 0x5f, 0xd9,
	0x12, 0x14, 0xfe, 0xe2, 0xc7, 0x10, 0x72, 0xee,
	0x19, 0xb8, 0x16, 0x0b, 0x88, 0x87, 0xce, 0xf3,
	0xfe, 0x57, 0x37, 0xd1, 0xa2, 0xf7, 0xd0, 0x5e,
	0x73, 0xde, 0x39, 0x35, 0xbc, 0xde, 0xed, 0x61,
	0x4b, 0x31, 0xdc, 0xfe, 0x3c, 0x4d, 0x98, 0xa9,
	0x36, 0xb0, 0x34, 0x5b, 0xb4, 0xb7, 0x79, 0x25,
	0x6e, 0x24, 0x7e, 0x10, 0xfe, 0x20, 0xd5, 0x16,
	0x86, 0xaf, 0xcd, 0x26, 0x34, 0xd3, 0x2e, 0xdc,
	0x7c, 0x69, 0xe3, 0xc5, 0x62, 0x0c, 0xba, 0x29,
	0x9c, 0x4b, 0x2f, 0x39, 0x45, 0xe1, 0xcf, 0xc5,
	0xfe, 0x35, 0xb6, 0x2f, 0xb1, 0x1a, 0x90, 0xe1,
	0xa7, 0x39, 0xe8, 0x1e, 0x5f, 0xac, 0xab, 0x1e,
	0x32, 0xba, 0xc5, 0x92, 0x39, 0x62, 0x37, 0x2c,
	0x49, 0xf1, 0x62, 0x90, 0xf7, 0x1e, 0x10, 0xce,
	0x8e, 0x95, 0xa3, 0xc6, 0xd8, 0xe5, 0xc8, 0xdf,
	0xcc, 0x94, 0x7d, 0x26, 0xab, 0x29, 0xbb, 0x9d,
	0xf3, 0x73, 0xce, 0xac, 0x76, 0xdf, 0x75, 0x2a,
	0x3e, 0x8f, 0x47, 0xff, 0x76, 0xfe, 0xea, 0xd4,
	0x4a, 0xa9, 0x36, 0x9d, 0x12, 0x45, 0xb7, 0x99,
	0x81, 0xb6, 0x77, 0x98, 0x13, 0xfb, 0x5a, 0xe5,
	0x40, 0x87, 0x61, 0x0d, 0x10, 0x76, 0xf6, 0x3e,
	0x48, 0xac, 0xc4, 0x27, 0x87, 0xcd, 0x07, 0xde,
	0x0b, 0x23, 0x97, 0x61, 0x3d, 0x18, 0x64, 0x7f,
	0xbf, 0xd6, 0x87, 0xc1, 0x11, 0xfb, 0xf9, 0xda,
	0x14, 0xa1, 0x01, 0xf8, 0x7e, 0xea, 0x5b, 0x5b,
	0xdd, 0x09, 0xf9, 0x31, 0x80, 0x3c, 0xee, 0x34,
	0x2d, 0xda, 0x71, 0xd9, 0x32, 0x7d, 0x45, 0xb2,
	0x53, 0xea, 0xd5, 0x7c, 0x85, 0x45, 0xce, 0x1d,
	0x2b, 0xe9, 0xd7, 0x95, 0xf8, 0x8c, 0x08, 0xe4,
	0xd0, 0x2f, 0x60, 0x75, 0x02, 0xf3, 0xde, 0xeb,
	0x46, 0x40, 0xa8, 0xd2, 0x37, 0xd6, 0xca, 0x5d,
	0xb9, 0xf4, 0x51, 0x31, 0x8a, 0x1a, 0x82, 0xbd,
	0x6f, 0x6d, 0x88, 0x2b, 0x63, 0x0f, 0xe1, 0xf0,
	0xcf, 0x13, 0x79, 0x1d, 0x78, 0x82, 0x66, 0xa1,
	0xef, 0xdb, 0x34, 0x50, 0xd2, 0x71, 0x47, 0x49,
	0x41, 0x74, 0xd9, 0x0b, 0x14, 0x38, 0x1f, 0xc3,
	0x09, 0x4d, 0xb3, 0xa6, 0x03, 0x3f, 0x56, 0x67,
	0xd7, 0x51, 0x4c, 0x8a, 0x1d, 0x37, 0x99, 0xfb,
	0xe1, 0x84, 0x57, 0x55, 0x9b, 0xf8, 0x73, 0x63,
	0x68, 0x73, 0x89, 0x52, 0x06, 0xe7, 0x34, 0xe7,
	0x1a, 0x15, 0x7e, 0xd9, 0x84, 0xa3, 0x0e, 0x68,
	0x14, 0x1c, 0xe8, 0x23, 0x9e, 0xe3, 0x8f, 0x71,
	0x02, 0x9b, 0x87, 0xd4, 0xd9, 0x1b, 0xd1, 0x9e,
	0x9e, 0xa0, 0x7e, 0x49, 0x8e, 0xaa, 0x89, 0xb5,
	0x16, 0x48, 0x07, 0xb3, 0x3d, 0x9e, 0x4c, 0x35,
	0x3e, 0x94, 0xa9, 0xf8, 0x82, 0x50, 0x6a, 0x41,
	0x28, 0x3e, 0x9f, 0x9a, 0x1a, 0x5d, 0x02, 0x7c,
	0xd0, 0x32, 0x52, 0xa5, 0xee, 0x09, 0x27, 0x2d,
	0x49, 0x17, 0xf7, 0x92, 0xa1, 0x63, 0x9d, 0x2a,
	0xfd, 0x53, 0x26, 0x14, 0x7c, 0x92, 0x72, 0xa6,
	0x38, 0x18, 0x8f, 0xb5, 0x54, 0xb3, 0x69, 0x63,
	0x6a, 0xdc, 0xb1, 0x5a, 0x12, 0x7a, 0x0b, 0xa3,
};

const struct testvec c3des_cbc_192_8_vectors[] = {
	{
		.blkno = 0,
		.ptxt = c3des_cbc_ptxt,
		.ctxt = c3des_cbc_192_encblkno8_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = c3des_cbc_ptxt,
		.ctxt = c3des_cbc_192_encblkno8_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = c3des_cbc_ptxt,
		.ctxt = c3des_cbc_192_encblkno8_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = c3des_cbc_ptxt,
		.ctxt = c3des_cbc_192_encblkno8_vec3_ctxt,
	},
};

static int
open_disk(const char *devpath, const char *imgpath, size_t size)
{
	int fd;

	fd = open(imgpath, O_CREAT | O_RDWR | O_TRUNC, 0600);
	if (fd < 0)
		return -1;

	if (ftruncate(fd, size) < 0)
		goto fail;

	if (rump_pub_etfs_register_withsize(devpath,
	    imgpath, RUMP_ETFS_BLK, 0, size) < 0) {
		goto fail;
	}

	unlink(imgpath);
	return fd;
fail:
	close(fd);
	unlink(imgpath);
	return -1;
}

static int
open_cgd(int devno)
{
	char devpath[32];

	sprintf(devpath, "/dev/rcgd%d%c", devno, getrawpartition() + 'a');

	return rump_sys_open(devpath, O_RDWR, 0);
}

static int
configure_cgd(int fd, const char *dkpath, const char *alg,
    const char *ivmethod, const char *key, size_t keylen)
{
	struct cgd_ioctl ci;

	memset(&ci, 0, sizeof(ci));
	ci.ci_disk = dkpath;
	ci.ci_alg = alg;
	ci.ci_ivmethod = ivmethod;
	ci.ci_keylen = 8 * keylen - 8; /* Exclude the NUL terminator. */
	ci.ci_key = key;
	ci.ci_blocksize = 64;

	return rump_sys_ioctl(fd, CGDIOCSET, &ci);
}

static int
unconfigure_cgd(int fd)
{
	struct cgd_ioctl ci;

	return rump_sys_ioctl(fd, CGDIOCCLR, &ci);
}

static int
write_testvec(int cgdfd, const struct testvec *tv)
{

	if (rump_sys_lseek(cgdfd, tv->blkno * SECSIZE, SEEK_SET) < 0)
		return -1;

	if (rump_sys_write(cgdfd, tv->ptxt, SECSIZE) != SECSIZE)
		return -1;

	return 0;
}

static int
read_testvec(int cgdfd, const struct testvec *tv)
{
	char *buf;
	int res = -1;

	buf = malloc(SECSIZE);
	if (buf == NULL)
		return -1;

	if (rump_sys_lseek(cgdfd, tv->blkno * SECSIZE, SEEK_SET) < 0)
		goto fail;

	if (rump_sys_read(cgdfd, buf, SECSIZE) != SECSIZE)
		goto fail;

	res = memcmp(buf, tv->ptxt, SECSIZE);
fail:
	free(buf);
	return res;
}

static int
check_testvec(int dkfd, const struct testvec *tv)
{
	char *buf;
	int res = -1;

	buf = malloc(SECSIZE);
	if (buf == NULL)
		return -1;

	if (lseek(dkfd, tv->blkno * SECSIZE, SEEK_SET) < 0)
		goto fail;

	if (read(dkfd, buf, SECSIZE) != SECSIZE)
		goto fail;

	res = memcmp(buf, tv->ctxt, SECSIZE);
fail:
	free(buf);
	return res;
}

ATF_TC(cgd_3des_cbc_192_encblkno1);
ATF_TC_HEAD(cgd_3des_cbc_192_encblkno1, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test 3des-cbc with 192 bits key, ivmethod encblkno1");
}

ATF_TC_BODY(cgd_3des_cbc_192_encblkno1, tc)
{
	const char imgpath[] = "3des-cbc-192-encblkno1.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "3des-cbc", "encblkno1",
	    c3des_cbc_192_key, sizeof(c3des_cbc_192_key)));

	ATF_CHECK_EQ(write_testvec(cgdfd, &c3des_cbc_192_1_vectors[0]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &c3des_cbc_192_1_vectors[1]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &c3des_cbc_192_1_vectors[2]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &c3des_cbc_192_1_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "3des-cbc", "encblkno1",
	    c3des_cbc_192_key, sizeof(c3des_cbc_192_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &c3des_cbc_192_1_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &c3des_cbc_192_1_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &c3des_cbc_192_1_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &c3des_cbc_192_1_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &c3des_cbc_192_1_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &c3des_cbc_192_1_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &c3des_cbc_192_1_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &c3des_cbc_192_1_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_3des_cbc_192_encblkno8);
ATF_TC_HEAD(cgd_3des_cbc_192_encblkno8, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test 3des-cbc with 192 bits key, ivmethod encblkno8");
}

ATF_TC_BODY(cgd_3des_cbc_192_encblkno8, tc)
{
	const char imgpath[] = "3des-cbc-192-encblkno8.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "3des-cbc", "encblkno8",
	    c3des_cbc_192_key, sizeof(c3des_cbc_192_key)));

	ATF_CHECK_EQ(write_testvec(cgdfd, &c3des_cbc_192_8_vectors[0]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &c3des_cbc_192_8_vectors[1]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &c3des_cbc_192_8_vectors[2]), 0);
	ATF_CHECK_EQ(write_testvec(cgdfd, &c3des_cbc_192_8_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "3des-cbc", "encblkno8",
	    c3des_cbc_192_key, sizeof(c3des_cbc_192_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &c3des_cbc_192_8_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &c3des_cbc_192_8_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &c3des_cbc_192_8_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &c3des_cbc_192_8_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &c3des_cbc_192_8_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &c3des_cbc_192_8_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &c3des_cbc_192_8_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &c3des_cbc_192_8_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TP_ADD_TCS(tp)
{

	ATF_TP_ADD_TC(tp, cgd_3des_cbc_192_encblkno1);
	ATF_TP_ADD_TC(tp, cgd_3des_cbc_192_encblkno8);

	return atf_no_error();
}
