/*-
 * SPDX-License-Identifier: BSD-2-Clause-FreeBSD
 *
 * Copyright (c) 2005  Joseph Koshy
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD: stable/12/sys/dev/hwpmc/pmc_events.h 336783 2018-07-27 21:25:01Z imp $
 */

#ifndef _DEV_HWPMC_PMC_EVENTS_H_
#define	_DEV_HWPMC_PMC_EVENTS_H_

/*
 * Note: Documentation on adding events can be found both in
 * the source tree at src/share/doc/papers/hwpmc/hwpmc.ms
 * as well as on-line at:
 *
 * https://wiki.freebsd.org/PmcTools/PmcHardwareHowTo
 *
 * Please refer to those resources before you attempt to modify
 * this file or the hwpmc driver/subsystem.
 */

/* * PMC event codes.
 *
 * __PMC_EV(CLASS, SYMBOLIC-NAME)
 *
 */
/* timestamp counters. */
#define        __PMC_EV_TSC()                                                  \
	__PMC_EV(TSC, TSC)

#define        PMC_EV_TSC_FIRST        PMC_EV_TSC_TSC
#define        PMC_EV_TSC_LAST         PMC_EV_TSC_TSC

/*
 * Software events are dynamically defined.
 */

#define        PMC_EV_DYN_COUNT        0x1000

#define        PMC_EV_SOFT_FIRST       0x20000
#define        PMC_EV_SOFT_LAST        (PMC_EV_SOFT_FIRST + PMC_EV_DYN_COUNT - 1)

/*
 * AMD K7 Events, from "The AMD Athlon(tm) Processor x86 Code
 * Optimization Guide" [Doc#22007K, Feb 2002]
 */

#define	__PMC_EV_K7()					\
__PMC_EV(K7, DC_ACCESSES)				\
__PMC_EV(K7, DC_MISSES)					\
__PMC_EV(K7, DC_REFILLS_FROM_L2)			\
__PMC_EV(K7, DC_REFILLS_FROM_SYSTEM)			\
__PMC_EV(K7, DC_WRITEBACKS)				\
__PMC_EV(K7, L1_DTLB_MISS_AND_L2_DTLB_HITS)		\
__PMC_EV(K7, L1_AND_L2_DTLB_MISSES)			\
__PMC_EV(K7, MISALIGNED_REFERENCES)			\
__PMC_EV(K7, IC_FETCHES)				\
__PMC_EV(K7, IC_MISSES)					\
__PMC_EV(K7, L1_ITLB_MISSES)				\
__PMC_EV(K7, L1_L2_ITLB_MISSES)				\
__PMC_EV(K7, RETIRED_INSTRUCTIONS)			\
__PMC_EV(K7, RETIRED_OPS)				\
__PMC_EV(K7, RETIRED_BRANCHES)				\
__PMC_EV(K7, RETIRED_BRANCHES_MISPREDICTED)		\
__PMC_EV(K7, RETIRED_TAKEN_BRANCHES)			\
__PMC_EV(K7, RETIRED_TAKEN_BRANCHES_MISPREDICTED)	\
__PMC_EV(K7, RETIRED_FAR_CONTROL_TRANSFERS)		\
__PMC_EV(K7, RETIRED_RESYNC_BRANCHES)			\
__PMC_EV(K7, INTERRUPTS_MASKED_CYCLES)			\
__PMC_EV(K7, INTERRUPTS_MASKED_WHILE_PENDING_CYCLES)	\
__PMC_EV(K7, HARDWARE_INTERRUPTS)

#define	PMC_EV_K7_FIRST	PMC_EV_K7_DC_ACCESSES
#define	PMC_EV_K7_LAST	PMC_EV_K7_HARDWARE_INTERRUPTS

/* AMD K8 PMCs */

#define	__PMC_EV_K8()							\
__PMC_EV(K8, FP_DISPATCHED_FPU_OPS)					\
__PMC_EV(K8, FP_CYCLES_WITH_NO_FPU_OPS_RETIRED)				\
__PMC_EV(K8, FP_DISPATCHED_FPU_FAST_FLAG_OPS)				\
__PMC_EV(K8, LS_SEGMENT_REGISTER_LOAD)					\
__PMC_EV(K8, LS_MICROARCHITECTURAL_RESYNC_BY_SELF_MODIFYING_CODE)	\
__PMC_EV(K8, LS_MICROARCHITECTURAL_RESYNC_BY_SNOOP)			\
__PMC_EV(K8, LS_BUFFER2_FULL)						\
__PMC_EV(K8, LS_LOCKED_OPERATION)					\
__PMC_EV(K8, LS_MICROARCHITECTURAL_LATE_CANCEL)				\
__PMC_EV(K8, LS_RETIRED_CFLUSH_INSTRUCTIONS)				\
__PMC_EV(K8, LS_RETIRED_CPUID_INSTRUCTIONS)				\
__PMC_EV(K8, DC_ACCESS)							\
__PMC_EV(K8, DC_MISS)							\
__PMC_EV(K8, DC_REFILL_FROM_L2)						\
__PMC_EV(K8, DC_REFILL_FROM_SYSTEM)					\
__PMC_EV(K8, DC_COPYBACK)						\
__PMC_EV(K8, DC_L1_DTLB_MISS_AND_L2_DTLB_HIT)				\
__PMC_EV(K8, DC_L1_DTLB_MISS_AND_L2_DTLB_MISS)				\
__PMC_EV(K8, DC_MISALIGNED_DATA_REFERENCE)				\
__PMC_EV(K8, DC_MICROARCHITECTURAL_LATE_CANCEL)				\
__PMC_EV(K8, DC_MICROARCHITECTURAL_EARLY_CANCEL)			\
__PMC_EV(K8, DC_ONE_BIT_ECC_ERROR)					\
__PMC_EV(K8, DC_DISPATCHED_PREFETCH_INSTRUCTIONS)			\
__PMC_EV(K8, DC_DCACHE_ACCESSES_BY_LOCKS)				\
__PMC_EV(K8, BU_CPU_CLK_UNHALTED)					\
__PMC_EV(K8, BU_INTERNAL_L2_REQUEST)					\
__PMC_EV(K8, BU_FILL_REQUEST_L2_MISS)					\
__PMC_EV(K8, BU_FILL_INTO_L2)						\
__PMC_EV(K8, IC_FETCH)							\
__PMC_EV(K8, IC_MISS)							\
__PMC_EV(K8, IC_REFILL_FROM_L2)						\
__PMC_EV(K8, IC_REFILL_FROM_SYSTEM)					\
__PMC_EV(K8, IC_L1_ITLB_MISS_AND_L2_ITLB_HIT)				\
__PMC_EV(K8, IC_L1_ITLB_MISS_AND_L2_ITLB_MISS)				\
__PMC_EV(K8, IC_MICROARCHITECTURAL_RESYNC_BY_SNOOP)			\
__PMC_EV(K8, IC_INSTRUCTION_FETCH_STALL)				\
__PMC_EV(K8, IC_RETURN_STACK_HIT)					\
__PMC_EV(K8, IC_RETURN_STACK_OVERFLOW)					\
__PMC_EV(K8, FR_RETIRED_X86_INSTRUCTIONS)				\
__PMC_EV(K8, FR_RETIRED_UOPS)						\
__PMC_EV(K8, FR_RETIRED_BRANCHES)					\
__PMC_EV(K8, FR_RETIRED_BRANCHES_MISPREDICTED)				\
__PMC_EV(K8, FR_RETIRED_TAKEN_BRANCHES)					\
__PMC_EV(K8, FR_RETIRED_TAKEN_BRANCHES_MISPREDICTED)			\
__PMC_EV(K8, FR_RETIRED_FAR_CONTROL_TRANSFERS)				\
__PMC_EV(K8, FR_RETIRED_RESYNCS)					\
__PMC_EV(K8, FR_RETIRED_NEAR_RETURNS)					\
__PMC_EV(K8, FR_RETIRED_NEAR_RETURNS_MISPREDICTED)			\
__PMC_EV(K8, FR_RETIRED_TAKEN_BRANCHES_MISPREDICTED_BY_ADDR_MISCOMPARE)	\
__PMC_EV(K8, FR_RETIRED_FPU_INSTRUCTIONS)				\
__PMC_EV(K8, FR_RETIRED_FASTPATH_DOUBLE_OP_INSTRUCTIONS)		\
__PMC_EV(K8, FR_INTERRUPTS_MASKED_CYCLES)				\
__PMC_EV(K8, FR_INTERRUPTS_MASKED_WHILE_PENDING_CYCLES)			\
__PMC_EV(K8, FR_TAKEN_HARDWARE_INTERRUPTS)				\
__PMC_EV(K8, FR_DECODER_EMPTY)						\
__PMC_EV(K8, FR_DISPATCH_STALLS)					\
__PMC_EV(K8, FR_DISPATCH_STALL_FROM_BRANCH_ABORT_TO_RETIRE)		\
__PMC_EV(K8, FR_DISPATCH_STALL_FOR_SERIALIZATION)			\
__PMC_EV(K8, FR_DISPATCH_STALL_FOR_SEGMENT_LOAD)			\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_REORDER_BUFFER_IS_FULL)		\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_RESERVATION_STATIONS_ARE_FULL)	\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_FPU_IS_FULL)			\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_LS_IS_FULL)				\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_WAITING_FOR_ALL_TO_BE_QUIET)	\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_FAR_XFER_OR_RESYNC_BRANCH_PENDING)	\
__PMC_EV(K8, FR_FPU_EXCEPTIONS)						\
__PMC_EV(K8, FR_NUMBER_OF_BREAKPOINTS_FOR_DR0)				\
__PMC_EV(K8, FR_NUMBER_OF_BREAKPOINTS_FOR_DR1)				\
__PMC_EV(K8, FR_NUMBER_OF_BREAKPOINTS_FOR_DR2)				\
__PMC_EV(K8, FR_NUMBER_OF_BREAKPOINTS_FOR_DR3)				\
__PMC_EV(K8, NB_MEMORY_CONTROLLER_PAGE_ACCESS_EVENT)			\
__PMC_EV(K8, NB_MEMORY_CONTROLLER_PAGE_TABLE_OVERFLOW)			\
__PMC_EV(K8, NB_MEMORY_CONTROLLER_DRAM_COMMAND_SLOTS_MISSED)		\
__PMC_EV(K8, NB_MEMORY_CONTROLLER_TURNAROUND)				\
__PMC_EV(K8, NB_MEMORY_CONTROLLER_BYPASS_SATURATION)			\
__PMC_EV(K8, NB_SIZED_COMMANDS)						\
__PMC_EV(K8, NB_PROBE_RESULT)						\
__PMC_EV(K8, NB_HT_BUS0_BANDWIDTH)					\
__PMC_EV(K8, NB_HT_BUS1_BANDWIDTH)					\
__PMC_EV(K8, NB_HT_BUS2_BANDWIDTH)

#define	PMC_EV_K8_FIRST		PMC_EV_K8_FP_DISPATCHED_FPU_OPS
#define	PMC_EV_K8_LAST		PMC_EV_K8_NB_HT_BUS2_BANDWIDTH

/*
 * Events supported by Intel architectural fixed function counters,
 * from the "Intel 64 and IA-32 Architectures Software Developer's
 * Manual Volume 3B: System Programming Guide, Part 2", July 2008.
 */
#define	__PMC_EV_IAF()							\
__PMC_EV(IAF, INSTR_RETIRED_ANY)					\
__PMC_EV(IAF, CPU_CLK_UNHALTED_CORE)					\
__PMC_EV(IAF, CPU_CLK_UNHALTED_REF)

#define	PMC_EV_IAF_FIRST	PMC_EV_IAF_INSTR_RETIRED_ANY
#define	PMC_EV_IAF_LAST		PMC_EV_IAF_CPU_CLK_UNHALTED_REF

#define	__PMC_EV_ALIAS_IAF()						\
__PMC_EV_ALIAS("instruction-retired",	IAF_INSTR_RETIRED_ANY)		\
__PMC_EV_ALIAS("unhalted-core-cycles",	IAF_CPU_CLK_UNHALTED_CORE)	\
__PMC_EV_ALIAS("unhalted-reference-cycles", IAF_CPU_CLK_UNHALTED_REF)


#define	PMC_EV_IAP_FIRST	PMC_EV_IAP_ARCH_BR_INS_RET
#define	PMC_EV_IAP_LAST		PMC_EV_IAP_EVENT_FDH_40H

/*
 * Map "architectural" event names to event ids.
 */
#define	__PMC_EV_ALIAS_INTEL_ARCHITECTURAL()				\
__PMC_EV_ALIAS("branch-instruction-retired",	IAP_ARCH_BR_INS_RET)	\
__PMC_EV_ALIAS("branch-misses-retired",		IAP_ARCH_BR_MIS_RET)	\
__PMC_EV_ALIAS("instruction-retired",		IAP_ARCH_INS_RET)	\
__PMC_EV_ALIAS("llc-misses",			IAP_ARCH_LLC_MIS)	\
__PMC_EV_ALIAS("llc-reference",			IAP_ARCH_LLC_REF)	\
__PMC_EV_ALIAS("unhalted-reference-cycles",	IAP_ARCH_UNH_REF_CYC)	\
__PMC_EV_ALIAS("unhalted-core-cycles",		IAP_ARCH_UNH_COR_CYC)

#define        __PMC_EV_UCP()                          \
	__PMC_EV(UCP, EVENT_0CH_04H_E)					   \
	__PMC_EV(UCP, EVENT_0CH_04H_F)					   \
	__PMC_EV(UCP, EVENT_0CH_04H_M)					   \
	__PMC_EV(UCP, EVENT_0CH_04H_S)					   \
	__PMC_EV(UCP, EVENT_0CH_08H_E)					   \
	__PMC_EV(UCP, EVENT_0CH_08H_F)					   \
	__PMC_EV(UCP, EVENT_0CH_08H_M)					   \
	__PMC_EV(UCP, EVENT_0CH_08H_S)					   \

/*
 * Intel XScale events from:
 *
 * Intel XScale Core Developer's Manual
 * January, 2004, #27347302
 *
 * 3rd Generation Intel XScale Microarchitecture
 * Developer's Manual
 * May 2007, #31628302
 *
 * First 14 events are for 1st and 2nd Generation Intel XScale cores. The
 * remaining are available only on 3rd Generation Intel XScale cores.
 */
#define	__PMC_EV_XSCALE()				\
	__PMC_EV(XSCALE, IC_FETCH)			\
	__PMC_EV(XSCALE, IC_MISS)			\
	__PMC_EV(XSCALE, DATA_DEPENDENCY_STALLED)	\
	__PMC_EV(XSCALE, ITLB_MISS)			\
	__PMC_EV(XSCALE, DTLB_MISS)			\
	__PMC_EV(XSCALE, BRANCH_RETIRED)		\
	__PMC_EV(XSCALE, BRANCH_MISPRED)		\
	__PMC_EV(XSCALE, INSTR_RETIRED)			\
	__PMC_EV(XSCALE, DC_FULL_CYCLE)			\
	__PMC_EV(XSCALE, DC_FULL_CONTIG)		\
	__PMC_EV(XSCALE, DC_ACCESS)			\
	__PMC_EV(XSCALE, DC_MISS)			\
	__PMC_EV(XSCALE, DC_WRITEBACK)			\
	__PMC_EV(XSCALE, PC_CHANGE)			\
	__PMC_EV(XSCALE, BRANCH_RETIRED_ALL)		\
	__PMC_EV(XSCALE, INSTR_CYCLE)			\
	__PMC_EV(XSCALE, CP_STALL)			\
	__PMC_EV(XSCALE, PC_CHANGE_ALL)			\
	__PMC_EV(XSCALE, PIPELINE_FLUSH)		\
	__PMC_EV(XSCALE, BACKEND_STALL)			\
	__PMC_EV(XSCALE, MULTIPLIER_USE)		\
	__PMC_EV(XSCALE, MULTIPLIER_STALLED)		\
	__PMC_EV(XSCALE, DATA_CACHE_STALLED)		\
	__PMC_EV(XSCALE, L2_CACHE_REQ)			\
	__PMC_EV(XSCALE, L2_CACHE_MISS)			\
	__PMC_EV(XSCALE, ADDRESS_BUS_TRANS)		\
	__PMC_EV(XSCALE, SELF_ADDRESS_BUS_TRANS)	\
	__PMC_EV(XSCALE, DATA_BUS_TRANS)

#define	PMC_EV_XSCALE_FIRST	PMC_EV_XSCALE_IC_FETCH
#define	PMC_EV_XSCALE_LAST	PMC_EV_XSCALE_DATA_BUS_TRANS

/*
 * ARMv7 Events
 */

#define	__PMC_EV_ARMV7()			\
	__PMC_EV(ARMV7, EVENT_00H)		\
	__PMC_EV(ARMV7, EVENT_01H)		\
	__PMC_EV(ARMV7, EVENT_02H)		\
	__PMC_EV(ARMV7, EVENT_03H)		\
	__PMC_EV(ARMV7, EVENT_04H)		\
	__PMC_EV(ARMV7, EVENT_05H)		\
	__PMC_EV(ARMV7, EVENT_06H)		\
	__PMC_EV(ARMV7, EVENT_07H)		\
	__PMC_EV(ARMV7, EVENT_08H)		\
	__PMC_EV(ARMV7, EVENT_09H)		\
	__PMC_EV(ARMV7, EVENT_0AH)		\
	__PMC_EV(ARMV7, EVENT_0BH)		\
	__PMC_EV(ARMV7, EVENT_0CH)		\
	__PMC_EV(ARMV7, EVENT_0DH)		\
	__PMC_EV(ARMV7, EVENT_0EH)		\
	__PMC_EV(ARMV7, EVENT_0FH)		\
	__PMC_EV(ARMV7, EVENT_10H)		\
	__PMC_EV(ARMV7, EVENT_11H)		\
	__PMC_EV(ARMV7, EVENT_12H)		\
	__PMC_EV(ARMV7, EVENT_13H)		\
	__PMC_EV(ARMV7, EVENT_14H)		\
	__PMC_EV(ARMV7, EVENT_15H)		\
	__PMC_EV(ARMV7, EVENT_16H)		\
	__PMC_EV(ARMV7, EVENT_17H)		\
	__PMC_EV(ARMV7, EVENT_18H)		\
	__PMC_EV(ARMV7, EVENT_19H)		\
	__PMC_EV(ARMV7, EVENT_1AH)		\
	__PMC_EV(ARMV7, EVENT_1BH)		\
	__PMC_EV(ARMV7, EVENT_1CH)		\
	__PMC_EV(ARMV7, EVENT_1DH)		\
	__PMC_EV(ARMV7, EVENT_1EH)		\
	__PMC_EV(ARMV7, EVENT_1FH)		\
	__PMC_EV(ARMV7, EVENT_20H)		\
	__PMC_EV(ARMV7, EVENT_21H)		\
	__PMC_EV(ARMV7, EVENT_22H)		\
	__PMC_EV(ARMV7, EVENT_23H)		\
	__PMC_EV(ARMV7, EVENT_24H)		\
	__PMC_EV(ARMV7, EVENT_25H)		\
	__PMC_EV(ARMV7, EVENT_26H)		\
	__PMC_EV(ARMV7, EVENT_27H)		\
	__PMC_EV(ARMV7, EVENT_28H)		\
	__PMC_EV(ARMV7, EVENT_29H)		\
	__PMC_EV(ARMV7, EVENT_2AH)		\
	__PMC_EV(ARMV7, EVENT_2BH)		\
	__PMC_EV(ARMV7, EVENT_2CH)		\
	__PMC_EV(ARMV7, EVENT_2DH)		\
	__PMC_EV(ARMV7, EVENT_2EH)		\
	__PMC_EV(ARMV7, EVENT_2FH)		\
	__PMC_EV(ARMV7, EVENT_30H)		\
	__PMC_EV(ARMV7, EVENT_31H)		\
	__PMC_EV(ARMV7, EVENT_32H)		\
	__PMC_EV(ARMV7, EVENT_33H)		\
	__PMC_EV(ARMV7, EVENT_34H)		\
	__PMC_EV(ARMV7, EVENT_35H)		\
	__PMC_EV(ARMV7, EVENT_36H)		\
	__PMC_EV(ARMV7, EVENT_37H)		\
	__PMC_EV(ARMV7, EVENT_38H)		\
	__PMC_EV(ARMV7, EVENT_39H)		\
	__PMC_EV(ARMV7, EVENT_3AH)		\
	__PMC_EV(ARMV7, EVENT_3BH)		\
	__PMC_EV(ARMV7, EVENT_3CH)		\
	__PMC_EV(ARMV7, EVENT_3DH)		\
	__PMC_EV(ARMV7, EVENT_3EH)		\
	__PMC_EV(ARMV7, EVENT_3FH)		\
	__PMC_EV(ARMV7, EVENT_40H)		\
	__PMC_EV(ARMV7, EVENT_41H)		\
	__PMC_EV(ARMV7, EVENT_42H)		\
	__PMC_EV(ARMV7, EVENT_43H)		\
	__PMC_EV(ARMV7, EVENT_44H)		\
	__PMC_EV(ARMV7, EVENT_45H)		\
	__PMC_EV(ARMV7, EVENT_46H)		\
	__PMC_EV(ARMV7, EVENT_47H)		\
	__PMC_EV(ARMV7, EVENT_48H)		\
	__PMC_EV(ARMV7, EVENT_49H)		\
	__PMC_EV(ARMV7, EVENT_4AH)		\
	__PMC_EV(ARMV7, EVENT_4BH)		\
	__PMC_EV(ARMV7, EVENT_4CH)		\
	__PMC_EV(ARMV7, EVENT_4DH)		\
	__PMC_EV(ARMV7, EVENT_4EH)		\
	__PMC_EV(ARMV7, EVENT_4FH)		\
	__PMC_EV(ARMV7, EVENT_50H)		\
	__PMC_EV(ARMV7, EVENT_51H)		\
	__PMC_EV(ARMV7, EVENT_52H)		\
	__PMC_EV(ARMV7, EVENT_53H)		\
	__PMC_EV(ARMV7, EVENT_54H)		\
	__PMC_EV(ARMV7, EVENT_55H)		\
	__PMC_EV(ARMV7, EVENT_56H)		\
	__PMC_EV(ARMV7, EVENT_57H)		\
	__PMC_EV(ARMV7, EVENT_58H)		\
	__PMC_EV(ARMV7, EVENT_59H)		\
	__PMC_EV(ARMV7, EVENT_5AH)		\
	__PMC_EV(ARMV7, EVENT_5BH)		\
	__PMC_EV(ARMV7, EVENT_5CH)		\
	__PMC_EV(ARMV7, EVENT_5DH)		\
	__PMC_EV(ARMV7, EVENT_5EH)		\
	__PMC_EV(ARMV7, EVENT_5FH)		\
	__PMC_EV(ARMV7, EVENT_60H)		\
	__PMC_EV(ARMV7, EVENT_61H)		\
	__PMC_EV(ARMV7, EVENT_62H)		\
	__PMC_EV(ARMV7, EVENT_63H)		\
	__PMC_EV(ARMV7, EVENT_64H)		\
	__PMC_EV(ARMV7, EVENT_65H)		\
	__PMC_EV(ARMV7, EVENT_66H)		\
	__PMC_EV(ARMV7, EVENT_67H)		\
	__PMC_EV(ARMV7, EVENT_68H)		\
	__PMC_EV(ARMV7, EVENT_69H)		\
	__PMC_EV(ARMV7, EVENT_6AH)		\
	__PMC_EV(ARMV7, EVENT_6BH)		\
	__PMC_EV(ARMV7, EVENT_6CH)		\
	__PMC_EV(ARMV7, EVENT_6DH)		\
	__PMC_EV(ARMV7, EVENT_6EH)		\
	__PMC_EV(ARMV7, EVENT_6FH)		\
	__PMC_EV(ARMV7, EVENT_70H)		\
	__PMC_EV(ARMV7, EVENT_71H)		\
	__PMC_EV(ARMV7, EVENT_72H)		\
	__PMC_EV(ARMV7, EVENT_73H)		\
	__PMC_EV(ARMV7, EVENT_74H)		\
	__PMC_EV(ARMV7, EVENT_75H)		\
	__PMC_EV(ARMV7, EVENT_76H)		\
	__PMC_EV(ARMV7, EVENT_77H)		\
	__PMC_EV(ARMV7, EVENT_78H)		\
	__PMC_EV(ARMV7, EVENT_79H)		\
	__PMC_EV(ARMV7, EVENT_7AH)		\
	__PMC_EV(ARMV7, EVENT_7BH)		\
	__PMC_EV(ARMV7, EVENT_7CH)		\
	__PMC_EV(ARMV7, EVENT_7DH)		\
	__PMC_EV(ARMV7, EVENT_7EH)		\
	__PMC_EV(ARMV7, EVENT_7FH)		\
	__PMC_EV(ARMV7, EVENT_80H)		\
	__PMC_EV(ARMV7, EVENT_81H)		\
	__PMC_EV(ARMV7, EVENT_82H)		\
	__PMC_EV(ARMV7, EVENT_83H)		\
	__PMC_EV(ARMV7, EVENT_84H)		\
	__PMC_EV(ARMV7, EVENT_85H)		\
	__PMC_EV(ARMV7, EVENT_86H)		\
	__PMC_EV(ARMV7, EVENT_87H)		\
	__PMC_EV(ARMV7, EVENT_88H)		\
	__PMC_EV(ARMV7, EVENT_89H)		\
	__PMC_EV(ARMV7, EVENT_8AH)		\
	__PMC_EV(ARMV7, EVENT_8BH)		\
	__PMC_EV(ARMV7, EVENT_8CH)		\
	__PMC_EV(ARMV7, EVENT_8DH)		\
	__PMC_EV(ARMV7, EVENT_8EH)		\
	__PMC_EV(ARMV7, EVENT_8FH)		\
	__PMC_EV(ARMV7, EVENT_90H)		\
	__PMC_EV(ARMV7, EVENT_91H)		\
	__PMC_EV(ARMV7, EVENT_92H)		\
	__PMC_EV(ARMV7, EVENT_93H)		\
	__PMC_EV(ARMV7, EVENT_94H)		\
	__PMC_EV(ARMV7, EVENT_95H)		\
	__PMC_EV(ARMV7, EVENT_96H)		\
	__PMC_EV(ARMV7, EVENT_97H)		\
	__PMC_EV(ARMV7, EVENT_98H)		\
	__PMC_EV(ARMV7, EVENT_99H)		\
	__PMC_EV(ARMV7, EVENT_9AH)		\
	__PMC_EV(ARMV7, EVENT_9BH)		\
	__PMC_EV(ARMV7, EVENT_9CH)		\
	__PMC_EV(ARMV7, EVENT_9DH)		\
	__PMC_EV(ARMV7, EVENT_9EH)		\
	__PMC_EV(ARMV7, EVENT_9FH)		\
	__PMC_EV(ARMV7, EVENT_A0H)		\
	__PMC_EV(ARMV7, EVENT_A1H)		\
	__PMC_EV(ARMV7, EVENT_A2H)		\
	__PMC_EV(ARMV7, EVENT_A3H)		\
	__PMC_EV(ARMV7, EVENT_A4H)		\
	__PMC_EV(ARMV7, EVENT_A5H)		\
	__PMC_EV(ARMV7, EVENT_A6H)		\
	__PMC_EV(ARMV7, EVENT_A7H)		\
	__PMC_EV(ARMV7, EVENT_A8H)		\
	__PMC_EV(ARMV7, EVENT_A9H)		\
	__PMC_EV(ARMV7, EVENT_AAH)		\
	__PMC_EV(ARMV7, EVENT_ABH)		\
	__PMC_EV(ARMV7, EVENT_ACH)		\
	__PMC_EV(ARMV7, EVENT_ADH)		\
	__PMC_EV(ARMV7, EVENT_AEH)		\
	__PMC_EV(ARMV7, EVENT_AFH)		\
	__PMC_EV(ARMV7, EVENT_B0H)		\
	__PMC_EV(ARMV7, EVENT_B1H)		\
	__PMC_EV(ARMV7, EVENT_B2H)		\
	__PMC_EV(ARMV7, EVENT_B3H)		\
	__PMC_EV(ARMV7, EVENT_B4H)		\
	__PMC_EV(ARMV7, EVENT_B5H)		\
	__PMC_EV(ARMV7, EVENT_B6H)		\
	__PMC_EV(ARMV7, EVENT_B7H)		\
	__PMC_EV(ARMV7, EVENT_B8H)		\
	__PMC_EV(ARMV7, EVENT_B9H)		\
	__PMC_EV(ARMV7, EVENT_BAH)		\
	__PMC_EV(ARMV7, EVENT_BBH)		\
	__PMC_EV(ARMV7, EVENT_BCH)		\
	__PMC_EV(ARMV7, EVENT_BDH)		\
	__PMC_EV(ARMV7, EVENT_BEH)		\
	__PMC_EV(ARMV7, EVENT_BFH)		\
	__PMC_EV(ARMV7, EVENT_C0H)		\
	__PMC_EV(ARMV7, EVENT_C1H)		\
	__PMC_EV(ARMV7, EVENT_C2H)		\
	__PMC_EV(ARMV7, EVENT_C3H)		\
	__PMC_EV(ARMV7, EVENT_C4H)		\
	__PMC_EV(ARMV7, EVENT_C5H)		\
	__PMC_EV(ARMV7, EVENT_C6H)		\
	__PMC_EV(ARMV7, EVENT_C7H)		\
	__PMC_EV(ARMV7, EVENT_C8H)		\
	__PMC_EV(ARMV7, EVENT_C9H)		\
	__PMC_EV(ARMV7, EVENT_CAH)		\
	__PMC_EV(ARMV7, EVENT_CBH)		\
	__PMC_EV(ARMV7, EVENT_CCH)		\
	__PMC_EV(ARMV7, EVENT_CDH)		\
	__PMC_EV(ARMV7, EVENT_CEH)		\
	__PMC_EV(ARMV7, EVENT_CFH)		\
	__PMC_EV(ARMV7, EVENT_D0H)		\
	__PMC_EV(ARMV7, EVENT_D1H)		\
	__PMC_EV(ARMV7, EVENT_D2H)		\
	__PMC_EV(ARMV7, EVENT_D3H)		\
	__PMC_EV(ARMV7, EVENT_D4H)		\
	__PMC_EV(ARMV7, EVENT_D5H)		\
	__PMC_EV(ARMV7, EVENT_D6H)		\
	__PMC_EV(ARMV7, EVENT_D7H)		\
	__PMC_EV(ARMV7, EVENT_D8H)		\
	__PMC_EV(ARMV7, EVENT_D9H)		\
	__PMC_EV(ARMV7, EVENT_DAH)		\
	__PMC_EV(ARMV7, EVENT_DBH)		\
	__PMC_EV(ARMV7, EVENT_DCH)		\
	__PMC_EV(ARMV7, EVENT_DDH)		\
	__PMC_EV(ARMV7, EVENT_DEH)		\
	__PMC_EV(ARMV7, EVENT_DFH)		\
	__PMC_EV(ARMV7, EVENT_E0H)		\
	__PMC_EV(ARMV7, EVENT_E1H)		\
	__PMC_EV(ARMV7, EVENT_E2H)		\
	__PMC_EV(ARMV7, EVENT_E3H)		\
	__PMC_EV(ARMV7, EVENT_E4H)		\
	__PMC_EV(ARMV7, EVENT_E5H)		\
	__PMC_EV(ARMV7, EVENT_E6H)		\
	__PMC_EV(ARMV7, EVENT_E7H)		\
	__PMC_EV(ARMV7, EVENT_E8H)		\
	__PMC_EV(ARMV7, EVENT_E9H)		\
	__PMC_EV(ARMV7, EVENT_EAH)		\
	__PMC_EV(ARMV7, EVENT_EBH)		\
	__PMC_EV(ARMV7, EVENT_ECH)		\
	__PMC_EV(ARMV7, EVENT_EDH)		\
	__PMC_EV(ARMV7, EVENT_EEH)		\
	__PMC_EV(ARMV7, EVENT_EFH)		\
	__PMC_EV(ARMV7, EVENT_F0H)		\
	__PMC_EV(ARMV7, EVENT_F1H)		\
	__PMC_EV(ARMV7, EVENT_F2H)		\
	__PMC_EV(ARMV7, EVENT_F3H)		\
	__PMC_EV(ARMV7, EVENT_F4H)		\
	__PMC_EV(ARMV7, EVENT_F5H)		\
	__PMC_EV(ARMV7, EVENT_F6H)		\
	__PMC_EV(ARMV7, EVENT_F7H)		\
	__PMC_EV(ARMV7, EVENT_F8H)		\
	__PMC_EV(ARMV7, EVENT_F9H)		\
	__PMC_EV(ARMV7, EVENT_FAH)		\
	__PMC_EV(ARMV7, EVENT_FBH)		\
	__PMC_EV(ARMV7, EVENT_FCH)		\
	__PMC_EV(ARMV7, EVENT_FDH)		\
	__PMC_EV(ARMV7, EVENT_FEH)		\
	__PMC_EV(ARMV7, EVENT_FFH)

#define	PMC_EV_ARMV7_FIRST	PMC_EV_ARMV7_EVENT_00H
#define	PMC_EV_ARMV7_LAST	PMC_EV_ARMV7_EVENT_FFH

#define	__PMC_EV_ALIAS_ARMV7_COMMON()					\
	__PMC_EV_ALIAS("PMNC_SW_INCR",		ARMV7_EVENT_00H)	\
	__PMC_EV_ALIAS("L1_ICACHE_REFILL",	ARMV7_EVENT_01H)	\
	__PMC_EV_ALIAS("ITLB_REFILL",		ARMV7_EVENT_02H)	\
	__PMC_EV_ALIAS("L1_DCACHE_REFILL",	ARMV7_EVENT_03H)	\
	__PMC_EV_ALIAS("L1_DCACHE_ACCESS",	ARMV7_EVENT_04H)	\
	__PMC_EV_ALIAS("DTLB_REFILL",		ARMV7_EVENT_05H)	\
	__PMC_EV_ALIAS("MEM_READ",		ARMV7_EVENT_06H)	\
	__PMC_EV_ALIAS("MEM_WRITE",		ARMV7_EVENT_07H)	\
	__PMC_EV_ALIAS("EXC_TAKEN",		ARMV7_EVENT_09H)	\
	__PMC_EV_ALIAS("EXC_EXECUTED",		ARMV7_EVENT_0AH)	\
	__PMC_EV_ALIAS("CID_WRITE",		ARMV7_EVENT_0BH)	\
	__PMC_EV_ALIAS("PC_WRITE",		ARMV7_EVENT_0CH)	\
	__PMC_EV_ALIAS("PC_IMM_BRANCH",		ARMV7_EVENT_0DH)	\
	__PMC_EV_ALIAS("MEM_UNALIGNED_ACCESS",	ARMV7_EVENT_0FH)	\
	__PMC_EV_ALIAS("PC_BRANCH_MIS_PRED",	ARMV7_EVENT_10H)	\
	__PMC_EV_ALIAS("CLOCK_CYCLES",		ARMV7_EVENT_11H)	\
	__PMC_EV_ALIAS("PC_BRANCH_PRED",	ARMV7_EVENT_12H)

#define	__PMC_EV_ALIAS_ARMV7_COMMON_A8()				\
	__PMC_EV_ALIAS_ARMV7_COMMON()					\
	__PMC_EV_ALIAS("INSTR_EXECUTED",	ARMV7_EVENT_08H)	\
	__PMC_EV_ALIAS("PC_PROC_RETURN",	ARMV7_EVENT_0EH)	\
	__PMC_EV_ALIAS("MEM_ACCESS",		ARMV7_EVENT_13H)	\
	__PMC_EV_ALIAS("L1_ICACHE_ACCESS",	ARMV7_EVENT_14H)	\
	__PMC_EV_ALIAS("L1_DCACHE_WB",		ARMV7_EVENT_15H)	\
	__PMC_EV_ALIAS("L2_CACHE_ACCESS",	ARMV7_EVENT_16H)	\
	__PMC_EV_ALIAS("L2_CACHE_REFILL",	ARMV7_EVENT_17H)	\
	__PMC_EV_ALIAS("L2_CACHE_WB",		ARMV7_EVENT_18H)	\
	__PMC_EV_ALIAS("BUS_ACCESS",		ARMV7_EVENT_19H)	\
	__PMC_EV_ALIAS("MEM_ERROR",		ARMV7_EVENT_1AH)	\
	__PMC_EV_ALIAS("INSTR_SPEC",		ARMV7_EVENT_1BH)	\
	__PMC_EV_ALIAS("TTBR_WRITE",		ARMV7_EVENT_1CH)	\
	__PMC_EV_ALIAS("BUS_CYCLES",		ARMV7_EVENT_1DH)	\
	__PMC_EV_ALIAS("CPU_CYCLES",		ARMV7_EVENT_FFH)

#define	__PMC_EV_ALIAS_ARMV7_CORTEX_A8()				 \
	__PMC_EV_ALIAS_ARMV7_COMMON_A8()				 \
	__PMC_EV_ALIAS("WRITE_BUF_FULL",		ARMV7_EVENT_40H) \
	__PMC_EV_ALIAS("L2_STORE_MERGED",		ARMV7_EVENT_41H) \
	__PMC_EV_ALIAS("L2_STORE_BUFFERABLE",		ARMV7_EVENT_42H) \
	__PMC_EV_ALIAS("L2_ACCESS",			ARMV7_EVENT_43H) \
	__PMC_EV_ALIAS("L2_CACHE_MISS",			ARMV7_EVENT_44H) \
	__PMC_EV_ALIAS("AXI_READ",			ARMV7_EVENT_45H) \
	__PMC_EV_ALIAS("AXI_WRITE",			ARMV7_EVENT_46H) \
	__PMC_EV_ALIAS("MEM_REPLAY_EVT",		ARMV7_EVENT_47H) \
	__PMC_EV_ALIAS("MEM_UNALIGNED_ACCESS_REPLAY",	ARMV7_EVENT_48H) \
	__PMC_EV_ALIAS("L1_DCACHE_HASH_MISS",		ARMV7_EVENT_49H) \
	__PMC_EV_ALIAS("L1_ICACHE_HASH_MISS",		ARMV7_EVENT_4AH) \
	__PMC_EV_ALIAS("L1_CACHE_PAGECOL_ALIAS",	ARMV7_EVENT_4BH) \
	__PMC_EV_ALIAS("L1_DCACHE_NEON_ACCESS",		ARMV7_EVENT_4CH) \
	__PMC_EV_ALIAS("L1_DCACHE_NEON_CACHEABLE",	ARMV7_EVENT_4DH) \
	__PMC_EV_ALIAS("L2_CACHE_NEON_MEM_ACCESS",	ARMV7_EVENT_4EH) \
	__PMC_EV_ALIAS("L2_CACHE_NEON_HIT",		ARMV7_EVENT_4FH) \
	__PMC_EV_ALIAS("L1_CACHE_ACCESS_NOCP15",	ARMV7_EVENT_50H) \
	__PMC_EV_ALIAS("RET_STACK_MISPREDICT",		ARMV7_EVENT_51H) \
	__PMC_EV_ALIAS("BRANCH_DIR_MISPREDICT",		ARMV7_EVENT_52H) \
	__PMC_EV_ALIAS("PRED_BRANCH_PRED_TAKEN",	ARMV7_EVENT_53H) \
	__PMC_EV_ALIAS("PRED_BRANCH_EXEC_TAKEN",	ARMV7_EVENT_54H) \
	__PMC_EV_ALIAS("OPS_ISSUED",			ARMV7_EVENT_55H) \
	__PMC_EV_ALIAS("CYCLES_NO_INSTRUCTION",		ARMV7_EVENT_56H) \
	__PMC_EV_ALIAS("INSTRUCTIONS_ISSUED_CYCLE",	ARMV7_EVENT_57H) \
	__PMC_EV_ALIAS("CYCLES_STALLED_NEON_MRC",	ARMV7_EVENT_58H) \
	__PMC_EV_ALIAS("CYCLES_STALLED_NEON_FULLQ",	ARMV7_EVENT_59H) \
	__PMC_EV_ALIAS("CYCLES_NONIDLE_NEON_INT",	ARMV7_EVENT_5AH) \
	__PMC_EV_ALIAS("PMUEXTIN0_EVT",			ARMV7_EVENT_70H) \
	__PMC_EV_ALIAS("PMUEXTIN1_EVT",			ARMV7_EVENT_71H) \
	__PMC_EV_ALIAS("PMUEXTIN_EVT",			ARMV7_EVENT_72H)
#define	PMC_EV_ARMV7_CORTEX_A8_FIRST	PMC_EV_ARMV7_PMNC_SW_INCR
#define	PMC_EV_ARMV7_CORTEX_A8_LAST	PMC_EV_ARMV7_PMUEXTIN_EVT

#define	__PMC_EV_ALIAS_ARMV7_CORTEX_A9()					\
	__PMC_EV_ALIAS_ARMV7_COMMON()						\
	__PMC_EV_ALIAS("JAVA_BYTECODE",			ARMV7_EVENT_40H)	\
	__PMC_EV_ALIAS("SOFTWARE_JAVA_BYTECODE",	ARMV7_EVENT_41H)	\
	__PMC_EV_ALIAS("JAZELLE_BACKWARD_BRANCH",	ARMV7_EVENT_42H)	\
	__PMC_EV_ALIAS("COHERENT_LINEFILL_MISSC",	ARMV7_EVENT_50H)	\
	__PMC_EV_ALIAS("COHERENT_LINEFILL_HITC",	ARMV7_EVENT_51H)	\
	__PMC_EV_ALIAS("INSTR_CACHE_DEPENDENT_STALL",	ARMV7_EVENT_60H)	\
	__PMC_EV_ALIAS("DATA_CACHE_DEPENDENT_STALL",	ARMV7_EVENT_61H)	\
	__PMC_EV_ALIAS("MAIN_TLB_MISS_STALL",		ARMV7_EVENT_62H)	\
	__PMC_EV_ALIAS("STREX_PASSED",			ARMV7_EVENT_63H)	\
	__PMC_EV_ALIAS("STREX_FAILED",			ARMV7_EVENT_64H)	\
	__PMC_EV_ALIAS("DATA_EVICTION",			ARMV7_EVENT_65H)	\
	__PMC_EV_ALIAS("ISSUE_DNOT_DISPATCH_ANY_INSTR",	ARMV7_EVENT_66H)	\
	__PMC_EV_ALIAS("ISSUE_IS_EMPTY",		ARMV7_EVENT_67H)	\
	__PMC_EV_ALIAS("INSTR_RENAMED",			ARMV7_EVENT_68H)	\
	__PMC_EV_ALIAS("PREDICTABLE_FUNCTION_RETURN",	ARMV7_EVENT_6EH)	\
	__PMC_EV_ALIAS("MAIN_EXECUTION_UNIT_PIPE",	ARMV7_EVENT_70H)	\
	__PMC_EV_ALIAS("SECOND_EXECUTION_UNIT_PIPE",	ARMV7_EVENT_71H)	\
	__PMC_EV_ALIAS("LOAD_STORE_PIPE",		ARMV7_EVENT_72H)	\
	__PMC_EV_ALIAS("FLOATING_POINT_INSTR_RENAMED",	ARMV7_EVENT_73H)	\
	__PMC_EV_ALIAS("NEON_INSTRS_RENAMED",		ARMV7_EVENT_74H)	\
	__PMC_EV_ALIAS("PLD_STALL",			ARMV7_EVENT_80H)	\
	__PMC_EV_ALIAS("WRITE_STALL",			ARMV7_EVENT_81H)	\
	__PMC_EV_ALIAS("INSTR_MAIN_TLB_MISS_STALL",	ARMV7_EVENT_82H)	\
	__PMC_EV_ALIAS("DATA_MAIN_TLB_MISS_STALL",	ARMV7_EVENT_83H)	\
	__PMC_EV_ALIAS("INSTR_MICRO_TLB_MISS_STALL",	ARMV7_EVENT_84H)	\
	__PMC_EV_ALIAS("DATA_MICRO_TLB_MISS_STALL",	ARMV7_EVENT_85H)	\
	__PMC_EV_ALIAS("DMB_STALL",			ARMV7_EVENT_86H)	\
	__PMC_EV_ALIAS("INTEGER_CORE_CLOCK_ENABLED",	ARMV7_EVENT_8AH)	\
	__PMC_EV_ALIAS("DATA_ENGINE_CLOCK_ENABLED",	ARMV7_EVENT_8BH)	\
	__PMC_EV_ALIAS("ISB",				ARMV7_EVENT_90H)	\
	__PMC_EV_ALIAS("DSB",				ARMV7_EVENT_91H)	\
	__PMC_EV_ALIAS("DMB",				ARMV7_EVENT_92H)	\
	__PMC_EV_ALIAS("EXTERNAL_INTERRUPT",		ARMV7_EVENT_93H)	\
	__PMC_EV_ALIAS("PLE_CACHE_LINE_REQ_COMPLETED",	ARMV7_EVENT_A0H)	\
	__PMC_EV_ALIAS("PLE_CACHE_LINE_REQ_SKIPPED",	ARMV7_EVENT_A1H)	\
	__PMC_EV_ALIAS("PLE_FIFO_FLUSH",		ARMV7_EVENT_A2H)	\
	__PMC_EV_ALIAS("PLE_REQUEST_COMPLETED",		ARMV7_EVENT_A3H)	\
	__PMC_EV_ALIAS("PLE_FIFO_OVERFLOW",		ARMV7_EVENT_A4H)	\
	__PMC_EV_ALIAS("PLE_REQUEST_PROGRAMMED",	ARMV7_EVENT_A5H)

/*
 * ARMv8 Events
 */

#define	__PMC_EV_ARMV8()			\
	__PMC_EV(ARMV8, EVENT_00H)		\
	__PMC_EV(ARMV8, EVENT_01H)		\
	__PMC_EV(ARMV8, EVENT_02H)		\
	__PMC_EV(ARMV8, EVENT_03H)		\
	__PMC_EV(ARMV8, EVENT_04H)		\
	__PMC_EV(ARMV8, EVENT_05H)		\
	__PMC_EV(ARMV8, EVENT_06H)		\
	__PMC_EV(ARMV8, EVENT_07H)		\
	__PMC_EV(ARMV8, EVENT_08H)		\
	__PMC_EV(ARMV8, EVENT_09H)		\
	__PMC_EV(ARMV8, EVENT_0AH)		\
	__PMC_EV(ARMV8, EVENT_0BH)		\
	__PMC_EV(ARMV8, EVENT_0CH)		\
	__PMC_EV(ARMV8, EVENT_0DH)		\
	__PMC_EV(ARMV8, EVENT_0EH)		\
	__PMC_EV(ARMV8, EVENT_0FH)		\
	__PMC_EV(ARMV8, EVENT_10H)		\
	__PMC_EV(ARMV8, EVENT_11H)		\
	__PMC_EV(ARMV8, EVENT_12H)		\
	__PMC_EV(ARMV8, EVENT_13H)		\
	__PMC_EV(ARMV8, EVENT_14H)		\
	__PMC_EV(ARMV8, EVENT_15H)		\
	__PMC_EV(ARMV8, EVENT_16H)		\
	__PMC_EV(ARMV8, EVENT_17H)		\
	__PMC_EV(ARMV8, EVENT_18H)		\
	__PMC_EV(ARMV8, EVENT_19H)		\
	__PMC_EV(ARMV8, EVENT_1AH)		\
	__PMC_EV(ARMV8, EVENT_1BH)		\
	__PMC_EV(ARMV8, EVENT_1CH)		\
	__PMC_EV(ARMV8, EVENT_1DH)		\
	__PMC_EV(ARMV8, EVENT_1EH)		\
	__PMC_EV(ARMV8, EVENT_1FH)		\
	__PMC_EV(ARMV8, EVENT_20H)		\
	__PMC_EV(ARMV8, EVENT_21H)		\
	__PMC_EV(ARMV8, EVENT_22H)		\
	__PMC_EV(ARMV8, EVENT_23H)		\
	__PMC_EV(ARMV8, EVENT_24H)		\
	__PMC_EV(ARMV8, EVENT_25H)		\
	__PMC_EV(ARMV8, EVENT_26H)		\
	__PMC_EV(ARMV8, EVENT_27H)		\
	__PMC_EV(ARMV8, EVENT_28H)		\
	__PMC_EV(ARMV8, EVENT_29H)		\
	__PMC_EV(ARMV8, EVENT_2AH)		\
	__PMC_EV(ARMV8, EVENT_2BH)		\
	__PMC_EV(ARMV8, EVENT_2CH)		\
	__PMC_EV(ARMV8, EVENT_2DH)		\
	__PMC_EV(ARMV8, EVENT_2EH)		\
	__PMC_EV(ARMV8, EVENT_2FH)		\
	__PMC_EV(ARMV8, EVENT_30H)		\
	__PMC_EV(ARMV8, EVENT_31H)		\
	__PMC_EV(ARMV8, EVENT_32H)		\
	__PMC_EV(ARMV8, EVENT_33H)		\
	__PMC_EV(ARMV8, EVENT_34H)		\
	__PMC_EV(ARMV8, EVENT_35H)		\
	__PMC_EV(ARMV8, EVENT_36H)		\
	__PMC_EV(ARMV8, EVENT_37H)		\
	__PMC_EV(ARMV8, EVENT_38H)		\
	__PMC_EV(ARMV8, EVENT_39H)		\
	__PMC_EV(ARMV8, EVENT_3AH)		\
	__PMC_EV(ARMV8, EVENT_3BH)		\
	__PMC_EV(ARMV8, EVENT_3CH)		\
	__PMC_EV(ARMV8, EVENT_3DH)		\
	__PMC_EV(ARMV8, EVENT_3EH)		\
	__PMC_EV(ARMV8, EVENT_3FH)		\
	__PMC_EV(ARMV8, EVENT_40H)		\
	__PMC_EV(ARMV8, EVENT_41H)		\
	__PMC_EV(ARMV8, EVENT_42H)		\
	__PMC_EV(ARMV8, EVENT_43H)		\
	__PMC_EV(ARMV8, EVENT_44H)		\
	__PMC_EV(ARMV8, EVENT_45H)		\
	__PMC_EV(ARMV8, EVENT_46H)		\
	__PMC_EV(ARMV8, EVENT_47H)		\
	__PMC_EV(ARMV8, EVENT_48H)		\
	__PMC_EV(ARMV8, EVENT_49H)		\
	__PMC_EV(ARMV8, EVENT_4AH)		\
	__PMC_EV(ARMV8, EVENT_4BH)		\
	__PMC_EV(ARMV8, EVENT_4CH)		\
	__PMC_EV(ARMV8, EVENT_4DH)		\
	__PMC_EV(ARMV8, EVENT_4EH)		\
	__PMC_EV(ARMV8, EVENT_4FH)		\
	__PMC_EV(ARMV8, EVENT_50H)		\
	__PMC_EV(ARMV8, EVENT_51H)		\
	__PMC_EV(ARMV8, EVENT_52H)		\
	__PMC_EV(ARMV8, EVENT_53H)		\
	__PMC_EV(ARMV8, EVENT_54H)		\
	__PMC_EV(ARMV8, EVENT_55H)		\
	__PMC_EV(ARMV8, EVENT_56H)		\
	__PMC_EV(ARMV8, EVENT_57H)		\
	__PMC_EV(ARMV8, EVENT_58H)		\
	__PMC_EV(ARMV8, EVENT_59H)		\
	__PMC_EV(ARMV8, EVENT_5AH)		\
	__PMC_EV(ARMV8, EVENT_5BH)		\
	__PMC_EV(ARMV8, EVENT_5CH)		\
	__PMC_EV(ARMV8, EVENT_5DH)		\
	__PMC_EV(ARMV8, EVENT_5EH)		\
	__PMC_EV(ARMV8, EVENT_5FH)		\
	__PMC_EV(ARMV8, EVENT_60H)		\
	__PMC_EV(ARMV8, EVENT_61H)		\
	__PMC_EV(ARMV8, EVENT_62H)		\
	__PMC_EV(ARMV8, EVENT_63H)		\
	__PMC_EV(ARMV8, EVENT_64H)		\
	__PMC_EV(ARMV8, EVENT_65H)		\
	__PMC_EV(ARMV8, EVENT_66H)		\
	__PMC_EV(ARMV8, EVENT_67H)		\
	__PMC_EV(ARMV8, EVENT_68H)		\
	__PMC_EV(ARMV8, EVENT_69H)		\
	__PMC_EV(ARMV8, EVENT_6AH)		\
	__PMC_EV(ARMV8, EVENT_6BH)		\
	__PMC_EV(ARMV8, EVENT_6CH)		\
	__PMC_EV(ARMV8, EVENT_6DH)		\
	__PMC_EV(ARMV8, EVENT_6EH)		\
	__PMC_EV(ARMV8, EVENT_6FH)		\
	__PMC_EV(ARMV8, EVENT_70H)		\
	__PMC_EV(ARMV8, EVENT_71H)		\
	__PMC_EV(ARMV8, EVENT_72H)		\
	__PMC_EV(ARMV8, EVENT_73H)		\
	__PMC_EV(ARMV8, EVENT_74H)		\
	__PMC_EV(ARMV8, EVENT_75H)		\
	__PMC_EV(ARMV8, EVENT_76H)		\
	__PMC_EV(ARMV8, EVENT_77H)		\
	__PMC_EV(ARMV8, EVENT_78H)		\
	__PMC_EV(ARMV8, EVENT_79H)		\
	__PMC_EV(ARMV8, EVENT_7AH)		\
	__PMC_EV(ARMV8, EVENT_7BH)		\
	__PMC_EV(ARMV8, EVENT_7CH)		\
	__PMC_EV(ARMV8, EVENT_7DH)		\
	__PMC_EV(ARMV8, EVENT_7EH)		\
	__PMC_EV(ARMV8, EVENT_7FH)		\
	__PMC_EV(ARMV8, EVENT_80H)		\
	__PMC_EV(ARMV8, EVENT_81H)		\
	__PMC_EV(ARMV8, EVENT_82H)		\
	__PMC_EV(ARMV8, EVENT_83H)		\
	__PMC_EV(ARMV8, EVENT_84H)		\
	__PMC_EV(ARMV8, EVENT_85H)		\
	__PMC_EV(ARMV8, EVENT_86H)		\
	__PMC_EV(ARMV8, EVENT_87H)		\
	__PMC_EV(ARMV8, EVENT_88H)		\
	__PMC_EV(ARMV8, EVENT_89H)		\
	__PMC_EV(ARMV8, EVENT_8AH)		\
	__PMC_EV(ARMV8, EVENT_8BH)		\
	__PMC_EV(ARMV8, EVENT_8CH)		\
	__PMC_EV(ARMV8, EVENT_8DH)		\
	__PMC_EV(ARMV8, EVENT_8EH)		\
	__PMC_EV(ARMV8, EVENT_8FH)		\
	__PMC_EV(ARMV8, EVENT_90H)		\
	__PMC_EV(ARMV8, EVENT_91H)		\
	__PMC_EV(ARMV8, EVENT_92H)		\
	__PMC_EV(ARMV8, EVENT_93H)		\
	__PMC_EV(ARMV8, EVENT_94H)		\
	__PMC_EV(ARMV8, EVENT_95H)		\
	__PMC_EV(ARMV8, EVENT_96H)		\
	__PMC_EV(ARMV8, EVENT_97H)		\
	__PMC_EV(ARMV8, EVENT_98H)		\
	__PMC_EV(ARMV8, EVENT_99H)		\
	__PMC_EV(ARMV8, EVENT_9AH)		\
	__PMC_EV(ARMV8, EVENT_9BH)		\
	__PMC_EV(ARMV8, EVENT_9CH)		\
	__PMC_EV(ARMV8, EVENT_9DH)		\
	__PMC_EV(ARMV8, EVENT_9EH)		\
	__PMC_EV(ARMV8, EVENT_9FH)		\
	__PMC_EV(ARMV8, EVENT_A0H)		\
	__PMC_EV(ARMV8, EVENT_A1H)		\
	__PMC_EV(ARMV8, EVENT_A2H)		\
	__PMC_EV(ARMV8, EVENT_A3H)		\
	__PMC_EV(ARMV8, EVENT_A4H)		\
	__PMC_EV(ARMV8, EVENT_A5H)		\
	__PMC_EV(ARMV8, EVENT_A6H)		\
	__PMC_EV(ARMV8, EVENT_A7H)		\
	__PMC_EV(ARMV8, EVENT_A8H)		\
	__PMC_EV(ARMV8, EVENT_A9H)		\
	__PMC_EV(ARMV8, EVENT_AAH)		\
	__PMC_EV(ARMV8, EVENT_ABH)		\
	__PMC_EV(ARMV8, EVENT_ACH)		\
	__PMC_EV(ARMV8, EVENT_ADH)		\
	__PMC_EV(ARMV8, EVENT_AEH)		\
	__PMC_EV(ARMV8, EVENT_AFH)		\
	__PMC_EV(ARMV8, EVENT_B0H)		\
	__PMC_EV(ARMV8, EVENT_B1H)		\
	__PMC_EV(ARMV8, EVENT_B2H)		\
	__PMC_EV(ARMV8, EVENT_B3H)		\
	__PMC_EV(ARMV8, EVENT_B4H)		\
	__PMC_EV(ARMV8, EVENT_B5H)		\
	__PMC_EV(ARMV8, EVENT_B6H)		\
	__PMC_EV(ARMV8, EVENT_B7H)		\
	__PMC_EV(ARMV8, EVENT_B8H)		\
	__PMC_EV(ARMV8, EVENT_B9H)		\
	__PMC_EV(ARMV8, EVENT_BAH)		\
	__PMC_EV(ARMV8, EVENT_BBH)		\
	__PMC_EV(ARMV8, EVENT_BCH)		\
	__PMC_EV(ARMV8, EVENT_BDH)		\
	__PMC_EV(ARMV8, EVENT_BEH)		\
	__PMC_EV(ARMV8, EVENT_BFH)		\
	__PMC_EV(ARMV8, EVENT_C0H)		\
	__PMC_EV(ARMV8, EVENT_C1H)		\
	__PMC_EV(ARMV8, EVENT_C2H)		\
	__PMC_EV(ARMV8, EVENT_C3H)		\
	__PMC_EV(ARMV8, EVENT_C4H)		\
	__PMC_EV(ARMV8, EVENT_C5H)		\
	__PMC_EV(ARMV8, EVENT_C6H)		\
	__PMC_EV(ARMV8, EVENT_C7H)		\
	__PMC_EV(ARMV8, EVENT_C8H)		\
	__PMC_EV(ARMV8, EVENT_C9H)		\
	__PMC_EV(ARMV8, EVENT_CAH)		\
	__PMC_EV(ARMV8, EVENT_CBH)		\
	__PMC_EV(ARMV8, EVENT_CCH)		\
	__PMC_EV(ARMV8, EVENT_CDH)		\
	__PMC_EV(ARMV8, EVENT_CEH)		\
	__PMC_EV(ARMV8, EVENT_CFH)		\
	__PMC_EV(ARMV8, EVENT_D0H)		\
	__PMC_EV(ARMV8, EVENT_D1H)		\
	__PMC_EV(ARMV8, EVENT_D2H)		\
	__PMC_EV(ARMV8, EVENT_D3H)		\
	__PMC_EV(ARMV8, EVENT_D4H)		\
	__PMC_EV(ARMV8, EVENT_D5H)		\
	__PMC_EV(ARMV8, EVENT_D6H)		\
	__PMC_EV(ARMV8, EVENT_D7H)		\
	__PMC_EV(ARMV8, EVENT_D8H)		\
	__PMC_EV(ARMV8, EVENT_D9H)		\
	__PMC_EV(ARMV8, EVENT_DAH)		\
	__PMC_EV(ARMV8, EVENT_DBH)		\
	__PMC_EV(ARMV8, EVENT_DCH)		\
	__PMC_EV(ARMV8, EVENT_DDH)		\
	__PMC_EV(ARMV8, EVENT_DEH)		\
	__PMC_EV(ARMV8, EVENT_DFH)		\
	__PMC_EV(ARMV8, EVENT_E0H)		\
	__PMC_EV(ARMV8, EVENT_E1H)		\
	__PMC_EV(ARMV8, EVENT_E2H)		\
	__PMC_EV(ARMV8, EVENT_E3H)		\
	__PMC_EV(ARMV8, EVENT_E4H)		\
	__PMC_EV(ARMV8, EVENT_E5H)		\
	__PMC_EV(ARMV8, EVENT_E6H)		\
	__PMC_EV(ARMV8, EVENT_E7H)		\
	__PMC_EV(ARMV8, EVENT_E8H)		\
	__PMC_EV(ARMV8, EVENT_E9H)		\
	__PMC_EV(ARMV8, EVENT_EAH)		\
	__PMC_EV(ARMV8, EVENT_EBH)		\
	__PMC_EV(ARMV8, EVENT_ECH)		\
	__PMC_EV(ARMV8, EVENT_EDH)		\
	__PMC_EV(ARMV8, EVENT_EEH)		\
	__PMC_EV(ARMV8, EVENT_EFH)		\
	__PMC_EV(ARMV8, EVENT_F0H)		\
	__PMC_EV(ARMV8, EVENT_F1H)		\
	__PMC_EV(ARMV8, EVENT_F2H)		\
	__PMC_EV(ARMV8, EVENT_F3H)		\
	__PMC_EV(ARMV8, EVENT_F4H)		\
	__PMC_EV(ARMV8, EVENT_F5H)		\
	__PMC_EV(ARMV8, EVENT_F6H)		\
	__PMC_EV(ARMV8, EVENT_F7H)		\
	__PMC_EV(ARMV8, EVENT_F8H)		\
	__PMC_EV(ARMV8, EVENT_F9H)		\
	__PMC_EV(ARMV8, EVENT_FAH)		\
	__PMC_EV(ARMV8, EVENT_FBH)		\
	__PMC_EV(ARMV8, EVENT_FCH)		\
	__PMC_EV(ARMV8, EVENT_FDH)		\
	__PMC_EV(ARMV8, EVENT_FEH)		\
	__PMC_EV(ARMV8, EVENT_FFH)

#define	PMC_EV_ARMV8_FIRST	PMC_EV_ARMV8_EVENT_00H
#define	PMC_EV_ARMV8_LAST	PMC_EV_ARMV8_EVENT_FFH

#define	__PMC_EV_ALIAS_ARMV8_COMMON()					\
	__PMC_EV_ALIAS("SW_INCR",		ARMV8_EVENT_00H)	\
	__PMC_EV_ALIAS("L1I_CACHE_REFILL",	ARMV8_EVENT_01H)	\
	__PMC_EV_ALIAS("L1I_TLB_REFILL",	ARMV8_EVENT_02H)	\
	__PMC_EV_ALIAS("L1D_CACHE_REFILL",	ARMV8_EVENT_03H)	\
	__PMC_EV_ALIAS("L1D_CACHE",		ARMV8_EVENT_04H)	\
	__PMC_EV_ALIAS("L1D_TLB_REFILL",	ARMV8_EVENT_05H)	\
	__PMC_EV_ALIAS("INST_RETIRED",		ARMV8_EVENT_08H)	\
	__PMC_EV_ALIAS("EXC_TAKEN",		ARMV8_EVENT_09H)	\
	__PMC_EV_ALIAS("EXC_RETURN",		ARMV8_EVENT_0AH)	\
	__PMC_EV_ALIAS("CID_WRITE_RETIRED",	ARMV8_EVENT_0BH)	\
	__PMC_EV_ALIAS("BR_MIS_PRED",		ARMV8_EVENT_10H)	\
	__PMC_EV_ALIAS("CPU_CYCLES",		ARMV8_EVENT_11H)	\
	__PMC_EV_ALIAS("BR_PRED",		ARMV8_EVENT_12H)	\
	__PMC_EV_ALIAS("MEM_ACCESS",		ARMV8_EVENT_13H)	\
	__PMC_EV_ALIAS("L1I_CACHE",		ARMV8_EVENT_14H)	\
	__PMC_EV_ALIAS("L1D_CACHE_WB",		ARMV8_EVENT_15H)	\
	__PMC_EV_ALIAS("L2D_CACHE",		ARMV8_EVENT_16H)	\
	__PMC_EV_ALIAS("L2D_CACHE_REFILL",	ARMV8_EVENT_17H)	\
	__PMC_EV_ALIAS("L2D_CACHE_WB",		ARMV8_EVENT_18H)	\
	__PMC_EV_ALIAS("BUS_ACCESS",		ARMV8_EVENT_19H)	\
	__PMC_EV_ALIAS("MEMORY_ERROR",		ARMV8_EVENT_1AH)	\
	__PMC_EV_ALIAS("BUS_CYCLES",		ARMV8_EVENT_1DH)	\
	__PMC_EV_ALIAS("CHAIN",			ARMV8_EVENT_1EH)	\
	__PMC_EV_ALIAS("BUS_ACCESS_LD",		ARMV8_EVENT_60H)	\
	__PMC_EV_ALIAS("BUS_ACCESS_ST",		ARMV8_EVENT_61H)	\
	__PMC_EV_ALIAS("BR_INDIRECT_SPEC",	ARMV8_EVENT_7AH)	\
	__PMC_EV_ALIAS("EXC_IRQ",		ARMV8_EVENT_86H)	\
	__PMC_EV_ALIAS("EXC_FIQ",		ARMV8_EVENT_87H)

#define	__PMC_EV_ALIAS_ARMV8_CORTEX_A53()				\
	__PMC_EV_ALIAS_ARMV8_COMMON()					\
	__PMC_EV_ALIAS("LD_RETIRED",		ARMV8_EVENT_06H)	\
	__PMC_EV_ALIAS("ST_RETIRED",		ARMV8_EVENT_07H)	\
	__PMC_EV_ALIAS("PC_WRITE_RETIRED",	ARMV8_EVENT_0CH)	\
	__PMC_EV_ALIAS("BR_IMMED_RETIRED",	ARMV8_EVENT_0DH)	\
	__PMC_EV_ALIAS("BR_RETURN_RETIRED",	ARMV8_EVENT_0EH)	\
	__PMC_EV_ALIAS("UNALIGNED_LDST_RETIRED",ARMV8_EVENT_0FH)

#define	__PMC_EV_ALIAS_ARMV8_CORTEX_A57()				\
	__PMC_EV_ALIAS_ARMV8_COMMON()					\
	__PMC_EV_ALIAS("INST_SPEC",		ARMV8_EVENT_1BH)	\
	__PMC_EV_ALIAS("TTBR_WRITE_RETIRED",	ARMV8_EVENT_1CH)	\
	__PMC_EV_ALIAS("L1D_CACHE_LD",		ARMV8_EVENT_40H)	\
	__PMC_EV_ALIAS("L1D_CACHE_ST",		ARMV8_EVENT_41H)	\
	__PMC_EV_ALIAS("L1D_CACHE_REFILL_LD",	ARMV8_EVENT_42H)	\
	__PMC_EV_ALIAS("L1D_CACHE_REFILL_ST",	ARMV8_EVENT_43H)	\
	__PMC_EV_ALIAS("L1D_CACHE_WB_VICTIM",	ARMV8_EVENT_46H)	\
	__PMC_EV_ALIAS("L1D_CACHE_WB_CLEAN",	ARMV8_EVENT_47H)	\
	__PMC_EV_ALIAS("L1D_CACHE_INVAL",	ARMV8_EVENT_48H)	\
	__PMC_EV_ALIAS("L1D_TLB_REFILL_LD",	ARMV8_EVENT_4CH)	\
	__PMC_EV_ALIAS("L1D_TLB_REFILL_ST",	ARMV8_EVENT_4DH)	\
	__PMC_EV_ALIAS("L2D_CACHE_LD",		ARMV8_EVENT_50H)	\
	__PMC_EV_ALIAS("L2D_CACHE_ST",		ARMV8_EVENT_51H)	\
	__PMC_EV_ALIAS("L2D_CACHE_REFILL_LD",	ARMV8_EVENT_52H)	\
	__PMC_EV_ALIAS("L2D_CACHE_REFILL_ST",	ARMV8_EVENT_53H)	\
	__PMC_EV_ALIAS("L2D_CACHE_WB_VICTIM",	ARMV8_EVENT_56H)	\
	__PMC_EV_ALIAS("L2D_CACHE_WB_CLEAN",	ARMV8_EVENT_57H)	\
	__PMC_EV_ALIAS("L2D_CACHE_INVAL",	ARMV8_EVENT_58H)	\
	__PMC_EV_ALIAS("BUS_ACCESS_SHARED",	ARMV8_EVENT_62H)	\
	__PMC_EV_ALIAS("BUS_ACCESS_NOT_SHARED",	ARMV8_EVENT_63H)	\
	__PMC_EV_ALIAS("BUS_ACCESS_NORMAL",	ARMV8_EVENT_64H)	\
	__PMC_EV_ALIAS("BUS_ACCESS_PERIPH",	ARMV8_EVENT_65H)	\
	__PMC_EV_ALIAS("MEM_ACCESS_LD",		ARMV8_EVENT_66H)	\
	__PMC_EV_ALIAS("MEM_ACCESS_ST",		ARMV8_EVENT_67H)	\
	__PMC_EV_ALIAS("UNALIGNED_LD_SPEC",	ARMV8_EVENT_68H)	\
	__PMC_EV_ALIAS("UNALIGNED_ST_SPEC",	ARMV8_EVENT_69H)	\
	__PMC_EV_ALIAS("UNALIGNED_LDST_SPEC",	ARMV8_EVENT_6AH)	\
	__PMC_EV_ALIAS("LDREX_SPEC",		ARMV8_EVENT_6CH)	\
	__PMC_EV_ALIAS("STREX_PASS_SPEC",	ARMV8_EVENT_6DH)	\
	__PMC_EV_ALIAS("STREX_FAIL_SPEC",	ARMV8_EVENT_6EH)	\
	__PMC_EV_ALIAS("LD_SPEC",		ARMV8_EVENT_70H)	\
	__PMC_EV_ALIAS("ST_SPEC",		ARMV8_EVENT_71H)	\
	__PMC_EV_ALIAS("LDST_SPEC",		ARMV8_EVENT_72H)	\
	__PMC_EV_ALIAS("DP_SPEC",		ARMV8_EVENT_73H)	\
	__PMC_EV_ALIAS("ASE_SPEC",		ARMV8_EVENT_74H)	\
	__PMC_EV_ALIAS("VFP_SPEC",		ARMV8_EVENT_75H)	\
	__PMC_EV_ALIAS("PC_WRITE_SPEC",		ARMV8_EVENT_76H)	\
	__PMC_EV_ALIAS("CRYPTO_SPEC",		ARMV8_EVENT_77H)	\
	__PMC_EV_ALIAS("BR_IMMED_SPEC",		ARMV8_EVENT_78H)	\
	__PMC_EV_ALIAS("BR_RETURN_SPEC",	ARMV8_EVENT_79H)	\
	__PMC_EV_ALIAS("ISB_SPEC",		ARMV8_EVENT_7CH)	\
	__PMC_EV_ALIAS("DSB_SPEC",		ARMV8_EVENT_7DH)	\
	__PMC_EV_ALIAS("DMB_SPEC",		ARMV8_EVENT_7EH)	\
	__PMC_EV_ALIAS("EXC_UNDEF",		ARMV8_EVENT_81H)	\
	__PMC_EV_ALIAS("EXC_SVC",		ARMV8_EVENT_82H)	\
	__PMC_EV_ALIAS("EXC_PABORT",		ARMV8_EVENT_83H)	\
	__PMC_EV_ALIAS("EXC_DABORT",		ARMV8_EVENT_84H)	\
	__PMC_EV_ALIAS("EXC_SMC",		ARMV8_EVENT_88H)	\
	__PMC_EV_ALIAS("EXC_HVC",		ARMV8_EVENT_8AH)	\
	__PMC_EV_ALIAS("EXC_TRAP_PABORT",	ARMV8_EVENT_8BH)	\
	__PMC_EV_ALIAS("EXC_TRAP_DABORT",	ARMV8_EVENT_8CH)	\
	__PMC_EV_ALIAS("EXC_TRAP_OTHER",	ARMV8_EVENT_8DH)	\
	__PMC_EV_ALIAS("EXC_TRAP_IRQ",		ARMV8_EVENT_8EH)	\
	__PMC_EV_ALIAS("EXC_TRAP_FIQ",		ARMV8_EVENT_8FH)	\
	__PMC_EV_ALIAS("RC_LD_SPEC",		ARMV8_EVENT_90H)	\
	__PMC_EV_ALIAS("RC_ST_SPEC",		ARMV8_EVENT_91H)

/*
 * MIPS Events from "Programming the MIPS32 24K Core Family",
 * Document Number: MD00355 Revision 04.63 December 19, 2008
 * These events are kept in the order found in Table 7.4.
 * For counters which are different between the left hand
 * column (0/2) and the right hand column (1/3) the left
 * hand is given first, e.g. BRANCH_COMPLETED and BRANCH_MISPRED
 * in the definition below.
 */

#define __PMC_EV_MIPS24K()                         \
	__PMC_EV(MIPS24K, CYCLE)                   \
	__PMC_EV(MIPS24K, INSTR_EXECUTED)          \
	__PMC_EV(MIPS24K, BRANCH_COMPLETED)        \
	__PMC_EV(MIPS24K, BRANCH_MISPRED)          \
	__PMC_EV(MIPS24K, RETURN)                  \
	__PMC_EV(MIPS24K, RETURN_MISPRED)          \
	__PMC_EV(MIPS24K, RETURN_NOT_31)           \
	__PMC_EV(MIPS24K, RETURN_NOTPRED)          \
	__PMC_EV(MIPS24K, ITLB_ACCESS)             \
	__PMC_EV(MIPS24K, ITLB_MISS)               \
	__PMC_EV(MIPS24K, DTLB_ACCESS)             \
	__PMC_EV(MIPS24K, DTLB_MISS)               \
	__PMC_EV(MIPS24K, JTLB_IACCESS)            \
	__PMC_EV(MIPS24K, JTLB_IMISS)              \
	__PMC_EV(MIPS24K, JTLB_DACCESS)            \
	__PMC_EV(MIPS24K, JTLB_DMISS)              \
	__PMC_EV(MIPS24K, IC_FETCH)                \
	__PMC_EV(MIPS24K, IC_MISS)                 \
	__PMC_EV(MIPS24K, DC_LOADSTORE)            \
	__PMC_EV(MIPS24K, DC_WRITEBACK)            \
	__PMC_EV(MIPS24K, DC_MISS)                 \
	__PMC_EV(MIPS24K, STORE_MISS)              \
	__PMC_EV(MIPS24K, LOAD_MISS)               \
	__PMC_EV(MIPS24K, INTEGER_COMPLETED)       \
	__PMC_EV(MIPS24K, FP_COMPLETED)            \
	__PMC_EV(MIPS24K, LOAD_COMPLETED)          \
	__PMC_EV(MIPS24K, STORE_COMPLETED)         \
	__PMC_EV(MIPS24K, BARRIER_COMPLETED)       \
	__PMC_EV(MIPS24K, MIPS16_COMPLETED)        \
	__PMC_EV(MIPS24K, NOP_COMPLETED)           \
	__PMC_EV(MIPS24K, INTEGER_MULDIV_COMPLETED)\
	__PMC_EV(MIPS24K, RF_STALL)                \
	__PMC_EV(MIPS24K, INSTR_REFETCH)           \
	__PMC_EV(MIPS24K, STORE_COND_COMPLETED)    \
	__PMC_EV(MIPS24K, STORE_COND_FAILED)       \
	__PMC_EV(MIPS24K, ICACHE_REQUESTS)         \
	__PMC_EV(MIPS24K, ICACHE_HIT)              \
	__PMC_EV(MIPS24K, L2_WRITEBACK)            \
	__PMC_EV(MIPS24K, L2_ACCESS)               \
	__PMC_EV(MIPS24K, L2_MISS)                 \
	__PMC_EV(MIPS24K, L2_ERR_CORRECTED)        \
	__PMC_EV(MIPS24K, EXCEPTIONS)              \
	__PMC_EV(MIPS24K, RF_CYCLES_STALLED)       \
	__PMC_EV(MIPS24K, IFU_CYCLES_STALLED)      \
	__PMC_EV(MIPS24K, ALU_CYCLES_STALLED)      \
	__PMC_EV(MIPS24K, UNCACHED_LOAD)           \
	__PMC_EV(MIPS24K, UNCACHED_STORE)          \
	__PMC_EV(MIPS24K, CP2_REG_TO_REG_COMPLETED)\
	__PMC_EV(MIPS24K, MFTC_COMPLETED)          \
	__PMC_EV(MIPS24K, IC_BLOCKED_CYCLES)       \
	__PMC_EV(MIPS24K, DC_BLOCKED_CYCLES)       \
	__PMC_EV(MIPS24K, L2_IMISS_STALL_CYCLES)   \
	__PMC_EV(MIPS24K, L2_DMISS_STALL_CYCLES)   \
	__PMC_EV(MIPS24K, DMISS_CYCLES)            \
	__PMC_EV(MIPS24K, L2_MISS_CYCLES)          \
	__PMC_EV(MIPS24K, UNCACHED_BLOCK_CYCLES)   \
	__PMC_EV(MIPS24K, MDU_STALL_CYCLES)        \
	__PMC_EV(MIPS24K, FPU_STALL_CYCLES)        \
	__PMC_EV(MIPS24K, CP2_STALL_CYCLES)        \
	__PMC_EV(MIPS24K, COREXTEND_STALL_CYCLES)  \
	__PMC_EV(MIPS24K, ISPRAM_STALL_CYCLES)     \
	__PMC_EV(MIPS24K, DSPRAM_STALL_CYCLES)     \
	__PMC_EV(MIPS24K, CACHE_STALL_CYCLES)      \
	__PMC_EV(MIPS24K, LOAD_TO_USE_STALLS)      \
	__PMC_EV(MIPS24K, BASE_MISPRED_STALLS)     \
	__PMC_EV(MIPS24K, CPO_READ_STALLS)         \
	__PMC_EV(MIPS24K, BRANCH_MISPRED_CYCLES)   \
	__PMC_EV(MIPS24K, IFETCH_BUFFER_FULL)      \
	__PMC_EV(MIPS24K, FETCH_BUFFER_ALLOCATED)  \
	__PMC_EV(MIPS24K, EJTAG_ITRIGGER)          \
	__PMC_EV(MIPS24K, EJTAG_DTRIGGER)          \
	__PMC_EV(MIPS24K, FSB_LT_QUARTER)          \
	__PMC_EV(MIPS24K, FSB_QUARTER_TO_HALF)     \
	__PMC_EV(MIPS24K, FSB_GT_HALF)             \
	__PMC_EV(MIPS24K, FSB_FULL_PIPELINE_STALLS)\
	__PMC_EV(MIPS24K, LDQ_LT_QUARTER)          \
	__PMC_EV(MIPS24K, LDQ_QUARTER_TO_HALF)     \
	__PMC_EV(MIPS24K, LDQ_GT_HALF)             \
	__PMC_EV(MIPS24K, LDQ_FULL_PIPELINE_STALLS)\
	__PMC_EV(MIPS24K, WBB_LT_QUARTER)          \
	__PMC_EV(MIPS24K, WBB_QUARTER_TO_HALF)     \
	__PMC_EV(MIPS24K, WBB_GT_HALF)             \
	__PMC_EV(MIPS24K, WBB_FULL_PIPELINE_STALLS) \
	__PMC_EV(MIPS24K, REQUEST_LATENCY)         \
	__PMC_EV(MIPS24K, REQUEST_COUNT)

#define	PMC_EV_MIPS24K_FIRST	PMC_EV_MIPS24K_CYCLE
#define	PMC_EV_MIPS24K_LAST	PMC_EV_MIPS24K_WBB_FULL_PIPELINE_STALLS

/*
 * MIPS74k events.  Similar to MIPS24k, the arrangement
 * is (0,2) then (1,3) events.
 */
#define __PMC_EV_MIPS74K()			\
	__PMC_EV(MIPS74K, CYCLES)		\
	__PMC_EV(MIPS74K, INSTR_EXECUTED)	\
	__PMC_EV(MIPS74K, PREDICTED_JR_31)	\
	__PMC_EV(MIPS74K, JR_31_MISPREDICTIONS)	\
	__PMC_EV(MIPS74K, REDIRECT_STALLS)	\
	__PMC_EV(MIPS74K, JR_31_NO_PREDICTIONS)	\
	__PMC_EV(MIPS74K, ITLB_ACCESSES)	\
	__PMC_EV(MIPS74K, ITLB_MISSES)		\
	__PMC_EV(MIPS74K, JTLB_INSN_MISSES)	\
	__PMC_EV(MIPS74K, ICACHE_ACCESSES)	\
	__PMC_EV(MIPS74K, ICACHE_MISSES)	\
	__PMC_EV(MIPS74K, ICACHE_MISS_STALLS)	\
	__PMC_EV(MIPS74K, UNCACHED_IFETCH_STALLS)	\
	__PMC_EV(MIPS74K, PDTRACE_BACK_STALLS)	\
	__PMC_EV(MIPS74K, IFU_REPLAYS)		\
	__PMC_EV(MIPS74K, KILLED_FETCH_SLOTS)	\
	__PMC_EV(MIPS74K, IFU_IDU_MISS_PRED_UPSTREAM_CYCLES)	\
	__PMC_EV(MIPS74K, IFU_IDU_NO_FETCH_CYCLES)	\
	__PMC_EV(MIPS74K, IFU_IDU_CLOGED_DOWNSTREAM_CYCLES)	\
	__PMC_EV(MIPS74K, DDQ0_FULL_DR_STALLS)	\
	__PMC_EV(MIPS74K, DDQ1_FULL_DR_STALLS)	\
	__PMC_EV(MIPS74K, ALCB_FULL_DR_STALLS)	\
	__PMC_EV(MIPS74K, AGCB_FULL_DR_STALLS)	\
	__PMC_EV(MIPS74K, CLDQ_FULL_DR_STALLS)	\
	__PMC_EV(MIPS74K, IODQ_FULL_DR_STALLS)	\
	__PMC_EV(MIPS74K, ALU_EMPTY_CYCLES)	\
	__PMC_EV(MIPS74K, AGEN_EMPTY_CYCLES)	\
	__PMC_EV(MIPS74K, ALU_OPERANDS_NOT_READY_CYCLES)	\
	__PMC_EV(MIPS74K, AGEN_OPERANDS_NOT_READY_CYCLES)	\
	__PMC_EV(MIPS74K, ALU_NO_ISSUES_CYCLES)	\
	__PMC_EV(MIPS74K, AGEN_NO_ISSUES_CYCLES)	\
	__PMC_EV(MIPS74K, ALU_BUBBLE_CYCLES)	\
	__PMC_EV(MIPS74K, AGEN_BUBBLE_CYCLES)	\
	__PMC_EV(MIPS74K, SINGLE_ISSUE_CYCLES)	\
	__PMC_EV(MIPS74K, DUAL_ISSUE_CYCLES)	\
	__PMC_EV(MIPS74K, OOO_ALU_ISSUE_CYCLES)	\
	__PMC_EV(MIPS74K, OOO_AGEN_ISSUE_CYCLES)	\
	__PMC_EV(MIPS74K, JALR_JALR_HB_INSNS)	\
	__PMC_EV(MIPS74K, DCACHE_LINE_REFILL_REQUESTS)	\
	__PMC_EV(MIPS74K, DCACHE_LOAD_ACCESSES)	\
	__PMC_EV(MIPS74K, DCACHE_ACCESSES)	\
	__PMC_EV(MIPS74K, DCACHE_WRITEBACKS)	\
	__PMC_EV(MIPS74K, DCACHE_MISSES)	\
	__PMC_EV(MIPS74K, JTLB_DATA_ACCESSES)	\
	__PMC_EV(MIPS74K, JTLB_DATA_MISSES)	\
	__PMC_EV(MIPS74K, LOAD_STORE_REPLAYS)	\
	__PMC_EV(MIPS74K, VA_TRANSALTION_CORNER_CASES)	\
	__PMC_EV(MIPS74K, LOAD_STORE_BLOCKED_CYCLES)	\
	__PMC_EV(MIPS74K, LOAD_STORE_NO_FILL_REQUESTS)	\
	__PMC_EV(MIPS74K, L2_CACHE_WRITEBACKS)	\
	__PMC_EV(MIPS74K, L2_CACHE_ACCESSES)	\
	__PMC_EV(MIPS74K, L2_CACHE_MISSES)	\
	__PMC_EV(MIPS74K, L2_CACHE_MISS_CYCLES)	\
	__PMC_EV(MIPS74K, FSB_FULL_STALLS)	\
	__PMC_EV(MIPS74K, FSB_OVER_50_FULL)	\
	__PMC_EV(MIPS74K, LDQ_FULL_STALLS)	\
	__PMC_EV(MIPS74K, LDQ_OVER_50_FULL)	\
	__PMC_EV(MIPS74K, WBB_FULL_STALLS)	\
	__PMC_EV(MIPS74K, WBB_OVER_50_FULL)	\
	__PMC_EV(MIPS74K, LOAD_MISS_CONSUMER_REPLAYS)	\
	__PMC_EV(MIPS74K, CP1_CP2_LOAD_INSNS)	\
	__PMC_EV(MIPS74K, JR_NON_31_INSNS)	\
	__PMC_EV(MIPS74K, MISPREDICTED_JR_31_INSNS)	\
	__PMC_EV(MIPS74K, BRANCH_INSNS)		\
	__PMC_EV(MIPS74K, CP1_CP2_COND_BRANCH_INSNS)	\
	__PMC_EV(MIPS74K, BRANCH_LIKELY_INSNS)	\
	__PMC_EV(MIPS74K, MISPREDICTED_BRANCH_LIKELY_INSNS)	\
	__PMC_EV(MIPS74K, COND_BRANCH_INSNS)	\
	__PMC_EV(MIPS74K, MISPREDICTED_BRANCH_INSNS)	\
	__PMC_EV(MIPS74K, INTEGER_INSNS)	\
	__PMC_EV(MIPS74K, FPU_INSNS)		\
	__PMC_EV(MIPS74K, LOAD_INSNS)		\
	__PMC_EV(MIPS74K, STORE_INSNS)		\
	__PMC_EV(MIPS74K, J_JAL_INSNS)		\
	__PMC_EV(MIPS74K, MIPS16_INSNS)		\
	__PMC_EV(MIPS74K, NOP_INSNS)		\
	__PMC_EV(MIPS74K, NT_MUL_DIV_INSNS)	\
	__PMC_EV(MIPS74K, DSP_INSNS)		\
	__PMC_EV(MIPS74K, ALU_DSP_SATURATION_INSNS)	\
	__PMC_EV(MIPS74K, DSP_BRANCH_INSNS)	\
	__PMC_EV(MIPS74K, MDU_DSP_SATURATION_INSNS)	\
	__PMC_EV(MIPS74K, UNCACHED_LOAD_INSNS)	\
	__PMC_EV(MIPS74K, UNCACHED_STORE_INSNS)	\
	__PMC_EV(MIPS74K, EJTAG_INSN_TRIGGERS)	\
	__PMC_EV(MIPS74K, CP1_BRANCH_MISPREDICTIONS)	\
	__PMC_EV(MIPS74K, SC_INSNS)		\
	__PMC_EV(MIPS74K, FAILED_SC_INSNS)	\
	__PMC_EV(MIPS74K, PREFETCH_INSNS)	\
	__PMC_EV(MIPS74K, CACHE_HIT_PREFETCH_INSNS)	\
	__PMC_EV(MIPS74K, NO_INSN_CYCLES)	\
	__PMC_EV(MIPS74K, LOAD_MISS_INSNS)	\
	__PMC_EV(MIPS74K, ONE_INSN_CYCLES)	\
	__PMC_EV(MIPS74K, TWO_INSNS_CYCLES)	\
	__PMC_EV(MIPS74K, GFIFO_BLOCKED_CYCLES)	\
	__PMC_EV(MIPS74K, CP1_CP2_STORE_INSNS)	\
	__PMC_EV(MIPS74K, MISPREDICTION_STALLS)	\
	__PMC_EV(MIPS74K, MISPREDICTED_BRANCH_INSNS_CYCLES)	\
	__PMC_EV(MIPS74K, EXCEPTIONS_TAKEN)	\
	__PMC_EV(MIPS74K, GRADUATION_REPLAYS)	\
	__PMC_EV(MIPS74K, COREEXTEND_EVENTS)	\
	__PMC_EV(MIPS74K, ISPRAM_EVENTS)	\
	__PMC_EV(MIPS74K, DSPRAM_EVENTS)	\
	__PMC_EV(MIPS74K, L2_CACHE_SINGLE_BIT_ERRORS)	\
	__PMC_EV(MIPS74K, SYSTEM_EVENT_0)	\
	__PMC_EV(MIPS74K, SYSTEM_EVENT_1)	\
	__PMC_EV(MIPS74K, SYSTEM_EVENT_2)	\
	__PMC_EV(MIPS74K, SYSTEM_EVENT_3)	\
	__PMC_EV(MIPS74K, SYSTEM_EVENT_4)	\
	__PMC_EV(MIPS74K, SYSTEM_EVENT_5)	\
	__PMC_EV(MIPS74K, SYSTEM_EVENT_6)	\
	__PMC_EV(MIPS74K, SYSTEM_EVENT_7)	\
	__PMC_EV(MIPS74K, OCP_ALL_REQUESTS)	\
	__PMC_EV(MIPS74K, OCP_ALL_CACHEABLE_REQUESTS)	\
	__PMC_EV(MIPS74K, OCP_READ_REQUESTS)	\
	__PMC_EV(MIPS74K, OCP_READ_CACHEABLE_REQUESTS)	\
	__PMC_EV(MIPS74K, OCP_WRITE_REQUESTS)	\
	__PMC_EV(MIPS74K, OCP_WRITE_CACHEABLE_REQUESTS)	\
	__PMC_EV(MIPS74K, FSB_LESS_25_FULL)	\
	__PMC_EV(MIPS74K, FSB_25_50_FULL)	\
	__PMC_EV(MIPS74K, LDQ_LESS_25_FULL)	\
	__PMC_EV(MIPS74K, LDQ_25_50_FULL)	\
	__PMC_EV(MIPS74K, WBB_LESS_25_FULL)	\
	__PMC_EV(MIPS74K, WBB_25_50_FULL)

#define	PMC_EV_MIPS74K_FIRST	PMC_EV_MIPS74K_CYCLES
#define	PMC_EV_MIPS74K_LAST	PMC_EV_MIPS74K_WBB_25_50_FULL

/*
 * Cavium Octeon counters. Obtained from cvmx-core.h
 */
#define __PMC_EV_OCTEON()                         \
    __PMC_EV(OCTEON, CLK)                         \
    __PMC_EV(OCTEON, ISSUE)                       \
    __PMC_EV(OCTEON, RET)                         \
    __PMC_EV(OCTEON, NISSUE)                      \
    __PMC_EV(OCTEON, SISSUE)                      \
    __PMC_EV(OCTEON, DISSUE)                      \
    __PMC_EV(OCTEON, IFI)                         \
    __PMC_EV(OCTEON, BR)                          \
    __PMC_EV(OCTEON, BRMIS)                       \
    __PMC_EV(OCTEON, J)                           \
    __PMC_EV(OCTEON, JMIS)                        \
    __PMC_EV(OCTEON, REPLAY)                      \
    __PMC_EV(OCTEON, IUNA)                        \
    __PMC_EV(OCTEON, TRAP)                        \
    __PMC_EV(OCTEON, UULOAD)                      \
    __PMC_EV(OCTEON, UUSTORE)                     \
    __PMC_EV(OCTEON, ULOAD)                       \
    __PMC_EV(OCTEON, USTORE)                      \
    __PMC_EV(OCTEON, EC)                          \
    __PMC_EV(OCTEON, MC)                          \
    __PMC_EV(OCTEON, CC)                          \
    __PMC_EV(OCTEON, CSRC)                        \
    __PMC_EV(OCTEON, CFETCH)                      \
    __PMC_EV(OCTEON, CPREF)                       \
    __PMC_EV(OCTEON, ICA)                         \
    __PMC_EV(OCTEON, II)                          \
    __PMC_EV(OCTEON, IP)                          \
    __PMC_EV(OCTEON, CIMISS)                      \
    __PMC_EV(OCTEON, WBUF)                        \
    __PMC_EV(OCTEON, WDAT)                        \
    __PMC_EV(OCTEON, WBUFLD)                      \
    __PMC_EV(OCTEON, WBUFFL)                      \
    __PMC_EV(OCTEON, WBUFTR)                      \
    __PMC_EV(OCTEON, BADD)                        \
    __PMC_EV(OCTEON, BADDL2)                      \
    __PMC_EV(OCTEON, BFILL)                       \
    __PMC_EV(OCTEON, DDIDS)                       \
    __PMC_EV(OCTEON, IDIDS)                       \
    __PMC_EV(OCTEON, DIDNA)                       \
    __PMC_EV(OCTEON, LDS)                         \
    __PMC_EV(OCTEON, LMLDS)                       \
    __PMC_EV(OCTEON, IOLDS)                       \
    __PMC_EV(OCTEON, DMLDS)                       \
    __PMC_EV(OCTEON, STS)                         \
    __PMC_EV(OCTEON, LMSTS)                       \
    __PMC_EV(OCTEON, IOSTS)                       \
    __PMC_EV(OCTEON, IOBDMA)                      \
    __PMC_EV(OCTEON, DTLB)                        \
    __PMC_EV(OCTEON, DTLBAD)                      \
    __PMC_EV(OCTEON, ITLB)                        \
    __PMC_EV(OCTEON, SYNC)                        \
    __PMC_EV(OCTEON, SYNCIOB)                     \
    __PMC_EV(OCTEON, SYNCW)

#define	PMC_EV_OCTEON_FIRST	PMC_EV_OCTEON_CLK
#define	PMC_EV_OCTEON_LAST	PMC_EV_OCTEON_SYNCW

#define __PMC_EV_PPC7450()						\
	__PMC_EV(PPC7450, CYCLE)					\
	__PMC_EV(PPC7450, INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, TLB_BIT_TRANSITIONS)				\
	__PMC_EV(PPC7450, INSTR_DISPATCHED)				\
	__PMC_EV(PPC7450, PMON_EXCEPT)					\
	__PMC_EV(PPC7450, PMON_SIG)					\
	__PMC_EV(PPC7450, VPU_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, VFPU_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, VIU1_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, VIU2_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, MTVSCR_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, MTVRSAVE_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, VPU_INSTR_WAIT_CYCLES)			\
	__PMC_EV(PPC7450, VFPU_INSTR_WAIT_CYCLES)			\
	__PMC_EV(PPC7450, VIU1_INSTR_WAIT_CYCLES)			\
	__PMC_EV(PPC7450, VIU2_INSTR_WAIT_CYCLES)			\
	__PMC_EV(PPC7450, MFVSCR_SYNC_CYCLES)				\
	__PMC_EV(PPC7450, VSCR_SAT_SET)					\
	__PMC_EV(PPC7450, STORE_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, L1_INSTR_CACHE_MISSES)			\
	__PMC_EV(PPC7450, L1_DATA_SNOOPS)				\
	__PMC_EV(PPC7450, UNRESOLVED_BRANCHES)				\
	__PMC_EV(PPC7450, SPEC_BUFFER_CYCLES)				\
	__PMC_EV(PPC7450, BRANCH_UNIT_STALL_CYCLES)			\
	__PMC_EV(PPC7450, TRUE_BRANCH_TARGET_HITS)			\
	__PMC_EV(PPC7450, BRANCH_LINK_STAC_PREDICTED)			\
	__PMC_EV(PPC7450, GPR_ISSUE_QUEUE_DISPATCHES)			\
	__PMC_EV(PPC7450, CYCLES_THREE_INSTR_DISPATCHED)		\
	__PMC_EV(PPC7450, THRESHOLD_INSTR_QUEUE_ENTRIES_CYCLES)		\
	__PMC_EV(PPC7450, THRESHOLD_VEC_INSTR_QUEUE_ENTRIES_CYCLES)	\
	__PMC_EV(PPC7450, CYCLES_NO_COMPLETED_INSTRS)			\
	__PMC_EV(PPC7450, IU2_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, BRANCHES_COMPLETED)				\
	__PMC_EV(PPC7450, EIEIO_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, MTSPR_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, SC_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, LS_LM_COMPLETED)				\
	__PMC_EV(PPC7450, ITLB_HW_TABLE_SEARCH_CYCLES)			\
	__PMC_EV(PPC7450, DTLB_HW_SEARCH_CYCLES_OVER_THRESHOLD)		\
	__PMC_EV(PPC7450, L1_INSTR_CACHE_ACCESSES)			\
	__PMC_EV(PPC7450, INSTR_BKPT_MATCHES)				\
	__PMC_EV(PPC7450, L1_DATA_CACHE_LOAD_MISS_CYCLES_OVER_THRESHOLD)\
	__PMC_EV(PPC7450, L1_DATA_SNOOP_HIT_ON_MODIFIED)		\
	__PMC_EV(PPC7450, LOAD_MISS_ALIAS)				\
	__PMC_EV(PPC7450, LOAD_MISS_ALIAS_ON_TOUCH)			\
	__PMC_EV(PPC7450, TOUCH_ALIAS)					\
	__PMC_EV(PPC7450, L1_DATA_SNOOP_HIT_CASTOUT_QUEUE)		\
	__PMC_EV(PPC7450, L1_DATA_SNOOP_HIT_CASTOUT)			\
	__PMC_EV(PPC7450, L1_DATA_SNOOP_HITS)				\
	__PMC_EV(PPC7450, WRITE_THROUGH_STORES)				\
	__PMC_EV(PPC7450, CACHE_INHIBITED_STORES)			\
	__PMC_EV(PPC7450, L1_DATA_LOAD_HIT)				\
	__PMC_EV(PPC7450, L1_DATA_TOUCH_HIT)				\
	__PMC_EV(PPC7450, L1_DATA_STORE_HIT)				\
	__PMC_EV(PPC7450, L1_DATA_TOTAL_HITS)				\
	__PMC_EV(PPC7450, DST_INSTR_DISPATCHED)				\
	__PMC_EV(PPC7450, REFRESHED_DSTS)				\
	__PMC_EV(PPC7450, SUCCESSFUL_DST_TABLE_SEARCHES)		\
	__PMC_EV(PPC7450, DSS_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, DST_STREAM_0_CACHE_LINE_FETCHES)		\
	__PMC_EV(PPC7450, VTQ_SUSPENDS_DUE_TO_CTX_CHANGE)		\
	__PMC_EV(PPC7450, VTQ_LINE_FETCH_HIT)				\
	__PMC_EV(PPC7450, VEC_LOAD_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, FP_STORE_INSTR_COMPLETED_IN_LSU)		\
	__PMC_EV(PPC7450, FPU_RENORMALIZATION)				\
	__PMC_EV(PPC7450, FPU_DENORMALIZATION)				\
	__PMC_EV(PPC7450, FP_STORE_CAUSES_STALL_IN_LSU)			\
	__PMC_EV(PPC7450, LD_ST_TRUE_ALIAS_STALL)			\
	__PMC_EV(PPC7450, LSU_INDEXED_ALIAS_STALL)			\
	__PMC_EV(PPC7450, LSU_ALIAS_VS_FSQ_WB0_WB1)			\
	__PMC_EV(PPC7450, LSU_ALIAS_VS_CSQ)				\
	__PMC_EV(PPC7450, LSU_LOAD_HIT_LINE_ALIAS_VS_CSQ0)		\
	__PMC_EV(PPC7450, LSU_LOAD_MISS_LINE_ALIAS_VS_CSQ0)		\
	__PMC_EV(PPC7450, LSU_TOUCH_LINE_ALIAS_VS_FSQ_WB0_WB1)		\
	__PMC_EV(PPC7450, LSU_TOUCH_ALIAS_VS_CSQ)			\
	__PMC_EV(PPC7450, LSU_LMQ_FULL_STALL)				\
	__PMC_EV(PPC7450, FP_LOAD_INSTR_COMPLETED_IN_LSU)		\
	__PMC_EV(PPC7450, FP_LOAD_SINGLE_INSTR_COMPLETED_IN_LSU)	\
	__PMC_EV(PPC7450, FP_LOAD_DOUBLE_COMPLETED_IN_LSU)		\
	__PMC_EV(PPC7450, LSU_RA_LATCH_STALL)				\
	__PMC_EV(PPC7450, LSU_LOAD_VS_STORE_QUEUE_ALIAS_STALL)		\
	__PMC_EV(PPC7450, LSU_LMQ_INDEX_ALIAS)				\
	__PMC_EV(PPC7450, LSU_STORE_QUEUE_INDEX_ALIAS)			\
	__PMC_EV(PPC7450, LSU_CSQ_FORWARDING)				\
	__PMC_EV(PPC7450, LSU_MISALIGNED_LOAD_FINISH)			\
	__PMC_EV(PPC7450, LSU_MISALIGN_STORE_COMPLETED)			\
	__PMC_EV(PPC7450, LSU_MISALIGN_STALL)				\
	__PMC_EV(PPC7450, FP_ONE_QUARTER_FPSCR_RENAMES_BUSY)		\
	__PMC_EV(PPC7450, FP_ONE_HALF_FPSCR_RENAMES_BUSY)		\
	__PMC_EV(PPC7450, FP_THREE_QUARTERS_FPSCR_RENAMES_BUSY)		\
	__PMC_EV(PPC7450, FP_ALL_FPSCR_RENAMES_BUSY)			\
	__PMC_EV(PPC7450, FP_DENORMALIZED_RESULT)			\
	__PMC_EV(PPC7450, L1_DATA_TOTAL_MISSES)				\
	__PMC_EV(PPC7450, DISPATCHES_TO_FPR_ISSUE_QUEUE)		\
	__PMC_EV(PPC7450, LSU_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, LOAD_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, SS_SM_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, TLBIE_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, LWARX_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, MFSPR_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, REFETCH_SERIALIZATION)			\
	__PMC_EV(PPC7450, COMPLETION_QUEUE_ENTRIES_OVER_THRESHOLD)	\
	__PMC_EV(PPC7450, CYCLES_ONE_INSTR_DISPATCHED)			\
	__PMC_EV(PPC7450, CYCLES_TWO_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, ITLB_NON_SPECULATIVE_MISSES)			\
	__PMC_EV(PPC7450, CYCLES_WAITING_FROM_L1_INSTR_CACHE_MISS)	\
	__PMC_EV(PPC7450, L1_DATA_LOAD_ACCESS_MISS)			\
	__PMC_EV(PPC7450, L1_DATA_TOUCH_MISS)				\
	__PMC_EV(PPC7450, L1_DATA_STORE_MISS)				\
	__PMC_EV(PPC7450, L1_DATA_TOUCH_MISS_CYCLES)			\
	__PMC_EV(PPC7450, L1_DATA_CYCLES_USED)				\
	__PMC_EV(PPC7450, DST_STREAM_1_CACHE_LINE_FETCHES)		\
	__PMC_EV(PPC7450, VTQ_STREAM_CANCELED_PREMATURELY)		\
	__PMC_EV(PPC7450, VTQ_RESUMES_DUE_TO_CTX_CHANGE)		\
	__PMC_EV(PPC7450, VTQ_LINE_FETCH_MISS)				\
	__PMC_EV(PPC7450, VTQ_LINE_FETCH)				\
	__PMC_EV(PPC7450, TLBIE_SNOOPS)					\
	__PMC_EV(PPC7450, L1_INSTR_CACHE_RELOADS)			\
	__PMC_EV(PPC7450, L1_DATA_CACHE_RELOADS)			\
	__PMC_EV(PPC7450, L1_DATA_CACHE_CASTOUTS_TO_L2)			\
	__PMC_EV(PPC7450, STORE_MERGE_GATHER)				\
	__PMC_EV(PPC7450, CACHEABLE_STORE_MERGE_TO_32_BYTES)		\
	__PMC_EV(PPC7450, DATA_BKPT_MATCHES)				\
	__PMC_EV(PPC7450, FALL_THROUGH_BRANCHES_PROCESSED)		\
	__PMC_EV(PPC7450,						\
	    FIRST_SPECULATIVE_BRANCH_BUFFER_RESOLVED_CORRECTLY)		\
	__PMC_EV(PPC7450, SECOND_SPECULATION_BUFFER_ACTIVE)		\
	__PMC_EV(PPC7450, BPU_STALL_ON_LR_DEPENDENCY)			\
	__PMC_EV(PPC7450, BTIC_MISS)					\
	__PMC_EV(PPC7450, BRANCH_LINK_STACK_CORRECTLY_RESOLVED)		\
	__PMC_EV(PPC7450, FPR_ISSUE_STALLED)				\
	__PMC_EV(PPC7450, SWITCHES_BETWEEN_PRIV_USER)			\
	__PMC_EV(PPC7450, LSU_COMPLETES_FP_STORE_SINGLE)		\
	__PMC_EV(PPC7450, VR_ISSUE_QUEUE_DISPATCHES)			\
	__PMC_EV(PPC7450, VR_STALLS)					\
	__PMC_EV(PPC7450, GPR_RENAME_BUFFER_ENTRIES_OVER_THRESHOLD)	\
	__PMC_EV(PPC7450, FPR_ISSUE_QUEUE_ENTRIES)			\
	__PMC_EV(PPC7450, FPU_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, STWCX_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, LS_LM_INSTR_PIECES)				\
	__PMC_EV(PPC7450, ITLB_HW_SEARCH_CYCLES_OVER_THRESHOLD)		\
	__PMC_EV(PPC7450, DTLB_MISSES)					\
	__PMC_EV(PPC7450, CANCELLED_L1_INSTR_CACHE_MISSES)		\
	__PMC_EV(PPC7450, L1_DATA_CACHE_OP_HIT)				\
	__PMC_EV(PPC7450, L1_DATA_LOAD_MISS_CYCLES)			\
	__PMC_EV(PPC7450, L1_DATA_PUSHES)				\
	__PMC_EV(PPC7450, L1_DATA_TOTAL_MISS)				\
	__PMC_EV(PPC7450, VT2_FETCHES)					\
	__PMC_EV(PPC7450, TAKEN_BRANCHES_PROCESSED)			\
	__PMC_EV(PPC7450, BRANCH_FLUSHES)				\
	__PMC_EV(PPC7450,						\
	    SECOND_SPECULATIVE_BRANCH_BUFFER_RESOLVED_CORRECTLY)	\
	__PMC_EV(PPC7450, THIRD_SPECULATION_BUFFER_ACTIVE)		\
	__PMC_EV(PPC7450, BRANCH_UNIT_STALL_ON_CTR_DEPENDENCY)		\
	__PMC_EV(PPC7450, FAST_BTIC_HIT)				\
	__PMC_EV(PPC7450, BRANCH_LINK_STACK_MISPREDICTED)		\
	__PMC_EV(PPC7450, CYCLES_THREE_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, CYCLES_NO_INSTR_DISPATCHED)			\
	__PMC_EV(PPC7450, GPR_ISSUE_QUEUE_ENTRIES_OVER_THRESHOLD)	\
	__PMC_EV(PPC7450, GPR_ISSUE_QUEUE_STALLED)			\
	__PMC_EV(PPC7450, IU1_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, DSSALL_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, TLBSYNC_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, SYNC_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, SS_SM_INSTR_PIECES)				\
	__PMC_EV(PPC7450, DTLB_HW_SEARCH_CYCLES)			\
	__PMC_EV(PPC7450, SNOOP_RETRIES)				\
	__PMC_EV(PPC7450, SUCCESSFUL_STWCX)				\
	__PMC_EV(PPC7450, DST_STREAM_3_CACHE_LINE_FETCHES)		\
	__PMC_EV(PPC7450,						\
	    THIRD_SPECULATIVE_BRANCH_BUFFER_RESOLVED_CORRECTLY)		\
	__PMC_EV(PPC7450, MISPREDICTED_BRANCHES)			\
	__PMC_EV(PPC7450, FOLDED_BRANCHES)				\
	__PMC_EV(PPC7450, FP_STORE_DOUBLE_COMPLETES_IN_LSU)		\
	__PMC_EV(PPC7450, L2_CACHE_HITS)				\
	__PMC_EV(PPC7450, L3_CACHE_HITS)				\
	__PMC_EV(PPC7450, L2_INSTR_CACHE_MISSES)			\
	__PMC_EV(PPC7450, L3_INSTR_CACHE_MISSES)			\
	__PMC_EV(PPC7450, L2_DATA_CACHE_MISSES)				\
	__PMC_EV(PPC7450, L3_DATA_CACHE_MISSES)				\
	__PMC_EV(PPC7450, L2_LOAD_HITS)					\
	__PMC_EV(PPC7450, L2_STORE_HITS)				\
	__PMC_EV(PPC7450, L3_LOAD_HITS)					\
	__PMC_EV(PPC7450, L3_STORE_HITS)				\
	__PMC_EV(PPC7450, L2_TOUCH_HITS)				\
	__PMC_EV(PPC7450, L3_TOUCH_HITS)				\
	__PMC_EV(PPC7450, SNOOP_MODIFIED)				\
	__PMC_EV(PPC7450, SNOOP_VALID)					\
	__PMC_EV(PPC7450, INTERVENTION)					\
	__PMC_EV(PPC7450, L2_CACHE_MISSES)				\
	__PMC_EV(PPC7450, L3_CACHE_MISSES)				\
	__PMC_EV(PPC7450, L2_CACHE_CASTOUTS)				\
	__PMC_EV(PPC7450, L3_CACHE_CASTOUTS)				\
	__PMC_EV(PPC7450, L2SQ_FULL_CYCLES)				\
	__PMC_EV(PPC7450, L3SQ_FULL_CYCLES)				\
	__PMC_EV(PPC7450, RAQ_FULL_CYCLES)				\
	__PMC_EV(PPC7450, WAQ_FULL_CYCLES)				\
	__PMC_EV(PPC7450, L1_EXTERNAL_INTERVENTIONS)			\
	__PMC_EV(PPC7450, L2_EXTERNAL_INTERVENTIONS)			\
	__PMC_EV(PPC7450, L3_EXTERNAL_INTERVENTIONS)			\
	__PMC_EV(PPC7450, EXTERNAL_INTERVENTIONS)			\
	__PMC_EV(PPC7450, EXTERNAL_PUSHES)				\
	__PMC_EV(PPC7450, EXTERNAL_SNOOP_RETRY)				\
	__PMC_EV(PPC7450, DTQ_FULL_CYCLES)				\
	__PMC_EV(PPC7450, BUS_RETRY)					\
	__PMC_EV(PPC7450, L2_VALID_REQUEST)				\
	__PMC_EV(PPC7450, BORDQ_FULL)					\
	__PMC_EV(PPC7450, BUS_TAS_FOR_READS)				\
	__PMC_EV(PPC7450, BUS_TAS_FOR_WRITES)				\
	__PMC_EV(PPC7450, BUS_READS_NOT_RETRIED)			\
	__PMC_EV(PPC7450, BUS_WRITES_NOT_RETRIED)			\
	__PMC_EV(PPC7450, BUS_READS_WRITES_NOT_RETRIED)			\
	__PMC_EV(PPC7450, BUS_RETRY_DUE_TO_L1_RETRY)			\
	__PMC_EV(PPC7450, BUS_RETRY_DUE_TO_PREVIOUS_ADJACENT)		\
	__PMC_EV(PPC7450, BUS_RETRY_DUE_TO_COLLISION)			\
	__PMC_EV(PPC7450, BUS_RETRY_DUE_TO_INTERVENTION_ORDERING)	\
	__PMC_EV(PPC7450, SNOOP_REQUESTS)				\
	__PMC_EV(PPC7450, PREFETCH_ENGINE_REQUEST)			\
	__PMC_EV(PPC7450, PREFETCH_ENGINE_COLLISION_VS_LOAD)		\
	__PMC_EV(PPC7450, PREFETCH_ENGINE_COLLISION_VS_STORE)		\
	__PMC_EV(PPC7450, PREFETCH_ENGINE_COLLISION_VS_INSTR_FETCH)	\
	__PMC_EV(PPC7450,						\
	    PREFETCH_ENGINE_COLLISION_VS_LOAD_STORE_INSTR_FETCH)	\
	__PMC_EV(PPC7450, PREFETCH_ENGINE_FULL)

#define PMC_EV_PPC7450_FIRST	PMC_EV_PPC7450_CYCLE
#define PMC_EV_PPC7450_LAST	PMC_EV_PPC7450_PREFETCH_ENGINE_FULL

#define __PMC_EV_PPC970() \
	__PMC_EV(PPC970, INSTR_COMPLETED) \
	__PMC_EV(PPC970, MARKED_GROUP_DISPATCH) \
	__PMC_EV(PPC970, MARKED_STORE_COMPLETED) \
	__PMC_EV(PPC970, GCT_EMPTY) \
	__PMC_EV(PPC970, RUN_CYCLES) \
	__PMC_EV(PPC970, OVERFLOW) \
	__PMC_EV(PPC970, CYCLES) \
	__PMC_EV(PPC970, THRESHOLD_TIMEOUT) \
	__PMC_EV(PPC970, GROUP_DISPATCH) \
	__PMC_EV(PPC970, BR_MARKED_INSTR_FINISH) \
	__PMC_EV(PPC970, GCT_EMPTY_BY_SRQ_FULL) \
	__PMC_EV(PPC970, STOP_COMPLETION) \
	__PMC_EV(PPC970, LSU_EMPTY) \
	__PMC_EV(PPC970, MARKED_STORE_WITH_INTR) \
	__PMC_EV(PPC970, CYCLES_IN_SUPER) \
	__PMC_EV(PPC970, VPU_MARKED_INSTR_COMPLETED) \
	__PMC_EV(PPC970, FXU0_IDLE_FXU1_BUSY) \
	__PMC_EV(PPC970, SRQ_EMPTY) \
	__PMC_EV(PPC970, MARKED_GROUP_COMPLETED) \
	__PMC_EV(PPC970, CR_MARKED_INSTR_FINISH) \
	__PMC_EV(PPC970, DISPATCH_SUCCESS) \
	__PMC_EV(PPC970, FXU0_IDLE_FXU1_IDLE) \
	__PMC_EV(PPC970, ONE_PLUS_INSTR_COMPLETED) \
	__PMC_EV(PPC970, GROUP_MARKED_IDU) \
	__PMC_EV(PPC970, MARKED_GROUP_COMPLETE_TIMEOUT) \
	__PMC_EV(PPC970, FXU0_BUSY_FXU1_BUSY) \
	__PMC_EV(PPC970, MARKED_STORE_SENT_TO_STS) \
	__PMC_EV(PPC970, FXU_MARKED_INSTR_FINISHED) \
	__PMC_EV(PPC970, MARKED_GROUP_ISSUED) \
	__PMC_EV(PPC970, FXU0_BUSY_FXU1_IDLE) \
	__PMC_EV(PPC970, GROUP_COMPLETED) \
	__PMC_EV(PPC970, FPU_MARKED_INSTR_COMPLETED) \
	__PMC_EV(PPC970, MARKED_INSTR_FINISH_ANY_UNIT) \
	__PMC_EV(PPC970, EXTERNAL_INTERRUPT) \
	__PMC_EV(PPC970, GROUP_DISPATCH_REJECT) \
	__PMC_EV(PPC970, LSU_MARKED_INSTR_FINISH) \
	__PMC_EV(PPC970, TIMEBASE_EVENT) \
	__PMC_EV(PPC970, LSU_COMPLETION_STALL) \
	__PMC_EV(PPC970, FXU_COMPLETION_STALL) \
	__PMC_EV(PPC970, DCACHE_MISS_COMPLETION_STALL) \
	__PMC_EV(PPC970, FPU_COMPLETION_STALL) \
	__PMC_EV(PPC970, FXU_LONG_INSTR_COMPLETION_STALL) \
	__PMC_EV(PPC970, REJECT_COMPLETION_STALL) \
	__PMC_EV(PPC970, FPU_LONG_INSTR_COMPLETION_STALL) \
	__PMC_EV(PPC970, GCT_EMPTY_BY_ICACHE_MISS) \
	__PMC_EV(PPC970, REJECT_COMPLETION_STALL_ERAT_MISS) \
	__PMC_EV(PPC970, GCT_EMPTY_BY_BRANCH_MISS_PREDICT) \
	__PMC_EV(PPC970, BUS_HIGH) \
	__PMC_EV(PPC970, BUS_LOW) \
	__PMC_EV(PPC970, ADDER)


#define PMC_EV_PPC970_FIRST		PMC_EV_PPC970_INSTR_COMPLETED
#define PMC_EV_PPC970_LAST		PMC_EV_PPC970_ADDER

#define __PMC_EV_E500() \
	__PMC_EV(E500, CYCLES) \
	__PMC_EV(E500, INSTR_COMPLETED) \
	__PMC_EV(E500, UOPS_COMPLETED) \
	__PMC_EV(E500, INSTR_FETCHED) \
	__PMC_EV(E500, UOPS_DECODED) \
	__PMC_EV(E500, PM_EVENT_TRANSITIONS) \
	__PMC_EV(E500, PM_EVENT_CYCLES) \
	__PMC_EV(E500, BRANCH_INSTRS_COMPLETED) \
	__PMC_EV(E500, LOAD_UOPS_COMPLETED) \
	__PMC_EV(E500, STORE_UOPS_COMPLETED) \
	__PMC_EV(E500, CQ_REDIRECTS) \
	__PMC_EV(E500, BRANCHES_FINISHED) \
	__PMC_EV(E500, TAKEN_BRANCHES_FINISHED) \
	__PMC_EV(E500, FINISHED_UNCOND_BRANCHES_MISS_BTB) \
	__PMC_EV(E500, BRANCH_MISPRED) \
	__PMC_EV(E500, BTB_BRANCH_MISPRED_FROM_DIRECTION) \
	__PMC_EV(E500, BTB_HITS_PSEUDO_HITS) \
	__PMC_EV(E500, CYCLES_DECODE_STALLED) \
	__PMC_EV(E500, CYCLES_ISSUE_STALLED) \
	__PMC_EV(E500, CYCLES_BRANCH_ISSUE_STALLED) \
	__PMC_EV(E500, CYCLES_SU1_SCHED_STALLED) \
	__PMC_EV(E500, CYCLES_SU2_SCHED_STALLED) \
	__PMC_EV(E500, CYCLES_MU_SCHED_STALLED) \
	__PMC_EV(E500, CYCLES_LRU_SCHED_STALLED) \
	__PMC_EV(E500, CYCLES_BU_SCHED_STALLED) \
	__PMC_EV(E500, TOTAL_TRANSLATED) \
	__PMC_EV(E500, LOADS_TRANSLATED) \
	__PMC_EV(E500, STORES_TRANSLATED) \
	__PMC_EV(E500, TOUCHES_TRANSLATED) \
	__PMC_EV(E500, CACHEOPS_TRANSLATED) \
	__PMC_EV(E500, CACHE_INHIBITED_ACCESS_TRANSLATED) \
	__PMC_EV(E500, GUARDED_LOADS_TRANSLATED) \
	__PMC_EV(E500, WRITE_THROUGH_STORES_TRANSLATED) \
	__PMC_EV(E500, MISALIGNED_LOAD_STORE_ACCESS_TRANSLATED) \
	__PMC_EV(E500, TOTAL_ALLOCATED_TO_DLFB) \
	__PMC_EV(E500, LOADS_TRANSLATED_ALLOCATED_TO_DLFB) \
	__PMC_EV(E500, STORES_COMPLETED_ALLOCATED_TO_DLFB) \
	__PMC_EV(E500, TOUCHES_TRANSLATED_ALLOCATED_TO_DLFB) \
	__PMC_EV(E500, STORES_COMPLETED) \
	__PMC_EV(E500, DATA_L1_CACHE_LOCKS) \
	__PMC_EV(E500, DATA_L1_CACHE_RELOADS) \
	__PMC_EV(E500, DATA_L1_CACHE_CASTOUTS) \
	__PMC_EV(E500, LOAD_MISS_DLFB_FULL) \
	__PMC_EV(E500, LOAD_MISS_LDQ_FULL) \
	__PMC_EV(E500, LOAD_GUARDED_MISS) \
	__PMC_EV(E500, STORE_TRANSLATE_WHEN_QUEUE_FULL) \
	__PMC_EV(E500, ADDRESS_COLLISION) \
	__PMC_EV(E500, DATA_MMU_MISS) \
	__PMC_EV(E500, DATA_MMU_BUSY) \
	__PMC_EV(E500, PART2_MISALIGNED_CACHE_ACCESS) \
	__PMC_EV(E500, LOAD_MISS_DLFB_FULL_CYCLES) \
	__PMC_EV(E500, LOAD_MISS_LDQ_FULL_CYCLES) \
	__PMC_EV(E500, LOAD_GUARDED_MISS_CYCLES) \
	__PMC_EV(E500, STORE_TRANSLATE_WHEN_QUEUE_FULL_CYCLES) \
	__PMC_EV(E500, ADDRESS_COLLISION_CYCLES) \
	__PMC_EV(E500, DATA_MMU_MISS_CYCLES) \
	__PMC_EV(E500, DATA_MMU_BUSY_CYCLES) \
	__PMC_EV(E500, PART2_MISALIGNED_CACHE_ACCESS_CYCLES) \
	__PMC_EV(E500, INSTR_L1_CACHE_LOCKS) \
	__PMC_EV(E500, INSTR_L1_CACHE_RELOADS) \
	__PMC_EV(E500, INSTR_L1_CACHE_FETCHES) \
	__PMC_EV(E500, INSTR_MMU_TLB4K_RELOADS) \
	__PMC_EV(E500, INSTR_MMU_VSP_RELOADS) \
	__PMC_EV(E500, DATA_MMU_TLB4K_RELOADS) \
	__PMC_EV(E500, DATA_MMU_VSP_RELOADS) \
	__PMC_EV(E500, L2MMU_MISSES) \
	__PMC_EV(E500, BIU_MASTER_REQUESTS) \
	__PMC_EV(E500, BIU_MASTER_INSTR_SIDE_REQUESTS) \
	__PMC_EV(E500, BIU_MASTER_DATA_SIDE_REQUESTS) \
	__PMC_EV(E500, BIU_MASTER_DATA_SIDE_CASTOUT_REQUESTS) \
	__PMC_EV(E500, BIU_MASTER_RETRIES) \
	__PMC_EV(E500, SNOOP_REQUESTS) \
	__PMC_EV(E500, SNOOP_HITS) \
	__PMC_EV(E500, SNOOP_PUSHES) \
	__PMC_EV(E500, SNOOP_RETRIES) \
	__PMC_EV(E500, DLFB_LOAD_MISS_CYCLES) \
	__PMC_EV(E500, ILFB_FETCH_MISS_CYCLES) \
	__PMC_EV(E500, EXT_INPU_INTR_LATENCY_CYCLES) \
	__PMC_EV(E500, CRIT_INPUT_INTR_LATENCY_CYCLES) \
	__PMC_EV(E500, EXT_INPUT_INTR_PENDING_LATENCY_CYCLES) \
	__PMC_EV(E500, CRIT_INPUT_INTR_PENDING_LATENCY_CYCLES) \
	__PMC_EV(E500, PMC0_OVERFLOW) \
	__PMC_EV(E500, PMC1_OVERFLOW) \
	__PMC_EV(E500, PMC2_OVERFLOW) \
	__PMC_EV(E500, PMC3_OVERFLOW) \
	__PMC_EV(E500, INTERRUPTS_TAKEN) \
	__PMC_EV(E500, EXT_INPUT_INTR_TAKEN) \
	__PMC_EV(E500, CRIT_INPUT_INTR_TAKEN) \
	__PMC_EV(E500, SYSCALL_TRAP_INTR) \
	__PMC_EV(E500, TLB_BIT_TRANSITIONS) \
	__PMC_EV(E500, L2_LINEFILL_BUFFER) \
	__PMC_EV(E500, LV2_VS) \
	__PMC_EV(E500, CASTOUTS_RELEASED) \
	__PMC_EV(E500, INTV_ALLOCATIONS) \
	__PMC_EV(E500, DLFB_RETRIES_TO_MBAR) \
	__PMC_EV(E500, STORE_RETRIES) \
	__PMC_EV(E500, STASH_L1_HITS) \
	__PMC_EV(E500, STASH_L2_HITS) \
	__PMC_EV(E500, STASH_BUSY_1) \
	__PMC_EV(E500, STASH_BUSY_2) \
	__PMC_EV(E500, STASH_BUSY_3) \
	__PMC_EV(E500, STASH_HITS) \
	__PMC_EV(E500, STASH_HIT_DLFB) \
	__PMC_EV(E500, STASH_REQUESTS) \
	__PMC_EV(E500, STASH_REQUESTS_L1) \
	__PMC_EV(E500, STASH_REQUESTS_L2) \
	__PMC_EV(E500, STALLS_NO_CAQ_OR_COB) \
	__PMC_EV(E500, L2_CACHE_ACCESSES) \
	__PMC_EV(E500, L2_HIT_CACHE_ACCESSES) \
	__PMC_EV(E500, L2_CACHE_DATA_ACCESSES) \
	__PMC_EV(E500, L2_CACHE_DATA_HITS) \
	__PMC_EV(E500, L2_CACHE_INSTR_ACCESSES) \
	__PMC_EV(E500, L2_CACHE_INSTR_HITS) \
	__PMC_EV(E500, L2_CACHE_ALLOCATIONS) \
	__PMC_EV(E500, L2_CACHE_DATA_ALLOCATIONS) \
	__PMC_EV(E500, L2_CACHE_DIRTY_DATA_ALLOCATIONS) \
	__PMC_EV(E500, L2_CACHE_INSTR_ALLOCATIONS) \
	__PMC_EV(E500, L2_CACHE_UPDATES) \
	__PMC_EV(E500, L2_CACHE_CLEAN_UPDATES) \
	__PMC_EV(E500, L2_CACHE_DIRTY_UPDATES) \
	__PMC_EV(E500, L2_CACHE_CLEAN_REDUNDANT_UPDATES) \
	__PMC_EV(E500, L2_CACHE_DIRTY_REDUNDANT_UPDATES) \
	__PMC_EV(E500, L2_CACHE_LOCKS) \
	__PMC_EV(E500, L2_CACHE_CASTOUTS) \
	__PMC_EV(E500, L2_CACHE_DATA_DIRTY_HITS) \
	__PMC_EV(E500, INSTR_LFB_WENT_HIGH_PRIORITY) \
	__PMC_EV(E500, SNOOP_THROTTLING_TURNED_ON) \
	__PMC_EV(E500, L2_CLEAN_LINE_INVALIDATIONS) \
	__PMC_EV(E500, L2_INCOHERENT_LINE_INVALIDATIONS) \
	__PMC_EV(E500, L2_COHERENT_LINE_INVALIDATIONS) \
	__PMC_EV(E500, COHERENT_LOOKUP_MISS_DUE_TO_VALID_BUT_INCOHERENT_MATCHES) \
	__PMC_EV(E500, IAC1S_DETECTED) \
	__PMC_EV(E500, IAC2S_DETECTED) \
	__PMC_EV(E500, DAC1S_DTECTED) \
	__PMC_EV(E500, DAC2S_DTECTED) \
	__PMC_EV(E500, DVT0_DETECTED) \
	__PMC_EV(E500, DVT1_DETECTED) \
	__PMC_EV(E500, DVT2_DETECTED) \
	__PMC_EV(E500, DVT3_DETECTED) \
	__PMC_EV(E500, DVT4_DETECTED) \
	__PMC_EV(E500, DVT5_DETECTED) \
	__PMC_EV(E500, DVT6_DETECTED) \
	__PMC_EV(E500, DVT7_DETECTED) \
	__PMC_EV(E500, CYCLES_COMPLETION_STALLED_NEXUS_FIFO_FULL) \
	__PMC_EV(E500, FPU_DOUBLE_PUMP) \
	__PMC_EV(E500, FPU_FINISH) \
	__PMC_EV(E500, FPU_DIVIDE_CYCLES) \
	__PMC_EV(E500, FPU_DENORM_INPUT_CYCLES) \
	__PMC_EV(E500, FPU_RESULT_STALL_CYCLES) \
	__PMC_EV(E500, FPU_FPSCR_FULL_STALL) \
	__PMC_EV(E500, FPU_PIPE_SYNC_STALLS) \
	__PMC_EV(E500, FPU_INPUT_DATA_STALLS) \
	__PMC_EV(E500, DECORATED_LOADS) \
	__PMC_EV(E500, DECORATED_STORES) \
	__PMC_EV(E500, LOAD_RETRIES) \
	__PMC_EV(E500, STWCX_SUCCESSES) \
	__PMC_EV(E500, STWCX_FAILURES) \

#define PMC_EV_E500_FIRST		PMC_EV_E500_CYCLES
#define PMC_EV_E500_LAST		PMC_EV_E500_STWCX_FAILURES
/*
 * All known PMC events.
 *
 * PMC event numbers are allocated sparsely to allow new PMC events to
 * be added to a PMC class without breaking ABI compatibility.  The
 * current allocation scheme is:
 *
 * START	#EVENTS		DESCRIPTION
 * 0		0x1000		Reserved
 * 0x1000	0x0001		TSC
 * 0x2000	0x0080		AMD K7 events
 * 0x2080	0x0100		AMD K8 events
 * 0x10000	0x0080		INTEL architectural fixed-function events
 * 0x10080	0x0F80		INTEL architectural programmable events
 * 0x11000	0x0080		INTEL Pentium 4 events
 * 0x11080	0x0080		INTEL Pentium MMX events
 * 0x11100	0x0100		INTEL Pentium Pro/P-II/P-III/Pentium-M events
 * 0x11200	0x00FF		INTEL XScale events
 * 0x11300	0x00FF		MIPS 24K events
 * 0x11400	0x00FF		Octeon events
 * 0x11500	0x00FF		MIPS 74K events
 * 0x13000	0x00FF		MPC7450 events
 * 0x13100	0x00FF		IBM PPC970 events
 * 0x13300	0x00FF		Freescale e500 events
 * 0x14000	0x0100		ARMv7 events
 * 0x14100	0x0100		ARMv8 events
 * 0x20000	0x1000		Software events
 */
#define	__PMC_EVENTS()				\
	__PMC_EV_BLOCK(TSC,	0x01000)	\
	__PMC_EV_TSC()				\
	__PMC_EV_BLOCK(IAF,     0x10000)	\
	__PMC_EV_IAF()				\
	__PMC_EV_BLOCK(K7,	0x2000)		\
	__PMC_EV_K7()				\
	__PMC_EV_BLOCK(K8,	0x2080)	        \
	__PMC_EV_K8()				\
	__PMC_EV_BLOCK(XSCALE,	0x11200)	\
	__PMC_EV_XSCALE()                       \
	__PMC_EV_BLOCK(MIPS24K,	0x11300)	\
	__PMC_EV_MIPS24K()			\
	__PMC_EV_BLOCK(OCTEON,	0x11400)	\
	__PMC_EV_OCTEON()			\
	__PMC_EV_BLOCK(MIPS74K,	0x11500)        \
	__PMC_EV_MIPS74K()			\
	__PMC_EV_BLOCK(UCP,     0x12080)        \
	__PMC_EV_UCP()				\
	__PMC_EV_BLOCK(PPC7450,	0x13000)	\
	__PMC_EV_PPC7450()			\
	__PMC_EV_BLOCK(PPC970,	0x13100)	\
	__PMC_EV_PPC970()			\
	__PMC_EV_BLOCK(E500,	0x13300)	\
	__PMC_EV_E500()				\
	__PMC_EV_BLOCK(ARMV7,	0x14000)	\
	__PMC_EV_ARMV7()			\
	__PMC_EV_BLOCK(ARMV8,	0x14100)	\
	__PMC_EV_ARMV8()

#define	PMC_EVENT_FIRST	PMC_EV_TSC_TSC
#define	PMC_EVENT_LAST	PMC_EV_SOFT_LAST

#endif /* _DEV_HWPMC_PMC_EVENTS_H_ */
