/*-
 * Copyright (c) 2016 Jared McNeill <jmcneill@invisible.ca>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD: stable/11/sys/arm/allwinner/clk/aw_axiclk.c 308324 2016-11-05 04:17:32Z mmel $
 */

/*
 * Allwinner AXI clock
 */

#include <sys/cdefs.h>
__FBSDID("$FreeBSD: stable/11/sys/arm/allwinner/clk/aw_axiclk.c 308324 2016-11-05 04:17:32Z mmel $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/bus.h>
#include <sys/rman.h>
#include <sys/kernel.h>
#include <sys/module.h>
#include <machine/bus.h>

#include <dev/ofw/ofw_bus.h>
#include <dev/ofw/ofw_bus_subr.h>
#include <dev/ofw/ofw_subr.h>

#include <dev/extres/clk/clk_mux.h>
#include <dev/extres/clk/clk_gate.h>

#include "clkdev_if.h"

#define	AXI_CLK_DIV_RATIO	(0x3 << 0)

static struct ofw_compat_data compat_data[] = {
	{ "allwinner,sun4i-a10-axi-clk",	1 },
	{ NULL, 0 }
};

struct aw_axiclk_sc {
	device_t	clkdev;
	bus_addr_t	reg;
};

#define	AXICLK_READ(sc, val)	CLKDEV_READ_4((sc)->clkdev, (sc)->reg, (val))
#define	AXICLK_WRITE(sc, val)	CLKDEV_WRITE_4((sc)->clkdev, (sc)->reg, (val))
#define	DEVICE_LOCK(sc)		CLKDEV_DEVICE_LOCK((sc)->clkdev)
#define	DEVICE_UNLOCK(sc)	CLKDEV_DEVICE_UNLOCK((sc)->clkdev)

static int
aw_axiclk_init(struct clknode *clk, device_t dev)
{
	clknode_init_parent_idx(clk, 0);
	return (0);
}

static int
aw_axiclk_recalc_freq(struct clknode *clk, uint64_t *freq)
{
	struct aw_axiclk_sc *sc;
	uint32_t val;

	sc = clknode_get_softc(clk);

	DEVICE_LOCK(sc);
	AXICLK_READ(sc, &val);
	DEVICE_UNLOCK(sc);

	*freq = *freq / ((val & AXI_CLK_DIV_RATIO) + 1);

	return (0);
}

static clknode_method_t aw_axiclk_clknode_methods[] = {
	/* Device interface */
	CLKNODEMETHOD(clknode_init,		aw_axiclk_init),
	CLKNODEMETHOD(clknode_recalc_freq,	aw_axiclk_recalc_freq),
	CLKNODEMETHOD_END
};
DEFINE_CLASS_1(aw_axiclk_clknode, aw_axiclk_clknode_class,
    aw_axiclk_clknode_methods, sizeof(struct aw_axiclk_sc), clknode_class);

static int
aw_axiclk_probe(device_t dev)
{
	if (!ofw_bus_status_okay(dev))
		return (ENXIO);

	if (ofw_bus_search_compatible(dev, compat_data)->ocd_data == 0)
		return (ENXIO);

	device_set_desc(dev, "Allwinner AXI Clock");
	return (BUS_PROBE_DEFAULT);
}

static int
aw_axiclk_attach(device_t dev)
{
	struct clknode_init_def def;
	struct aw_axiclk_sc *sc;
	struct clkdom *clkdom;
	struct clknode *clk;
	clk_t clk_parent;
	bus_addr_t paddr;
	bus_size_t psize;
	phandle_t node;
	int error;

	node = ofw_bus_get_node(dev);

	if (ofw_reg_to_paddr(node, 0, &paddr, &psize, NULL) != 0) {
		device_printf(dev, "cannot parse 'reg' property\n");
		return (ENXIO);
	}

	clkdom = clkdom_create(dev);

	error = clk_get_by_ofw_index(dev, 0, 0, &clk_parent);
	if (error != 0) {
		device_printf(dev, "cannot parse clock parent\n");
		return (ENXIO);
	}

	memset(&def, 0, sizeof(def));
	error = clk_parse_ofw_clk_name(dev, node, &def.name);
	if (error != 0) {
		device_printf(dev, "cannot parse clock name\n");
		error = ENXIO;
		goto fail;
	}
	def.id = 1;
	def.parent_names = malloc(sizeof(char *), M_OFWPROP, M_WAITOK);
	def.parent_names[0] = clk_get_name(clk_parent);
	def.parent_cnt = 1;

	clk = clknode_create(clkdom, &aw_axiclk_clknode_class, &def);
	if (clk == NULL) {
		device_printf(dev, "cannot create clknode\n");
		error = ENXIO;
		goto fail;
	}

	sc = clknode_get_softc(clk);
	sc->reg = paddr;
	sc->clkdev = device_get_parent(dev);

	clknode_register(clkdom, clk);

	if (clkdom_finit(clkdom) != 0) {
		device_printf(dev, "cannot finalize clkdom initialization\n");
		error = ENXIO;
		goto fail;
	}

	if (bootverbose)
		clkdom_dump(clkdom);

	return (0);

fail:
	return (error);
}

static device_method_t aw_axiclk_methods[] = {
	/* Device interface */
	DEVMETHOD(device_probe,		aw_axiclk_probe),
	DEVMETHOD(device_attach,	aw_axiclk_attach),

	DEVMETHOD_END
};

static driver_t aw_axiclk_driver = {
	"aw_axiclk",
	aw_axiclk_methods,
	0
};

static devclass_t aw_axiclk_devclass;

EARLY_DRIVER_MODULE(aw_axiclk, simplebus, aw_axiclk_driver,
    aw_axiclk_devclass, 0, 0, BUS_PASS_BUS + BUS_PASS_ORDER_MIDDLE);
