/*
 * schedule.c : Driver program for sports schedule program
 *
 * George Ferguson, ferguson@cs.rochester.edu, 15 Mar 1994
 *
 * This driver now supports many leagues, not just the NHL.
 *
 * You should never have to change this file. All season- and
 * league-specific date is in a separate file whose name should be
 * given by setting DATAFILE before compiling. That file is then
 * #include'd below.
 *
 * Please see the file INSTALLING for instructions on building and
 * installing the package.
 */
/*
  Based on NHL.C - Last modified on 6 Sep 1993 by ferguson.

  Schedule program for the NHL regular season.

  Reconstructed for 1993-94 by:
    George Ferguson <ferguson@cs.rochester.edu>

    With any luck, you should never have to edit this file for a new
    season. All the season-specific information is in schedule.c which
    is included by this file. I also added head-to-head (team or division)
    modes, the manpage, and put together the distribution kit.

  Maintained thorugh 1992-93 season by:
    Valerie Hammerl (hammerl@acsu.buffalo.edu)
    Rob Springall (rgs7077@ultb.isc.rit.edu, rgs7077@ritvax.bitnet)

  ----------------------------------------------------------------------

  Original program concept by Len Carr, used with permission.
  All other features created by Rob Springall, except for neutral site 
  games, assisted by Tom Wilson.

  Additional credit goes to the members of the USENET community who
  contributed ideas and support for this project.

  This program is not copyrighted or registered in any form.  It does,
  however, remain the intellectual property of its authors.

  ----------------------------------------------------------------------
*/

#include <stdio.h>
#include <string.h>
#include <time.h>
#ifndef VMS
#ifndef MSDOS
#include <sys/time.h>
#endif
#endif

/*
 * Load the season-specific code
 */
#include DATAFILE

/*
 * Include version control information for -v option
 */
#include "patchlevel.h"

/*	-	-	-	-	-	-	-	-	*/
/*
 * Functions defined here:
 */
static void initialize(), parse_options(), set_team_arg();
static int str_to_team(), str_to_division();
static void error(), print_help();
static void do_sched(), do_league_sched(), do_team_sched(), do_div_sched();
static void do_team_vs_team(), do_team_vs_div(), do_div_vs_div();
static int today(), parse_date();
static int md_to_leagueday(), leagueday_to_dow();
static int find_char();
static void inc_date(), print_game();

/*
 * Global variables:
 */
char *program;		/* Name of program for error messages */
int start_day;		/* Date of start of season */
int start_month;
int start_year;
int end_day;		/* Date of end of season */
int end_month;
int end_year;
int start_dow;		/* Day of week of start of season (Sun == 0) */
int over_year;		/* True if season starts one year and ends another */
int old_year_offset;	/* Days in start year before season starts */
int new_year_offset;	/* Days in start year after season starts */

int team1 = -1;			/* First team specified on cmd-line */
int team2 = -1;			/* Second team specified on cmd-line */
int team1_is_div = 0;		/* First team is actually a division */
int team2_is_div = 0;		/* Second team is actually a division */
int homeonlyflag = 0;		/* Only print home games */
int awayonlyflag = 0;		/* Only print away games */

#define DEFAULT_GAMES_TO_SHOW 3
int num_games_to_show = DEFAULT_GAMES_TO_SHOW;	/* Value of -n option */

/*
 * These arrays will be munged in initialize() to adjust for leap-years.
 */
/* Number of days in month in non-leap year, Jan == 1. */
int month_len[] = {
  0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
};
/* Index of first day of month in non-leap year, Jan == 1. */
int month_start[] = {
  0,  0, 31, 59, 90,120,151,181,212,243,273,304,334
};

/*
 * String corresponding to weekday (0 == Sunday):
 */
char *dow[] = {
  "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday",
};

/*	-	-	-	-	-	-	-	-	*/

main(argc,argv)
int argc;
char *argv[];
{
    int month,day;

    program = argv[0];
    argc -= 1;
    argv += 1;
    initialize();
    parse_options(&argc,&argv);
    if (argc == 0) {
	today(&month,&day);
	/* If today isn't in season, assume start of season */
	if (md_to_leagueday(month,day) < 0) {
	    fprintf(stderr,
		    "%s: today not during season, assuming start of season\n",
		    program);
	    month = start_month;
	    day = start_day;
	}
	do_sched(month,day);
    } else {
	while (argc--) {
	    if (parse_date(argv[0],&month,&day) >= 0) {
		do_sched(month,day);
	    }
	    argv += 1;
	}
    }
    exit(0);
}

/*
 * initialize():
 *  Initialize the program. This involves setting various "constants"
 *  from the season-specific information.
 */
static void
initialize()
{
    int div,team,i;

    /* Parse the start and end dates */
    if (sscanf(START_DATE,"%d/%d/%d",
	       &start_month,&start_day,&start_year) != 3) {
	fprintf(stderr,"%s: YOW! START_DATE not set correctly: %s\n",
		program,START_DATE);
	exit(1);
    }
    if (sscanf(END_DATE,"%d/%d/%d",
	       &end_month,&end_day,&end_year) != 3) {
	fprintf(stderr,"%s: YOW! END_DATE not set correctly: %s\n",
		program,END_DATE);
	exit(1);
    }
    /* Parse the start dow */
    for (i=0; i < 7; i++) {
	if (strcmp(START_DOW,dow[i]) == 0) {
	    start_dow = i;
	    break;
	}
    }
    if (i == 7) {
	fprintf(stderr,"%s: YOW! START_DOW not set correctly: %s\n",
		program,START_DOW);
	exit(1);
    }
    /* Adjust length of Feb for leap years */
    if (end_year % 400 == 0 ||
	(end_year % 100 != 0 && end_year % 4 == 0)) {
	month_len[2] += 1;
	for (i=3; i <= end_month; i++)
	    month_start[i] += 1;
    }
    /* Compute some stuff about the dates (see md_to_leagueday()) */
    over_year = (start_month > end_month);
    if (over_year) {
	/* Number of days in start_year before season starts: */
	old_year_offset = month_start[start_month] + start_day;
	/* Number of days in start_year after season starts: */
	new_year_offset = month_start[12]+month_len[12]-old_year_offset;
    } else {
	/* Number of days in start_year before season starts: */
	old_year_offset = 0;
	/* Number of days in start_year after season starts: */
	new_year_offset = -(month_start[start_month]+start_day);
    }
    /* Parse the division info */
    for (div=0; div < NUM_DIVISIONS; div++) {
	for (i=0; i < MAX_TEAMS_PER_DIVISION &&
	          divisions[div].teams[i] != NULL; i++) {
	    if ((team=str_to_team(divisions[div].teams[i])) < 0) {
		fprintf(stderr,
			"%s: YOW! division \"%s\" has bogus team: \"%s\"\n",
			program,divisions[div].name,divisions[div].teams[i]);
		exit(1);
	    }
	    divisions[div].flags[team] = 1;
	}
    }
}

/*
 * parse_options(argcp,argvp):
 *  Parse the command-line and update ARGC/ARGV, exit on error.
 */
static void
parse_options(argcp,argvp)
int *argcp;
char ***argvp;
{
    char *arg;
    int team,division;

    while (*argcp > 0) {
	arg = **argvp;
	if (strncmp(arg,"-v",2) == 0) {
	    printf("Sports schedule program v%d.%d by George Ferguson, ferguson@cs.rochester.edu\n",
		   VERSION_MAJOR,VERSION_MINOR);
	    printf("Info for %s by %s\n",VERSION_SEASON,VERSION_ENTERER);
	    exit(0);
	} else if (strncmp(arg,"-h",2) == 0) {
	    print_help();
	    exit(0);
	} else if (strncmp(arg,"-H",2) == 0) {
	    homeonlyflag = 1;
	} else if (strncmp(arg,"-A",2) == 0) {
	    awayonlyflag = 1;
	} else if (strncmp(arg,"-n",2) == 0) {
	    if ((num_games_to_show=atoi(arg+2)) == 0) {
		error("bad or missing value for -n: \"%s\"\n",arg+2);
	    }
	} else if (strncmp(arg,"-t",2) == 0) {
	    if ((team=str_to_team(arg+2)) < 0) {
		error("invalid team code: %s\n",arg+2);
	    }
	    set_team_arg(team,0);
	} else if (strncmp(arg,"-d",2) == 0) {
	    if ((division=str_to_division(arg+2)) < 0) {
		error("invalid divison code: %s\n",arg+2);
	    }
	    set_team_arg(division,1);
	} else if (*arg < '0' || *arg > '9') {
	    if ((team=str_to_team(arg)) >= 0) {
		set_team_arg(team,0);
	    } else if ((division=str_to_division(arg)) >= 0) {
		set_team_arg(division,1);
	    } else {
		error("unknown team or division code: %s\n",arg);
	    }
	} else {
	    /* Must be dates starting now */
	    return;
	}
	*argcp -= 1;
	*argvp += 1;
    }
}

/*
 * str_to_team(str):
 *  Converts string STR to a team index (ie., an index into teams[])
 *  if it is a valid team code and returns it, otherwise returns -1.
 */
static int
str_to_team(str)
char *str;
{
    int team;

    for (team=0; team < NUM_TEAMS; team++) {
	if (strcmp(str,teams[team].abbrev) == 0)
	    return(team);
    }
    return(-1);
}

/*
 * str_to_division(str):
 *  Converts string STR to a division index (ie., an index into divisons[])
 *  if it is a valid team code and returns it, otherwise returns -1.
 */
static int
str_to_division(str)
char *str;
{
    int division;

    for (division=0; division < NUM_DIVISIONS; division++) {
	if (strcmp(str,divisions[division].abbrev) == 0)
	    return(division);
    }
    return(-1);
}

/*
 * set_team_arg(team,flag):
 *  Sets global variable team1 or team2 if one isn't set. If both are
 *  already set, prints a message and exits. If FLAG is non-zero, then
 *  this is really a division code not a team, and the appropriate flag
 *  (team1_is_div or team2_is_div) is set.
 */
static void
set_team_arg(team,flag)
int team,flag;
{
    if (team1 == -1) {
	team1 = team;
	team1_is_div = flag;
    } else if (team2 == -1) {
	team2 = team;
	team2_is_div = flag;
    } else {
	error("too many teams or divisions specified\n",NULL);
    }
}

/*
 * error(fmt,arg):
 *  Print an error message to stderr using FMT and ARG, then print the
 *  help message and exit.
 */
static void
error(fmt,arg)
char *fmt,*arg;
{
    fprintf(stderr,"%s: ",program);
    fprintf(stderr,fmt,arg);
    print_help();
    exit(1);
}

/*
 * print_help():
 *  Print the usage message:
 */
static void
print_help()
{
    int i;

#ifdef EUROPEAN_DATES
    printf("usage: %s [-HAv] [-nNUM] [TEAM|DIV [TEAM|DIV]] [dd/mm ...]\n",program);
#else
    printf("usage: %s [-HAv] [-nNUM] [TEAM|DIV [TEAM|DIV]] [mm/dd ...]\n",program);
#endif
    printf(" With no teams or divisions specified, print the league schedule\n");
    printf("   for given dates (default is today if no dates given).\n");
    printf(" With one team or division, print next NUM games (default %d) for\n",DEFAULT_GAMES_TO_SHOW);
    printf("   that team or teams in that division.\n");
    printf(" With two teams or divisions, print next NUM games where first team (or team\n");
    printf("   in first division) plays second team (or team in second division).\n");
    printf(" -H or -A: Show only home or away games, resp., for first team or division.\n");
    printf(" -v: Print version only: This is version %d.%d for %s\n",
	   VERSION_MAJOR,VERSION_MINOR,VERSION_SEASON);
    printf(" Teams can specified with or without leading -t, from the following list:\n");
    for (i = 0; i < NUM_TEAMS; i++) {
	printf("   %3s - %-16s",teams[i].abbrev,teams[i].city);
	if ((i % 3) == 2)
	    printf( "\n");
    }
    if ((i % 3) != 0)
	printf( "\n");
    printf(" Divisions can specified with or without a leading -d, from the following list:\n");
    for (i = 0; i < NUM_DIVISIONS; i++) {
	printf("   %8s - %-12s",divisions[i].abbrev,divisions[i].name);
	if (((i+1) % 2) == 0)
	    printf( "\n");
    }
    if (i % 2 != 0)
	printf( "\n");
}

/*	-	-	-	-	-	-	-	-	*/
/*
 * do_sched(month,day):
 *  Depending on the globals set by cmd-line flags, do the right thing
 *  for the given date.
 */
static void
do_sched(month,day)
int month,day;
{
    int leagueday;

    /* Convert date to index in season */
    if ((leagueday=md_to_leagueday(month,day)) < 0) {
#ifdef EUROPEAN_DATES
	printf("The %s regular season runs from %d/%d - %d/%d\n",
	       VERSION_SEASON,start_day,start_month,end_day,end_month);
#else
	printf("The %s regular season runs from %d/%d - %d/%d\n",
	       VERSION_SEASON,start_month,start_day,end_month,end_day);
#endif
	return;
    }
    /* Now do the right thing */
    if (team1 == -1 && team2 == -1) {
	do_league_sched(leagueday,month,day);
    } else if (team2 == -1) {
	if (team1_is_div)
	    do_div_sched(team1,leagueday,month,day);
	else
	    do_team_sched(team1,leagueday,month,day);
    } else if (team1_is_div && team2_is_div) {
	do_div_vs_div(team1,team2,leagueday,month,day);
    } else if (team1_is_div) {
	do_team_vs_div(team2,team1,leagueday,month,day);
    } else if (team2_is_div) {
	do_team_vs_div(team1,team2,leagueday,month,day);
    } else {
	do_team_vs_team(team1,team2,leagueday,month,day);
    }
}

/*
 * do_league_sched(leagueday,month,day):
 *  Print league schedule for MON and DAY
 */
static void
do_league_sched(leagueday,month,day)
int leagueday,month,day;
{
    char code;
    int home,info;
    int count,i;

#ifdef EUROPEAN_DATES
    printf("%s schedule for %s, %d/%d...\n",VERSION_SEASON,
	   dow[leagueday_to_dow(leagueday)],day,month);
#else
    printf("%s schedule for %s, %d/%d...\n",VERSION_SEASON,
	   dow[leagueday_to_dow(leagueday)],month,day);
#endif
    /* Scan today's schedule for games */
    count = 0;
    for (i=0; i < NUM_TEAMS; i++) {
	code = schedule[leagueday][i];
	if (ISAWAYCODE(code)) {
	    home = TEAMCODETOINDEX(code);
	    printf("     %-14s at %s",teams[i].city,teams[home].city);
	    code = schedule[leagueday][home];
	    if (ISINFOCODE(code)) {
		info = INFOCODETOINDEX(code);
		printf(" %s",extra_info[info]);
	    }
	    printf("\n");
	    count += 1;
	}
    }
    /* If there are no games, see if it is a special day. */
    if (count == 0) {
	printf("     No games scheduled");
	for (i=0; special_dates[i].month != 0; i++) {
	    if (month == special_dates[i].month &&
		day == special_dates[i].day) {
		printf(": %s",special_dates[i].text);
		break;
	    }
	}
	printf("\n");
    }
}

/*	-	-	-	-	-	-	-	-	*/
/*
 * do_team_sched(team,leagueday,month,day):
 *  Print upcoming games for TEAM starting at MONTH/DAY.
 */
static void
do_team_sched(team,leagueday,month,day)
int team;
int leagueday,month,day;
{
    char code;
    int home,visitor;
    int count;

    if (homeonlyflag) {
	printf("Upcoming %s home games...\n",teams[team].name);
    } else if (awayonlyflag) {
	printf("Upcoming %s away games...\n",teams[team].name);
    } else {
	printf("Upcoming %s games...\n",teams[team].name);
    }
    count = 0;
    while ((leagueday >= 0) && (count < num_games_to_show)) {
	code = schedule[leagueday][team];
	if ((ISHOMECODE(code) || ISINFOCODE(code)) &&	/* TEAM is home */
	    !awayonlyflag) {
	    visitor = find_char(INDEXTOTEAMCODE(team),schedule[leagueday]);
	    print_game(leagueday,month,day,visitor,team);
	    count += 1;
	} else if (ISAWAYCODE(code) && !homeonlyflag) {	/* TEAM is away */
	    home = TEAMCODETOINDEX(code);
	    print_game(leagueday,month,day,team,home);
	    count += 1;
	}
	inc_date(&month,&day);
	leagueday = md_to_leagueday(month,day);
    }
}

/*
 * do_div_sched(div,leagueday,month,day):
 *  Print upcoming games for DIV starting at MONTH/DAY.
 */
static void
do_div_sched(div,leagueday,month,day)
int div;
int leagueday,month,day;
{
    char code;
    int team;
    int count,i;

    if (homeonlyflag) {
	printf("Upcoming %s division home games...\n",divisions[div].name);
    } else if (awayonlyflag) {
	printf("Upcoming %s division away games...\n",divisions[div].name);
    } else {
	printf("Upcoming %s division games...\n",divisions[div].name);
    }
    count = 0;
    while ((leagueday >= 0) && (count < num_games_to_show)) {
	for (i=0; i < NUM_TEAMS; i++) {
	    code = schedule[leagueday][i];
	    if (ISAWAYCODE(code)) {
		team = TEAMCODETOINDEX(code);
		if ((divisions[div].flags[i] && !homeonlyflag) ||
		    (divisions[div].flags[team] && !awayonlyflag)) {
		    print_game(leagueday,month,day,i,team);
		    count += 1;
		}
	    }
	}
	inc_date(&month,&day);
	leagueday = md_to_leagueday(month,day);
    }
}

/*
 * do_team_vs_team(team1,team2,leagueday,month,day):
 *  Prints upcoming games between TEAM1 and TEAM2 starting at MON/DAY.
 */
static void
do_team_vs_team(team1,team2,leagueday,month,day)
int team1,team2;
int leagueday,month,day;
{
    char code1,code2;
    int count;

    if (homeonlyflag) {
	printf("Upcoming games for the %s home to the %s...\n",
	       teams[team1].name,teams[team2].name);
    } else if (awayonlyflag) {
	printf("Upcoming games for the %s away at the %s...\n",
	       teams[team1].name,teams[team2].name);
    } else {
	printf("Upcoming games between the %s and the %s...\n",
	       teams[team1].name,teams[team2].name);
    }
    code1 = INDEXTOTEAMCODE(team1);
    code2 = INDEXTOTEAMCODE(team2);
    count = 0;
    while ((leagueday >= 0) && (count < num_games_to_show)) {
	if (schedule[leagueday][team1] == code2 &&	/* TEAM1 at TEAM2 */
	    !homeonlyflag) {
	    print_game(leagueday,month,day,team1,team2);
	    count += 1;
	} else if (schedule[leagueday][team2] == code1 && /* TEAM2 at TEAM1 */
		   !awayonlyflag) {
	    print_game(leagueday,month,day,team2,team1);
	    count += 1;
	}
	inc_date(&month,&day);
	leagueday = md_to_leagueday(month,day);
    }
}

/*
 * do_team_vs_div(team,div,leagueday,month,day):
 *  Prints upcoming games between TEAM and teams in DIV starting at MON/DAY.
 */
static void
do_team_vs_div(team,div,leagueday,month,day)
int team,div;
int leagueday,month,day;
{
    char code,teamcode;
    int home,count,i;

    if (homeonlyflag) {
	printf("Upcoming games for the %s home to %s division teams...\n",
	       teams[team].name,divisions[div].name);
    } else if (awayonlyflag) {
	printf("Upcoming games for the %s away at %s division teams...\n",
	       teams[team].name,divisions[div].name);
    } else {
	printf("Upcoming games between the %s and %s division teams...\n",
	       teams[team].name,divisions[div].name);
    }
    teamcode = INDEXTOTEAMCODE(team);
    count = 0;
    while ((leagueday >= 0) && (count < num_games_to_show)) {
	code = schedule[leagueday][team];
	if (ISAWAYCODE(code) && !homeonlyflag) {	/* TEAM is away */
	    home = TEAMCODETOINDEX(code);
	    if (divisions[div].flags[home]) {		/* home is in DIV */
		print_game(leagueday,month,day,team,home);
		count += 1;
	    }
	} else if ((ISHOMECODE(code) || ISINFOCODE(code)) && /* TEAM is home */
		   !awayonlyflag) {
	    for (i=0; i < NUM_TEAMS; i++) {
		if (schedule[leagueday][i] == teamcode && /* i away at TEAM */
		    divisions[div].flags[i]) {		/* i in DIV */
		    print_game(leagueday,month,day,i,team);
		    count += 1;
		}
	    }
	}
	inc_date(&month,&day);
	leagueday = md_to_leagueday(month,day);
    }
}

/*
 * do_div_vs_div(div1,div2,leagueday,month,day):
 *  Prints upcoming games between teams in DIV1 and teams in DIV2 starting
 *  at MON/DAY.
 */
static void
do_div_vs_div(div1,div2,leagueday,month,day)
int div1,div2;
int leagueday,month,day;
{
    char code;
    int home;
    int count,i;

    if (div1 == div2) {
	printf("Upcoming games between %s division teams...\n",
	       divisions[div1].name);
    } else if (homeonlyflag) {
	printf("Upcoming games for %s division teams home to %s division teams...\n",
	       divisions[div1].name,divisions[div2].name);
    } else if (awayonlyflag) {
	printf("Upcoming games for %s division teams away at %s division teams...\n",
	       divisions[div1].name,divisions[div2].name);
    } else {
	printf("Upcoming games between %s division teams and %s division teams...\n",
	       divisions[div1].name,divisions[div2].name);
    }
    count = 0;
    while ((leagueday >= 0) && (count < num_games_to_show)) {
	for (i=0; i < NUM_TEAMS; i++) {
	    if (divisions[div1].flags[i]) {		/* Team i in DIV1 */
		code = schedule[leagueday][i];
		if (ISAWAYCODE(code) &&			/* Team i away */
		    (div1 == div2 || !homeonlyflag)) {
		    home = TEAMCODETOINDEX(code);
		    if (divisions[div2].flags[home]) {	/* Home team in DIV2 */
			print_game(leagueday,month,day,i,home);
			count += 1;
		    }
		}
	    } else if (divisions[div2].flags[i]) {	/* Team i in DIV2 */
		code = schedule[leagueday][i];
		if (ISAWAYCODE(code) &&			/* Team i away */
		    (div1 == div2 || !awayonlyflag)) {
		    home = TEAMCODETOINDEX(code);
		    if (divisions[div1].flags[home]) {	/* Home team in DIV1 */
			print_game(leagueday,month,day,i,home);
			count += 1;
		    }
		}
	    }
	}
	inc_date(&month,&day);
	leagueday = md_to_leagueday(month,day);
    }
}

/*	-	-	-	-	-	-	-	-	*/
/*
 * today(monp,dayp):
 *  Set MON and DAY to the current date.
 */
static int
today(monp,dayp)
int *monp,*dayp;
{
    time_t tv;
    struct tm *tmp;

    tv = time(NULL);
    tmp = localtime(&tv);
    *monp = tmp->tm_mon+1;             /* We use Jan == 1 */
    *dayp = tmp->tm_mday;
    return(0);
}

/*
 * parse_date(date,monp,dayp):
 *  Parse given DATE string (format MM/DD) into MON and DAY.
 */
static int
parse_date(date,monp,dayp)
char *date;
int *monp,*dayp;
{
#ifdef EUROPEAN_DATES
    if (sscanf(date,"%d/%d",dayp,monp) != 2) {
	fprintf(stderr,"%s: bad date: %s...format is \"dd/mm\"\n",
		program,date);
	return(-1);
    }
#else
    if (sscanf(date,"%d/%d",monp,dayp) != 2) {
	fprintf(stderr,"%s: bad date: %s...format is \"mm/dd\"\n",
		program,date);
	return(-1);
    }
#endif
    if (*monp < 1 || *monp > 12) {
	fprintf(stderr,"%s: bad month in date: %s\n",program,date);
	return(-1);
    }
    if (*dayp < 1 || *dayp > month_len[*monp]) {
	fprintf(stderr,"%s: bad day in date: %s\n",program,date);
	return(-1);
    }
    return(0);
}

/*
 * md_to_leagueday(month,day):
 *  Return the day of the season for the given MONTH/DAY (ie., the index
 *  into schedule[] for that date's games).
 *  Returns -1 if the date is not during the league season.
 */
static int
md_to_leagueday(month,day)
int month,day;
{
    int leagueday;

    if ((month == start_month && day < start_day) ||
	(month == end_month && day > end_day) ||
	(over_year && (month < start_month && month > end_month)) ||
	(!over_year && (month < start_month || month > end_month))) {
	/* Not during season */
	leagueday = -1;
    } else if (over_year && month >= start_month) {
	/* Before Jan 1 */
	leagueday = month_start[month] + day - old_year_offset;
    } else {
	/* After Jan 1 */
	leagueday = month_start[month] + day + new_year_offset;
    }
    return(leagueday);
}

/*
 * leagueday_to_dow(leagueday):
 *  Return the day of the week for the given leagueday (Sun == 0).
 */
static int
leagueday_to_dow(leagueday)
int leagueday;
{
    return(((leagueday % 7) + start_dow) % 7);
}

/*
 * find_char(c,str):
 *  Return index of first occurrence of C in STR, or -1 if not found.
 */
static int
find_char(c,str)
char c;
char *str;
{
    int n = 0;

    while (*str) {
	if (c == *str++)
	    return(n);
	else
	    n += 1;
    }
    return(-1);
}

/*
 * inc_date(month,day):
 *  Increment day (and month, if necessary) to the next day.
 */
static void
inc_date(month,day)
int *month,*day;
{
    if (*day >= month_len[*month]) {
	*day   = 1;
	*month += 1;
	if (*month > 12) {
	    *month = 1;
	}
    } else {
	(*day)++;
    }
}

/*
 * print_game(leagueday,month,day,visitor,home):
 *  Format and output the entry for a game on DOW/MONTH/DAY between teams
 *  VISITOR and HOME, checking for extra info.
 *  This is used by do_team_sched() and do_head_to_head_sched().
 */
static void
print_game(leagueday,month,day,visitor,home)
int leagueday;
int month,day;
int visitor,home;
{
    char buf[32],code;
    int info;

#ifdef EUROPEAN_DATES
    sprintf(buf,"%s, %d/%d:",dow[leagueday_to_dow(leagueday)],day,month);
#else
    sprintf(buf,"%s, %d/%d:",dow[leagueday_to_dow(leagueday)],month,day);
#endif
    printf("     %-18s",buf);
    printf("%s at %s",teams[visitor].city,teams[home].city);
    code = schedule[leagueday][home];
    if (ISINFOCODE(code)) {
	info = INFOCODETOINDEX(code);
	printf(" %s",extra_info[info]);
    }
    printf("\n");
}
