#include <glib.h>
#include <glib-object.h>
#include <blockdev/utils.h>


#ifndef BD_FS_API
#define BD_FS_API

GQuark  bd_fs_error_quark (void);


#define BD_FS_ERROR bd_fs_error_quark ()
typedef enum {
    BD_FS_ERROR_INVAL,
    BD_FS_ERROR_PARSE,
    BD_FS_ERROR_FAIL,
    BD_FS_ERROR_NOFS,
    BD_FS_ERROR_PIPE,
    BD_FS_ERROR_UNMOUNT_FAIL,
    BD_FS_ERROR_NOT_SUPPORTED,
    BD_FS_ERROR_NOT_MOUNTED,
    BD_FS_ERROR_AUTH,
    BD_FS_ERROR_TECH_UNAVAIL,
    BD_FS_ERROR_UNKNOWN_FS = 11,
} BDFsError;

#define BD_FS_TYPE_EXT2_INFO (bd_fs_ext2_info_get_type ())
GType  bd_fs_ext2_info_get_type ();

#define BD_FS_TYPE_EXT3_INFO (bd_fs_ext3_info_get_type ())
GType  bd_fs_ext3_info_get_type ();

#define BD_FS_TYPE_EXT4_INFO (bd_fs_ext4_info_get_type ())
GType  bd_fs_ext4_info_get_type ();


typedef struct BDFSExtInfo {
    gchar *label;
    gchar *uuid;
    gchar *state;
    guint64 block_size;
    guint64 block_count;
    guint64 free_blocks;
} BDFSExtInfo;

typedef struct BDFSExtInfo BDFSExt4Info;
typedef struct BDFSExtInfo BDFSExt3Info;
typedef struct BDFSExtInfo BDFSExt2Info;

/**
 * BDFSExtInfo:
 * @label: label of the filesystem
 * @uuid: uuid of the filesystem
 * @state: state of the filesystem (e.g. "clean")
 * @block_size: block size used by the filesystem
 * @block_count: number of blocks in the filesystem
 * @free_blocks: number of free blocks in the filesystem
 */
/**
 * bd_fs_ext2_info_copy: (skip)
 *
 * Creates a new copy of @data.
 */
BDFSExt2Info* bd_fs_ext2_info_copy (BDFSExt2Info *data);


/**
 * bd_fs_ext3_info_copy: (skip)
 *
 * Creates a new copy of @data.
 */
BDFSExt3Info* bd_fs_ext3_info_copy (BDFSExt3Info *data);


/**
 * bd_fs_ext4_info_copy: (skip)
 *
 * Creates a new copy of @data.
 */
BDFSExt4Info* bd_fs_ext4_info_copy (BDFSExt4Info *data);


/**
 * bd_fs_ext3_info_free: (skip)
 *
 * Frees @data.
 */
void  bd_fs_ext3_info_free (BDFSExt3Info *data);


/**
 * bd_fs_ext4_info_free: (skip)
 *
 * Frees @data.
 */
void  bd_fs_ext4_info_free (BDFSExt4Info *data);





#define BD_FS_TYPE_XFS_INFO (bd_fs_xfs_info_get_type ())
GType  bd_fs_xfs_info_get_type ();


typedef struct BDFSXfsInfo {
    gchar *label;
    gchar *uuid;
    guint64 block_size;
    guint64 block_count;
} BDFSXfsInfo;

/**
 * BDFSXfsInfo:
 * @label: label of the filesystem
 * @uuid: uuid of the filesystem
 * @block_size: block size used by the filesystem
 * @block_count: number of blocks in the filesystem
 */
/**
 * bd_fs_xfs_info_copy: (skip)
 *
 * Creates a new copy of @data.
 */
BDFSXfsInfo* bd_fs_xfs_info_copy (BDFSXfsInfo *data);


/**
 * bd_fs_xfs_info_free: (skip)
 *
 * Frees @data.
 */
void  bd_fs_xfs_info_free (BDFSXfsInfo *data);



#define BD_FS_TYPE_VFAT_INFO (bd_fs_vfat_info_get_type ())
GType  bd_fs_vfat_info_get_type ();


typedef struct BDFSVfatInfo {
    gchar *label;
    gchar *uuid;
    guint64 cluster_size;
    guint64 cluster_count;
    guint64 free_cluster_count;
} BDFSVfatInfo;

/**
 * BDFSVfatInfo:
 * @label: label of the filesystem
 * @uuid: uuid of the filesystem
 * @cluster_size: cluster size used by the filesystem
 * @cluster_count: number of clusters in the filesystem
 * @free_cluster_count: number of free clusters in the filesystem
 */
/**
 * bd_fs_vfat_info_copy: (skip)
 *
 * Creates a new copy of @data.
 */
BDFSVfatInfo* bd_fs_vfat_info_copy (BDFSVfatInfo *data);


/**
 * bd_fs_vfat_info_free: (skip)
 *
 * Frees @data.
 */
void  bd_fs_vfat_info_free (BDFSVfatInfo *data);



typedef struct BDFSNtfsInfo {
    guint64 size;
    guint64 free_space;
} BDFSNtfsInfo;

/**
 * BDFSNtfsInfo:
 * @size: size of the filesystem in bytes
 * @free_space: number of free space in the filesystem in bytes
 */
/**
 * bd_fs_ntfs_info_copy: (skip)
 *
 * Creates a new copy of @data.
 */
BDFSNtfsInfo* bd_fs_ntfs_info_copy (BDFSNtfsInfo *data);


/**
 * bd_fs_ntfs_info_free: (skip)
 *
 * Frees @data.
 */
void  bd_fs_ntfs_info_free (BDFSNtfsInfo *data);


GType  bd_fs_ntfs_info_get_type ();


typedef enum {
    BD_FS_TECH_GENERIC = 0,
    BD_FS_TECH_MOUNT,
    BD_FS_TECH_EXT2,
    BD_FS_TECH_EXT3,
    BD_FS_TECH_EXT4,
    BD_FS_TECH_XFS,
    BD_FS_TECH_VFAT,
    BD_FS_TECH_NTFS,
    BD_FS_TECH_EXFAT,
} BDFSTech;

typedef enum {
    BD_FS_TECH_MODE_MKFS      = 1 << 0,
    BD_FS_TECH_MODE_WIPE      = 1 << 1,
    BD_FS_TECH_MODE_CHECK     = 1 << 2,
    BD_FS_TECH_MODE_REPAIR    = 1 << 3,
    BD_FS_TECH_MODE_SET_LABEL = 1 << 4,
    BD_FS_TECH_MODE_QUERY     = 1 << 5,
    BD_FS_TECH_MODE_RESIZE    = 1 << 6,
} BDFSTechMode;

#define BD_FS_TYPE_EXFAT_INFO (bd_fs_exfat_info_get_type ())
GType  bd_fs_exfat_info_get_type ();


typedef struct BDFSExfatInfo {
    gchar *label;
    gchar *uuid;
    guint64 sector_size;
    guint64 sector_count;
    guint64 cluster_count;
} BDFSExfatInfo;

/**
 * BDFSExfatInfo:
 * @label: label of the filesystem
 * @uuid: uuid of the filesystem
 * @sector_size: sector size used by the filesystem
 * @cluster_size: cluster size used by the filesystem
 * @sector_count: number of sectors in the filesystem
 * @free_sectors: number of free sectors in the filesystem
 * @cluster_count: number of clusters in the filesystem
 * @free_clusters: number of free clusters in the filesystem
 */
/**
 * bd_fs_exfat_info_free: (skip)
 * @data: (allow-none): %BDFSExfatInfo to free
 *
 * Frees @data.
 */
void  bd_fs_exfat_info_free (BDFSExfatInfo *data);


/**
 * bd_fs_exfat_info_copy: (skip)
 * @data: (allow-none): %BDFSExfatInfo to copy
 *
 * Creates a new copy of @data.
 */
BDFSExfatInfo* bd_fs_exfat_info_copy (BDFSExfatInfo *data);



/**
 * bd_fs_is_tech_avail:
 * @tech: the queried tech
 * @mode: a bit mask of queried modes of operation (#BDFSTechMode) for @tech
 * @error: (out): place to store error (details about why the @tech-@mode combination is not available)
 *
 * Returns: whether the @tech-@mode combination is available -- supported by the
 *          plugin implementation and having all the runtime dependencies available
 */
gboolean  bd_fs_is_tech_avail (BDFSTech tech, guint64 mode, GError **error);


/**
 * bd_fs_wipe:
 * @device: the device to wipe signatures from
 * @all: whether to wipe all (%TRUE) signatures or just the first (%FALSE) one
 * @error: (out): place to store error (if any)
 *
 * Note: This function will wipe signatures on a mounted @device without
 *       asking. Use %bd_fs_wipe_force if you want to control this
 *       behaviour manually.
 *
 * Returns: whether signatures were successfully wiped on @device or not
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_WIPE
 */
gboolean  bd_fs_wipe (const gchar *device, gboolean all, GError **error);


/**
 * bd_fs_wipe_force:
 * @device: the device to wipe signatures from
 * @all: whether to wipe all (%TRUE) signatures or just the first (%FALSE) one
 * @force: whether to force wipe even if the filesystem is mounted
 * @error: (out): place to store error (if any)
 *
 * Returns: whether signatures were successfully wiped on @device or not
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_WIPE
 */
gboolean  bd_fs_wipe_force (const gchar *device, gboolean all, gboolean force, GError **error);


/**
 * bd_fs_clean:
 * @device: the device to clean
 * @error: (out): place to store error (if any)
 *
 * Clean all signatures from @device.
 * Difference between this and bd_fs_wipe() is that this function doesn't
 * return error if @device is already empty. This will also always remove
 * all signatures from @device, not only the first one.
 *
 * Returns: whether @device was successfully cleaned or not
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_WIPE
 */
gboolean  bd_fs_clean (const gchar *device, GError **error);


/**
 * bd_fs_get_fstype:
 * @device: the device to probe
 * @error: (out): place to store error (if any)
 *
 * Get first signature on @device as a string.
 *
 * Returns: (transfer full): type of filesystem found on @device, %NULL in case
 *                           no signature has been detected or in case of error
 *                           (@error is set in this case)
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_QUERY
 */
gchar* bd_fs_get_fstype (const gchar *device,  GError **error);


/**
 * bd_fs_freeze:
 * @mountpoint: mountpoint of the device (filesystem) to freeze
 * @error: (out): place to store error (if any)
 *
 * Freezes filesystem mounted on @mountpoint. The filesystem must
 * support freezing.
 *
 * Returns: whether @mountpoint was successfully freezed or not
 *
 */
gboolean  bd_fs_freeze (const gchar *mountpoint, GError **error);


/**
 * bd_fs_unfreeze:
 * @mountpoint: mountpoint of the device (filesystem) to un-freeze
 * @error: (out): place to store error (if any)
 *
 * Un-freezes filesystem mounted on @mountpoint. The filesystem must
 * support freezing.
 *
 * Returns: whether @mountpoint was successfully unfreezed or not
 *
 */
gboolean  bd_fs_unfreeze (const gchar *mountpoint, GError **error);


/**
 * bd_fs_unmount:
 * @spec: mount point or device to unmount
 * @lazy: enable/disable lazy unmount
 * @force: enable/disable force unmount
 * @extra: (allow-none) (array zero-terminated=1): extra options for the unmount
 *                                                 currently only 'run_as_uid'
 *                                                 and 'run_as_gid' are supported
 *                                                 value must be a valid non zero
 *                                                 uid (gid)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether @spec was successfully unmounted or not
 *
 * Tech category: %BD_FS_TECH_MOUNT (no mode, ignored)
 */
gboolean  bd_fs_unmount (const gchar *spec, gboolean lazy, gboolean force, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_mount:
 * @device: (allow-none): device to mount, if not specified @mountpoint entry
 *                        from fstab will be used
 * @mountpoint: (allow-none): mountpoint for @device, if not specified @device
 *                            entry from fstab will be used
 * @fstype: (allow-none): filesystem type
 * @options: (allow-none): comma delimited options for mount
 * @extra: (allow-none) (array zero-terminated=1): extra options for the unmount
 *                                                 currently only 'run_as_uid'
 *                                                 and 'run_as_gid' are supported
 *                                                 value must be a valid non zero
 *                                                 uid (gid)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether @device (or @mountpoint) was successfully mounted or not
 *
 * Tech category: %BD_FS_TECH_MOUNT (no mode, ignored)
 */
gboolean  bd_fs_mount (const gchar *device, const gchar *mountpoint, const gchar *fstype, const gchar *options, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_get_mountpoint:
 * @device: device to find mountpoint for
 * @error: (out): place to store error (if any)
 *
 * Get mountpoint for @device. If @device is mounted multiple times only
 * one mountpoint will be returned.
 *
 * Returns: (transfer full): mountpoint for @device, %NULL in case device is
 *                           not mounted or in case of an error (@error is set
 *                           in this case)
 *
 * Tech category: %BD_FS_TECH_MOUNT (no mode, ignored)
 */
gchar* bd_fs_get_mountpoint (const gchar *device, GError **error);


/**
 * bd_fs_is_mountpoint:
 * @path: path (folder) to check
 * @error: (out): place to store error (if any)
 *
 * Returns: whether @path is a mountpoint or not
 *
 * Tech category: %BD_FS_TECH_MOUNT (no mode, ignored)
 */
gboolean  bd_fs_is_mountpoint (const gchar *path, GError **error);


/**
 * bd_fs_resize:
 * @device: the device the file system of which to resize
 * @new_size: new requested size for the file system (if 0, the file system is
 *            adapted to the underlying block device)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the file system on @device was successfully resized or not
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_RESIZE
 */
gboolean  bd_fs_resize (const gchar *device, guint64 new_size, GError **error);


 /**
 * bd_fs_repair:
 * @device: the device the file system of which to repair
 * @error: (out): place to store error (if any)
 *
 * Repair filesystem on @device. This calls other fs repair functions from this
 * plugin based on detected filesystem (e.g. bd_fs_xfs_repair for XFS). This
 * function will return an error for unknown/unsupported filesystems.
 *
 * Returns: whether the file system on @device was successfully repaired or not
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_REPAIR
 */
gboolean  bd_fs_repair (const gchar *device, GError **error);


/**
 * bd_fs_check:
 * @device: the device the file system of which to check
 * @error: (out): place to store error (if any)
 *
 * Check filesystem on @device. This calls other fs check functions from this
 * plugin based on detected filesystem (e.g. bd_fs_xfs_check for XFS). This
 * function will return an error for unknown/unsupported filesystems.
 *
 * Returns: whether the file system on @device passed the consistency check or not
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_CHECK
 */
gboolean  bd_fs_check (const gchar *device, GError **error);


/**
 * bd_fs_set_label:
 * @device: the device with file system to set the label for
 * @error: (out): place to store error (if any)
 *
 * Set label for filesystem on @device. This calls other fs label functions from this
 * plugin based on detected filesystem (e.g. bd_fs_xfs_set_label for XFS). This
 * function will return an error for unknown/unsupported filesystems.
 *
 * Returns: whether the file system on @device was successfully relabled or not
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_SET_LABEL
 */
gboolean  bd_fs_set_label (const gchar *device, const gchar *label, GError **error);


typedef enum {
    BD_FS_OFFLINE_SHRINK = 1 << 1,
    BD_FS_OFFLINE_GROW = 1 << 2,
    BD_FS_ONLINE_SHRINK = 1 << 3,
    BD_FS_ONLINE_GROW = 1 << 4
} BDFsResizeFlags;

/**
 * BDFsResizeFlags:
 * Flags indicating whether a filesystem resize action supports growing and/or
 * shrinking if mounted or unmounted.
 */
/**
 * bd_fs_can_resize:
 * @type: the filesystem type to be tested for installed resize support
 * @mode: (out): flags for allowed resizing (i.e. growing/shrinking support for online/offline)
 * @required_utility: (out) (transfer full): the utility binary which is required for resizing (if missing i.e. returns FALSE but no error)
 * @error: (out): place to store error (if any)
 *
 * Searches for the required utility to resize the given filesystem and returns whether
 * it is installed. The mode flags indicate if growing and/or shrinking resize is available if
 * mounted/unmounted.
 * Unknown filesystems or filesystems which do not support resizing result in errors.
 *
 * Returns: whether filesystem resize is available
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_QUERY
 */
gboolean  bd_fs_can_resize (const gchar *type, BDFsResizeFlags *mode, gchar **required_utility, GError **error);


/**
 * bd_fs_can_check:
 * @type: the filesystem type to be tested for installed consistency check support
 * @required_utility: (out) (transfer full): the utility binary which is required for checking (if missing i.e. returns FALSE but no error)
 * @error: (out): place to store error (if any)
 *
 * Searches for the required utility to check the given filesystem and returns whether
 * it is installed.
 * Unknown filesystems or filesystems which do not support checking result in errors.
 *
 * Returns: whether filesystem check is available
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_QUERY
 */
gboolean  bd_fs_can_check (const gchar *type, gchar **required_utility, GError **error);


/**
 * bd_fs_can_repair:
 * @type: the filesystem type to be tested for installed repair support
 * @required_utility: (out) (transfer full): the utility binary which is required for repairing (if missing i.e. return FALSE but no error)
 * @error: (out): place to store error (if any)
 *
 * Searches for the required utility to repair the given filesystem and returns whether
 * it is installed.
 * Unknown filesystems or filesystems which do not support reparing result in errors.
 *
 * Returns: whether filesystem repair is available
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_QUERY
 */
gboolean  bd_fs_can_repair (const gchar *type, gchar **required_utility, GError **error);


/**
 * bd_fs_can_set_label:
 * @type: the filesystem type to be tested for installed label support
 * @required_utility: (out) (transfer full): the utility binary which is required for relabeling (if missing i.e. return FALSE but no error)
 * @error: (out): place to store error (if any)
 *
 * Searches for the required utility to set the label of the given filesystem and returns whether
 * it is installed.
 * Unknown filesystems or filesystems which do not support setting the label result in errors.
 *
 * Returns: whether setting filesystem label is available
 *
 * Tech category: %BD_FS_TECH_GENERIC-%BD_FS_TECH_MODE_QUERY
 */
gboolean  bd_fs_can_set_label (const gchar *type, gchar **required_utility, GError **error);


/**
 * bd_fs_ext2_mkfs:
 * @device: the device to create a new ext2 fs on
 * @extra: (allow-none) (array zero-terminated=1): extra options for the creation (right now
 *                                                 passed to the 'mke2fs' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether a new ext2 fs was successfully created on @device or not
 *
 * Tech category: %BD_FS_TECH_EXT2-%BD_FS_TECH_MODE_MKFS
 */
gboolean  bd_fs_ext2_mkfs (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ext3_mkfs:
 * @device: the device to create a new ext3 fs on
 * @extra: (allow-none) (array zero-terminated=1): extra options for the creation (right now
 *                                                 passed to the 'mke2fs' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether a new ext3 fs was successfully created on @device or not
 *
 * Tech category: %BD_FS_TECH_EXT3-%BD_FS_TECH_MODE_MKFS
 */
gboolean  bd_fs_ext3_mkfs (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ext4_mkfs:
 * @device: the device to create a new ext4 fs on
 * @extra: (allow-none) (array zero-terminated=1): extra options for the creation (right now
 *                                                 passed to the 'mkfs.ext4' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether a new ext4 fs was successfully created on @device or not
 *
 * Tech category: %BD_FS_TECH_EXT4-%BD_FS_TECH_MODE_MKFS
 */
gboolean  bd_fs_ext4_mkfs (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ext2_wipe:
 * @device: the device to wipe an ext2 signature from
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an ext2 signature was successfully wiped from the @device or
 *          not
 *
 * Tech category: %BD_FS_TECH_EXT2-%BD_FS_TECH_MODE_WIPE
 */
gboolean  bd_fs_ext2_wipe (const gchar *device, GError **error);


/**
 * bd_fs_ext3_wipe:
 * @device: the device to wipe an ext3 signature from
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an ext3 signature was successfully wiped from the @device or
 *          not
 *
 * Tech category: %BD_FS_TECH_EXT3-%BD_FS_TECH_MODE_WIPE
 */
gboolean  bd_fs_ext3_wipe (const gchar *device, GError **error);


/**
 * bd_fs_ext4_wipe:
 * @device: the device to wipe an ext4 signature from
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an ext4 signature was successfully wiped from the @device or
 *          not
 *
 * Tech category: %BD_FS_TECH_EXT4-%BD_FS_TECH_MODE_WIPE
 */
gboolean  bd_fs_ext4_wipe (const gchar *device, GError **error);


/**
 * bd_fs_ext2_check:
 * @device: the device the file system on which to check
 * @extra: (allow-none) (array zero-terminated=1): extra options for the check (right now
 *                                                 passed to the 'e2fsck' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an ext2 file system on the @device is clean or not
 *
 * Tech category: %BD_FS_TECH_EXT2-%BD_FS_TECH_MODE_CHECK
 */
gboolean  bd_fs_ext2_check (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ext3_check:
 * @device: the device the file system on which to check
 * @extra: (allow-none) (array zero-terminated=1): extra options for the check (right now
 *                                                 passed to the 'e2fsck' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an ext3 file system on the @device is clean or not
 *
 * Tech category: %BD_FS_TECH_EXT3-%BD_FS_TECH_MODE_CHECK
 */
gboolean  bd_fs_ext3_check (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ext4_check:
 * @device: the device the file system on which to check
 * @extra: (allow-none) (array zero-terminated=1): extra options for the check (right now
 *                                                 passed to the 'e2fsck' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an ext4 file system on the @device is clean or not
 *
 * Tech category: %BD_FS_TECH_EXT4-%BD_FS_TECH_MODE_CHECK
 */
gboolean  bd_fs_ext4_check (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ext2_repair:
 * @device: the device the file system on which to repair
 * @unsafe: whether to do unsafe operations too
 * @extra: (allow-none) (array zero-terminated=1): extra options for the repair (right now
 *                                                 passed to the 'e2fsck' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an ext2 file system on the @device was successfully repaired
 *          (if needed) or not (error is set in that case)
 *
 * Tech category: %BD_FS_TECH_EXT2-%BD_FS_TECH_MODE_REPAIR
 */
gboolean  bd_fs_ext2_repair (const gchar *device, gboolean unsafe, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ext3_repair:
 * @device: the device the file system on which to repair
 * @unsafe: whether to do unsafe operations too
 * @extra: (allow-none) (array zero-terminated=1): extra options for the repair (right now
 *                                                 passed to the 'e2fsck' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an ext3 file system on the @device was successfully repaired
 *          (if needed) or not (error is set in that case)
 *
 * Tech category: %BD_FS_TECH_EXT3-%BD_FS_TECH_MODE_REPAIR
 */
gboolean  bd_fs_ext3_repair (const gchar *device, gboolean unsafe, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ext4_repair:
 * @device: the device the file system on which to repair
 * @unsafe: whether to do unsafe operations too
 * @extra: (allow-none) (array zero-terminated=1): extra options for the repair (right now
 *                                                 passed to the 'e2fsck' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an ext4 file system on the @device was successfully repaired
 *          (if needed) or not (error is set in that case)
 *
 * Tech category: %BD_FS_TECH_EXT4-%BD_FS_TECH_MODE_REPAIR
 */
gboolean  bd_fs_ext4_repair (const gchar *device, gboolean unsafe, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ext2_set_label:
 * @device: the device the file system on which to set label for
 * @label: label to set
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the label of ext2 file system on the @device was
 *          successfully set or not
 *
 * Tech category: %BD_FS_TECH_EXT2-%BD_FS_TECH_MODE_SET_LABEL
 */
gboolean  bd_fs_ext2_set_label (const gchar *device, const gchar *label, GError **error);


/**
 * bd_fs_ext3_set_label:
 * @device: the device the file system on which to set label for
 * @label: label to set
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the label of ext3 file system on the @device was
 *          successfully set or not
 *
 * Tech category: %BD_FS_TECH_EXT3-%BD_FS_TECH_MODE_SET_LABEL
 */
gboolean  bd_fs_ext3_set_label (const gchar *device, const gchar *label, GError **error);


/**
 * bd_fs_ext4_set_label:
 * @device: the device the file system on which to set label for
 * @label: label to set
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the label of ext4 file system on the @device was
 *          successfully set or not
 *
 * Tech category: %BD_FS_TECH_EXT3-%BD_FS_TECH_MODE_SET_LABEL
 */
gboolean  bd_fs_ext4_set_label (const gchar *device, const gchar *label, GError **error);


/**
 * bd_fs_ext2_get_info:
 * @device: the device the file system of which to get info for
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): information about the file system on @device or
 *                           %NULL in case of error
 *
 * Tech category: %BD_FS_TECH_EXT2-%BD_FS_TECH_MODE_QUERY
 */
BDFSExt2Info* bd_fs_ext2_get_info (const gchar *device, GError **error);


/**
 * bd_fs_ext3_get_info:
 * @device: the device the file system of which to get info for
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): information about the file system on @device or
 *                           %NULL in case of error
 *
 * Tech category: %BD_FS_TECH_EXT3-%BD_FS_TECH_MODE_QUERY
 */
BDFSExt3Info* bd_fs_ext3_get_info (const gchar *device, GError **error);


/**
 * bd_fs_ext4_get_info:
 * @device: the device the file system of which to get info for
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): information about the file system on @device or
 *                           %NULL in case of error
 *
 * Tech category: %BD_FS_TECH_EXT3-%BD_FS_TECH_MODE_QUERY
 */
BDFSExt4Info* bd_fs_ext4_get_info (const gchar *device, GError **error);


/**
 * bd_fs_ext2_resize:
 * @device: the device the file system of which to resize
 * @new_size: new requested size for the file system (if 0, the file system is
 *            adapted to the underlying block device)
 * @extra: (allow-none) (array zero-terminated=1): extra options for the resize (right now
 *                                                 passed to the 'resize2fs' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the file system on @device was successfully resized or not
 *
 * Tech category: %BD_FS_TECH_EXT2-%BD_FS_TECH_MODE_RESIZE
 */
gboolean  bd_fs_ext2_resize (const gchar *device, guint64 new_size, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ext3_resize:
 * @device: the device the file system of which to resize
 * @new_size: new requested size for the file system (if 0, the file system is
 *            adapted to the underlying block device)
 * @extra: (allow-none) (array zero-terminated=1): extra options for the resize (right now
 *                                                 passed to the 'resize2fs' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the file system on @device was successfully resized or not
 *
 * Tech category: %BD_FS_TECH_EXT3-%BD_FS_TECH_MODE_RESIZE
 */
gboolean  bd_fs_ext3_resize (const gchar *device, guint64 new_size, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ext4_resize:
 * @device: the device the file system of which to resize
 * @new_size: new requested size for the file system (if 0, the file system is
 *            adapted to the underlying block device)
 * @extra: (allow-none) (array zero-terminated=1): extra options for the resize (right now
 *                                                 passed to the 'resize2fs' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the file system on @device was successfully resized or not
 *
 * Tech category: %BD_FS_TECH_EXT4-%BD_FS_TECH_MODE_RESIZE
 */
gboolean  bd_fs_ext4_resize (const gchar *device, guint64 new_size, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_xfs_mkfs:
 * @device: the device to create a new xfs fs on
 * @extra: (allow-none) (array zero-terminated=1): extra options for the creation (right now
 *                                                 passed to the 'mkfs.xfs' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether a new xfs fs was successfully created on @device or not
 *
 * Tech category: %BD_FS_TECH_XFS-%BD_FS_TECH_MODE_MKFS
 */
gboolean  bd_fs_xfs_mkfs (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_xfs_wipe:
 * @device: the device to wipe an xfs signature from
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an xfs signature was successfully wiped from the @device or
 *          not
 *
 * Tech category: %BD_FS_TECH_XFS-%BD_FS_TECH_MODE_WIPE
 */
gboolean  bd_fs_xfs_wipe (const gchar *device, GError **error);


/**
 * bd_fs_xfs_check:
 * @device: the device containing the file system to check
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an xfs file system on the @device is clean or not
 *
 * Note: If the file system is mounted RW, it will always be reported as not
 *       clean!
 *
 * Tech category: %BD_FS_TECH_XFS-%BD_FS_TECH_MODE_CHECK
 */
gboolean  bd_fs_xfs_check (const gchar *device, GError **error);


/**
 * bd_fs_xfs_repair:
 * @device: the device containing the file system to repair
 * @extra: (allow-none) (array zero-terminated=1): extra options for the repair (right now
 *                                                 passed to the 'xfs_repair' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an xfs file system on the @device was successfully repaired
 *          (if needed) or not (error is set in that case)
 *
 * Tech category: %BD_FS_TECH_XFS-%BD_FS_TECH_MODE_REPAIR
 */
gboolean  bd_fs_xfs_repair (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_xfs_set_label:
 * @device: the device containing the file system to set label for
 * @label: label to set
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the label of xfs file system on the @device was
 *          successfully set or not
 *
 * Tech category: %BD_FS_TECH_XFS-%BD_FS_TECH_MODE_SET_LABEL
 */
gboolean  bd_fs_xfs_set_label (const gchar *device, const gchar *label, GError **error);


/**
 * bd_fs_xfs_get_info:
 * @device: the device containing the file system to get info for (device must
            be mounted, trying to get info for an unmounted device will result
            in an error)
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): information about the file system on @device or
 *                           %NULL in case of error
 *
 * Tech category: %BD_FS_TECH_XFS-%BD_FS_TECH_MODE_QUERY
 */
BDFSXfsInfo* bd_fs_xfs_get_info (const gchar *device, GError **error);


/**
 * bd_fs_xfs_resize:
 * @mpoint: the mount point of the file system to resize
 * @new_size: new requested size for the file system *in file system blocks* (see bd_fs_xfs_get_info())
 *            (if 0, the file system is adapted to the underlying block device)
 * @extra: (allow-none) (array zero-terminated=1): extra options for the resize (right now
 *                                                 passed to the 'xfs_growfs' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the file system mounted on @mpoint was successfully resized or not
 *
 * Tech category: %BD_FS_TECH_XFS-%BD_FS_TECH_MODE_RESIZE
 */
gboolean  bd_fs_xfs_resize (const gchar *mpoint, guint64 new_size, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_vfat_mkfs:
 * @device: the device to create a new vfat fs on
 * @extra: (allow-none) (array zero-terminated=1): extra options for the creation (right now
 *                                                 passed to the 'mkfs.vfat' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether a new vfat fs was successfully created on @device or not
 *
 * Tech category: %BD_FS_TECH_VFAT-%BD_FS_TECH_MODE_MKFS
 */
gboolean  bd_fs_vfat_mkfs (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_vfat_wipe:
 * @device: the device to wipe an vfat signature from
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an vfat signature was successfully wiped from the @device or
 *          not
 *
 * Tech category: %BD_FS_TECH_VFAT-%BD_FS_TECH_MODE_WIPE
 */
gboolean  bd_fs_vfat_wipe (const gchar *device, GError **error);


/**
 * bd_fs_vfat_check:
 * @device: the device containing the file system to check
 * @extra: (allow-none) (array zero-terminated=1): extra options for the repair (right now
 *                                                 passed to the 'fsck.vfat' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an vfat file system on the @device is clean or not
 *
 * Tech category: %BD_FS_TECH_VFAT-%BD_FS_TECH_MODE_CHECK
 */
gboolean  bd_fs_vfat_check (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_vfat_repair:
 * @device: the device containing the file system to repair
 * @extra: (allow-none) (array zero-terminated=1): extra options for the repair (right now
 *                                                 passed to the 'fsck.vfat' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an vfat file system on the @device was successfully repaired
 *          (if needed) or not (error is set in that case)
 *
 * Tech category: %BD_FS_TECH_VFAT-%BD_FS_TECH_MODE_REPAIR
 */
gboolean  bd_fs_vfat_repair (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_vfat_set_label:
 * @device: the device containing the file system to set label for
 * @label: label to set
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the label of vfat file system on the @device was
 *          successfully set or not
 *
 * Tech category: %BD_FS_TECH_VFAT-%BD_FS_TECH_MODE_SET_LABEL
 */
gboolean  bd_fs_vfat_set_label (const gchar *device, const gchar *label, GError **error);


/**
 * bd_fs_vfat_get_info:
 * @device: the device containing the file system to get info for
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): information about the file system on @device or
 *                           %NULL in case of error
 *
 * Tech category: %BD_FS_TECH_VFAT-%BD_FS_TECH_MODE_QUERY
 */
BDFSVfatInfo* bd_fs_vfat_get_info (const gchar *device, GError **error);


/**
 * bd_fs_vfat_resize:
 * @device: the device the file system of which to resize
 * @new_size: new requested size for the file system (if 0, the file system is
 *            adapted to the underlying block device)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the file system on @device was successfully resized or not
 *
 * Tech category: %BD_FS_TECH_VFAT-%BD_FS_TECH_MODE_RESIZE
 */
gboolean  bd_fs_vfat_resize (const gchar *device, guint64 new_size, GError **error);


/**
 * bd_fs_ntfs_mkfs:
 * @device: the device to create a new ntfs fs on
 * @extra: (allow-none) (array zero-terminated=1): extra options for the creation (right now
 *                                                 passed to the 'mkntfs' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether a new NTFS fs was successfully created on @device or not
 *
 * Tech category: %BD_FS_TECH_NTFS-%BD_FS_TECH_MODE_MKFS
 */
gboolean  bd_fs_ntfs_mkfs (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_ntfs_wipe:
 * @device: the device to wipe an ntfs signature from
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an ntfs signature was successfully wiped from the @device or not
 *
 * Tech category: %BD_FS_TECH_NTFS-%BD_FS_TECH_MODE_WIPE
 */
gboolean  bd_fs_ntfs_wipe (const gchar *device, GError **error);


/**
 * bd_fs_ntfs_check:
 * @device: the device containing the file system to check
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an ntfs file system on the @device is clean or not
 *
 * Tech category: %BD_FS_TECH_NTFS-%BD_FS_TECH_MODE_CHECK
 */
gboolean  bd_fs_ntfs_check (const gchar *device, GError **error);


/**
 * bd_fs_ntfs_repair:
 * @device: the device containing the file system to repair
 * @error: (out): place to store error (if any)
 *
 * Returns: whether an NTFS file system on the @device was successfully repaired
 *          (if needed) or not (error is set in that case)
 *
 * Tech category: %BD_FS_TECH_NTFS-%BD_FS_TECH_MODE_REPAIR
 */
gboolean  bd_fs_ntfs_repair (const gchar *device, GError **error);


/**
 * bd_fs_ntfs_set_label:
 * @device: the device containing the file system to set the label for
 * @label: label to set
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the label of the NTFS file system on the @device was
 *          successfully set or not
 *
 * Tech category: %BD_FS_TECH_NTFS-%BD_FS_TECH_MODE_SET_LABEL
 */
gboolean  bd_fs_ntfs_set_label (const gchar *device, const gchar *label, GError **error);


/**
 * bd_fs_ntfs_resize:
 * @device: the device the file system of which to resize
 * @new_size: new requested size for the file system in bytes (if 0, the file system
 *            is adapted to the underlying block device)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the file system on @device was successfully resized or not
 *
 * Tech category: %BD_FS_TECH_NTFS-%BD_FS_TECH_MODE_RESIZE
 */
gboolean  bd_fs_ntfs_resize (const gchar *device, guint64 new_size, GError **error);


/**
 * bd_fs_ntfs_get_info:
 * @device: the device containing the file system to get info for (device must
            not be mounted, trying to get info for a mounted device will result
            in an error)
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): information about the file system on @device or
 *                           %NULL in case of error
 *
 * Tech category: %BD_FS_TECH_NTFS-%BD_FS_TECH_MODE_QUERY
 */
BDFSNtfsInfo* bd_fs_ntfs_get_info (const gchar *device, GError **error);


/**
 * bd_fs_exfat_mkfs:
 * @device: the device to create a new exfat fs on
 * @extra: (allow-none) (array zero-terminated=1): extra options for the creation (right now
 *                                                 passed to the 'mkexfatfs' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether a new exfat fs was successfully created on @device or not
 *
 * Tech category: %BD_FS_TECH_EXFAT-%BD_FS_TECH_MODE_MKFS
 */
gboolean  bd_fs_exfat_mkfs (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_exfat_wipe:
 * @device: the device to wipe a exfat signature from
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the exfat signature was successfully wiped from the @device or
 *          not
 *
 * Tech category: %BD_FS_TECH_EXFAT-%BD_FS_TECH_MODE_WIPE
 */
gboolean  bd_fs_exfat_wipe (const gchar *device, GError **error);


/**
 * bd_fs_exfat_check:
 * @device: the device containing the file system to check
 * @extra: (allow-none) (array zero-terminated=1): extra options for the repair (right now
 *                                                 passed to the 'exfatfsck' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the exfat file system on the @device is clean or not
 *
 * Tech category: %BD_FS_TECH_EXFAT-%BD_FS_TECH_MODE_CHECK
 */
gboolean  bd_fs_exfat_check (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_exfat_repair:
 * @device: the device containing the file system to repair
 * @extra: (allow-none) (array zero-terminated=1): extra options for the repair (right now
 *                                                 passed to the 'exfatfsck' utility)
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the exfat file system on the @device was successfully repaired
 *          (if needed) or not (error is set in that case)
 *
 * Tech category: %BD_FS_TECH_EXFAT-%BD_FS_TECH_MODE_REPAIR
 */
gboolean  bd_fs_exfat_repair (const gchar *device, const BDExtraArg **extra, GError **error);


/**
 * bd_fs_exfat_set_label:
 * @device: the device containing the file system to set label for
 * @label: label to set
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the label of exfat file system on the @device was
 *          successfully set or not
 *
 * Tech category: %BD_FS_TECH_EXFAT-%BD_FS_TECH_MODE_SET_LABEL
 */
gboolean  bd_fs_exfat_set_label (const gchar *device, const gchar *label, GError **error);


/**
 * bd_fs_exfat_get_info:
 * @device: the device containing the file system to get info for
 * @error: (out): place to store error (if any)
 *
 * Returns: (transfer full): information about the file system on @device or
 *                           %NULL in case of error
 * Tech category: %BD_FS_TECH_EXFAT-%BD_FS_TECH_MODE_QUERY
 */
BDFSExfatInfo* bd_fs_exfat_get_info (const gchar *device, GError **error);


#endif  /* BD_FS_API */
