/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2002 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * oldlog.c - support for v0.4 and v0.5 flat file format
 */

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <glib.h>

#include "logfile.h"
#include "types.h"


/*
 * file fields
 */
const gint old_log_fields[] = { DATE, GMT, CALL, BAND, MODE, RST, MYRST, 
	QSLOUT, QSLIN, REMARKS };
const gint old_log_widths[] = { 15, 15, 15, 15, 15, 15, 15, 15, 15, 80 };
const gint old_log_field_nr = 10;

static gint old_log_open(LOGDB*);
static void old_log_close(LOGDB*);
static gint old_log_create(LOGDB*);
static gint old_log_qso_append(LOGDB*, const qso_t*);
static gint old_log_qso_foreach(LOGDB*, gint (*fn)(LOGDB*, qso_t*, gpointer arg), gpointer arg);

const struct log_ops old_log_ops = {
open:			old_log_open,
close:			old_log_close,
create:			old_log_create,
qso_append:	old_log_qso_append,
qso_foreach:	old_log_qso_foreach,
};

/*
 * open for read
 */
gint old_log_open(LOGDB *handle)
{
	FILE *fp;

	fp = fopen(handle->path, "r");
	if (!fp) return -1;
	handle->priv = (gpointer)fp;

	/* TODO: read first line, and determine whether v0.4 and v0.5 */

	handle->column_nr = old_log_field_nr;
	memcpy(handle->column_fields, old_log_fields, sizeof(old_log_fields));
	memcpy(handle->column_widths, old_log_widths, sizeof(old_log_widths));

	return 0;
}

/*
 * open for write
 */
gint old_log_create(LOGDB *handle)
{
	FILE *fp;

	fp = fopen(handle->path, "w");
	if (!fp) return -1;
	handle->priv = (gpointer)fp;
	return 0;
}

void old_log_close(LOGDB *handle)
{
	FILE *fp = (FILE*)handle->priv;
	fclose(fp);
}

/* 
 * version 0.5 format.
 * each field 15 positions, remarks on a separate line, 80 wide
 */
gint old_log_qso_append(LOGDB *handle, const qso_t *q)
{
	FILE *fp = (FILE *)handle->priv;

	fprintf(fp, "%-15s%-15s%-15s%-15s%-15s%-15s%-15s%-15s%-15s\n%-80s\n",
		q[DATE], q[GMT], q[CALL], q[BAND], q[MODE], 
		q[RST], q[MYRST], q[QSLOUT], q[QSLIN], q[REMARKS]);
	return 0;
}

gint old_log_qso_foreach(LOGDB *handle, gint (*fn)(LOGDB*, qso_t*, gpointer arg), gpointer arg)
{
	FILE *fp = (FILE*)handle->priv;
	gint i, ret;
	qso_t q[QSO_FIELDS];
	gchar buffer[81];
		
	while (!feof(fp)) {
		for (i = 1; i < 8; i++) {
			fgets(buffer, 16, fp);

			if (strlen(buffer) != 15 || feof(fp)) {
				return 0;
			}
		q[i] = g_strdup(g_strstrip(buffer));
		}

		/* QSLOUT and QSLIN are not present in version 0.4 */
		fgets(buffer, 16, fp); /*QSL out? */
		if (strlen(buffer) == 15) {
			q[QSLOUT] = g_strdup(g_strstrip(buffer));
			fgets(buffer, 16, fp); /*QSL in*/
			q[QSLIN] = g_strdup(g_strstrip(buffer));
			fgets(buffer, 3, fp); /*EOL*/
		} else {
			q[QSLOUT] = g_strdup("");
			q[QSLIN] = g_strdup("");
			/* EOL already eaten by previous fgets */
		}

		/* line with remarks */
		fgets(buffer, 80, fp);
		q[REMARKS] = g_strdup(g_strstrip(buffer));
		fgets(buffer, 3, fp); /*EOL*/

		ret = (*fn)(handle, q, arg);
		if (ret) return ret;
	}
	return 0;
}
