#include <klocale.h>
#include <kconfig.h>
#include <khtmlview.h>
#include <kstdaction.h>
#include "lyrics.h"
#include <noatun/player.h>
#include <kdebug.h>


extern "C"
{
  Plugin *create_plugin()
  {
    return new Lyrics();
  }
}

Lyrics::Lyrics() : KMainWindow(), Plugin()
{
  NOATUNPLUGINC(Lyrics);

  /* Create default actions */
  (void)KStdAction::close(this, SLOT(close()), actionCollection());
  //(void)KStdAction::print(this, SLOT(print()), actionCollection());
  //(void)KStdAction::printPreview(this, SLOT(printPreview()), actionCollection());
  //(void)KStdAction::mail(this, SLOT(mail()), actionCollection());
  //(void)KStdAction::find(this, SLOT(find()), actionCollection());
  follow_act = new KToggleAction(i18n("Follow Noatun playlist"), "goto", 0, actionCollection(), "follow");
  back_act = KStdAction::back(this, SLOT(back()), actionCollection());
  back_act->setEnabled(false);
  forward_act = KStdAction::forward(this, SLOT(forward()), actionCollection());
  forward_act->setEnabled(false);

  /* Add entry to menu */
  menuID = napp->pluginMenuAdd(i18n("&View lyrics"), this, SLOT(viewLyrics()));

  /* Create history manager and htmlpart */
  history = new HistoryManager();
	htmlpart = new KHTMLPart(this);
	//htmlpart->view()->setMinimumSize(350, 420);

  /* Connect signals/slots */
	connect( htmlpart->browserExtension(), SIGNAL(openURLRequestDelayed( const KURL &, const KParts::URLArgs & )), this, SLOT(openURLRequest( const KURL &, const KParts::URLArgs & )));
  connect( htmlpart, SIGNAL(started(KIO::Job *)), this, SLOT(loadingURL(KIO::Job *)) );
  connect( htmlpart, SIGNAL(completed()), this, SLOT(loadedURL()) );
  connect( history, SIGNAL(uiChanged(int, bool)), this, SLOT(changeUI(int, bool)) );
  connect( napp->player(), SIGNAL(newSong()), this, SLOT(newSong()) );

  /* Status bar */
  statusBar()->insertItem(i18n("Ready"), 0, 1);
  statusBar()->setItemAlignment(0, Qt::AlignLeft);

  /* Finalize the GUI */
	setCentralWidget(htmlpart->view());
  createGUI("lyricsui.rc");
  setAutoSaveSettings("Lyrics");

  /* Load configuration */
  KConfig *config = KGlobal::config();
  config->setGroup("Lyrics");
  follow_act->setChecked(config->readBoolEntry("follow", true));
}

void Lyrics::loadingURL(KIO::Job *)
{
  statusBar()->changeItem(i18n("Loading..."), 0);
}

void Lyrics::loadedURL()
{
  statusBar()->changeItem(i18n("Loaded"), 0);
  setCaption(i18n("Lyrics: %1").arg(napp->player()->current()->property("title")));
}

void Lyrics::back() 
{
  KURL url = history->back();
  if (url.isEmpty())
    return;
  kdDebug() << "Going to " << url.url() << endl;
  htmlpart->openURL(url);
}

void Lyrics::forward()
{
  KURL url = history->forward();
  if (url.isEmpty())
    return;
  kdDebug() << "Going to " << url.url() << endl;
  htmlpart->openURL(url);
}

void Lyrics::changeUI(int button, bool enable)
{
  if (button == HistoryManager::Back)
    back_act->setEnabled( enable );
  else if (button == HistoryManager::Forward)
    forward_act->setEnabled( enable );
}

void Lyrics::go(const KURL &url)
{
  history->addURL(url);
  kdDebug() << "Going to " << url.url() << endl;
  htmlpart->openURL(url);
}

void Lyrics::newSong() {
  // If I'm not following, just forget about it
  if (!follow_act->isChecked())
    return;
  if (isVisible())
    viewLyrics();
  // TODO: If not visible, maybe it would be a good idea
  // to load the lyrics on advance
}


void Lyrics::viewLyrics()
{
	KURL url("http://www.songmeanings.net/mb.php");
	QString title(napp->player()->current()->property("title"));
	QString artist(napp->player()->current()->property("artist"));
  setCaption(i18n("Loading Lyrics for %1").arg(title));
	// Check if we have a music playing
	if (napp->player()->current()) {
  	htmlpart->begin();
  	htmlpart->write(i18n("<HTML><BODY><CENTER>Please hang on!<BR>Searching for %1 - %2 at http://www.songmeanings.net</CENTER></BODY></HTML>").arg(artist).arg(title));
  	htmlpart->end();
  	url.setQuery(QString("?artist=")+KURL::encode_string(artist)+"&title="+KURL::encode_string(title));
  	go(url);
  	this->show();
  }
}

void Lyrics::openURLRequest( const KURL &url, const KParts::URLArgs & )
{
  go(url);
}

Lyrics::~Lyrics()
{
  /* Save configurations */
  KConfig *config = KGlobal::config();
  config->setGroup("Lyrics");
  config->writeEntry("follow", follow_act->isChecked());
  // Force saving, as closeEvent is probably never called
  saveMainWindowSettings(config, "Lyrics");
	napp->pluginMenuRemove(menuID);
}

bool Lyrics::queryClose()
{
	hide();
  return false;
}

/** HISTORY MANAGER ***
*** TODO: Add more complex support (like in Konqueror)
          Maybe use Konqueror's History Manager */

HistoryManager::HistoryManager() {
}

HistoryManager::~HistoryManager() {
}

void HistoryManager::addURL(const KURL &url) {
  /* push current URL to the back */
  if (!currentURL.isEmpty()) {
    if (back_stack.count() == 0)
      emit uiChanged( Back, true );
    back_stack.push(currentURL);
    kdDebug() << "pushing url \"" << currentURL.url() << "\" into back_stack (" << back_stack.count() << ")" << endl;
  }
  /* Make a copy of the new URL */
  currentURL = url;
  kdDebug() << "changing url to \"" << currentURL.url() << "\"" <<  endl;
  /* Clear the forward stack */
  if (forward_stack.count() > 0)
    emit uiChanged( Forward, false );
  forward_stack.clear();
  kdDebug() << "clearing stack" << endl;
}

KURL HistoryManager::back() {
  if (back_stack.count() <= 0)
    return KURL::KURL();
  /* The currentURL is now pushed in the forward_stack */
  if (forward_stack.count() == 0)
    emit uiChanged( Forward, true );
  forward_stack.push(currentURL);
  kdDebug() << "pushing url \"" << currentURL.url() << "\" into forward_stack" << forward_stack.count() << endl;
  /* The last URL is the back stack is now popped as the current one */
  if (back_stack.count() == 1)
    emit uiChanged( Back, false );
  currentURL = back_stack.pop();
  kdDebug() << "poping url \"" << currentURL.url() << "\" from back_stack" << back_stack.count() << endl;
  return currentURL;
}

KURL HistoryManager::forward() {
  if (forward_stack.count() <= 0)
    return KURL::KURL();
  /* Push the currentURL into the back_stack */
  if (back_stack.count() == 0)
    emit uiChanged( Back, true );
  back_stack.push(currentURL);
  kdDebug() << "pushing url \"" << currentURL.url() << "\" into back_stack" << back_stack.count() << endl;
  /* Pop from the forward_stack into the currentURL */
  if (forward_stack.count() == 1)
    emit uiChanged( Forward, false );
  currentURL = forward_stack.pop();
  kdDebug() << "poping url \"" << currentURL.url() << "\" from forward_stack" << forward_stack.count() << endl;
  return currentURL;
}

