/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* 
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <glib.h>

#include <pan/base/pan-glib-extensions.h>


/***
****
****  STRINGS
****
***/

/* null-safe strstr */
char*
pan_strstr (const char * s1,
            const char * s2)
{
	g_return_val_if_fail (s1!=NULL, NULL);
	g_return_val_if_fail (s2!=NULL, NULL);

	return strstr (s1, s2);
}

/* null-safe strcmp.  NULLs < strings. */
gint 
pan_strcmp (const char * a,
            const char * b)
{
	gint retval;

	if (a == b)
		retval = 0;
	else if (!a)
		retval = -1;
	else if (!b)
		retval = 1;
	else
		retval = strcmp (a, b);

	return retval;
}

char*
pan_stristr (const char * string, const char * pattern)
{
	size_t slen;
	size_t plen;
	char * start = (char*) string;

	g_return_val_if_fail (string!=NULL, NULL);
	g_return_val_if_fail (pattern!=NULL, NULL);

 	slen = strlen (string);
	plen = strlen (pattern);

	for (; slen>=plen; ++start, --slen)
	{
		char * sptr;
		char * pptr;

		/* find start of pattern in string */
		while (toupper(*start) != toupper(*pattern))
		{
			++start;
			--slen;

			/* if pattern longer than string */
			if (slen < plen)
				return NULL;
		}

		sptr = start;
		pptr = (char *) pattern;

		while (toupper(*sptr) == toupper(*pptr))
		{
			++sptr;
			++pptr;

			if (!*pptr)
				return start;
		}
	}

	return NULL;
}

gboolean
string_ends_with (const char * str,
                  const char * end)
{
	size_t str_len;
	size_t end_len;
	gboolean retval = FALSE;

	g_return_val_if_fail (end!=NULL, FALSE);
	g_return_val_if_fail (str!=NULL, FALSE);

	str_len = strlen (str);
	end_len = strlen (end);
	if (end_len <= str_len)
		retval = !memcmp(end, str+str_len-end_len, end_len);
 
	return retval;
}


void
replace_gstr (char   ** target_gfree_old,
              char    * assign_from_me)
{
	char * gfree_old;

 	g_return_if_fail (target_gfree_old != NULL);

	gfree_old = *target_gfree_old;
	*target_gfree_old = assign_from_me;
	g_free (gfree_old);
}

/**
 * Replaces the search string inside original with the replace string.
 * This should be safe with overlapping elements of text, though I haven't
 * not tested it with elaborately cruel cases.
 */
char*
pan_substitute (const char * original,
                const char * search,
                const char * replace)
{
	size_t slen;		/* length of search */
	size_t rlen;		/* length of replace */
	size_t tlen;		/* length of target (predicted) */
	gint i;
	const char * o;
	const char * pchar;
	char * t;
	char * retval = NULL;

	g_return_val_if_fail (original!=NULL, NULL);
	g_return_val_if_fail (*original!='\0', NULL);
	g_return_val_if_fail (search!=NULL, NULL);
	g_return_val_if_fail (*search!='\0', NULL);
	g_return_val_if_fail (replace!=NULL, NULL);

	slen = strlen (search);
	rlen = strlen (replace);

	/* calculate the length */

	i = 0;
	tlen = 0;
	pchar = original;
	while ((pchar = pan_strstr (pchar, search))) {
		i++;
		pchar += slen;
	}
	tlen = strlen(original) + i*(rlen - slen);

	/**
	***  Make the substitution.
	**/

	o = original;
	t = retval = g_malloc(tlen + 1);
	while ((pchar = pan_strstr (o, search))) {
		(void) memcpy (t, o, (size_t)(pchar-o));
		t += pchar-o;
		(void) memcpy (t, replace, (size_t)rlen);
		t += rlen;
		o = pchar + slen;
	}
	(void) strcpy ( t, o );

	pan_warn_if_fail (strlen(retval)==tlen);

	return retval;
}

/***
****
****  GLIB AUGMENTATION
****
***/

void
pan_g_ptr_array_append (GPtrArray   * target,
                        gpointer    * source_ptr,
                        guint         source_qty)
{
	guint old_len;

	g_return_if_fail (target!=NULL);
	g_return_if_fail (source_ptr!=NULL);
	g_return_if_fail (source_qty>0);
       
	old_len = target->len;
	g_ptr_array_set_size (target, old_len+source_qty);
	memcpy (target->pdata+old_len, source_ptr, sizeof(gpointer)*source_qty);
}

void
pan_g_ptr_array_assign  (GPtrArray  * target,
                         gpointer   * source_ptr,
                         guint        source_qty)
{
	g_return_if_fail (target!=NULL);
	g_return_if_fail (source_qty==0 || source_ptr!=NULL);

	g_ptr_array_set_size (target, source_qty);
	memcpy (target->pdata, source_ptr, sizeof(gpointer)*source_qty);
}

GPtrArray*
pan_g_ptr_array_dup     (const GPtrArray  * source)
{
	GPtrArray * retval;

	g_return_val_if_fail (source!=NULL, NULL);

	retval = g_ptr_array_new ();
	pan_g_ptr_array_assign (retval, source->pdata, source->len);
	return retval;
}


static void
pan_hash_to_ptr_array_ghfunc (gpointer key, gpointer val, gpointer data)
{
	g_ptr_array_add ((GPtrArray*)data, val);
}
void
pan_hash_to_ptr_array  (GHashTable   * hash,
                        GPtrArray    * fillme)
{
	g_return_if_fail (fillme!=NULL);
	g_ptr_array_set_size (fillme, 0);

	g_return_if_fail (hash!=NULL);
	pan_g_ptr_array_reserve (fillme, g_hash_table_size(hash));
	g_hash_table_foreach (hash, pan_hash_to_ptr_array_ghfunc, fillme);
}


void
pan_g_ptr_array_reserve (GPtrArray  * a,
                         int          n)
{
	int len;

	g_return_if_fail (a!=NULL);

	len = a->len;
	g_ptr_array_set_size (a, MAX(len, n));
	a->len = len;
}

void
pan_g_ptr_array_insert (GPtrArray   * a,
                        gpointer      ptr,
                        int           index)
{
        g_return_if_fail (a!=NULL);

        if (index<0 || index>=a->len)
	{
                g_ptr_array_add (a, ptr);
	}
        else
        {
		pan_g_ptr_array_reserve (a, a->len+1);
                g_memmove (&a->pdata[index+1],
                           &a->pdata[index],
                           sizeof(gpointer)*(a->len-index));
                a->pdata[index] = ptr;
		++a->len;
        }
}

void
pan_g_ptr_array_foreach (GPtrArray   * a,
                         GFunc         func,
                         gpointer      user_data)
{
	guint i;

	g_return_if_fail (a!=NULL);
	g_return_if_fail (func!=NULL);

	for (i=0; i!=a->len; ++i)
	{
		gpointer call_data = g_ptr_array_index (a, i);
		(*func)(call_data, user_data);
	}
}

void
pan_g_string_replace (GString       * string,
                      const char    * search,
		      const char    * replace)
{
	const char * pch;
	size_t slen;
	size_t rlen;
	size_t offset;

	g_return_if_fail (string != NULL);
	g_return_if_fail (is_nonempty_string(search));
	g_return_if_fail (replace != NULL);

	slen = strlen (search);
	rlen = strlen (replace);

	offset = 0;
	while ((pch = pan_strstr (string->str + offset, search)) && *pch) {
		const gint pos = pch - string->str;
		g_string_erase (string, pos, slen);
		g_string_insert (string, pos, replace);
		offset = pos + rlen;
	}
}

void
pan_g_string_strstrip (GString * string)
{
	g_return_if_fail (string != NULL);

	if (string->len != 0)
	{
		g_strstrip (string->str);
		string->len = strlen (string->str);
	}
}

void
pan_g_string_append_len  (GString      * str,
                          const char   * appendme,
                          guint          len)
{
	g_return_if_fail (str!=NULL);
	g_return_if_fail (appendme!=NULL);

	if (len < 1024)
	{
		char buf[1024];
		memcpy (buf, appendme, len);
		buf[len] = '\0';
		g_string_append (str, buf);
	}
	else
	{
		char * pch = g_strndup (appendme, len);
		g_string_append (str, pch);
		g_free (pch);
	}
}

/***
****
****  TOKENS
****
***/

 
void
skip_next_token (const char   * pch,
                 const char     delimiter,
                 const char  ** setme_next_token)
{
	if (is_nonempty_string(pch))
	{
		while (*pch && *pch!=delimiter)
			++pch;
		if (*pch==delimiter)
			++pch; /* skip past delimiter */
		if (setme_next_token)
			*setme_next_token = pch;
	}
}
 
const char*
get_next_token_run (const char   * pch,
                    const char     delimiter,
                    const char  ** setme_next_token,
                    const char  ** setme_start,
                    int          * setme_len)
{
	const char * start = pch;

	if (pch!=NULL)
		while (*pch && *pch!=delimiter)
			++pch;

	*setme_start = start;
	*setme_len = pch!=NULL ? pch-start : 0;
	if (setme_next_token != NULL)
		*setme_next_token = pch!=NULL && *pch==delimiter ? pch+1 : pch;

	return is_nonempty_string(start) ? start : NULL;
}


glong
get_next_token_long (const char   * pch,
                     const char     delimiter,
                     const char  ** setme_next_token)
{
	glong retval = 0;

	if (is_nonempty_string(pch))
	{
		const char * m;
		const char * end = NULL;

		/* find the end of this token */
		skip_next_token (pch, delimiter, &end);
		if (setme_next_token != NULL)
			*setme_next_token = end;

		/* only extract an int of there's a digit in this token.
		   otherwise if isspace(token) is true, atoi may walk
		   right into the next token and give a wrong retval */
		for (m=pch; m!=end; ++m) {
			if (*m=='-' || *m=='+' || isdigit((guchar)*m)) {
				retval = atol (m);
				break;
			}
		}
	}

	return retval;
}

gulong
get_next_token_ulong (const char   * pch,
                      const char     delimiter,
                      const char  ** setme_next_token)
{
	gulong retval = 0;

	if (is_nonempty_string(pch))
	{
		const char * m;
		const char * end = NULL;

		/* find the end of this token */
		skip_next_token (pch, delimiter, &end);
		if (setme_next_token != NULL)
			*setme_next_token = end;

		/* only extract an int of there's a digit in this token.
		   otherwise if isspace(token) is true, strtoul may walk
		   right into the next token and give a wrong retval */
		for (m=pch; m!=end; ++m) {
			if (*m=='-' || *m=='+' || isdigit((guchar)*m)) {
				retval = strtoul (m, NULL, 10);
				break;
			}
		}
	}

	return retval;
}

int
get_next_token_int (const char   * pch,
                    const char     delimiter,
                    const char  ** setme_next_token)
{
	int retval = 0;

	if (is_nonempty_string(pch))
	{
		const char * m;
		const char * end = NULL;

		/* find the end of this token */
		skip_next_token (pch, delimiter, &end);
		if (setme_next_token != NULL)
			*setme_next_token = end;

		/* only extract an int of there's a digit in this token.
		   otherwise if isspace(token) is true, atoi may walk
		   right into the next token and give a wrong retval */
		for (m=pch; m!=end; ++m) {
			if (*m=='-' || *m=='+' || isdigit((guchar)*m)) {
				retval = atoi(m);
				break;
			}
		}
	}

	return retval;
}

char*
get_next_token_str (const char   * pch,
                    const char     delimiter,
                    const char  ** setme_next_token)
{
	char* retval = NULL;
	register const char* end = NULL;

	if (is_nonempty_string(pch)) {
		end = pch;
		while (is_nonempty_string(end) && *end!=delimiter)
			++end;
	}

	if (end != NULL)
		retval = g_strndup (pch, end-pch);

	if (setme_next_token)
		*setme_next_token = is_nonempty_string(end) ? end+1 : end;

	return retval;
}

gboolean
get_next_token_g_str (const char   * pch,
                      const char     delimiter,
                      const char  ** setme_next_token,
                      GString      * setme)
{
	register const char* end = pch;
	gboolean retval;

	while (is_nonempty_string(end) && *end!=delimiter)
		++end;

	g_string_truncate (setme, 0);

	if (end == pch)
		retval = end==NULL ? FALSE : *end==delimiter;
	else {
		pan_g_string_append_len (setme, pch, end-pch);
		retval = TRUE;
	}

	if (setme_next_token)
		*setme_next_token = is_nonempty_string(end) ? end+1 : end;

	return retval;
}


/***
****
****  UNSORTED
****
***/


int
lower_bound (const void         * key,
             const void         * base,
             size_t               n,
             size_t               size,
             int                  (*compare)(const void *, const void *),
             gboolean           * exact_match )
{
	register int low = 0;
	register int high = n - 1;

	while (low<=high)
	{
		const int mid = (unsigned)(low+high)/2u;
		const void * checkme = (void*)(((char*)(base))+(mid*size));
		const int comp = (*compare)(key,checkme);

		if (comp>0) low = mid+1;
		else if (comp<0 ) high = mid-1;
		else {
			if ( exact_match!=NULL )
				*exact_match = TRUE;
			return mid;
		}
	}

	if (exact_match!=NULL)
		*exact_match = FALSE;

	return low;
}


void
commatize_ulong (gulong    num,
		 char    * setme)
{
	char buf[32], *src=buf;
	int len = g_snprintf (buf, sizeof(buf), "%ld", num);
	int i = len % 3;

	for (;;)
	{
		while (*src && i--)
			*setme++ = *src++;
		if (!*src) {
			*setme = '\0';
			return;
		}
		else if (src!=buf)
		   *setme++ = ',';
		i = 3;
	}

	pan_warn_if_reached ();
}


/***
****
****  XML
****
***/

char*
pan_str_escape (const char * str)
{
	gint size_needed;
	char * retval;
	char * out;
	const char * in;

	/* sanity clause */
	if (!is_nonempty_string (str))
		return g_strdup ("");

	/* how big must the buffer be? */
	size_needed = 1;
	for (in=str; *in; ++in) {
		gint inc;
		switch (*in) {
			case '&': inc = 5; break;
			case '>': case '<': inc = 4; break;
			case '"': inc = 6; break;
			default: inc = 1; break;
		}
		size_needed += inc;
	}

	/* build the output string */
	retval = out = g_malloc (size_needed);
	for (in=str; *in; ++in) {
		switch (*in) {
			case '&': memcpy (out, "&amp;", 5); out += 5; break;
			case '>': memcpy (out, "&gt;", 4); out += 4; break;
			case '<': memcpy (out, "&lt;", 4); out += 4; break;
			case '"': memcpy (out, "&quot;", 6); out += 6; break;
			default: *out++ = *in; break;
		}
	}
	*out = '\0';

	return retval;
}

char*
pan_str_unescape (const char * escaped)
{
	const char * src;
	char * buf;
	char * tgt;

	g_return_val_if_fail (escaped!=NULL, NULL);

	src = escaped;
	tgt = buf = g_new0 (char, strlen(src)+1);
	while (*src)
	{
		if (*src!='&') { *tgt++ = *src++; }
		else if (!strncmp(src,"&#",2)) {
			char * endp = NULL;
			long ch;
			src += 2;
			if ((ch = strtoul(src, &endp, 0))) {
				*tgt++ = ch;
				src = endp + 1;
			}
		}
		else if (!strncmp(src,"&lt;",4)) { *tgt++ = '<';  src+=4; }
		else if (!strncmp(src,"&gt;",4))   { *tgt++ = '>';  src+=4; }
		else if (!strncmp(src,"&amp;",5))  { *tgt++ = '&';  src+=5; }
		else if (!strncmp(src,"&apos;",6)) { *tgt++ = '\''; src+=6; }
		else if (!strncmp(src,"&quot;",6)) { *tgt++ = '"';  src+=6; }
		else *tgt++ = *src++;
	}

	*tgt = '\0';

	return buf;
}

/***
****  This is taken from Sylpheed 0.7.0.
***/

static char*
strstr_with_skip_quote (const char *haystack, const char *needle)
{
	register guint haystack_len, needle_len;
	gboolean in_squote = FALSE, in_dquote = FALSE;

	haystack_len = strlen(haystack);
	needle_len   = strlen(needle);

	if (haystack_len < needle_len || needle_len == 0)
		return NULL;

	while (haystack_len >= needle_len) {
		if (!in_squote && !in_dquote &&
		    !strncmp(haystack, needle, needle_len))
			return (char *)haystack;

		/* 'foo"bar"' -> foo"bar"
		   "foo'bar'" -> foo'bar' */
		if (*haystack == '\'') {
			if (in_squote)
				in_squote = FALSE;
			else if (!in_dquote)
				in_squote = TRUE;
		} else if (*haystack == '\"') {
			if (in_dquote)
				in_dquote = FALSE;
			else if (!in_squote)
				in_dquote = TRUE;
		}

		haystack++;
		haystack_len--;
	}

	return NULL;
}

/* this fuction was taken from gstrfuncs.c in glib. */
static char **
strsplit_with_quote (const char * str,
                     const char * delim,
                     int max_tokens)
{
	GSList *string_list = NULL, *slist;
	char **str_array, *s;
	guint i, n = 1;

	g_return_val_if_fail(str != NULL, NULL);
	g_return_val_if_fail(delim != NULL, NULL);

	if (max_tokens < 1)
		max_tokens = G_MAXINT;

	s = strstr_with_skip_quote(str, delim);
	if (s) {
		guint delimiter_len = strlen(delim);

		do {
			guint len;
			char *new_str;

			len = s - str;
			new_str = g_new(char, len + 1);
			strncpy(new_str, str, len);
			new_str[len] = 0;
			string_list = g_slist_prepend(string_list, new_str);
			n++;
			str = s + delimiter_len;
			s = strstr_with_skip_quote(str, delim);
		} while (--max_tokens && s);
	}

	if (*str) {
		n++;
		string_list = g_slist_prepend(string_list, g_strdup(str));
	}

	str_array = g_new(char*, n);

	i = n - 1;

	str_array[i--] = NULL;
	for (slist = string_list; slist; slist = slist->next)
		str_array[i--] = slist->data;

	g_slist_free(string_list);

	return str_array;
}

char**
strsplit_command_line_arguments (const char * command_line)
{
	int i;
	char ** argv;

	/* split the command line into arguments */
	argv = strsplit_with_quote (command_line, " ", 0);

	/* remove the quotes from quoted arguments */
	for (i=0; argv[i]!=NULL; i++) {
		char * str = argv[i];
		if (str[0] == '\'' || str[0] == '\"') {
			int len = strlen(str);
			if (str[len - 1] == str[0]) {
				str[len - 1] = '\0';
				memmove(str, str + 1, len - 1);
			}
		}
	}

	return argv;
}
