/*
 * GQview
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "dnd.h"

#include "collect.h"
#include "filelist.h"
#include "image.h"
#include "img-main.h"
#include "ui_fileops.h"


enum {
	TARGET_APP_COLLECTION_MEMBER,
	TARGET_URI_LIST,
	TARGET_TEXT_PLAIN
};


static GtkTargetEntry file_drag_types[] = {
	{ "text/uri-list", 0, TARGET_URI_LIST },
	{ "text/plain", 0, TARGET_TEXT_PLAIN }
};
static gint n_file_drag_types = 2;

static GtkTargetEntry file_drop_types[] = {
	{ "application/x-gqview-collection-member", 0, TARGET_APP_COLLECTION_MEMBER },
	{ "text/uri-list", 0, TARGET_URI_LIST }
};
static gint n_file_drop_types = 2;


GList *get_uri_file_list(gchar *data)
{
	GList *list = NULL;
	gint b, e;

	b = e = 0;

	while (data[b] != '\0')
		{
		while (data[e] != '\r' && data[e] != '\n' && data[e] != '\0') e++;
		if (!strncmp(data + b, "file:", 5))
			{
			b += 5;
			while (data[b] == '/' && data[b+1] == '/') b++;
			list = g_list_append(list, g_strndup(data + b, e - b));
			}
		while (data[e] == '\r' || data[e] == '\n') e++;
		b = e;
		}

	return list;
}

gchar *make_uri_file_list(GList *list, gint plain_text, gint *length)
{
	gchar *uri_text = NULL;
	gchar *leader = "file:";
	gchar *sep = "\r\n";
	gint total;
	gint leader_l;
	gint sep_l = strlen(sep);
	gchar *ptr;
	GList *work;

	if (!list)
		{
		*length = 0;
		return NULL;
		}

	if (plain_text)
		{
		leader_l = 0;
		}
	else
		{
		leader_l = strlen(leader);
		}
	

	work = list;
	total = 0;

	/* compute length */

	while (work)
		{
		gchar *name = work->data;
		total += leader_l + strlen(name) + sep_l;
		work = work->next;
		}

	/* create list */
	uri_text = g_malloc(total + 1);
	ptr = uri_text;

	work = list;
	while (work)
		{
		gchar *name = work->data;
		if (leader_l > 0)
			{
			strcpy(ptr, leader);
			ptr += leader_l;
			}
		strcpy(ptr, name);
		ptr += strlen(name);
		strcpy(ptr, sep);
		ptr += sep_l;
		work = work->next;
		}
	ptr[0] = '\0';

	*length = total;

	return uri_text;
}

/*
 *-----------------------------------------------------------------------------
 * image drag and drop routines
 *-----------------------------------------------------------------------------
 */ 

static void image_get_dnd_data(GtkWidget *widget, GdkDragContext *context,
			       gint x, gint y,
			       GtkSelectionData *selection_data, guint info,
			       guint time, gpointer data)
{
	ImageWindow *imd = data;

	if (info == TARGET_URI_LIST || info == TARGET_APP_COLLECTION_MEMBER)
		{
		CollectionData *source;
		GList *list;
		GList *info_list;

		if (info == TARGET_URI_LIST)
			{
			list = get_uri_file_list(selection_data->data);
			source = NULL;
			info_list = NULL;
			}
		else
			{
			source = collection_from_dnd_data((gchar *)selection_data->data, &list, &info_list);
			}

		if (list)
			{
			gchar *path;

			path = list->data;

			if (imd == normal_image)
				{
				if (isfile(path))
					{
					gint row;
					gchar *dir = remove_level_from_path(path);
					if (strcmp(dir, current_path) != 0)
						filelist_change_to(dir);
					g_free(dir);
	
					row = find_file_in_list(path);
					if (source && info_list)
						main_image_change_to_from_collection(source, info_list->data);
					else if (row == -1)
						main_image_change_to(path, NULL);
					else
						file_image_change_to(row);
					}
				else if (isdir(path))
					{
					filelist_change_to(path);
					main_image_change_to(NULL, NULL);
					}
				}
			else
				{
				if (isfile(path))
					{
					if (source && info_list)
						image_change_from_collection(imd, source, info_list->data, image_zoom_get_default(imd, zoom_mode));
					else
						image_change_path(imd, path, image_zoom_get_default(imd, zoom_mode));
					}
				}

			if (debug)
				{
				GList *work = list;
				while (work)
					{
					printf("dropped: %s\n", (gchar *)(work->data));
					work = work->next;
					}
				}

			path_list_free(list);
			g_list_free(info_list);
			}
		}
}

static void image_set_dnd_data(GtkWidget *widget, GdkDragContext *context,
			       GtkSelectionData *selection_data, guint info,
			       guint time, gpointer data)
{
	ImageWindow *imd = data;
	const gchar *path = image_get_path(imd);

	if (path)
		{
		gchar *text = NULL;
		switch (info)
			{
			case TARGET_URI_LIST:
				text = g_strconcat("file:", path, "\r\n", NULL);
				break;
			case TARGET_TEXT_PLAIN:
				text = g_strdup(path);
				break;
			}
		if (text)
			{
			gtk_selection_data_set (selection_data, selection_data->target,
						8, text, strlen(text));
			g_free(text);
			}
		}
	else
		{
		gtk_selection_data_set (selection_data, selection_data->target,
					8, NULL, 0);
		}
}

static void image_drag_end(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	ImageWindow *imd = data;
	if (context->action == GDK_ACTION_MOVE)
		{
		gint row = find_file_in_list(image_get_path(imd));
		if (row < 0) return;
		if (main_image_get_path() && strcmp(main_image_get_path(), image_get_path(imd)) == 0)
			{
			if (row < file_count() - 1)
				{
				main_image_next();
				}
			else
				{
				main_image_prev();
				}
			}
		filelist_refresh();
		}
}

void image_dnd_init(ImageWindow *imd)
{
	gtk_drag_source_set(imd->image, GDK_BUTTON2_MASK,
				file_drag_types, n_file_drag_types, GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK);
	gtk_signal_connect(GTK_OBJECT(imd->image), "drag_data_get",
				image_set_dnd_data, imd);
	gtk_signal_connect(GTK_OBJECT(imd->image), "drag_end",
				image_drag_end, imd);

	/* FIXME fix image.c to redraw when exposing because GTK_DEST_DEFAULT_HIGHLIGHT is enabled */
	gtk_drag_dest_set(imd->image,
			  GTK_DEST_DEFAULT_MOTION /* FIXME | GTK_DEST_DEFAULT_HIGHLIGHT */ | GTK_DEST_DEFAULT_DROP,
			  file_drop_types, n_file_drop_types,
                          GDK_ACTION_COPY);
	gtk_signal_connect(GTK_OBJECT(imd->image), "drag_data_received",
				image_get_dnd_data, imd);
}

/*
 *-----------------------------------------------------------------------------
 * file list drag and drop routines (private)
 *-----------------------------------------------------------------------------
 */ 

static void file_list_set_dnd_data(GtkWidget *widget, GdkDragContext *context,
			      GtkSelectionData *selection_data, guint info,
			      guint time, gpointer data)
{
	gchar *uri_text = NULL;
	gint total;

	switch (info)
		{
		case TARGET_URI_LIST:
			break;
		case TARGET_TEXT_PLAIN:
			break;
		default:
			return;
			break;
		}
	
	if (file_clicked_is_selected())
		{
		GList *list;

		list = file_get_selected_list();
		if (!list) return;

		uri_text = make_uri_file_list(list, (info == TARGET_TEXT_PLAIN), &total);

		path_list_free(list);
		}
	else
		{
		gchar *path = file_clicked_get_path();
		if (!path) return;
		switch (info)
			{
			case TARGET_URI_LIST:
				uri_text = g_strconcat("file:", path, "\r\n", NULL);
				break;
			case TARGET_TEXT_PLAIN:
				uri_text = g_strdup(path);
				break;
			}
		total = strlen(uri_text);
		g_free(path);
		}

	if (debug) printf(uri_text);

	gtk_selection_data_set (selection_data, selection_data->target,
					8, uri_text, total);
	g_free(uri_text);

	file_clist_highlight_unset();
}

static void file_list_drag_begin(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	file_clist_highlight_set();
}

static void file_list_drag_end(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	file_clist_highlight_unset();

	if (context->action == GDK_ACTION_MOVE)
		{
		filelist_refresh();
		}
}

/*
 *-----------------------------------------------------------------------------
 * directory list drag and drop routines (private)
 *-----------------------------------------------------------------------------
 */ 

static void dir_list_set_dnd_data(GtkWidget *widget, GdkDragContext *context,
			      GtkSelectionData *selection_data, guint info,
			      guint time, gpointer data)
{
	gint row = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(dir_clist), "click_row"));

	if (row >= 0)
		{
		gchar *name = gtk_clist_get_row_data(GTK_CLIST(dir_clist), row);
		gchar *new_path;
		gchar *text = NULL;

		if (strcmp(name, ".") == 0)
			new_path = g_strdup(current_path);
		else if (strcmp(name, "..") == 0)
			new_path = remove_level_from_path(current_path);
		else
			{
			if (strcmp(current_path, "/") == 0)
				new_path = g_strconcat(current_path, name, NULL);
			else
				new_path = g_strconcat(current_path, "/", name, NULL);
			}

		
		switch (info)
			{
			case TARGET_URI_LIST:
				text = g_strconcat("file:", new_path, "\r\n", NULL);
				break;
			case TARGET_TEXT_PLAIN:
				text = g_strdup(new_path);
				break;
			}
		if (text)
			{
			gtk_selection_data_set (selection_data, selection_data->target,
					8, text, strlen(text));
			g_free(text);
			}
		g_free(new_path);
		}
}

static void dir_list_drag_begin(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	dir_clist_set_highlight(TRUE);
}

static void dir_list_drag_end(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	dir_clist_set_highlight(FALSE);

	if (context->action == GDK_ACTION_MOVE)
		{
		filelist_refresh();
		}
}

/*
 *-----------------------------------------------------------------------------
 * drag and drop initialization (public)
 *-----------------------------------------------------------------------------
 */ 

void init_dnd(void)
{
	/* dir clist */
	gtk_drag_source_set(dir_clist, GDK_BUTTON2_MASK,
				file_drag_types, n_file_drag_types, GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK);
	gtk_signal_connect(GTK_OBJECT(dir_clist), "drag_begin",
				dir_list_drag_begin, NULL);
	gtk_signal_connect(GTK_OBJECT(dir_clist), "drag_data_get",
				dir_list_set_dnd_data, NULL);
	gtk_signal_connect(GTK_OBJECT(dir_clist), "drag_end",
				dir_list_drag_end, NULL);

	/* file clist */
	gtk_drag_source_set(file_clist, GDK_BUTTON2_MASK,
				file_drag_types, n_file_drag_types, GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK);
	gtk_signal_connect(GTK_OBJECT(file_clist), "drag_begin",
				file_list_drag_begin, NULL);
	gtk_signal_connect(GTK_OBJECT(file_clist), "drag_data_get",
				file_list_set_dnd_data, NULL);
	gtk_signal_connect(GTK_OBJECT(file_clist), "drag_end",
				file_list_drag_end, NULL);

	/* image */
	image_dnd_init(main_image);
}


