/*****************************************************************************/
/*  cache.c - contains the cache routines                                    */
/*  Copyright (C) 1998-2002 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "gftp.h"

FILE *
gftp_new_cache_entry (gftp_request * request)
{
  char *cachedir, *tempstr, *temp1str, *cachefile, description[255];
  FILE *fd;

  g_snprintf (description, sizeof (description), "%s://%s@%s:%d%s",
              request->url_prefix, 
              request->username == NULL ? "" : request->username,
              request->hostname == NULL ? "" : request->hostname,
              request->port, 
              request->directory == NULL ? "" : request->directory);

  if ((fd = gftp_find_cache_entry (request)) != NULL)
    return (fd);

  cachedir = expand_path (BASE_CONF_DIR "/cache");
  tempstr = g_strdup_printf ("%s/index.db", cachedir);
  if ((fd = fopen (tempstr, "ab+")) == NULL)
    {
      g_free (tempstr);
      g_free (cachedir);
      return (NULL);
    }
  g_free (tempstr);

  srand (time (NULL));
  tempstr = NULL;
  cachefile = NULL;
  do
    {
      if (tempstr != NULL)
        g_free (tempstr);

      if (cachefile != NULL)
        g_free (cachefile);

      cachefile = g_strdup_printf ("cache%ld",
                                   1 + (long) (99999999.0 * rand () /
				   (RAND_MAX + 1.0)));
      tempstr = g_strdup_printf ("%s/%s", cachedir, cachefile);
    }
  while (access (tempstr, F_OK) != -1);

  temp1str = g_strdup_printf ("%s\t%s\n", description, cachefile);
  fseek (fd, 0, SEEK_END);
  fwrite (temp1str, 1, strlen (temp1str), fd);
  g_free (temp1str);
  if (fclose (fd) != 0)
    return (NULL);

  fd = fopen (tempstr, "wb+");
  g_free (tempstr);
  g_free (cachefile);
  g_free (cachedir);
  return (fd);
}


FILE *
gftp_find_cache_entry (gftp_request * request)
{
  char *cachedir, *tempstr, buf[255], *pos, description[255];
  FILE *indexfd, *cachefd;
  size_t len;

  g_snprintf (description, sizeof (description), "%s://%s@%s:%d%s",
              request->url_prefix, 
              request->username == NULL ? "" : request->username,
              request->hostname == NULL ? "" : request->hostname,
              request->port, 
              request->directory == NULL ? "" : request->directory);

  cachedir = expand_path (BASE_CONF_DIR "/cache");
  if (access (cachedir, F_OK) == -1)
    mkdir (cachedir, 0x1C0);

  tempstr = g_strdup_printf ("%s/index.db", cachedir);
  if ((indexfd = fopen (tempstr, "rb")) == NULL)
    {
      g_free (tempstr);
      g_free (cachedir);
      return (NULL);
    }
  g_free (tempstr);

  while (fgets (buf, sizeof (buf), indexfd))
    {
      len = strlen (buf);
      if (buf[len - 1] == '\n')
        {
          len--;
          buf[len] = '\0';
        }
      if (buf[len - 1] == '\r')
        {
          len--;
          buf[len] = '\0';
        }

      if (!((pos = strchr (buf, '\t')) != NULL && *(pos + 1) != '\0'))
	continue;

      if (pos - buf != strlen (description))
        continue;

      if (strncmp (buf, description, strlen (description)) == 0)
	{
	  pos++;
	  tempstr = g_strdup_printf ("%s/%s", cachedir, pos);
	  if (fclose (indexfd) != 0)
            return (NULL);

	  cachefd = fopen (tempstr, "rb+");
          g_free (cachedir);
	  g_free (tempstr);
          fseek (cachefd, 0, SEEK_END);
          if (ftell (cachefd) == 0)
            { 
              fclose (cachefd); 
              return (NULL);
            } 
          fseek (cachefd, 0, SEEK_SET);
	  return (cachefd);
	}
    }
  fclose (indexfd);
  g_free (cachedir);
  return (NULL);
}


void
gftp_clear_cache_files (void)
{
  char *cachedir, *tempstr, buf[255], *pos;
  FILE *indexfd;
  size_t len;

  cachedir = expand_path (BASE_CONF_DIR "/cache");
  if (access (cachedir, F_OK) == -1)
    {
      g_free (cachedir);
      return;
    }

  tempstr = g_strdup_printf ("%s/index.db", cachedir);
  if ((indexfd = fopen (tempstr, "rb")) == NULL)
    {
      g_free (tempstr);
      g_free (cachedir);
      return;
    }
  unlink (tempstr);
  g_free (tempstr);

  while (fgets (buf, sizeof (buf), indexfd))
    {
      len = strlen (buf);
      if (buf[len - 1] == '\n')
        {
          buf[len - 1] = '\0';
          len--;
        }
      if (buf[len - 1] == '\r')
        {
          buf[len - 1] = '\0';
          len--;
        }

      if (!((pos = strchr (buf, '\t')) != NULL && *(pos + 1) != '\0'))
	continue;
      pos++;
      tempstr = g_strdup_printf ("%s/%s", cachedir, pos);
      unlink (tempstr);
      g_free (tempstr);
    }
  g_free (cachedir);
  fclose (indexfd);
}


void
gftp_delete_cache_entry (gftp_request * request)
{
  char *cachedir, *tempstr, *pos, *oldfile, *newfile, buf[255], 
       description[255];
  FILE *indexfd, *newfd;

  g_snprintf (description, sizeof (description), "%s://%s@%s:%d%s",
              request->url_prefix, 
              request->username == NULL ? "" : request->username,
              request->hostname == NULL ? "" : request->hostname,
              request->port, 
              request->directory == NULL ? "" : request->directory);

  cachedir = expand_path (BASE_CONF_DIR "/cache");
  if (access (cachedir, F_OK) == -1)
    {
      g_free (cachedir);
      return;
    }

  oldfile = g_strdup_printf ("%s/index.db", cachedir);
  if ((indexfd = fopen (oldfile, "rb")) == NULL)
    {
      g_free (oldfile);
      g_free (cachedir);
      return;
    }

  newfile = g_strdup_printf ("%s/index.db.new", cachedir);
  if ((newfd = fopen (newfile, "wb")) == NULL)
    {
      g_free (oldfile);
      g_free (newfile);
      g_free (cachedir);
      return;
    }

  while (fgets (buf, sizeof (buf), indexfd))
    {
      if (!((pos = strchr (buf, '\t')) != NULL && *(pos + 1) != '\0'))
	continue;
      *pos = '\0';
      if (strcmp (buf, description) == 0)
	{
	  tempstr = g_strdup_printf ("%s/%s", cachedir, pos + 1);
	  unlink (tempstr);
	  g_free (tempstr);
	}
      else
	fwrite (buf, 1, strlen (buf), newfd);
    }

  fclose (indexfd);
  fclose (newfd);

  unlink (oldfile);
  rename (newfile, oldfile);

  g_free (oldfile);
  g_free (newfile);
  g_free (cachedir);
}

