/*****************************************************************************/

/*
 *      main.c  --  STP router main.
 *
 *      Copyright (C) 2000
 *        Thomas Sailer (sailer@ife.ee.ethz.ch)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "stp.h"

#include <sys/time.h>
#include <sys/types.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

#define _GNU_SOURCE
#include <getopt.h>

#if WIN32

#include <windows.h>
#include <windowsx.h>
#include <winsock.h>

typedef int socklen_t;

#ifndef MSG_DONTWAIT
#define MSG_DONTWAIT 0
#endif

#undef EAGAIN
#define EAGAIN       WSAEWOULDBLOCK
#undef EINTR
#define EINTR        WSAEINTR
#define EINPROGRESS  WSAEINPROGRESS

#else

#include <fcntl.h>
#include <sys/socket.h>
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>

extern inline int closesocket(int fd)
{
	return close(fd);
}

#endif

/* ---------------------------------------------------------------------- */

LIST_HEAD(confudpinputlist);
LIST_HEAD(conftcpinputlist);
LIST_HEAD(confudpoutputlist);
LIST_HEAD(conftcpoutputlist);
LIST_HEAD(udpinputlist);
LIST_HEAD(tcpinputlist);
LIST_HEAD(tcpoutputlist);

static int udpsendsock = -1;

/* ---------------------------------------------------------------------- */

static struct udpinput *find_udpinput(u_int16_t port)
{
	struct list_head *list;
	struct udpinput *ui;

	for (list = udpinputlist.next; list != &udpinputlist; list = list->next) {
		ui = list_entry(list, struct udpinput, list);
		if (ui->port == port)
			return ui;
	}
	return NULL;
}

static struct tcpinput *find_tcpinput(u_int16_t port)
{
	struct list_head *list;
	struct tcpinput *ti;

	for (list = tcpinputlist.next; list != &tcpinputlist; list = list->next) {
		ti = list_entry(list, struct tcpinput, list);
		if (ti->port == port)
			return ti;
	}
	return NULL;
}

static void start_listen(void)
{
	struct list_head *list;
	struct confudpinput *cui;
	struct udpinput *ui;
	struct conftcpinput *cti;
	struct tcpinput *ti;
	struct conftcpoutput *cto;
	struct sockaddr_in sin;
	time_t curtime;

	/* create UDP send socket */
	udpsendsock = socket(PF_INET, SOCK_DGRAM, 0);
	if (udpsendsock == -1) {
		fprintf(stderr, "stprouter: unable to create UDP send socket, socket error %s (%d)\n",
			strerror(errno), errno);
		exit(1);
	}
	/* create UDP receive sockets */
	for (list = confudpinputlist.next; list != &confudpinputlist;) {
		cui = list_entry(list, struct confudpinput, list);
		list = list->next;
		ui = find_udpinput(cui->port);
		if (!ui) {
			ui = calloc(1, sizeof(struct udpinput));
			if (!ui) {
				fprintf(stderr, "stprouter: out of memory\n");
				exit(1);
			}
			ui->fd = socket(PF_INET, SOCK_DGRAM, 0);
			if (ui->fd == -1) {
				fprintf(stderr, "stprouter: unable to create UDP receive socket, socket error %s (%d)\n",
					strerror(errno), errno);
				exit(1);
			}
			sin.sin_family = AF_INET;
			sin.sin_addr.s_addr = htonl(INADDR_ANY);
			sin.sin_port = cui->port;
			if (bind(ui->fd, (struct sockaddr *)&sin, sizeof(sin))) {
				fprintf(stderr, "stprouter: unable to create UDP receive socket, bind error %s (%d)\n",
					strerror(errno), errno);
				exit(1);
			}
			list_add_tail(&ui->list, &udpinputlist);
		}
		list_add_tail(&ui->conflist, &cui->uifdlist);
	}
	/* create TCP listen sockets */
	for (list = conftcpinputlist.next; list != &conftcpinputlist;) {
		cti = list_entry(list, struct conftcpinput, list);
		list = list->next;
		ti = find_tcpinput(cti->port);
		if (!ti) {
			ti = calloc(1, sizeof(struct tcpinput));
			if (!ti) {
				fprintf(stderr, "stprouter: out of memory\n");
				exit(1);
			}
			ti->fd = socket(PF_INET, SOCK_STREAM, 0);
			if (ti->fd == -1) {
				fprintf(stderr, "stprouter: unable to create TCP listen socket, socket error %s (%d)\n",
					strerror(errno), errno);
				exit(1);
			}
			sin.sin_family = AF_INET;
			sin.sin_addr.s_addr = htonl(INADDR_ANY);
			sin.sin_port = cti->port;
			if (bind(ti->fd, (struct sockaddr *)&sin, sizeof(sin))) {
				fprintf(stderr, "stprouter: unable to create TCP listen socket, bind error %s (%d)\n",
					strerror(errno), errno);
				exit(1);
			}
			if (listen(ti->fd, 5)) {
				fprintf(stderr, "stprouter: unable to create TCP listen socket, listen error %s (%d)\n",
					strerror(errno), errno);
				exit(1);
			}
			list_add_tail(&ti->list, &tcpinputlist);
		}
		list_add_tail(&ti->conflist, &cti->tifdlist);
	}
	/* initialize connect times */
	time(&curtime);
	for (list = conftcpoutputlist.next; list != &conftcpoutputlist;) {
		cto = list_entry(list, struct conftcpoutput, list);
		list = list->next;
		cto->nextconntime = curtime;
		if (cto->interval < 30)
			cto->interval = 30;
	}
}

static int parse_packet(struct packet **pkt, const unsigned char *buf, unsigned int len)
{
	const unsigned char *ptr = buf, *end = buf + len, *lend;
	char hdrline[128];
	unsigned char *wptr, *wend;
	struct packet *p, *p2;
	int telemlen = -1;
	unsigned int hdrllen;

	*pkt = NULL;
	if (!len)
		return 0;
	p = calloc(1, sizeof(struct packet) + MAXPACKETSIZE);
	if (!p)
		return 0;
	wptr = p->buf;
	wend = wptr + MAXPACKETSIZE;
	for (;;) {
		if (ptr >= end)
			goto unfin;
		if (ptr[0] == '\r' || ptr[0] == '\n')
			break;
		lend = memchr(ptr, '\n', end - ptr);
		if (!lend)
			goto unfin;
		hdrllen = lend - ptr;
		if (lend[-1] == '\r')
			hdrllen--;
		if (hdrllen >= sizeof(hdrline) || wptr + hdrllen + 2 > wend)
			goto err;
		memcpy(hdrline, ptr, hdrllen);
		hdrline[hdrllen] = 0;
		memcpy(wptr, ptr, hdrllen);
		ptr = lend + 1;
		if (!strncmp(hdrline, "Source: ", 8)) 
			strncpy(p->satname, hdrline+8, sizeof(p->satname));
		else if (!strncmp(hdrline, "Length: ", 8))
			telemlen = (strtoul(hdrline+8, NULL, 10) + 7) >> 3;
		else if (!strncmp(hdrline, "X-HB9JNX-Hopcount: ", 19)) {
			p->hopcnt = strtoul(hdrline+19, NULL, 10);
			continue;
		}
		/* append line to output */
		wptr += hdrllen;
		*wptr++ = '\r';
		*wptr++ = '\n';
	}
	p->hopcnt += 1;
	hdrllen = snprintf(hdrline, sizeof(hdrline), "X-HB9JNX-Hopcount: %u\r\n", p->hopcnt);
	if (telemlen <= 0 || !p->satname[0] || wptr + 2 + telemlen + hdrllen > wend)
		goto err;
	memcpy(wptr, hdrline, hdrllen);
	wptr += hdrllen;
	if (ptr[0] == '\r')
		ptr++;
	if (ptr >= end)
		goto unfin;
	if (ptr[0] != '\n')
		goto err;
	ptr++;
       	if (ptr + telemlen > end)
		goto unfin;
	*wptr++ = '\r';
	*wptr++ = '\n';
	memcpy(wptr, ptr, telemlen);
	wptr += telemlen;
	p->len = wptr - p->buf;
	if (!(p2 = realloc(p, sizeof(struct packet) + p->len)))
		p = p2;
	*pkt = p;
	return ptr - buf;

  unfin:
	free(p);
	return 0;
  err:
	free(p);
	return -1;
}

static void udp_output_packet(struct confudpoutput *cuo, struct packet *pkt)
{
	struct sockaddr_in sin;

	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = cuo->addr;
	sin.sin_port = cuo->port;
	if (sendto(udpsendsock, pkt->buf, pkt->len, 0, (struct sockaddr *)&sin, sizeof(sin)) == -1)
		fprintf(stderr, "stprouter: UDP sendto error %s (%d)\n", strerror(errno), errno);
}

static void tcp_output_packet(struct tcpoutput *to, struct packet *pkt)
{
	unsigned int i;

	if (to->inprogress || !to->confout)
		return;
	i = (to->oqwr+1) % OUTQUEUESIZE;
	if (i != to->oqrd) {
		to->oqueue[to->oqwr] = pkt;
		to->oqwr = i;
		pkt->refcnt++;
	}
}

static inline void process_udpinput(struct udpinput *ui)
{
	struct confudpinput *cui, *conf;
	struct confudpoutput *cuo;
	struct tcpoutput *to;
	struct list_head *list;
	struct sockaddr_in sin;
	socklen_t addrlen = sizeof(sin);
	unsigned char buf[MAXPACKETSIZE];
	int conf_goodness;
	int ret;
	struct packet *pkt;

	ret = recvfrom(ui->fd, buf, sizeof(buf), 0, (struct sockaddr *)&sin, &addrlen);
	if (ret == -1) {
		fprintf(stderr, "stprouter: UDP recvfrom error %s (%d)\n", strerror(errno), errno);
		return;
	}
	if (ret <= 0)
		return;
	/* find configuration that fits the source address; drop if none found */
	conf = NULL;
	conf_goodness = -1;
	for (list = confudpinputlist.next; list != &confudpinputlist;) {
		cui = list_entry(list, struct confudpinput, list);
		list = list->next;
		if ((cui->addr ^ sin.sin_addr.s_addr) & cui->mask)
			continue;
		if (cui->maskbits > conf_goodness || !conf) {
			conf_goodness = cui->maskbits;
			conf = cui;
		}
	}
	if (!conf)
		return;
	if (parse_packet(&pkt, buf, ret) <= 0 || !pkt)
		return;
	if (pkt->hopcnt > conf->maxhops) {
		printf("UDP packet: hop count exceeded\n");
		free(pkt);
		return;
	}
	for (list = confudpoutputlist.next; list != &confudpoutputlist; list = list->next) {
		cuo = list_entry(list, struct confudpoutput, list);
		if (cuo->addr != sin.sin_addr.s_addr || cuo->port != sin.sin_port)
			udp_output_packet(cuo, pkt);
	}
	for (list = tcpoutputlist.next; list != &tcpoutputlist; list = list->next) {
		to = list_entry(list, struct tcpoutput, list);
		tcp_output_packet(to, pkt);
	}
	if (!pkt->refcnt)
		free(pkt);
}

static inline void process_tcplisten(struct tcpinput *ti)
{
	struct conftcpinput *cti, *confin;
	struct conftcpoutput *cto, *confout;
	struct tcpoutput *to;
	struct list_head *list;
	struct sockaddr_in sin;
	socklen_t slen = sizeof(sin);
	int fd, conf_goodness;

	fd = accept(ti->fd, (struct sockaddr *)&sin, &slen);
	if (fd == -1)
		return;
	if (sin.sin_family != AF_INET) {
		closesocket(fd);
		return;
	}
#if WIN32
	WSAAsyncSelect(fd, NULL, 0, 0);
	{
		/* Set Nonblock mode */
		u_long nonblock = 1;
		ioctlsocket(fd, FIONBIO, &nonblock);
	}
#else
	fcntl(fd, F_SETFL, fcntl(fd, F_GETFL, 0) | O_NONBLOCK);
#endif
	conf_goodness = -1;
	confin = NULL;
	for (list = conftcpinputlist.next; list != &conftcpinputlist; list = list->next) {
		cti = list_entry(list, struct conftcpinput, list);
		if ((cti->addr ^ sin.sin_addr.s_addr) & cti->mask)
			continue;
		if (cti->maskbits > conf_goodness || !confin) {
			confin = cti;
			conf_goodness = cti->maskbits;
		}
	}
	conf_goodness = -1;
	confout = NULL;
	for (list = conftcpoutputlist.next; list != &conftcpoutputlist; list = list->next) {
		cto = list_entry(list, struct conftcpoutput, list);
		if ((cto->addr ^ sin.sin_addr.s_addr) & cto->mask)
			continue;
		if (cto->port != -1 && cto->port != sin.sin_port)
			continue;
		if (cto->maskbits > conf_goodness || !confin) {
			confout = cto;
			conf_goodness = cto->maskbits;
		}
	}
	if (!confin && !confout) {
		closesocket(fd);
		printf("Connect rejected from %s:%u (no configuration)\n", inet_ntoa(sin.sin_addr), ntohs(sin.sin_port));
		return;
	}
	if ((confin && (confin->cntconn >= confin->maxconn)) ||
	    (confout && (confout->cntconn >= confout->maxconn))) {
		closesocket(fd);
		printf("Connect rejected from %s:%u (connection count exceeded %u %u %u %u)\n",
		       inet_ntoa(sin.sin_addr), ntohs(sin.sin_port), 
		       confin ? confin->cntconn : 0, confin ? confin->maxconn : 0,
		       confout ? confout->cntconn : 0, confout ? confout->maxconn : 0);
		return;
	}
	to = calloc(1, sizeof(struct tcpoutput));
	if (!to) {
		closesocket(fd);
		return;
	}
       	to->fd = fd;
	to->peeraddr = sin.sin_addr.s_addr;
	to->peerport = sin.sin_port;
	to->confin = confin;
	to->confout = confout;
	list_add(&to->list, &tcpoutputlist);
	if (confin)
		confin->cntconn++;
	if (confout)
		confout->cntconn++;
}

static void tcp_closeconn(struct tcpoutput *to)
{
	struct packet *pkt;

	printf("TCP: closing connection to %08x:%u\n", ntohl(to->peeraddr), ntohs(to->peerport));
	list_del(&to->list);
	if (to->confin)
		to->confin->cntconn--;
	if (to->confout)
		to->confout->cntconn--;
	closesocket(to->fd);
	while (to->oqrd != to->oqwr) {
		pkt = to->oqueue[to->oqrd];
		to->oqrd = (to->oqrd + 1) % OUTQUEUESIZE;
		pkt->refcnt--;
		if (!pkt->refcnt)
			free(pkt);
	}
	free(to);
}

static struct tcpoutput *tcp_connect(struct conftcpoutput *cto)
{
	struct conftcpinput *cti, *confin;
	struct tcpoutput *to, *to2;
	struct list_head *list;
	int fd, conf_goodness;
	struct sockaddr_in sin;
	unsigned int inprogress;

	if (cto->passive)
		return NULL;
	for (list = tcpoutputlist.next; list != &tcpoutputlist; list = list->next) {
		to2 = list_entry(list, struct tcpoutput, list);
		if (to2->confout == cto)
			return NULL;
	}
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = cto->addr;
	sin.sin_port = cto->port;
	conf_goodness = -1;
	confin = NULL;
	for (list = conftcpinputlist.next; list != &conftcpinputlist; list = list->next) {
		cti = list_entry(list, struct conftcpinput, list);
		if ((cti->addr ^ cto->addr) & cti->mask)
			continue;
		if (cti->maskbits > conf_goodness || !confin) {
			confin = cti;
			conf_goodness = cti->maskbits;
		}
	}
	if ((confin && (confin->cntconn >= confin->maxconn)) ||
	    (cto->cntconn >= cto->maxconn)) {
		printf("Connect to %s:%u exceeds count\n", inet_ntoa(sin.sin_addr), ntohs(sin.sin_port));
		return NULL;
	}
	fd = socket(PF_INET, SOCK_STREAM, 0);
	if (fd == -1)
		return NULL;
#if WIN32
	WSAAsyncSelect(fd, NULL, 0, 0);
	{
		/* Set Nonblock mode */
		u_long nonblock = 1;
		ioctlsocket(fd, FIONBIO, &nonblock);
	}
#else
	fcntl(fd, F_SETFL, fcntl(fd, F_GETFL, 0) | O_NONBLOCK);
#endif
	inprogress = 0;
	if (connect(fd, (struct sockaddr *)&sin, sizeof(sin))) {
		if (errno != EINPROGRESS) {
			fprintf(stderr, "stprouter: TCP connect error %s (%d)\n", strerror(errno), errno);
			closesocket(fd);
			return NULL;
		}
		inprogress = 1;
	}
	to = calloc(1, sizeof(struct tcpoutput));
	to->inprogress = inprogress;
	to->fd = fd;
	to->peeraddr = sin.sin_addr.s_addr;
	to->peerport = sin.sin_port;
	to->confout = cto;
	to->confin = confin;
	list_add(&to->list, &tcpoutputlist);
	cto->cntconn++;
	if (confin)
		confin->cntconn++;
	return to;
}

static inline void process_tcpconnected(struct tcpoutput *to)
{
	int ret;
	socklen_t len = sizeof(ret);

	if (!to->inprogress)
		return;
	if (getsockopt(to->fd, SOL_SOCKET, SO_ERROR, &ret, &len) == -1) {
		fprintf(stderr, "stprouter: TCP getsockopt error %s (%d)\n", strerror(errno), errno);
		tcp_closeconn(to);
		return;
	}
	if (ret) {
		fprintf(stderr, "stprouter: TCP connect error %s (%d)\n", strerror(ret), ret);
		tcp_closeconn(to);
		return;
	}
	to->inprogress = 0;
}

static inline void process_tcpio(struct tcpoutput *to, int revent, int wevent, int eevent)
{
	struct packet *pkt;
	struct confudpoutput *cuo;
	struct tcpoutput *to2;
	struct list_head *list;
	int ret;

	if (eevent)
		goto errout;
	if (revent && to->confin) {
		ret = recv(to->fd, &to->inbuf[to->inlen], MAXPACKETSIZE-to->inlen, 0);
		if ((ret == -1 && errno != EINTR && errno != EAGAIN) || ret == 0)
			goto errout;
		if (ret > 0) {
			to->inlen += ret;
			for (;;) {
				ret = parse_packet(&pkt, to->inbuf, to->inlen);
				if (pkt && pkt->hopcnt > to->confin->maxhops) {
					printf("TCP packet: hop count exceeded\n");
					free(pkt);
					pkt = NULL;
				}
				if (pkt) {
					for (list = confudpoutputlist.next; list != &confudpoutputlist; list = list->next) {
						cuo = list_entry(list, struct confudpoutput, list);
						udp_output_packet(cuo, pkt);
					}
					for (list = tcpoutputlist.next; list != &tcpoutputlist; list = list->next) {
						to2 = list_entry(list, struct tcpoutput, list);
						if (to->peeraddr != to2->peeraddr || to->peerport != to2->peerport)
							tcp_output_packet(to2, pkt);
					}
					if (!pkt->refcnt)
						free(pkt);
				}
				if (ret < 0)
					goto errout;
				if (!ret)
					break;
				to->inlen -= ret;
				memcpy(to->inbuf, &to->inbuf[ret], to->inlen);
			}
		}
	}
	if (wevent && to->oqrd != to->oqwr) {
		pkt = to->oqueue[to->oqrd];
		ret = send(to->fd, &pkt->buf[to->optr], pkt->len - to->optr, MSG_DONTWAIT);
		if ((ret == -1 && errno != EINTR && errno != EAGAIN) || ret == 0)
			goto errout;
		if (ret > 0) {
			to->optr += ret;
			if (to->optr >= pkt->len) {
				pkt->refcnt--;
				if (!pkt->refcnt)
					free(pkt);
				to->oqueue[to->oqrd] = NULL;
				to->oqrd = (to->oqrd + 1) % OUTQUEUESIZE;
				to->optr = 0;
			}
		}
	}
	return;

  errout:
	tcp_closeconn(to);
}



static void mainloop(void)
{
	struct list_head *list;
	struct conftcpoutput *cto;
	struct udpinput *ui;
	struct tcpinput *ti;
	struct tcpoutput *to;
	fd_set rmask, wmask, emask;
	struct timeval tv;
	int maxfd, ret;
	time_t curtime, nextconntime;

	time(&nextconntime);
       	for (;;) {
		time(&curtime);
		/* check for connects */
		if ((signed)(nextconntime - curtime) < 0) {
			nextconntime = curtime + 24 * 60 * 60;
			for (list = conftcpoutputlist.next; list != &conftcpoutputlist;) {
				cto = list_entry(list, struct conftcpoutput, list);
				list = list->next;
				if (cto->passive)
					continue;
				if ((signed)(cto->nextconntime - curtime) < 0) {
					if (cto->cntconn == 0)
						tcp_connect(cto);
					cto->nextconntime = curtime + cto->interval;
				}
				if ((signed)(cto->nextconntime - nextconntime) < 0)
					nextconntime = cto->nextconntime;
			}
		}
		/* do select loop */
		tv.tv_sec = nextconntime - curtime;
		tv.tv_usec = 0;
		if (tv.tv_sec <= 0 || tv.tv_sec > 2 * 24 * 60 * 60)
			tv.tv_sec = 1;
		maxfd = -1;
		FD_ZERO(&rmask);
		FD_ZERO(&wmask);
		FD_ZERO(&emask);
		for (list = udpinputlist.next; list != &udpinputlist; list = list->next) {
			ui = list_entry(list, struct udpinput, list);
			if (maxfd < ui->fd)
				maxfd = ui->fd;
			FD_SET(ui->fd, &rmask);
		}
		for (list = tcpinputlist.next; list != &tcpinputlist; list = list->next) {
			ti = list_entry(list, struct tcpinput, list);
			if (maxfd < ti->fd)
				maxfd = ti->fd;
			FD_SET(ti->fd, &rmask);
		}
		for (list = tcpoutputlist.next; list != &tcpoutputlist; list = list->next) {
			to = list_entry(list, struct tcpoutput, list);
			if (maxfd < to->fd)
				maxfd = to->fd;
			if (to->inprogress) {
				FD_SET(to->fd, &wmask);
			} else {
				if (to->inlen < MAXPACKETSIZE && to->confin)
					FD_SET(to->fd, &rmask);
				if (to->oqrd != to->oqwr)
					FD_SET(to->fd, &wmask);
				FD_SET(to->fd, &emask);
			}
		}
		/* walk tcp active output to find timeout */
		ret = select(maxfd+1, &rmask, &wmask, &emask, &tv);
		if (ret < 0) {
			fprintf(stderr, "stprouter: select error %s (%d)\n", strerror(errno), errno);
			exit(1);
		}
		for (list = udpinputlist.next; list != &udpinputlist;) {
			ui = list_entry(list, struct udpinput, list);
			list = list->next;
			if (FD_ISSET(ui->fd, &rmask))
				process_udpinput(ui);
		}
		for (list = tcpinputlist.next; list != &tcpinputlist;) {
			ti = list_entry(list, struct tcpinput, list);
			list = list->next;
			if (FD_ISSET(ti->fd, &rmask))
				process_tcplisten(ti);
		}
		for (list = tcpoutputlist.next; list != &tcpoutputlist;) {
			to = list_entry(list, struct tcpoutput, list);
			list = list->next;
			if (to->inprogress) {
				if (FD_ISSET(to->fd, &wmask))
					process_tcpconnected(to);
			} else {
				process_tcpio(to, FD_ISSET(to->fd, &rmask), FD_ISSET(to->fd, &wmask), FD_ISSET(to->fd, &emask));
			}
		}
	}
}

/* ---------------------------------------------------------------------- */

unsigned int stpconf_lineno = 1;

void yyerror(const char *msg)
{
    fprintf(stderr, "Error %s in line %u\n", msg, stpconf_lineno);
}

#if YYDEBUG != 0
extern int yydebug;
#endif

int main(int argc, char *argv[])
{
        int c, err = 0, pipe = 0;
        char *objname = "stdin";
#ifdef WIN32
        WORD wVersionRequested;
        WSADATA wsaData;
#endif
        
        while ((c = getopt(argc, argv, "p")) != EOF) {
                switch (c) {
                case 'p':
                        pipe = 1;
                        yyin = stdin;
                        break;

                default:
                        err++;
                        break;
                }
        }
        if (!pipe) {
            if (optind >= argc)
                err++;
            if (!err && !(yyin = fopen(argv[optind], "r")))
                err++;
            objname = argv[optind];
        }
        if (err) {
                fprintf(stderr, "usage: stprouter [-p] [<input file>]\n");
                exit(1);
        }
#ifdef WIN32
        wVersionRequested = MAKEWORD(1, 1);
        if (WSAStartup(wVersionRequested, &wsaData)) {
                fprintf(stderr, "stprouter: Win32 Socket startup failed\n");
                exit(1);
        }
        if (LOBYTE(wsaData.wVersion) != 1 || HIBYTE(wsaData.wVersion) != 1) {
                fprintf(stderr, "stprouter: Win32 Socket version 1.1 not supported\n");
                WSACleanup();
                exit(1);
        }
#endif /* WIN32 */
#if YYDEBUG != 0
        yydebug = 1;
#endif
        yyparse();
	start_listen();
	mainloop();
#ifdef WIN32
        WSACleanup();
#endif /* WIN32 */
        exit(0);
}
