// META: title=test WebNN API where operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-where
// Select the values from the input or the other tensor depending on the
// corresponding boolean values of the condition tensor.
//
// MLOperand where(MLOperand condition, MLOperand trueValue, MLOperand
//                 falseValue);


const getWherePrecisionTolerance = () => {
  return {metricType: 'ULP', value: 0};
};

const whereTests = [
  {
    'name': 'where float32 0D scalars',
    'graph': {
      'inputs': {
        'inputCondition':
            {'data': [247], 'descriptor': {shape: [], dataType: 'uint8'}},
        'inputTrueValue': {
          'data': [-22.83924102783203],
          'descriptor': {shape: [], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [-50.02232360839844],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [-22.83924102783203],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 1D constant tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [24], dataType: 'uint8'},
          'constant': true
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   66.7642593383789,   -1.5351932048797607,
            -21.683168411254883, 29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  65.5730209350586,   -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, 18.877336502075195,
            -23.302522659301758, 30.851404190063477, -75.41789245605469
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 1D tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [24], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   66.7642593383789,   -1.5351932048797607,
            -21.683168411254883, 29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  65.5730209350586,   -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, 18.877336502075195,
            -23.302522659301758, 30.851404190063477, -75.41789245605469
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 2D tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [4, 6], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   66.7642593383789,   -1.5351932048797607,
            -21.683168411254883, 29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  65.5730209350586,   -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, 18.877336502075195,
            -23.302522659301758, 30.851404190063477, -75.41789245605469
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 3D tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   66.7642593383789,   -1.5351932048797607,
            -21.683168411254883, 29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  65.5730209350586,   -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, 18.877336502075195,
            -23.302522659301758, 30.851404190063477, -75.41789245605469
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 4D tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   66.7642593383789,   -1.5351932048797607,
            -21.683168411254883, 29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  65.5730209350586,   -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, 18.877336502075195,
            -23.302522659301758, 30.851404190063477, -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 5D tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   66.7642593383789,   -1.5351932048797607,
            -21.683168411254883, 29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  65.5730209350586,   -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, 18.877336502075195,
            -23.302522659301758, 30.851404190063477, -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 4D tensors only broadcast condition 0D to 4D',
    'graph': {
      'inputs': {
        'inputCondition':
            {'data': [247], 'descriptor': {shape: [], dataType: 'uint8'}},
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 4D tensors only broadcast condition 1D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [253, 222, 0],
          'descriptor': {shape: [3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   89.99047088623047,  -39.82423782348633,
            -83.77035522460938,  29.59670639038086,  -46.46003341674805,
            -6.663239002227783,  -83.15333557128906, 35.14354705810547,
            -45.470706939697266, 50.38861083984375,  60.741416931152344,
            47.503013610839844,  -21.95842933654785, 34.315250396728516,
            -76.8262710571289,   67.0030746459961,   91.12223052978516,
            -29.081029891967773, -12.19106674194336, 18.877336502075195,
            -23.302522659301758, 30.851404190063477, -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 4D tensors only broadcast condition 2D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [103, 0],
          'descriptor': {shape: [2, 1], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -21.683168411254883, -67.3449478149414,  -46.46003341674805,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            11.960981369018555,  76.65287780761719,  60.741416931152344,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            7.7088470458984375,  -82.2353286743164,  91.12223052978516,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            62.31299591064453,   48.867103576660156, -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 4D tensors only broadcast condition 3D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [235, 0, 93, 213, 0, 117],
          'descriptor': {shape: [1, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   66.7642593383789,    -1.5351932048797607,
            -83.77035522460938,  -67.3449478149414,   -37.10446548461914,
            -6.663239002227783,  65.5730209350586,    -90.47624206542969,
            -45.470706939697266, 76.65287780761719,   46.510799407958984,
            47.503013610839844,  -30.397233963012695, 42.89878463745117,
            -76.8262710571289,   -82.2353286743164,   40.09717559814453,
            -29.081029891967773, 74.97810363769531,   -10.8577880859375,
            -23.302522659301758, 48.867103576660156,  -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 4D tensors only broadcast condition 4D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [0, 165, 0, 90],
          'descriptor': {shape: [1, 2, 2, 1], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -83.77035522460938,  29.59670639038086,   -37.10446548461914,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            -45.470706939697266, 50.38861083984375,   46.510799407958984,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            -76.8262710571289,   67.0030746459961,    40.09717559814453,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            -23.302522659301758, 30.851404190063477,  -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 4D tensors only broadcast trueValues 2D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            38.77214050292969, 18.20477294921875, -11.542049407958984,
            -82.74693298339844, -18.61202621459961, -75.77534484863281
          ],
          'descriptor': {shape: [2, 3], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            38.77214050292969,   66.7642593383789,   -11.542049407958984,
            -21.683168411254883, -18.61202621459961, -75.77534484863281,
            38.77214050292969,   65.5730209350586,   -11.542049407958984,
            -82.74693298339844,  -18.61202621459961, -75.77534484863281,
            38.77214050292969,   18.20477294921875,  -11.542049407958984,
            -82.74693298339844,  -18.61202621459961, -75.77534484863281,
            38.77214050292969,   18.20477294921875,  18.877336502075195,
            -82.74693298339844,  -18.61202621459961, -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 4D tensors only broadcast trueValues 4D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            -29.164173126220703, 32.200286865234375, -6.146966457366943,
            -24.822790145874023
          ],
          'descriptor': {shape: [2, 2, 1, 1], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            90.44255065917969,   66.7642593383789,    -39.82423782348633,
            -21.683168411254883, -67.3449478149414,   -46.46003341674805,
            -55.907073974609375, 65.5730209350586,    35.14354705810547,
            11.960981369018555,  76.65287780761719,   60.741416931152344,
            -93.93305206298828,  -30.397233963012695, 34.315250396728516,
            7.7088470458984375,  -82.2353286743164,   91.12223052978516,
            22.937334060668945,  74.97810363769531,   18.877336502075195,
            62.31299591064453,   48.867103576660156,  -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            -29.164173126220703, 66.7642593383789,    -29.164173126220703,
            -21.683168411254883, -29.164173126220703, -29.164173126220703,
            32.200286865234375,  65.5730209350586,    32.200286865234375,
            32.200286865234375,  32.200286865234375,  32.200286865234375,
            -6.146966457366943,  -6.146966457366943,  -6.146966457366943,
            -6.146966457366943,  -6.146966457366943,  -6.146966457366943,
            -24.822790145874023, -24.822790145874023, 18.877336502075195,
            -24.822790145874023, -24.822790145874023, -75.41789245605469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 4D tensors only broadcast falseValues 3D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            -86.59848022460938, -67.72773742675781, -30.731334686279297,
            -93.49223327636719
          ],
          'descriptor': {shape: [2, 2, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   -86.59848022460938,  -1.5351932048797607,
            -67.72773742675781,  29.59670639038086,   -37.10446548461914,
            -6.663239002227783,  -30.731334686279297, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,   46.510799407958984,
            47.503013610839844,  -21.95842933654785,  42.89878463745117,
            -76.8262710571289,   67.0030746459961,    40.09717559814453,
            -29.081029891967773, -12.19106674194336,  -30.731334686279297,
            -23.302522659301758, 30.851404190063477,  -93.49223327636719
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 4D tensors only broadcast falseValues 4D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.27151489257812,   89.99047088623047,  -1.5351932048797607,
            -83.77035522460938,  29.59670639038086,  -37.10446548461914,
            -6.663239002227783,  -83.15333557128906, -90.47624206542969,
            -45.470706939697266, 50.38861083984375,  46.510799407958984,
            47.503013610839844,  -21.95842933654785, 42.89878463745117,
            -76.8262710571289,   67.0030746459961,   40.09717559814453,
            -29.081029891967773, -12.19106674194336, -10.8577880859375,
            -23.302522659301758, 30.851404190063477, -58.82704544067383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            17.777414321899414, -1.9890096187591553, -57.28207015991211,
            -80.01390838623047, 66.87323760986328, -31.328258514404297
          ],
          'descriptor': {shape: [1, 2, 1, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.27151489257812,   -1.9890096187591553, -1.5351932048797607,
            17.777414321899414,  29.59670639038086,   -37.10446548461914,
            -6.663239002227783,  66.87323760986328,   -90.47624206542969,
            -45.470706939697266, 50.38861083984375,   46.510799407958984,
            47.503013610839844,  -21.95842933654785,  42.89878463745117,
            -76.8262710571289,   67.0030746459961,    40.09717559814453,
            -29.081029891967773, -12.19106674194336,  -31.328258514404297,
            -23.302522659301758, 30.851404190063477,  -31.328258514404297
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'where float32 4D tensors all broadcast 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [58, 217],
          'descriptor': {shape: [2, 1, 1, 1], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            38.77214050292969, 18.20477294921875, -11.542049407958984,
            -82.74693298339844, -18.61202621459961, -75.77534484863281
          ],
          'descriptor': {shape: [2, 3], dataType: 'float32'}
        },
        'inputFalseValue': {
          'data': [
            -86.59848022460938, -67.72773742675781, -30.731334686279297,
            -93.49223327636719
          ],
          'descriptor': {shape: [2, 2, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            38.77214050292969,  18.20477294921875,  -11.542049407958984,
            -82.74693298339844, -18.61202621459961, -75.77534484863281,
            38.77214050292969,  18.20477294921875,  -11.542049407958984,
            -82.74693298339844, -18.61202621459961, -75.77534484863281,
            38.77214050292969,  18.20477294921875,  -11.542049407958984,
            -82.74693298339844, -18.61202621459961, -75.77534484863281,
            38.77214050292969,  18.20477294921875,  -11.542049407958984,
            -82.74693298339844, -18.61202621459961, -75.77534484863281
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'where float16 0D scalars',
    'graph': {
      'inputs': {
        'inputCondition':
            {'data': [247], 'descriptor': {shape: [], dataType: 'uint8'}},
        'inputTrueValue': {
          'data': [-22.84375],
          'descriptor': {shape: [], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [-50.03125],
          'descriptor': {shape: [], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [-22.84375],
          'descriptor': {shape: [], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 1D constant tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [24], dataType: 'uint8'},
          'constant': true
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,     66.75,      -1.53515625, -21.6875,   29.59375,
            -37.09375, -6.6640625, 65.5625,     -90.5,      -45.46875,
            50.375,    46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,  67,         40.09375,    -29.078125, -12.1875,
            18.875,    -23.296875, 30.84375,    -75.4375
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 1D tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [24], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,     66.75,      -1.53515625, -21.6875,   29.59375,
            -37.09375, -6.6640625, 65.5625,     -90.5,      -45.46875,
            50.375,    46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,  67,         40.09375,    -29.078125, -12.1875,
            18.875,    -23.296875, 30.84375,    -75.4375
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 2D tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [4, 6], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,     66.75,      -1.53515625, -21.6875,   29.59375,
            -37.09375, -6.6640625, 65.5625,     -90.5,      -45.46875,
            50.375,    46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,  67,         40.09375,    -29.078125, -12.1875,
            18.875,    -23.296875, 30.84375,    -75.4375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 3D tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,     66.75,      -1.53515625, -21.6875,   29.59375,
            -37.09375, -6.6640625, 65.5625,     -90.5,      -45.46875,
            50.375,    46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,  67,         40.09375,    -29.078125, -12.1875,
            18.875,    -23.296875, 30.84375,    -75.4375
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 4D tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,     66.75,      -1.53515625, -21.6875,   29.59375,
            -37.09375, -6.6640625, 65.5625,     -90.5,      -45.46875,
            50.375,    46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,  67,         40.09375,    -29.078125, -12.1875,
            18.875,    -23.296875, 30.84375,    -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 5D tensors',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,     66.75,      -1.53515625, -21.6875,   29.59375,
            -37.09375, -6.6640625, 65.5625,     -90.5,      -45.46875,
            50.375,    46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,  67,         40.09375,    -29.078125, -12.1875,
            18.875,    -23.296875, 30.84375,    -75.4375
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 4D tensors only broadcast condition 0D to 4D',
    'graph': {
      'inputs': {
        'inputCondition':
            {'data': [247], 'descriptor': {shape: [], dataType: 'uint8'}},
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 4D tensors only broadcast condition 1D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [253, 222, 0],
          'descriptor': {shape: [3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,      90,         -39.8125, -83.75,     29.59375, -46.46875,
            -6.6640625, -83.125,    35.15625, -45.46875,  50.375,   60.75,
            47.5,       -21.953125, 34.3125,  -76.8125,   67,       91.125,
            -29.078125, -12.1875,   18.875,   -23.296875, 30.84375, -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 4D tensors only broadcast condition 2D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [103, 0],
          'descriptor': {shape: [2, 1], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,      90,         -1.53515625, -21.6875,   -67.375, -46.46875,
            -6.6640625, -83.125,    -90.5,       11.9609375, 76.625,  60.75,
            47.5,       -21.953125, 42.90625,    7.70703125, -82.25,  91.125,
            -29.078125, -12.1875,   -10.859375,  62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 4D tensors only broadcast condition 3D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [235, 0, 93, 213, 0, 117],
          'descriptor': {shape: [1, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,      66.75,      -1.53515625, -83.75,     -67.375, -37.09375,
            -6.6640625, 65.5625,    -90.5,       -45.46875,  76.625,  46.5,
            47.5,       -30.390625, 42.90625,    -76.8125,   -82.25,  40.09375,
            -29.078125, 75,         -10.859375,  -23.296875, 48.875,  -58.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 4D tensors only broadcast condition 4D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [0, 165, 0, 90],
          'descriptor': {shape: [1, 2, 2, 1], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            90.4375,   66.75,      -39.8125, -83.75,     29.59375, -37.09375,
            -55.90625, 65.5625,    35.15625, -45.46875,  50.375,   46.5,
            -93.9375,  -30.390625, 34.3125,  -76.8125,   67,       40.09375,
            22.9375,   75,         18.875,   -23.296875, 30.84375, -58.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 4D tensors only broadcast trueValues 2D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data':
              [38.78125, 18.203125, -11.5390625, -82.75, -18.609375, -75.75],
          'descriptor': {shape: [2, 3], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            38.78125, 66.75,     -11.5390625, -21.6875, -18.609375, -75.75,
            38.78125, 65.5625,   -11.5390625, -82.75,   -18.609375, -75.75,
            38.78125, 18.203125, -11.5390625, -82.75,   -18.609375, -75.75,
            38.78125, 18.203125, 18.875,      -82.75,   -18.609375, -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 4D tensors only broadcast trueValues 4D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [-29.171875, 32.1875, -6.1484375, -24.828125],
          'descriptor': {shape: [2, 2, 1, 1], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [
            90.4375,   66.75,      -39.8125, -21.6875,   -67.375, -46.46875,
            -55.90625, 65.5625,    35.15625, 11.9609375, 76.625,  60.75,
            -93.9375,  -30.390625, 34.3125,  7.70703125, -82.25,  91.125,
            22.9375,   75,         18.875,   62.3125,    48.875,  -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            -29.171875, 66.75,      -29.171875, -21.6875,   -29.171875,
            -29.171875, 32.1875,    65.5625,    32.1875,    32.1875,
            32.1875,    32.1875,    -6.1484375, -6.1484375, -6.1484375,
            -6.1484375, -6.1484375, -6.1484375, -24.828125, -24.828125,
            18.875,     -24.828125, -24.828125, -75.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 4D tensors only broadcast falseValues 3D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [-86.625, -67.75, -30.734375, -93.5],
          'descriptor': {shape: [2, 2, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,      -86.625,    -1.53515625, -67.75,     29.59375,
            -37.09375,  -6.6640625, -30.734375,  -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -30.734375, -23.296875, 30.84375,    -93.5
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 4D tensors only broadcast falseValues 4D to 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [
            241, 0,   85,  0,   227, 51, 202, 0,  104, 227, 129, 129,
            175, 134, 130, 140, 103, 46, 158, 17, 0,   41,  94,  0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data': [
            70.25,      90,         -1.53515625, -83.75,     29.59375,
            -37.09375,  -6.6640625, -83.125,     -90.5,      -45.46875,
            50.375,     46.5,       47.5,        -21.953125, 42.90625,
            -76.8125,   67,         40.09375,    -29.078125, -12.1875,
            -10.859375, -23.296875, 30.84375,    -58.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [17.78125, -1.9892578125, -57.28125, -80, 66.875, -31.328125],
          'descriptor': {shape: [1, 2, 1, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            70.25,      -1.9892578125, -1.53515625, 17.78125,
            29.59375,   -37.09375,     -6.6640625,  66.875,
            -90.5,      -45.46875,     50.375,      46.5,
            47.5,       -21.953125,    42.90625,    -76.8125,
            67,         40.09375,      -29.078125,  -12.1875,
            -31.328125, -23.296875,    30.84375,    -31.328125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'where float16 4D tensors all broadcast 4D',
    'graph': {
      'inputs': {
        'inputCondition': {
          'data': [58, 217],
          'descriptor': {shape: [2, 1, 1, 1], dataType: 'uint8'}
        },
        'inputTrueValue': {
          'data':
              [38.78125, 18.203125, -11.5390625, -82.75, -18.609375, -75.75],
          'descriptor': {shape: [2, 3], dataType: 'float16'}
        },
        'inputFalseValue': {
          'data': [-86.625, -67.75, -30.734375, -93.5],
          'descriptor': {shape: [2, 2, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'where',
        'arguments': [
          {'condition': 'inputCondition'}, {'trueValue': 'inputTrueValue'},
          {'falseValue': 'inputFalseValue'}
        ],
        'outputs': 'whereOutput'
      }],
      'expectedOutputs': {
        'whereOutput': {
          'data': [
            38.78125, 18.203125, -11.5390625, -82.75, -18.609375, -75.75,
            38.78125, 18.203125, -11.5390625, -82.75, -18.609375, -75.75,
            38.78125, 18.203125, -11.5390625, -82.75, -18.609375, -75.75,
            38.78125, 18.203125, -11.5390625, -82.75, -18.609375, -75.75
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  }
];

if (navigator.ml) {
  whereTests.forEach((test) => {
    webnn_conformance_test(
        buildAndExecuteGraph, getWherePrecisionTolerance, test);
  });
} else {
  test(() => assert_implements(navigator.ml, 'missing navigator.ml'));
}
